/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.security;

import com.sap.tc.logging.Location;
import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.ResourceException;
import com.sapportals.wcm.repository.manager.IRepositoryManager;

import com.sapportals.wcm.repository.manager.IResourceEventReceiver;
import com.sapportals.wcm.repository.manager.ResourceEvent;
import com.sapportals.wcm.util.acl.*;
import com.sapportals.wcm.util.events.IEvent;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.uri.RID;
import java.util.ArrayList;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

/**
 * A singleton shared by all <code>ResourceAclManager</code> instances, which
 * does the event handling for all of them. This is necessary because the
 * processing of some events may involves more than one ResourceAclManager
 * instance. Instead of registering with a RepositoryManager itself, a
 * ResourceAclManager uses the unique ResourceAclEventHandler instance, which
 * keeps track of all RepositoryManager-ResourceAclManager relations.
 *
 * @author manfred.baedke@greenbytes.de
 */
public class ResourceAclEventHandler implements IResourceEventReceiver {

  private static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.repository.security.ResourceAclEventHandler.class);

  private static ResourceAclEventHandler singleton = null;

  private final HashMap mgrMap = new HashMap();

  private ResourceAclEventHandler() { }

  public static ResourceAclEventHandler getInstance() {
    if (singleton == null) {
      ResourceAclEventHandler.singleton = new ResourceAclEventHandler();
    }
    return ResourceAclEventHandler.singleton;
  }

  public void received(IEvent event) {

    if (event == null) {
      return;
    }
    if (!(event instanceof ResourceEvent)) {
      return;
    }

    ResourceEvent resourceEvent = (ResourceEvent)event;
    IResource src = resourceEvent.getResource();
    if (src == null) {
      return;
    }

    List list = this.getEntry(src.getRepositoryManager());
    if (list == null) {
      return;
    }
    Iterator iter = list.iterator();
    while (iter.hasNext()) {
      ResourceAclManager srcAclMgr = (ResourceAclManager)iter.next();

      IResource dest = null;
      ResourceAclManager destAclMgr = null;

      try {
        switch (resourceEvent.getType()) {

          case ResourceEvent.DELETE:
            this.delete(src, srcAclMgr);
            break;
          case ResourceEvent.MOVE:
            dest = (IResource)resourceEvent.getParameter();
            /*
             * if (dest != null) {
             * destAclMgr = (ResourceAclManager) this.getEntry(dest.getRepositoryManager());
             * if (srcAclMgr.equals(destAclMgr)) {
             * destAclMgr = null;
             * }
             * }
             */
            this.move(src, srcAclMgr, dest, null);
            break;
          case ResourceEvent.RENAME:
            this.rename(src, srcAclMgr, RID.getRID((String)resourceEvent.getParameter()));
            break;
        }
      }
      catch (Exception e) {
        log.errorT("received(113)", "error while processing framework event" + LoggingFormatter.extractCallstack(e));
        throw new RuntimeException(e.getClass().getName() + ": " + e.getMessage());
      }
    }

  }

  public void register(ResourceAclManager aclMgr, IRepositoryManager repMgr)
    throws WcmException {

    if (aclMgr == null || repMgr == null) {
      throw new java.lang.IllegalArgumentException();
    }

    this.addEntry(aclMgr, repMgr);

    repMgr.getEventBroker().register(this, new ResourceEvent(ResourceEvent.DELETE, null));
    repMgr.getEventBroker().register(this, new ResourceEvent(ResourceEvent.MOVE, null));
    repMgr.getEventBroker().register(this, new ResourceEvent(ResourceEvent.RENAME, null));
  }

  private void addEntry(ResourceAclManager aclMgr, IRepositoryManager repMgr) {
    List list = (List)this.mgrMap.get(repMgr);
    if (list == null) {
      list = new ArrayList();
      this.mgrMap.put(repMgr, list);
    }
    list.add(aclMgr);
  }

  private List getEntry(IRepositoryManager repMgr) {
    return (List)this.mgrMap.get(repMgr);
  }

  private void delete(IResource resource, ResourceAclManager aclMgr)
    throws AclPersistenceException, NotAuthorizedException, InvalidClassException, ResourceException {

    if (resource == null || aclMgr == null) {
      return;
    }
    aclMgr.deleteResource(resource);
  }

  private void move(IResource src, ResourceAclManager srcAclMgr, IResource dest, ResourceAclManager destAclMgr)
    throws AclPersistenceException, NotAuthorizedException, AclExistsException, InvalidClassException,
    AlreadyAssignedToAclException, PermissionNotSupportedException, LastOwnerException, ResourceException {

    if (src == null || srcAclMgr == null || dest == null) {
      return;
    }
    srcAclMgr.moveResource(src, dest, destAclMgr);
  }

  private void rename(IResource src, ResourceAclManager srcAclMgr, RID oldRID)
    throws AclPersistenceException, NotAuthorizedException, AclExistsException, ResourceException {

    if (src == null || srcAclMgr == null || oldRID == null) {
      return;
    }
    srcAclMgr.renameResource(oldRID.getPath(), src);
  }
}
