/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.search;

import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import com.sap.tc.logging.Location;
import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.repository.enum.PropertyType;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.regex.re.RE;
import com.sapportals.wcm.util.regex.re.RESyntaxException;

/**
 * Query expressions supported by the Simple Property Search Manager. <p>
 *
 * Copyright (c) SAP AG 2001-2004
 *
 * @author julian.reschke@greenbytes.de
 * @version $Id: SimpleQueryExpression.java,v 1.4 2004/07/02 09:34:44 jre Exp $
 */

class SimpleQueryExpression implements IQueryExpression {
  private final static Location log = Location.getLocation(SimpleQueryExpression.class);

  /**
   * Append a character escaped suitably for regexp to an expression
   *
   * @param s the regexp expression
   * @param c the character to escape
   */
  private static void appendEscaped(StringBuffer s, char c) {

    switch (c) {

      case '*':
      case '\\':
      case '[':
      case ']':
      case '^':
      case '.':
      case '$':
      case '+':
      case '?':
      case '|':
        s.append("\\" + c);
        break;
      default:
        s.append(c);
        break;
    }
  }

  /**
   * Compile the "like" expression into a regexp
   *
   * @param likeExpression match expression as defined in <a
   *      href="http://www.greenbytes.de/tech/webdav/draft-davis-dasl-protocol-00.htm">
   *      DASL</a>
   * @return regexp expression or <code>null</code> when malformed
   */
  private static RE regexpFromLike(String likeExpression) {

    StringBuffer expr = new StringBuffer();
    expr.append("^");

    for (int i = 0; i < likeExpression.length(); i++) {

      char c = likeExpression.charAt(i);

      switch (c) {
        case '%':
          expr.append(".*");
          break;
        case '_':
          expr.append(".");
          break;
        case '\\':
          try {
            c = likeExpression.charAt(++i);
            if (c != '\\' && c != '%' && c != '_') {
              throw new IllegalArgumentException("malformed 'like' expression ('\\' must escape '\\', '%' or '_')");
            }
          }
          catch (ArrayIndexOutOfBoundsException ex) {
            log.debugT("regexpFromLike(102)", "malformed 'like' expression: '" + likeExpression + "' (will be treated as undefined)");
            throw new IllegalArgumentException("malformed like expression");
          }
        // fall through

        default:
          appendEscaped(expr, c);
          break;
      }
    }

    expr.append("$");

    try {
      return new RE(expr.toString());
    }
    catch (RESyntaxException ex) {
      log.debugT("regexpFromLike(119)", "malformed 'like' expression: '" + likeExpression + "' (will be treated as undefined)");
      throw new IllegalArgumentException("could not translate like expression");
    }
  }

  /**
   * Parse a string in the ISO8601 subset format as specified in <a
   * href="http://www.greenbytes.de/tech/webdav/rfc2518.html#rfc.section.B">
   * RFC2518</a> 
   */
  private static Date parseISO8601(String text)
    throws java.text.ParseException {
    // TODO: this code is inefficient because it keeps creating timezone and date format objects
    
    final String FORMAT_ISO8601 = "yyyy-MM-dd'T'HH:mm:ss'Z'";
    final String FORMAT_ISO8601_MS = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'";
    final java.util.TimeZone gmt = java.util.TimeZone.getTimeZone("GMT");

    if (text.length() > 20) {
      java.text.SimpleDateFormat sdfISO8601ms;
      sdfISO8601ms = new java.text.SimpleDateFormat(FORMAT_ISO8601_MS, java.util.Locale.ENGLISH);
      sdfISO8601ms.setTimeZone(gmt);
      return sdfISO8601ms.parse(text);
    }
    else {
      java.text.SimpleDateFormat sdfISO8601;
      sdfISO8601 = new java.text.SimpleDateFormat(FORMAT_ISO8601, java.util.Locale.ENGLISH);
      sdfISO8601.setTimeZone(gmt);
      return sdfISO8601.parse(text);
    }
  }


  //---------------------------------------------------------------------------

  public static class And extends SimpleQueryExpression {

    private List members;

    public And(IQueryExpression exp1, IQueryExpression exp2) {

      this.members = new ArrayList();

      if (exp1 instanceof And) {
        this.members.addAll(((And)exp1).members);
      }
      else {
        this.members.add(exp1);
      }

      if (exp2 instanceof And) {
        this.members.addAll(((And)exp2).members);
      }
      else {
        this.members.add(exp2);
      }
    }

    public String toString() {

      StringBuffer sb = new StringBuffer();

      for (Iterator it = this.members.iterator(); it.hasNext(); ) {
        sb.append("(").append(it.next().toString()).append(")");
        if (it.hasNext()) {
          sb.append(" and ");
        }
      }

      return sb.toString();
    }

    public Boolean match(IResource resource, IPropertyMap props) {

      Boolean result = Boolean.TRUE;

      for (Iterator it = this.members.iterator(); it.hasNext(); ) {
        Boolean res = ((SimpleQueryExpression)it.next()).match(resource, props);

        if (Boolean.FALSE.equals(res)) {
          return res;
        }
        else if (null == res) {
          result = null;
        }
      }

      return result;
    }
  }

  public static class Not extends SimpleQueryExpression {

    private IQueryExpression member;

    public Not(IQueryExpression exp1) {
      this.member = exp1;
    }

    public String toString() {
      StringBuffer sb = new StringBuffer();
      sb.append("not (").append(this.member.toString()).append(")");
      return sb.toString();
    }

    public Boolean match(IResource resource, IPropertyMap props) {
      Boolean res = ((SimpleQueryExpression)this.member).match(resource, props);
      if (res == null) {
        return res;
      }
      else {
        return res.booleanValue() == true ? Boolean.FALSE : Boolean.TRUE;
      }
    }
  }

  public static class Or extends SimpleQueryExpression {

    private List members;

    public Or(IQueryExpression exp1, IQueryExpression exp2) {
      this.members = new ArrayList();

      if (exp1 instanceof Or) {
        this.members.addAll(((Or)exp1).members);
      }
      else {
        this.members.add(exp1);
      }

      if (exp2 instanceof Or) {
        this.members.addAll(((Or)exp2).members);
      }
      else {
        this.members.add(exp2);
      }
    }

    public String toString() {

      StringBuffer sb = new StringBuffer();
      for (Iterator it = this.members.iterator(); it.hasNext(); ) {
        sb.append("(").append(it.next().toString()).append(")");
        if (it.hasNext()) {
          sb.append(" or ");
        }
      }

      return sb.toString();
    }

    public Boolean match(IResource resource, IPropertyMap props) {

      Boolean result = Boolean.FALSE;
      for (Iterator it = this.members.iterator(); it.hasNext(); ) {
        Boolean res = ((SimpleQueryExpression)it.next()).match(resource, props);

        if (Boolean.TRUE.equals(res)) {
          return res;
        }
        else if (null == res) {
          result = null;
        }
      }

      return result;
    }
  }

  public abstract static class Comparison extends SimpleQueryExpression {

    IPropertyName prop;
    String vString;
    Boolean vBoolean = null;
    Long vLong = null;
    Integer vInteger = null;
    Date vDate = null;

    boolean caseSensitive;
    boolean mv;

    private Comparison(IPropertyName prop, Object value, boolean caseSensitive, boolean mv) {

      this.prop = prop;
      this.vString = value.toString();

      if (value instanceof Boolean) {
        this.vBoolean = (Boolean)value;
      }
      else if ("true".equals(this.vString) || "1".equals(this.vString)) {
        this.vBoolean = Boolean.TRUE;
      }
      else if ("false".equals(this.vString) || "0".equals(this.vString)) {
        this.vBoolean = Boolean.FALSE;
      }
      else {
        this.vBoolean = null;
      }
      
      try {
        this.vLong = (value instanceof Long)
           ? (Long)value
           : new Long(Long.parseLong(this.vString));
      }
      catch (Exception ex) {
        // $JL-EXC
      }

      try {
        this.vInteger = (value instanceof Integer)
           ? (Integer)value
           : new Integer(Integer.parseInt(this.vString));
      }
      catch (Exception ex) {
        // $JL-EXC
      }

      try {
        this.vDate = (value instanceof Date)
           ? (Date)value
           : super.parseISO8601(this.vString);
      }
      catch (Exception ex) {
        // $JL-EXC
      }

      this.caseSensitive = caseSensitive;
      this.mv = mv;
    }

    public Boolean match(IResource resource, IPropertyMap props) {
      IProperty prop = props.get(this.prop);
      return cmp(prop);
    }

    public String makeString(String op) {
      return this.prop +
        (this.mv ? "[]" : "") +
        " " + op + (this.caseSensitive ? "" : "~") + " " +
        "'" + this.vString + "'";
    }

    public abstract Boolean cmp(long x, long y);
//    {
//      log.fatal("unimplemented property search comparison");
//      return null;
//    }

    public abstract Boolean cmp(String x, String y);
//    {
//      log.fatal("unimplemented property search comparison");
//      return null;
//    }

    private Boolean mvcmp(IProperty prop) {

      if (!prop.isMultivalued()) {
        return null;
      }

      PropertyType typ = prop.getPropertyDef().getType();
      List values;

      try {
        values = prop.getValues();
      }
      catch (ResourceException ex) {
        log.debugT("mvcmp(390)", "getting values - " + LoggingFormatter.extractCallstack(ex));
        return null;
      }

      if (typ.equals(PropertyType.BOOLEAN) && this.vBoolean != null) {
        int v2 = this.vBoolean.equals(Boolean.FALSE) ? 0 : 1;
        for (Iterator it = values.listIterator(); it.hasNext(); ) {
          Object o = it.next();
          if (o instanceof Boolean) {
            int v1 = ((Boolean)o).booleanValue() == false ? 0 : 1;
            if (Boolean.TRUE.equals(cmp(v1, v2))) {
              return Boolean.TRUE;
            }
          }
        }
        return Boolean.FALSE;
      }
      else if (typ.equals(PropertyType.DATE) && this.vDate != null) {
        long v2 = this.vDate.getTime();
        for (Iterator it = values.listIterator(); it.hasNext(); ) {
          Object o = it.next();
          if (o instanceof Date) {
            long v1 = ((Date)o).getTime();
            if (Boolean.TRUE.equals(cmp(v1, v2))) {
              return Boolean.TRUE;
            }
          }
        }
        return Boolean.FALSE;
      }
      else if (typ.equals(PropertyType.INTEGER) && this.vInteger != null) {
        long v2 = this.vInteger.intValue();
        for (Iterator it = values.listIterator(); it.hasNext(); ) {
          Object o = it.next();
          if (o instanceof Integer) {
            long v1 = ((Integer)o).intValue();
            if (Boolean.TRUE.equals(cmp(v1, v2))) {
              return Boolean.TRUE;
            }
          }
        }
        return Boolean.FALSE;
      }
      else if (typ.equals(PropertyType.LONG) && this.vLong != null) {
        long v2 = this.vLong.longValue();
        for (Iterator it = values.listIterator(); it.hasNext(); ) {
          Object o = it.next();
          if (o instanceof Long) {
            long v1 = ((Long)o).longValue();
            if (Boolean.TRUE.equals(cmp(v1, v2))) {
              return Boolean.TRUE;
            }
          }
        }
        return Boolean.FALSE;
      }
      else {
        String v2 = this.vString;
        for (Iterator it = values.listIterator(); it.hasNext(); ) {
          String v1 = it.next().toString();
          if (Boolean.TRUE.equals(cmp(v1, v2))) {
            return Boolean.TRUE;
          }
        }
        return Boolean.FALSE;
      }
    }


    private Boolean cmp(IProperty prop) {

      if (prop == null) {
        return null;
      }
      else if (this.mv) {
        return mvcmp(prop);
      }
      else if (prop.isMultivalued()) {
        return null;
      }

      PropertyType typ = prop.getPropertyDef().getType();

      if (typ.equals(PropertyType.BOOLEAN) && this.vBoolean != null) {
        int v1 = prop.getBooleanValue() == false ? 0 : 1;
        int v2 = this.vBoolean.equals(Boolean.FALSE) ? 0 : 1;
        return cmp(v1, v2);
      }
      else if (typ.equals(PropertyType.DATE) && this.vDate != null) {
        return cmp(prop.getDateValue().getTime(), this.vDate.getTime());
      }
      else if (typ.equals(PropertyType.INTEGER) && this.vInteger != null) {
        return cmp(prop.getIntValue(), this.vInteger.intValue());
      }
      else if (typ.equals(PropertyType.LONG) && this.vLong != null) {
        return cmp(prop.getLongIntValue(), this.vLong.longValue());
      }
      else if (typ.equals(PropertyType.XML)) {
        // XML typed properties never match in value comparisons
        return null;
      }
      else {
        return cmp(prop.getValueAsString(), this.vString);
      }
    }

  }

  public static class Eq extends Comparison {

    public Eq(IPropertyName prop, Object value, boolean caseSensitive, boolean mv) {
      super(prop, value, caseSensitive, mv);
    }

    public String toString() {
      return super.makeString("=");
    }

    public Boolean cmp(long x, long y) {
      return x == y ? Boolean.TRUE : Boolean.FALSE;
    }

    public Boolean cmp(String x, String y) {
      if (this.caseSensitive) {
        return x.equals(y) ? Boolean.TRUE : Boolean.FALSE;
      }
      else {
        return x.equalsIgnoreCase(y) ? Boolean.TRUE : Boolean.FALSE;
      }
    }
  }

  public static class Greater extends Comparison {

    public Greater(IPropertyName prop, Object value, boolean caseSensitive, boolean mv) {
      super(prop, value, caseSensitive, mv);
    }

    public String toString() {
      return super.makeString(">");
    }

    public Boolean cmp(long x, long y) {
      return x > y ? Boolean.TRUE : Boolean.FALSE;
    }

    public Boolean cmp(String x, String y) {
      if (this.caseSensitive) {
        return x.compareTo(y) > 0 ? Boolean.TRUE : Boolean.FALSE;
      }
      else {
        return x.compareToIgnoreCase(y) > 0 ? Boolean.TRUE : Boolean.FALSE;
      }
    }
  }

  public static class GreaterEq extends Comparison {

    public GreaterEq(IPropertyName prop, Object value, boolean caseSensitive, boolean mv) {
      super(prop, value, caseSensitive, mv);
    }

    public String toString() {
      return super.makeString(">=");
    }

    public Boolean cmp(long x, long y) {
      return x >= y ? Boolean.TRUE : Boolean.FALSE;
    }

    public Boolean cmp(String x, String y) {
      if (this.caseSensitive) {
        return x.compareTo(y) >= 0 ? Boolean.TRUE : Boolean.FALSE;
      }
      else {
        return x.compareToIgnoreCase(y) >= 0 ? Boolean.TRUE : Boolean.FALSE;
      }
    }
  }

  public static class IsCollection extends SimpleQueryExpression {

    public IsCollection() {
      //
    }

    public String toString() {
      return "iscollection()";
    }

    public Boolean match(IResource resource, IPropertyMap props) {
      return resource.isCollection() ? Boolean.TRUE : Boolean.FALSE;
    }
  }

  public static class IsDefined extends SimpleQueryExpression {

    IPropertyName prop;

    public IsDefined(IPropertyName prop) {
      this.prop = prop;
    }

    public String toString() {
      return "isdefined(" + this.prop + ")";
    }

    public Boolean match(IResource resource, IPropertyMap props) {
      return (null != props.get(this.prop)) ? Boolean.TRUE : Boolean.FALSE;
    }
  }

  public static class Less extends Comparison {

    public Less(IPropertyName prop, Object value, boolean caseSensitive, boolean mv) {
      super(prop, value, caseSensitive, mv);
    }

    public String toString() {
      return super.makeString("<");
    }

    public Boolean cmp(long x, long y) {
      return x < y ? Boolean.TRUE : Boolean.FALSE;
    }

    public Boolean cmp(String x, String y) {
      if (this.caseSensitive) {
        return x.compareTo(y) < 0 ? Boolean.TRUE : Boolean.FALSE;
      }
      else {
        return x.compareToIgnoreCase(y) < 0 ? Boolean.TRUE : Boolean.FALSE;
      }
    }
  }

  public static class LessEq extends Comparison {

    public LessEq(IPropertyName prop, Object value, boolean caseSensitive, boolean mv) {
      super(prop, value, caseSensitive, mv);
    }

    public String toString() {
      return super.makeString("<=");
    }

    public Boolean cmp(long x, long y) {
      return x <= y ? Boolean.TRUE : Boolean.FALSE;
    }

    public Boolean cmp(String x, String y) {
      if (this.caseSensitive) {
        return x.compareTo(y) <= 0 ? Boolean.TRUE : Boolean.FALSE;
      }
      else {
        return x.compareToIgnoreCase(y) <= 0 ? Boolean.TRUE : Boolean.FALSE;
      }
    }
  }

  public static class Like extends SimpleQueryExpression {

    IPropertyName prop;
    String pattern;
    boolean caseSensitive;
    boolean mv;
    private RE re = null;

    public Like(IPropertyName prop, String pattern, boolean caseSensitive, boolean mv) {

      this.prop = prop;
      this.pattern = pattern;
      this.caseSensitive = caseSensitive;
      this.mv = mv;

      try {
        this.re = super.regexpFromLike(caseSensitive ? pattern : pattern.toUpperCase());
      }
      catch (Exception ex) {
        // match test will evaluate to "undefined"
      }
    }

    public String toString() {
      return this.prop +
        (this.mv ? "[]" : "") +
        (this.caseSensitive ? " like " : " ~like ") +
        "'" + this.pattern + "'";
    }

    private Boolean mvmatch(IProperty prop) {

      if (!prop.isMultivalued()) {
        return null;
      }
      else {
        List values;
        try {
          values = prop.getValues();
        }
        catch (ResourceException ex) {
          log.debugT("mvmatch(711)", "getting values - " + LoggingFormatter.extractCallstack(ex));
          return null;
        }

        for (Iterator it = values.listIterator(); it.hasNext(); ) {

          String cmp = this.caseSensitive ? it.next().toString() :
            it.next().toString().toUpperCase();

          if (re.match(cmp)) {
            return Boolean.TRUE;
          }
        }

        return Boolean.FALSE;
      }
    }

    public Boolean match(IResource resource, IPropertyMap props) {

      // if the expression didn't compile, return "undefined"
      if (this.re == null) {
        return null;
      }

      // get the property and try to match it

      IProperty prop = props.get(this.prop);
      if (prop == null) {
        return null;
      }
      else if (this.mv) {
        return mvmatch(prop);
      }
      else if (prop.isMultivalued()) {
        return null;
      }
      else if (prop.getPropertyDef().getType().equals(PropertyType.XML)) {
        // XML-typed properties never match in value comparisons
        return null;
      }
      else {
        String cmp = this.caseSensitive ? prop.getValueAsString() :
          prop.getValueAsString().toUpperCase();

        return re.match(cmp) ? Boolean.TRUE : Boolean.FALSE;
      }
    }
  }

  public static class NotEq extends Comparison {

    public NotEq(IPropertyName prop, Object value, boolean caseSensitive, boolean mv) {
      super(prop, value, caseSensitive, mv);
    }

    public String toString() {
      return super.makeString("!=");
    }

    public Boolean cmp(long x, long y) {
      return x != y ? Boolean.TRUE : Boolean.FALSE;
    }

    public Boolean cmp(String x, String y) {
      if (this.caseSensitive) {
        return x.compareTo(y) != 0 ? Boolean.TRUE : Boolean.FALSE;
      }
      else {
        return x.compareToIgnoreCase(y) != 0 ? Boolean.TRUE : Boolean.FALSE;
      }
    }
  }

  public static class ConstFalse extends SimpleQueryExpression {

    public ConstFalse() {
      //
    }

    public String toString() {
      return "(false)";
    }

    public Boolean match(IResource resource, IPropertyMap props) {
      return Boolean.FALSE;
    }
  }

  public static class ConstTrue extends SimpleQueryExpression {

    public ConstTrue() {
      //
    }

    public String toString() {
      return "(true)";
    }

    public Boolean match(IResource resource, IPropertyMap props) {
      return Boolean.TRUE;
    }
  }

  public IQueryExpression or(IQueryExpression other) {
    return new Or(this, other);
  }

  public IQueryExpression and(IQueryExpression other) {
    return new And(this, other);
  }

  public IQueryExpression not() {
    return new Not(this);
  }

  public Boolean match(IResource res, IPropertyMap props) {
    // if we get here, it's a "undefined"
    return null;
  }

}
