/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.search;

import com.sap.tc.logging.Location;
import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.repository.enum.LinkType;
import com.sapportals.wcm.util.logging.LoggingFormatter;

/**
 * Implementation of a simple property search manager which performs it's search
 * by just traversing the resource hierarchy. <p>
 *
 * Copyright (c) SAP AG 2001-2005
 * @author julian.reschke@greenbytes.de
 * @version $Id: SimplePropertySearch.java,v 1.4 2005/01/28 11:50:47 jre Exp $
 */
public final class SimplePropertySearch {

  private final static Location log = Location.getLocation(SimplePropertySearch.class);

  private final SimpleQueryBuilder builder;

  public SimplePropertySearch() {
    this.builder = new SimpleQueryBuilder();
  }

  public IQueryBuilder getQueryBuilder() {
    return this.builder;
  }

  public IResourceList execute(IQueryExpression query, IResource resource,
    int depth, int maxResults, boolean includeVersions)
    throws ResourceException, AccessDeniedException, NotSupportedException {
    if (log.beDebug()) {
      log.debugT("execute(50)", "execute: " + query + " on " + resource.getRID() + " depth " + depth);
    }

    IResourceList rl = new ResourceList();
    SimpleQueryExpression exp = (SimpleQueryExpression)query;

    if (exp == null || Boolean.TRUE.equals(exp.match(resource, resource.getProperties()))) {
      rl.add(resource);
    }

    if (includeVersions && rl.size() < maxResults && resource.isVersioned()) {
      log.errorT("execute(61)", "descend into versions for " + resource.getRID());
      IResourceList versions = getVersionResources(resource);

      if (versions != null) {
        execute(exp, versions, maxResults, rl);
      }
    }

    // todo: do not recurse for links to collections
    if (depth > 0 && rl.size() < maxResults && resource.isCollection()) {
      execute(exp, (ICollection)resource, depth - 1, maxResults, rl, includeVersions);
    }

    return rl;
  }

  private void execute(SimpleQueryExpression query, ICollection collection,
    int depth, int maxResults, IResourceList rl, boolean includeVersions)
    throws ResourceException {

    IResourceList children = null;

    try {
      children = collection.getChildren();
    }
    catch (ResourceException ex) {
      log.debugT("execute(87)", "exception getting children of " + collection.getRID() + " - " + LoggingFormatter.extractCallstack(ex));
      return;
    }

    for (IResourceListIterator it = children.listIterator();
      it.hasNext() && rl.size() < maxResults; ) {

      IResource res = it.next();
      log.debugT("execute(95)", "execute: " + query + " on " + res.getRID() + " depth " + depth);

      try {
        if (query == null || Boolean.TRUE.equals(query.match(res, res.getProperties()))) {
          rl.add(res);
        }
      }
      catch (ResourceException ex) {
        if (log.beDebug()) {
          log.debugT("got resource exception for " + res.getRID() + " - " + LoggingFormatter.extractCallstack(ex));
        }
      }

      try {
        if (includeVersions && rl.size() < maxResults && res.isVersioned()) {
          IResourceList versions = getVersionResources(res);

          if (versions != null) {
            execute(query, versions, maxResults, rl);
          }
        }
      }
      catch (ResourceException ex) {
        if (log.beDebug()) {
          log.debugT("got resource exception for " + res.getRID() + " - " + LoggingFormatter.extractCallstack(ex));
        }
      }

      try {
        if (res.isCollection() && LinkType.NONE.equals(res.getLinkType())) {
          if (depth > 0 && rl.size() < maxResults) {
            execute(query, (ICollection)res, depth - 1, maxResults, rl, includeVersions);
          }
        }
      }
      catch (ResourceException ex) {
        if (log.beDebug()) {
          log.debugT("got resource exception for " + res.getRID() + " - " + LoggingFormatter.extractCallstack(ex));
        }
      }
      
    }
  }

  private void execute(SimpleQueryExpression query, IResourceList members,
    int maxResults, IResourceList rl)
    throws ResourceException {

    for (IResourceListIterator it = members.listIterator();
      it.hasNext() && rl.size() < maxResults; ) {

      IResource res = it.next();

      try {
        if (query == null || Boolean.TRUE.equals(query.match(res, res.getProperties()))) {
          rl.add(res);
        }
      }
      catch (ResourceException ex) {
        if (log.beDebug()) {
          log.debugT("got resource exception for " + res.getRID() + " - " + LoggingFormatter.extractCallstack(ex));
        }
      }
    }
  }

  private IResourceList getVersionResources(IResource resource)
    throws ResourceException, NotSupportedException {

    IVersionControlledResource vcr = (IVersionControlledResource)resource.as(IVersionControlledResource.class);
    IVersionHistoryResource vhr = vcr == null ? null : (IVersionHistoryResource)vcr.getVersionHistoryResource().as(IVersionHistoryResource.class);

    // try again using IVersionResource interface
    if (vhr == null) {
      IVersionResource vr = (IVersionResource)resource.as(IVersionResource.class);
      if (vr != null) {
        vhr = (IVersionHistoryResource)vr.getVersionHistoryResource().as(IVersionHistoryResource.class);
      }
    }

    if (vhr == null) {
      // give up -- use old style interface
      return resource.getVersionHistory();
    }
    else {
      return vhr.getVersionResourceSet();
    }
  }
}
