/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.search;

import java.io.StringReader;
import java.text.ParseException;
import java.util.*;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import com.sap.tc.logging.Location;
import com.sapportals.wcm.IWcmConst;
import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.util.http.DateFormat;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.xml.SimpleSerializer;

/**
 * A special query builder that just collects the expressions and offers
 * additional interfaces for obtaining a serialization and for creating a new
 * query expression from the external format. <p>
 *
 * Copyright (c) SAP AG 2001-2004
 *
 * @author julian.reschke@greenbytes.de
 * @version $Id: PersistentQueryExpression.java,v 1.4 2003/01/29 13:05:31 jre
 *      Exp $
 */

public abstract class PersistentQueryExpression implements IQueryExpression {
  private final static Location log = Location.getLocation(PersistentQueryExpression.class);

  private final static String NS = IWcmConst.SAP_WCM_NAMESPACE + IWcmConst.NAMESPACE_SEPARATOR + "simple-persistent-query";

  private final static String AN_CS = "case-sensitive";
  private final static String AN_DT = "dt";
  private final static String EN_LIT = "literal";
  private final static String EN_PROP = "prop";
  private final static String T_BOOLEAN = "boolean";
  private final static String T_DATE = "dateTime";
  private final static String T_INTEGER = "integer";
  private final static String T_LONG = "long";
  
  private static void checkLikeExpression(String expr) {

    try {
      for (int i = 0; i < expr.length(); i++) {
        char c = expr.charAt(i);

        switch (c) {
          case '\\':
            c = expr.charAt(++i);
            if (c != '\\' && c != '%' && c != '_') {
              throw new java.lang.IllegalArgumentException(
                "malformed 'like' expression ('\\' must escape '\\', '%' or '_')");
            }
            break;
          default:
            break;
        }
      }
    }
    catch (java.lang.ArrayIndexOutOfBoundsException ex) {
      throw new java.lang.IllegalArgumentException();
    }
  }

  private static class And extends PersistentQueryExpression {

    private List members;

    public And(IQueryExpression exp1, IQueryExpression exp2) {
      members = new ArrayList();

      if (exp1 instanceof And) {
        members.addAll(((And)exp1).members);
      }
      else {
        members.add(exp1);
      }

      if (exp2 instanceof And) {
        members.addAll(((And)exp2).members);
      }
      else {
        members.add(exp2);
      }
    }

    public Element toElem(Document doc) {
      Element and = doc.createElementNS(NS, "and");

      for (int i = 0; i < members.size(); i++) {
        and.appendChild(((PersistentQueryExpression)members.get(i)).toElem(doc));
      }

      return and;
    }
  }

  static class Not extends PersistentQueryExpression {
    private PersistentQueryExpression member;

    public Not(IQueryExpression exp1) {
      member = (PersistentQueryExpression)exp1;
    }

    public Element toElem(Document doc) {

      Element not = doc.createElementNS(NS, "not");
      not.appendChild(member.toElem(doc));
      return not;
    }
  }

  private static class Or extends PersistentQueryExpression {
    private List members;

    public Or(IQueryExpression exp1, IQueryExpression exp2) {
      members = new ArrayList();

      if (exp1 instanceof Or) {
        members.addAll(((Or)exp1).members);
      }
      else {
        members.add(exp1);
      }

      if (exp2 instanceof Or) {
        members.addAll(((Or)exp2).members);
      }
      else {
        members.add(exp2);
      }
    }

    public Element toElem(Document doc) {
      Element and = doc.createElementNS(NS, "or");

      for (int i = 0; i < members.size(); i++) {
        and.appendChild(((PersistentQueryExpression)members.get(i)).toElem(doc));
      }

      return and;
    }

  }

  private abstract static class Comparison extends PersistentQueryExpression {
    final IPropertyName prop;
    final String string;
    final boolean caseSensitive;
    final String type;

    public Comparison(IPropertyName prop, Object value, boolean caseSensitive) {
      this.prop = prop;
      if (value instanceof java.util.Date) {
        this.string = (new DateFormat()).formatISO8601((Date)value);
        this.type = T_DATE;
      }
      else if (value instanceof Long) {
        this.string = value.toString();
        this.type = T_LONG;
      }
      else if (value instanceof Integer) {
        this.string = value.toString();
        this.type = T_INTEGER;
      }
      else if (value instanceof Boolean) {
        this.string = Boolean.TRUE.equals(value) ? "1" : "0";
        this.type = T_BOOLEAN;
      }
      else {
        this.string = value.toString();
        this.type = null;
      }
      this.caseSensitive = caseSensitive;
    }

    abstract Element toElem(Document doc);

    protected Element buildElem(Document doc, String name) {

      Element e = doc.createElementNS(NS, name);

      if (!this.caseSensitive) {
        e.setAttribute(AN_CS, "0");
      }

      e.appendChild(getPropElem(doc, prop));
      e.appendChild(getLiteralElem(doc, string, type));

      return e;
    }
  }

  static class Eq extends Comparison {

    public Eq(IPropertyName prop, Object value, boolean caseSensitive) {
      super(prop, value, caseSensitive);
    }

    public Element toElem(Document doc) {
      return super.buildElem(doc, "eq");
    }
  }

  static class SomeEq extends Comparison {

    public SomeEq(IPropertyName prop, Object value, boolean caseSensitive) {
      super(prop, value, caseSensitive);
    }

    public Element toElem(Document doc) {
      return super.buildElem(doc, "someEq");
    }
  }

  static class Greater extends Comparison {

    public Greater(IPropertyName prop, Object value, boolean caseSensitive) {
      super(prop, value, caseSensitive);
    }

    public Element toElem(Document doc) {
      return super.buildElem(doc, "greater");
    }
  }

  static class SomeGreater extends Comparison {

    public SomeGreater(IPropertyName prop, Object value, boolean caseSensitive) {
      super(prop, value, caseSensitive);
    }

    public Element toElem(Document doc) {
      return super.buildElem(doc, "someGreater");
    }
  }

  static class GreaterEq extends Comparison {

    public GreaterEq(IPropertyName prop, Object value, boolean caseSensitive) {
      super(prop, value, caseSensitive);
    }

    public Element toElem(Document doc) {
      return super.buildElem(doc, "greaterEq");
    }
  }

  static class SomeGreaterEq extends Comparison {

    public SomeGreaterEq(IPropertyName prop, Object value, boolean caseSensitive) {
      super(prop, value, caseSensitive);
    }

    public Element toElem(Document doc) {
      return super.buildElem(doc, "someGreaterEq");
    }
  }

  static class IsCollection extends PersistentQueryExpression {

    public IsCollection() {
      //
    }

    public Element toElem(Document doc) {
      return doc.createElementNS(NS, "isCollection");
    }
  }

  static class IsDefined extends PersistentQueryExpression {

    IPropertyName prop;

    public IsDefined(IPropertyName prop) {
      this.prop = prop;
    }

    public Element toElem(Document doc) {
      Element isd = doc.createElementNS(NS, "isDefined");
      isd.appendChild(getPropElem(doc, prop));
      return isd;
    }
  }

  static class Less extends Comparison {

    public Less(IPropertyName prop, Object value, boolean caseSensitive) {
      super(prop, value, caseSensitive);
    }

    public Element toElem(Document doc) {
      return super.buildElem(doc, "less");
    }
  }

  static class SomeLess extends Comparison {

    public SomeLess(IPropertyName prop, Object value, boolean caseSensitive) {
      super(prop, value, caseSensitive);
    }

    public Element toElem(Document doc) {
      return super.buildElem(doc, "someLess");
    }
  }

  static class LessEq extends Comparison {

    public LessEq(IPropertyName prop, Object value, boolean caseSensitive) {
      super(prop, value, caseSensitive);
    }

    public Element toElem(Document doc) {
      return super.buildElem(doc, "lessEq");
    }
  }

  static class SomeLessEq extends Comparison {

    public SomeLessEq(IPropertyName prop, Object value, boolean caseSensitive) {
      super(prop, value, caseSensitive);
    }

    public Element toElem(Document doc) {
      return super.buildElem(doc, "someLessEq");
    }
  }

  static class Like extends Comparison {

    public Like(IPropertyName prop, String pattern, boolean caseSensitive) {
      super(prop, pattern, caseSensitive);
      PersistentQueryExpression.checkLikeExpression(pattern);
    }

    public Element toElem(Document doc) {
      return super.buildElem(doc, "like");
    }
  }

  static class SomeLike extends Comparison {

    public SomeLike(IPropertyName prop, String pattern, boolean caseSensitive) {
      super(prop, pattern, caseSensitive);
      PersistentQueryExpression.checkLikeExpression(pattern);
    }

    public Element toElem(Document doc) {
      return super.buildElem(doc, "someLike");
    }
  }

  static class NotEq extends Comparison {

    public NotEq(IPropertyName prop, Object value, boolean caseSensitive) {
      super(prop, value, caseSensitive);
    }

    public Element toElem(Document doc) {
      return super.buildElem(doc, "notEq");
    }
  }

  static class SomeNotEq extends Comparison {

    public SomeNotEq(IPropertyName prop, Object value, boolean caseSensitive) {
      super(prop, value, caseSensitive);
    }

    public Element toElem(Document doc) {
      return super.buildElem(doc, "someNotEq");
    }
  }

  static class ConstFalse extends PersistentQueryExpression {
    public ConstFalse() {
      //
    }

    public Element toElem(Document doc) {
      return doc.createElementNS(NS, "false");
    }
  }

  static class ConstTrue extends PersistentQueryExpression {
    public ConstTrue() {
      //
    }

    public Element toElem(Document doc) {
      return doc.createElementNS(NS, "true");
    }
  }

  public IQueryExpression or(IQueryExpression other) {
    return new Or(this, other);
  }

  public IQueryExpression and(IQueryExpression other) {
    return new And(this, other);
  }

  public IQueryExpression not() {
    return new Not(this);
  }

  public Boolean match(IResource res, IPropertyMap props) {
    // if we get here, it's a "undefined"
    return null;
  }

  abstract Element toElem(Document doc);


  /**
   * Returns the query expression as a string of serialized XML.
   *
   * @return asXml
   */
  public String getAsXml()
    throws java.io.IOException {
    Document doc = getAsDocument();
    java.io.StringWriter sw = new java.io.StringWriter();
    SimpleSerializer ser = new SimpleSerializer(sw, null);
    ser.serialize(doc);
    return sw.toString();
  }

  /**
   * Returns the query expression as a {@link org.w3c.dom.Document}.
   *
   * @return asDocument
   */
  public Document getAsDocument() {

    Document doc = null;
    try {
      DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
      docBuilderFactory.setNamespaceAware(true);
      DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
      doc = docBuilder.newDocument();
    }
    catch (ParserConfigurationException ex) {
      log.fatalT("getAsDocument(509)", "creating DOM - " + LoggingFormatter.extractCallstack(ex));
      throw new RuntimeException(ex.toString());
    }

    doc.appendChild(this.toElem(doc));
    return doc;
  }

  private final static Set comparators = new HashSet(
    Arrays.asList(
    new String[]{"eq", "notEq", "greater",
    "greaterEq", "less", "lessEq", "like",
    "someEq", "someNotEq", "someGreater", "someGreaterEq", "someLess",
    "someLessEq", "someLike"}
    ));

  private static IQueryExpression fromElement(IQueryBuilder qb, Element e)
    throws ResourceException {

    try {
      IMultiValueQueryBuilder mvqb = qb instanceof IMultiValueQueryBuilder
         ? (IMultiValueQueryBuilder)qb : null;

      if (!NS.equals(e.getNamespaceURI())) {
        return null;
      }

      String name = e.getLocalName();
      IQueryExpression start = null;

      if ("and".equals(name)) {
        for (Node n = e.getFirstChild(); n != null; n = n.getNextSibling()) {
          if (n.getNodeType() == Node.ELEMENT_NODE) {
            if (start == null) {
              start = fromElement(qb, (Element)n);
            }
            else {
              start = start.and(fromElement(qb, (Element)n));
            }
          }
        }
        return start;
      }
      else if ("or".equals(name)) {
        for (Node n = e.getFirstChild(); n != null; n = n.getNextSibling()) {
          if (n.getNodeType() == Node.ELEMENT_NODE) {
            if (start == null) {
              start = fromElement(qb, (Element)n);
            }
            else {
              start = start.or(fromElement(qb, (Element)n));
            }
          }
        }
        return start;
      }
      else if ("not".equals(name)) {
        for (Node n = e.getFirstChild(); n != null; n = n.getNextSibling()) {
          if (n.getNodeType() == Node.ELEMENT_NODE) {
            return fromElement(qb, (Element)n).not();
          }
        }
        return start;
      }
      // comparisons
      else if (comparators.contains(name)) {
        IPropertyName prop = getPropName(e);
        Object value = getValue(e);
        boolean caseSensitive = !e.getAttribute(AN_CS).equalsIgnoreCase("0");

        if ("eq".equals(name)) {
          return caseSensitive ? qb.eq(prop, value) : qb.eqIgnoreCase(prop, value.toString());
        }
        else if ("notEq".equals(name)) {
          return caseSensitive ? qb.notEq(prop, value) : qb.notEqIgnoreCase(prop, value.toString());
        }
        else if ("less".equals(name)) {
          return caseSensitive ? qb.less(prop, value) : qb.lessIgnoreCase(prop, value.toString());
        }
        else if ("lessEq".equals(name)) {
          return caseSensitive ? qb.lessEq(prop, value) : qb.lessEqIgnoreCase(prop, value.toString());
        }
        else if ("greater".equals(name)) {
          return caseSensitive ? qb.greater(prop, value) : qb.greaterIgnoreCase(prop, value.toString());
        }
        else if ("greaterEq".equals(name)) {
          return caseSensitive ? qb.greaterEq(prop, value) : qb.greaterEqIgnoreCase(prop, value.toString());
        }
        else if ("like".equals(name)) {
          return caseSensitive ? qb.like(prop, value.toString()) : qb.likeIgnoreCase(prop, value.toString());
        }

        if (mvqb == null) {
          throw new NotSupportedException();
        }

        if ("someEq".equals(name)) {
          return caseSensitive ? mvqb.someEq(prop, value) : mvqb.someEqIgnoreCase(prop, value);
        }
        else if ("someNotEq".equals(name)) {
          return caseSensitive ? mvqb.someNotEq(prop, value) : mvqb.someNotEqIgnoreCase(prop, value);
        }
        else if ("someLess".equals(name)) {
          return caseSensitive ? mvqb.someLess(prop, value) : mvqb.someLessIgnoreCase(prop, value);
        }
        else if ("someLessEq".equals(name)) {
          return caseSensitive ? mvqb.someLessEq(prop, value) : mvqb.someLessEqIgnoreCase(prop, value);
        }
        else if ("someGreater".equals(name)) {
          return caseSensitive ? mvqb.someGreater(prop, value) : mvqb.someGreaterIgnoreCase(prop, value);
        }
        else if ("someGreaterEq".equals(name)) {
          return caseSensitive ? mvqb.someGreaterEq(prop, value) : mvqb.someGreaterEqIgnoreCase(prop, value);
        }
        else if ("someLike".equals(name)) {
          return caseSensitive ? mvqb.someLike(prop, value.toString()) : mvqb.someLikeIgnoreCase(prop, value.toString());
        }

        // shouldn't get here
        return null;
      }
      else if (name.equals("isCollection")) {
        return qb.isCollection();
      }
      else if (name.equals("isDefined")) {
        IPropertyName prop = getPropName(e);
        return qb.isDefined(prop);
      }
      else if (name.equals("false")) {
        return qb.constFalse();
      }
      else if (name.equals("true")) {
        return qb.constTrue();
      }
      else {
        log.errorT("fromElement(643)", "unknown element name: " + name);
        return null;
      }
    }
    catch (ResourceException ex) {
      throw ex;
    }
    catch (Exception ex) {
      throw new ResourceException(ex.getMessage());
    }
  }

  private static IPropertyName getPropName(Element e) throws ResourceException {
    for (Node n = e.getFirstChild(); n != null; n = n.getNextSibling()) {
      if (NS.equals(n.getNamespaceURI()) && EN_PROP.equals(n.getLocalName())) {
        for (Node c = n.getFirstChild(); c != null; c = c.getNextSibling()) {
          if (c.getNodeType() == Node.ELEMENT_NODE) {
            return new PropertyName(c.getNamespaceURI(), c.getLocalName());
          }
        }
      }
    }
    return null;
  }

  
  private static Object getValue(Element e) throws ParseException {
    for (Node n = e.getFirstChild(); n != null; n = n.getNextSibling()) {
      if (NS.equals(n.getNamespaceURI()) && EN_LIT.equals(n.getLocalName())) {
        String dt = ((Element)n).getAttribute(AN_DT);
        n.normalize();
        String sval = n.getFirstChild() != null ? n.getFirstChild().getNodeValue() : "";
        if (T_DATE.equals(dt)) {
          return ((new DateFormat().parseISO8601(sval)));
        }
        else if (T_LONG.equals(dt)) {
          return Long.valueOf(sval);
        }
        else if (T_INTEGER.equals(dt)) {
          return Integer.valueOf(sval);
        }
        else if (T_BOOLEAN.equals(dt)) {
          return "1".equals(sval) ? Boolean.TRUE : Boolean.FALSE;
        }
        else {
          return sval;
        }        
      } 
    }
    return null;
  }


  /**
   * Rebuild a query expression from the persistence format
   *
   * @param qb the query builder for which the query expression should be
   *      generated
   * @param doc Document obtained from getAsDocument()
   * @return query expression
   */
  public static IQueryExpression fromDocument(IQueryBuilder qb, Document doc)
    throws ResourceException {
    return fromElement(qb, doc.getDocumentElement());
  }


  /**
   * Rebuild a query expression from the persistence format
   *
   * @param qb the query builder for which the query expression should be
   *      generated
   * @return query expression
   */
  public static IQueryExpression fromXml(IQueryBuilder qb, String query)
    throws java.io.IOException, SAXException, ResourceException {
    try {
      DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
      docBuilderFactory.setNamespaceAware(true);
      DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
      Document doc = docBuilder.parse(new InputSource(new StringReader(query)));
      return fromDocument(qb, doc);
    }
    catch (ParserConfigurationException ex) {
      log.fatalT("fromXml(718)", "creating DOM: " + LoggingFormatter.extractCallstack(ex));
      throw new RuntimeException(ex.toString());
    }
  }


  // helper

  private static Element getLiteralElem(Document doc, String s, String type) {
    Element lit = doc.createElementNS(NS, EN_LIT);
    lit.appendChild(doc.createTextNode(s));
    if (type != null) {
      lit.setAttribute(AN_DT, type);
    }
    return lit;
  }

  private static Element getPropElem(Document doc, IPropertyName prop) {
    Element pr = doc.createElementNS(NS, EN_PROP);
    pr.appendChild(doc.createElementNS(prop.getNamespace(), prop.getName()));
    return pr;
  }

}
