/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.search;

import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.repository.manager.IPropertySearchManager;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.w3c.dom.Document;

/**
 * Default implementation for building generic query expressions. <p>
 *
 * Copyright (c) SAP AG 2002
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Revision: 1.2 $
 */
public final class GenericQueryFactoryImpl implements IGenericQueryFactory {

  private static GenericQueryFactoryImpl instance;

  public static synchronized GenericQueryFactoryImpl getInstance()
    throws Exception {
    if (GenericQueryFactoryImpl.instance == null) {
      GenericQueryFactoryImpl.instance = new GenericQueryFactoryImpl();
    }
    return GenericQueryFactoryImpl.instance;
  }

  private final IQueryBuilder builder;
  private final SimplePropertySearch search;

  private GenericQueryFactoryImpl() {
    this.builder = new PersistentQueryBuilder();
    this.search = new SimplePropertySearch();
  }

  public IQueryBuilder getQueryBuilder() {
    return this.builder;
  }

  public IGenericQuery toGenericQuery(IQueryExpression query)
    throws NotSupportedException {
    if (!(query instanceof PersistentQueryExpression)) {
      throw new NotSupportedException("query is not of correct type, must be created with own builder");
    }

    return GenericQuery.getInstance((PersistentQueryExpression)query, this.search);
  }

  //------------------------ private -----------------------------------------

  private final static class GenericQuery implements IGenericQuery {

    public static IGenericQuery getInstance(PersistentQueryExpression query, SimplePropertySearch search) {
      return new GenericQuery(query, search);
    }

    private final Document queryDoc;
    private final SimplePropertySearch search;
    private final Map queryMap;

    private GenericQuery(PersistentQueryExpression query, SimplePropertySearch search) {
      this.queryDoc = query.getAsDocument();
      this.queryMap = new HashMap();
      this.search = search;
    }

    public IResourceList execute(IResource resource, int depth, int maxResults, boolean includeVersions)
      throws ResourceException, AccessDeniedException, NotSupportedException {
      IPropertySearchManager psm = resource.getRepositoryManager().getPropertySearchManager(resource);

      IQueryExpression query = null;
      synchronized (this.queryMap) {
        query = (IQueryExpression)this.queryMap.get(psm);
        if (psm != null) {
          if (query == null) {
            query = PersistentQueryExpression.fromDocument(psm.getQueryBuilder(), this.queryDoc);
            this.queryMap.put(psm, query);
          }
          return psm.execute(query, resource, depth, maxResults, includeVersions);
        }
        else {
          if (query == null) {
            query = PersistentQueryExpression.fromDocument(this.search.getQueryBuilder(), this.queryDoc);
            this.queryMap.put(null, query);
          }
          return this.search.execute(query, resource, depth, maxResults, includeVersions);
        }
      }
    }
  }

  public IQueryExpression fromXml(String persistedQuery)
    throws ResourceException {
    try {
      return PersistentQueryExpression.fromXml(getQueryBuilder(), persistedQuery);
    }
    catch (Exception ex) {
      throw new ResourceException(ex);
    }
  }

  public String toXml(IQueryExpression queryExpression)
    throws ResourceException {
    if (queryExpression instanceof PersistentQueryExpression) {
      try {
        return ((PersistentQueryExpression)queryExpression).getAsXml();
      }
      catch (IOException e) {
        throw new ResourceException(e);
      }
    }
    else {
      throw new NotSupportedException("query expression must be instanceof PersistentQueryExpression");
    }
  }
}
