/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.rf/60NW_SP_COR/src/_runtime/java/api/com/sapportals/wcm/repository/runtime/CmSystem.java#10 $
 */
package com.sapportals.wcm.repository.runtime;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;

import com.sap.jmx.monitoring.api.ConfigurationList;
import com.sap.tc.logging.Category;
import com.sap.tc.logging.Location;
import com.sapportals.config.event.IConfigEventService;
import com.sapportals.config.fwk.Configuration;
import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.crt.CascadingException;
import com.sapportals.wcm.crt.CrtClassLoaderRegistry;
import com.sapportals.wcm.crt.CrtSystem;
import com.sapportals.wcm.crt.IClassLoaderRegistryListener;
import com.sapportals.wcm.crt.component.ComponentException;
import com.sapportals.wcm.crt.component.ComponentState;
import com.sapportals.wcm.crt.component.IComponent;
import com.sapportals.wcm.crt.component.IComponentInfo;
import com.sapportals.wcm.crt.component.IComponentManager;
import com.sapportals.wcm.crt.component.IContainer;
import com.sapportals.wcm.crt.component.ILifecycleInfo;
import com.sapportals.wcm.crt.configuration.ConfigurationException;
import com.sapportals.wcm.crt.moni.ComponentRuntime;
import com.sapportals.wcm.repository.IResourceContext;
import com.sapportals.wcm.repository.ResourceContext;
import com.sapportals.wcm.repository.filter.IFilterManager;
import com.sapportals.wcm.repository.manager.IRepositoryManager;
import com.sapportals.wcm.repository.manager.IResourceEventBroker;
import com.sapportals.wcm.repository.manager.OverallResourceEventBroker;
import com.sapportals.wcm.repository.service.IRepositoryService;
import com.sapportals.wcm.repository.so.ObjectFactoryRegistry;
import com.sapportals.wcm.service.IService;
import com.sapportals.wcm.service.landscape.ILandscapeService;
import com.sapportals.wcm.service.landscape.ISystem;
import com.sapportals.wcm.service.landscape.LandscapeServiceFactory;
import com.sapportals.wcm.util.cache.CacheException;
import com.sapportals.wcm.util.cache.CacheFactory;
import com.sapportals.wcm.util.acl.AclManagerFactory ;
import com.sapportals.wcm.util.logging.LoggingFormatter;

/**
 * The CM system class provides access to all the components of the CM system
 * and to resources. <p>
 *
 * Copyright (c) SAP AG 2002
 *
 * @author Markus Breitenfelder
 * @version $Id: //kmgmt/bc.rf/60NW_SP_COR/src/_runtime/java/api/com/sapportals/wcm/repository/runtime/CmSystem.java#10 $
 */
public class CmSystem implements IClassLoaderRegistryListener,
                                 ComponentRuntime.VirtualMBean {

	private final static Category cat = Category.getCategory(Category.APPLICATIONS,"KMC/RF");

  static boolean testabilityMode = false;

  private static Location log = Location.getLocation(com.sapportals.wcm.repository.runtime.CmSystem.class);

  // Client ID for CRT - MUST match the prefix of the CRT Repository !
  final static String CRT_CLIENT_ID = "runtime";

  private static CmSystem cmSystem;
  private static ArrayList startupHistory;
  private boolean starting;
  private boolean shutdown;
  private boolean up;

  private boolean startupFailed;
  private IComponentManager componentManager;

  private CmFilterHandler filterHandler;
  private CmConfigurationProvider provider;
  private CmRepMgrContainer repMgrContainer;
  private CmServiceContainer serviceContainer;
  private CmRepSrvContainer repSrvContainer;
  private CmRepFltContainer filterContainer;

  private Thread suspender;
  private int numWorkers;

  private CmSystem() {
  }

  CmRepMgrContainer getRepMgrContainer() throws ComponentException {
    if (this.repMgrContainer == null) {
      IComponentManager cm = this.getComponentManager();
      if (cm == null) return null;      
      this.repMgrContainer =
        (CmRepMgrContainer)cm.lookupComponent(
          CmConfigurationProvider.CONTAINER_URIS.REPMGR.getKey());
    }
    return this.repMgrContainer;
  }

  CmRepSrvContainer getRepSrvContainer() throws ComponentException {
    if (this.repSrvContainer == null) {
      IComponentManager cm = this.getComponentManager();
      if (cm == null) return null;      
      this.repSrvContainer =
        (CmRepSrvContainer)cm.lookupComponent(
          CmConfigurationProvider.CONTAINER_URIS.REPSRV.getKey());
    }
    return this.repSrvContainer;
  }

  CmServiceContainer getServiceContainer() throws ComponentException {
    if (serviceContainer == null) {
      IComponentManager cm = this.getComponentManager();
      if (cm == null) return null;
      this.serviceContainer =
        (CmServiceContainer)cm.lookupComponent(
          CmConfigurationProvider.CONTAINER_URIS.SERVICE.getKey());
    }
    return this.serviceContainer;
  }

  CmRepFltContainer getFilterContainer() throws ComponentException {
    if (this.filterContainer == null) {
      IComponentManager cm = this.getComponentManager();
      if (cm == null) return null;      
      this.filterContainer =
        (CmRepFltContainer)cm.lookupComponent(
          CmConfigurationProvider.CONTAINER_URIS.FILTER.getKey());
    }
    return this.filterContainer;
  }

  /**
   * Get the component manager for CM from CRT
   *
   * @return componentManager
   */
  private IComponentManager getComponentManager() {
    if (componentManager == null) {
      componentManager = CrtSystem.getInstance().getComponentManager(CmSystem.CRT_CLIENT_ID);
    }
    return componentManager;
  }

  CmConfigurationProvider getProvider() {
    return this.provider;
  }

  public static synchronized CmSystem getInstance() throws CmStartupException {
    if (null == CmSystem.cmSystem) {
      CmSystem.cmSystem = new CmSystem();
      CmSystem.cmSystem.startUp();
    }

    return CmSystem.cmSystem;
  }

  private static String extractCallstack(Throwable throwable) {
	  StringWriter stringWriter = new StringWriter();
	  throwable.printStackTrace(new PrintWriter(stringWriter));
	  return stringWriter.toString();
  }

  /**
   * Starts up the CM system: Add application to CRT.
   *
   * @exception CmStartupException Exception raised in failure situation
   */
  private void startUp() throws CmStartupException {
    final String METHOD = "startUp()";
    
    if (startupFailed) {
      throw new CmStartupException("CM startup failed");
    }

    if (!up && !shutdown) {
      starting = true;
      cat.infoT(log, "CM system is starting ... \n" );
          
      if( startupHistory == null ) {
        startupHistory = new ArrayList(); 
      }
      startupHistory.add( new Date() );
      
      CmRegistry.registry.clear();
      this.provider = new CmConfigurationProvider();
      this.filterHandler = new CmFilterHandler(this.provider);
      try {
        CrtSystem.getInstance().createComponentManager(
          CmSystem.CRT_CLIENT_ID,
          this.provider,
          null,
          null,
          cat);
      }
      catch (CascadingException ex) {
        this.handleStartupFailure("Error creating CRT application: " + ex.getMessage(), ex);
      }
      catch (Exception ex) {
        this.handleStartupFailure("Error creating CRT application: " + ex.getMessage(), ex);
      }

      try {
        this.getRepMgrContainer();
        this.getRepSrvContainer();
        this.getServiceContainer();
        this.getFilterContainer();
      }
      catch (ComponentException ex) {
        this.handleStartupFailure(ex.getMessage(), ex);
      }

      // This will call lookup() for all auto-startable components
      CrtSystem.getInstance().startUpComponentManager(CmSystem.CRT_CLIENT_ID);

      // Register provider class as listener at configuration framework
      try {
        IConfigEventService eventService = Configuration.getInstance().getConfigEventServiceInstance();
        eventService.addConfigEventListener(this.provider, new String[] { "/cm" });
      }
      catch (Exception ex) {
        this.handleStartupFailure("Exception while registering config event listener", ex);
      }

      // Register CRT as monitoring data provider.
      try {
        com.sapportals.wcm.crt.moni.ComponentRuntime.setImplementation( this );
      }
      catch ( Throwable thr ) {
        // Something went wrong with registering the mbean implementation.
        // Ignore but trace the fact.
        
        log.debugT( METHOD, "tried to register mbean implementation, ignoring {0}", new Object[] { thr } );
      }

      CrtClassLoaderRegistry.addClassLoaderRegistryListener(this);

      cat.infoT(log, "CM successfully started.");

      starting = false;
      up = true;
    }

  }

  private void handleStartupFailure(String details, Exception ex) throws CmStartupException {
    log.fatalT(ex.getMessage() + ": " + LoggingFormatter.extractCallstack(ex));
    log.debugT(LoggingFormatter.extractCallstack(ex));
    CmStartupException startupException = new CmStartupException("CM startup failed: " + details, ex);
    log.debugT(LoggingFormatter.extractCallstack(startupException));
    throw startupException;
  }

  public synchronized void shutDown() {
    final String METHOD = "shutDown()";
    
    if (up && !starting && !shutdown) {
      shutdown = true;

      CrtClassLoaderRegistry.removeClassLoaderRegistryListener(this);

      cat.infoT(log, "CM system is shutting down...\n" );

      // Unregister CRT as monitoring data provider.
      try {
        com.sapportals.wcm.crt.moni.ComponentRuntime.setImplementation( null );
      }
      catch ( Throwable thr ) {
        // Something went wrong with unregistering the mbean implementation.
        // Ignore but trace the fact.
        log.debugT( METHOD, "tried to unregister mbean implementation, ignoring {0}", new Object[] { thr } );
      }

		  cat.infoT(log, "CM system is shutting down... - shutdown of CRT");

      try {
        IConfigEventService eventService = Configuration.getInstance().getConfigEventServiceInstance();
        if( eventService != null ) {
          eventService.removeConfigEventListener(this.provider);
        }  

        CrtSystem.getInstance().shutDownComponentManager(CmSystem.CRT_CLIENT_ID);
        
        ObjectFactoryRegistry.reset();
      }
      catch (Throwable x) {
        log.fatalT(LoggingFormatter.extractCallstack(x));
      }

      /*
      * take hot deployment roadblocks down
      */
      try {
        AclManagerFactory.resetFactory() ;
      }
      catch (Throwable x) {
        log.errorT(LoggingFormatter.extractCallstack(x));
      }
      try {
        CacheFactory.resetFactory();
      }
      catch (Throwable x) {
        log.errorT(LoggingFormatter.extractCallstack(x));
      }
      
      CmAdapter.instance = null;
      cmSystem = null;
      shutdown = false;
      up = false;

      cat.infoT(log, "CM successfully shut down.");            
    }
  }

  // ---------------------------------------------------------------------------
  // Public methods
  // ---------------------------------------------------------------------------

  public List getStartupHistory() {
    return startupHistory;  
  }
  
  /**
   * Adds a new filter component to the CM system.
   *
   * @param filterConfig The filter manager configuration data.
   * @exception WcmException If the configuration data is invalid.
   */
  public void createFilterManager(Properties filterConfig) throws WcmException {
    try {
      this.provider.createFilterManager(filterConfig);
    }
    catch (ConfigurationException ex) {
      throw new WcmException("Failed to create filter manager: " + ex.getMessage(), ex);
    }
  }

  /**
   * Attach an existing filter manager to an additional repository manager.
   *
   * @param filterManagerID The identifier of an existing filter manager
   *      component.
   * @param repositoryManagerID The identifier of an existing repository manager
   *      component.
   * @exception WcmException Exception raised in failure situation
   * @excpetion WcmException If the filter or repository manager does not exist
   *      or if the filter is already attached.
   */
  public void attachFilterToRepository(String filterManagerID, String repositoryManagerID) throws WcmException {
    try {
      this.provider.attachFilterToRepository(filterManagerID, repositoryManagerID);
    }
    catch (Exception ex) {
      throw new WcmException(ex);
    }
  }

  /**
   * Detaches the filter manager from the specified repository manager.
   *
   * @param filterManagerID The identifier of an existing filter manager
   *      component.
   * @param repositoryManagerID The identifier of an existing repository manager
   *      component.
   * @exception WcmException Exception raised in failure situation
   */
  public void detachFilterFromRepository(String filterManagerID, String repositoryManagerID) throws WcmException {
    try {
      this.provider.detachFilterFromRepository(filterManagerID, repositoryManagerID);
    }
    catch (Exception ex) {
      throw new WcmException(ex);
    }
  }

  /**
   * Releases the component instance.
   *
   * @param comp The component instance.
   */
  public void releaseComponent(IComponent comp) {
    try {
      if (comp instanceof IRepositoryManager) {
        this.getRepMgrContainer().getComponentManager().releaseComponent(comp);
      }
      else if (comp instanceof IRepositoryService) {
        this.getRepSrvContainer().getComponentManager().releaseComponent(comp);
      }
      else if (comp instanceof IService) {
        this.getServiceContainer().getComponentManager().releaseComponent(comp);
      }
      else if (comp instanceof IFilterManager) {
        this.getFilterContainer().getComponentManager().releaseComponent(comp);
      }
      else {
        log.errorT("releaseComponent(642)", "Invalid component instance: " + comp);
      }
    }
    catch (ComponentException ex) {
      log.errorT("releaseComponent(646)", LoggingFormatter.extractCallstack(ex));
    }
  }

  public synchronized IResourceEventBroker getOverallResourceEventBroker() {
    return OverallResourceEventBroker.getInstance();
  }

  public CmFilterHandler getFilterHandler() {
    return this.filterHandler;
  }

  public IResourceContext getServiceContext() throws WcmException {
    return new ResourceContext(null);
  }

  /* (non-Javadoc)
   * @see com.sapportals.wcm.crt.IClassLoaderRegistryListener#classLoaderAdded(java.lang.String, java.lang.ClassLoader)
   */
  public void classLoaderAdded(String id, ClassLoader newLoader) {
    log.infoT("ClassLoader " + id + " added.");
    shutDown();
  }

  /* (non-Javadoc)
   * @see com.sapportals.wcm.crt.IClassLoaderRegistryListener#classLoaderReplaced(java.lang.String, java.lang.ClassLoader, java.lang.ClassLoader)
   */
  public void classLoaderReplaced(String id, ClassLoader newLoader, ClassLoader oldLoader) {
    log.infoT("ClassLoader " + id + " replaced.");
    shutDown();
  }
  
  //////////////////////////////////
  // JMX instrumentation
  //////////////////////////////////
  
  public Object getRoot() {
    return CrtSystem.getInstance().getComponentManager( CRT_CLIENT_ID );
  }
  
  public IComponentManager getManager( Object obj ) {
    if ( obj instanceof IContainer ) {
      return ( ( IContainer ) obj ).getComponentManager();
    } else if ( obj instanceof IComponentManager ) {
      return ( IComponentManager ) obj;
    } else {
      return null;
    }
  }
    
  public String[] getKeys( Object obj ) {
    IComponentManager man = getManager( obj );
    
    if ( man != null ) {
      return man.listComponentKeys();
    } else {
      return NO_IDS;
    }
  }
    
  public Object getComponent( Object obj, String key) {
    IComponentManager man = getManager( obj );
    
    if ( man != null ) {
      try {
        return man.lookupComponent( key );
      }
      catch ( ComponentException exc ) {
        return null;
      }
    } else {
      return null;
    }
  }
    
  public String getDesc( Object comp ) {
    if ( comp instanceof IComponentInfo ) {
      String desc = ( ( IComponentInfo ) comp ).getDescription( LOCALE );

      if ( desc != null ) {
        return desc;
      } else {
        return UNKNOWN_DESC;
      }
    } else {
      return UNKNOWN_DESC;
    }
  }
  
  public String getState( Object comp ) {
    if ( comp instanceof ILifecycleInfo ) {
      ComponentState state = ( ( ILifecycleInfo ) comp ).getState();

      if ( state != null ) {
        return state.getName();
      } else {
        return UNKNOWN_STATE;
      }
    } else {
      return UNKNOWN_STATE;
    }
  }
  
  public boolean isAvailable( Object comp ) {
    if ( comp instanceof ILifecycleInfo ) {
      return ( ( ILifecycleInfo ) comp ).getState() == ComponentState.RUNNING;
    } else {
      return true;
    }
  }
      
  public String getStartupIssue( Object comp ) {
    if ( comp instanceof ILifecycleInfo ) {
      Throwable thr = ( ( ILifecycleInfo ) comp ).getStartupException();

      if ( thr != null ) {
        return thr.getMessage();
      } else {
        return NO_ISSUE;
      }
    } else {
      return UNKNOWN_ISSUE;
    }
  }
      
  public String getLastConfigurationIssue( Object comp ) {
    if ( comp instanceof ILifecycleInfo ) {
      Throwable thr = ( ( ILifecycleInfo ) comp ).getLastConfigurationException();

      if ( thr != null ) {
        return thr.getMessage();
      } else {
        return NO_ISSUE;
      }
    } else {
      return UNKNOWN_ISSUE;
    }
  }
      
  public String getCreationDate( Object comp ) {
    if ( comp instanceof ILifecycleInfo ) {
      Date date = ( ( ILifecycleInfo ) comp ).getCreationDate();
      
      if ( date != null ) {
        return DATE_FORMATTER.format( date );
      } else {
        return NO_DATE;
      }
    } else {
      return NO_DATE;
    }
  }

  public String getLastReconfigurationDate( Object comp ) {
    if ( comp instanceof ILifecycleInfo ) {
      Date date = ( ( ILifecycleInfo ) comp ).getLastReconfigurationDate();
      
      if ( date != null ) {
        return DATE_FORMATTER.format( date );
      } else {
        return NO_DATE;
      }
    } else {
      return NO_DATE;
    }
  }

  public String getNextAutoRestartDate( Object comp ) {
    if ( comp instanceof ILifecycleInfo ) {
      Date date = ( ( ILifecycleInfo ) comp ).getNextAutoRestartDate();
      
      if ( date != null ) {
        return DATE_FORMATTER.format( date );
      } else {
        return NO_DATE;
      }
    } else {
      return NO_DATE;
    }
  }
  
  public ConfigurationList getSystemConfiguration() {
    final String METHOD = "getSystemConfiguration()";
    
    ConfigurationList conf = new ConfigurationList();

    try {    
      ILandscapeService lsSrv  = LandscapeServiceFactory.lookup();
      ISystem           system = lsSrv.getSystemFactory().getLocalSystem();
        
      conf.setConfigurationParameter( SYSTEM_ID_PROP, ( system != null ) ? system.getId()
                                                                         : DEFAULT_LOCAL_NODE_ID );
      conf.setConfigurationParameter( IS_CLUSTER_PROP, ( lsSrv.isClusterInstallation() ) ? "Yes"
                                                                                         : "No" );
      conf.setConfigurationParameter( CACHING_TIMEOUT_PROP, Long.toString( lsSrv.getClusterCachingTimeout() ) + "ms" );
    
      LandscapeServiceFactory.release( lsSrv );
    }
    catch ( WcmException exc ) {
      // Landscape Service is not available. Trace the fact and set the monitored
      // properties to special values to indicate.
      conf.setConfigurationParameter( SYSTEM_ID_PROP, UNKNOWN_PROP );
      conf.setConfigurationParameter( IS_CLUSTER_PROP, UNKNOWN_PROP );
      conf.setConfigurationParameter( CACHING_TIMEOUT_PROP, UNKNOWN_PROP );
      log.errorT( METHOD, "LandscapeService not available, using default node id" );
    }
    
    return conf;
  }

  private static final String SYSTEM_INFO_ID = "com.sapportals.wcm.repository.runtime.CmSystemInfo";

  private static final String[] NO_IDS                = new String[0];
  private static final String   UNKNOWN_DESC          = "<unknown>",
                                UNKNOWN_STATE         = "<unknown>",
                                NO_ISSUE              = "None",
                                UNKNOWN_ISSUE         = "<unknown>",
                                NO_DATE               = "Never",
                                DEFAULT_LOCAL_NODE_ID = "<local>",
                                UNKNOWN_PROP          = "<unknown>";
  private static final String   SYSTEM_ID_PROP        = "System Id",
                                IS_CLUSTER_PROP       = "Is Within a Cluster",
                                CACHING_TIMEOUT_PROP  = "Cluster Caching Timeout";
                              
                                
  private static final Locale     LOCALE         = Locale.US; 
  private static final DateFormat DATE_FORMATTER = DateFormat.getDateTimeInstance( DateFormat.LONG, DateFormat.LONG, LOCALE );
        


  private Map mockManagerMap;

  /**
   * Force the RF core to support unit testing.
   */
  public void activateTestabilityMode() {
    if (CmSystem.testabilityMode) {
      return;
    }
    CmSystem.testabilityMode = true;
    this.mockManagerMap = new HashMap();    
    log.infoT("RF core testability mode is activated");         
  }

  public void registerMockManager(String prefix, Object o) {
    if (!CmSystem.testabilityMode) {
      throw new IllegalStateException("RF core testability mode is not activated");
    }
    if (prefix == null || o == null) {
      throw new NullPointerException();
    }    
    this.mockManagerMap.put(prefix, o);
  }
  
  public void unregisterMockManager(String prefix) {
    if (!CmSystem.testabilityMode) {
      throw new IllegalStateException("RF core testability mode is not activated");
    }
    if (prefix == null) {
      throw new NullPointerException();
    }
    this.mockManagerMap.remove(prefix); 
  }
  
  public Object getMockManager(String prefix) {
    if (!CmSystem.testabilityMode) {
      throw new IllegalStateException("RF core testability mode is not activated");
    }
    if (prefix == null) {
      throw new NullPointerException();
    } 
    return this.mockManagerMap.get(prefix);
  }
}
