/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.runtime;

import com.sap.tc.logging.Location;
import com.sapportals.config.fwk.IConfigUri;
import com.sapportals.wcm.crt.configuration.ConfigurationException;
import com.sapportals.wcm.crt.configuration.IConfiguration;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import java.util.*;

/**
 * The registry is used by the CmConfigurationProvider and CmSystem classes for
 * storing and updating the system properties of CM component. This class knows
 * about the system properties of all components and how to keep them
 * consistent. <p>
 *
 * Copyright (c) SAP AG 2002
 *
 * @author Markus Breitenfelder
 * @version $Id:$
 */
public final class CmRegistry {

  private static final Location TRACE = Location.getLocation(CmRegistry.class);

  /**
   * TBD: Description of the class.
   */
  public class IdListDiff {
    private List added = null;
    private List removed = null;

    public IdListDiff(List added, List removed) {
      this.added = added;
      this.removed = removed;
    }

    public List getAdded() {
      return this.added;
    }

    public List getRemoved() {
      return this.removed;
    }
  }


  /**
   * Maps: prefix -> manager ID
   */
  private HashMap mapPrefixManager = new HashMap();

  /**
   * Maps: manager ID + "/" + service type -> service ID
   */
  private HashMap mapRepSrvToRepository = new HashMap();

  /**
   * Map: manager ID -> List of content filter IDs
   */
  private HashMap mapContentFilter = new HashMap();

  /**
   * Map: manager ID -> List of property filter IDs
   */
  private HashMap mapPropertyFilter = new HashMap();

  /**
   * Map: manager ID -> List of namespace filter IDs
   */
  private HashMap mapNamespaceFilter = new HashMap();

  /**
   * List of URI filter IDs
   */
  private ArrayList listUriFilter = new ArrayList();

  /**
   * Map: filter ID -> CmFilterProperties
   */
  private HashMap mapFilterProperties = new HashMap();

  /**
   * Map: repository ID -> CmFilterProperties
   */
  private HashMap mapManagerProperties = new HashMap();

  /**
   * Map: rep. service ID -> CmRepServiceProperties
   */
  private HashMap mapRepServiceProperties = new HashMap();

  /**
   * Map: service ID -> CmServiceProperties
   */
  private HashMap mapServiceProperties = new HashMap();

  /**
   * Map: child config class name -> Collection of IConfigUri
   */
  private HashMap mapChildConfigService = new HashMap();
  private HashMap mapChildConfigManager = new HashMap();
  
  public static final CmRegistry registry = new CmRegistry();

  public static final CmRegistry getInstance() {
    return CmRegistry.registry;
  }

  private CmRegistry() { }
  
  public void clear() {
    mapPrefixManager = new HashMap();
    mapRepSrvToRepository = new HashMap();
    mapContentFilter = new HashMap();
    mapPropertyFilter = new HashMap();
    mapNamespaceFilter = new HashMap();
    listUriFilter = new ArrayList();
    mapFilterProperties = new HashMap();
    mapManagerProperties = new HashMap();
    mapRepServiceProperties = new HashMap();
    mapServiceProperties = new HashMap();
    mapChildConfigService = new HashMap();
    mapChildConfigManager = new HashMap();
  }
  
  /**
   * Adds a new repository. All the service IDs in the list must already exist.
   *
   * @param id repository to be added
   * @param mgrProperties repository to be added
   * @exception ConfigurationException If the specified identifier is already in
   *      use.
   */
  public synchronized void addRepository(String id, CmManagerProperties mgrProperties)
    throws ConfigurationException {
      
    if( TRACE.beDebug() ) {
      TRACE.debugT("addRepository( id = " + id + ", mgrProperties = " + mgrProperties + " )");
    }
    
    if (id == null) {
      throw new IllegalArgumentException("id is null");
    }
    if (mgrProperties == null) {
      throw new IllegalArgumentException("mgrProperties is null");
    }
    if (this.mapManagerProperties.containsKey(id)) {
      throw new ConfigurationException("The repository manager ID is already in use: " + id);
    }
    if (this.mapPrefixManager.containsKey(mgrProperties.getPrefix())) {
      throw new InvalidPrefixConfigurationException("The repository manager prefix \"" + mgrProperties.getPrefix() + "\" is already in use: " + id);
    }
    // Shallow clone of the maps (the properties objects are read-only)
    HashMap copy1 = (HashMap)this.mapManagerProperties.clone();
    copy1.put(id, mgrProperties);
    HashMap copy2 = (HashMap)this.mapPrefixManager.clone();
    copy2.put(mgrProperties.getPrefix(), id);
    HashMap copy3 = (HashMap)this.mapRepSrvToRepository.clone();
    // For each service in the list get the service type and add the mapping
    HashMap serviceMap = (HashMap)copy3.get(id);
    if( serviceMap == null ) serviceMap = new HashMap();
    List srvIDs = mgrProperties.getServiceIDs();
    for (int i = 0; i < srvIDs.size(); i++) {
      String srvID = (String)srvIDs.get(i);
      CmRepServiceProperties props = (CmRepServiceProperties)this.mapRepServiceProperties.get(srvID);
      if (props == null) {
        throw new ConfigurationException("Repository service ID not found: " + srvID);
      }
      serviceMap.put(props.getType(), srvID);
    }
    copy3.put(id, serviceMap);    
    addManagerChildConfigMapping(mgrProperties.getConfigUri(), mgrProperties.getChildConfigClassNames());
    this.mapManagerProperties = copy1;
    this.mapPrefixManager = copy2;
    this.mapRepSrvToRepository = copy3;
  }

  /**
   * Adds and/or removes services assignments to/from a repository.
   *
   * @param id The repository ID
   * @param serviceID The rep. service ID
   * @param addOrRemove true: assign service to repository, false: unassign
   *      service
   * @exception ConfigurationException Exception raised in failure situation
   */
  public synchronized void updateRepository(String id, String serviceID, boolean addOrRemove)
    throws ConfigurationException {
    
    if( TRACE.beDebug() ) {
      TRACE.debugT("updateRepository( id = " + id + ", serviceID = " + serviceID + " )");
    }

    if (id == null) {
      throw new IllegalArgumentException("id is null");
    }
    CmManagerProperties oldProps = (CmManagerProperties)this.mapManagerProperties.get(id);
    if (oldProps == null) {
      throw new ConfigurationException("The repository manager ID does not exist: " + id);
    }

    List oldServiceIDs = oldProps.getServiceIDs();
    ArrayList newServiceIDs = new ArrayList(oldServiceIDs.size());
    Iterator it = oldServiceIDs.iterator();
    while (it.hasNext()) {
      newServiceIDs.add(it.next());
    }
    HashMap copy2 = (HashMap)this.mapRepSrvToRepository.clone();

    CmRepServiceProperties srvProps = (CmRepServiceProperties)this.mapRepServiceProperties.get(serviceID);
    if (srvProps == null) {
      throw new ConfigurationException("Service id does not exist: " + serviceID);
    }

    if (!addOrRemove) {
      newServiceIDs.remove(serviceID);
      HashMap serviceMap = (HashMap)copy2.get(id);
      if( serviceMap != null ) {
        copy2.remove(srvProps.getType());
      }
    } else {
      newServiceIDs.add(serviceID);
      HashMap serviceMap = (HashMap)this.mapRepSrvToRepository.get(id);
      if(   ( serviceMap != null )
         && ( serviceMap.containsKey(srvProps.getType()) ) 
        ) {
        throw new ConfigurationException("The service " + serviceID + " is already assigned to repository " + id);
      } else {
        if( serviceMap == null ) serviceMap = new HashMap();
        serviceMap.put(srvProps.getType(), serviceID);
        this.mapRepSrvToRepository.put(id, serviceMap);
      }
    }
    this.mapManagerProperties.put(id, new CmManagerProperties(oldProps.getPrefix(), oldProps.getConfigUri(), newServiceIDs, oldProps.getChildConfigClassNames()));
    this.mapRepSrvToRepository = copy2;
  }


  /**
   * Removes the repository.
   *
   * @param id TBD: Description of the incoming method parameter
   */
  public synchronized void removeRepository(String id) {
    if( TRACE.beDebug() ) {
      TRACE.debugT("removeRepository( id = " + id + " )");
    }
        
    CmManagerProperties mgrProperties = (CmManagerProperties)this.mapManagerProperties.get(id);
    if (mgrProperties == null) {
      return;
    }
    this.mapPrefixManager.remove(mgrProperties.getPrefix());
    this.mapManagerProperties.remove(id);
    this.mapRepSrvToRepository.remove(id);
  }

  /**
   * Adds a new repository service.
   *
   * @param id repositoryService to be added
   * @param serviceProperties repositoryService to be added
   * @exception ConfigurationException If the specified identifier is already in
   *      use.
   */
  public synchronized void addRepositoryService(String id, CmRepServiceProperties serviceProperties)
    throws ConfigurationException {
    if (TRACE.beDebug()) {
      TRACE.debugT("addRepository( id = " + id + ", serviceProperties = " + serviceProperties + " )");
    }
            
    if (id == null) {
      throw new IllegalArgumentException("id is null");
    }
    if (serviceProperties == null) {
      throw new IllegalArgumentException("serviceProperties is null");
    }
    if (this.mapRepServiceProperties.containsKey(id)) {
      throw new ConfigurationException("The repository service ID is already in use: " + id);
    }
    this.addServiceChildConfigMapping(serviceProperties.getConfigUri(), serviceProperties.getChildConfigClassNames());
    this.mapRepServiceProperties.put(id, serviceProperties);
  }

  /**
   * Removes the repository service.
   *
   * @param id TBD: Description of the incoming method parameter
   */
  public synchronized void removeRepositoryService(String id) {
    throw new UnsupportedOperationException();
  }

  /**
   * Adds a new repository service
   *
   * @param id service to be added
   * @param serviceProperties service to be added
   * @exception ConfigurationException If the specified identifier is already in
   *      use.
   */
  public synchronized void addService(String id, CmServiceProperties serviceProperties)
    throws ConfigurationException {
    if (TRACE.beDebug()) {
      TRACE.debugT("addService( id = " + id + ", serviceProperties = " + serviceProperties + " )");
    }
      
    if (id == null) {
      throw new IllegalArgumentException("id is null");
    }
    if (serviceProperties == null) {
      throw new IllegalArgumentException("serviceProperties is null");
    }
    if (this.mapServiceProperties.containsKey(id)) {
      throw new ConfigurationException("The service ID is already in use: " + id);
    }
    this.addServiceChildConfigMapping(serviceProperties.getConfigUri(), serviceProperties.getChildConfigClassNames());
    this.mapServiceProperties.put(id, serviceProperties);
  }

  /**
   * Removes the service.
   *
   * @param id TBD: Description of the incoming method parameter
   */
  public synchronized void removeService(String id) {
    throw new UnsupportedOperationException();
  }

  /**
   * Adds a new filter.
   *
   * @param id filter to be added
   * @param fltProperties filter to be added
   * @exception ConfigurationException If the specified identifier is already in
   *      use.
   */
  public synchronized void addFilter(String id, CmFilterProperties fltProperties)
    throws ConfigurationException {
    if (TRACE.beDebug()) {
      TRACE.debugT("addFilter( id = " + id + ", fltProperties = " + fltProperties + " )");
    }
    
    if (id == null) {
      throw new IllegalArgumentException("id is null");
    }
    if (fltProperties == null) {
      throw new IllegalArgumentException("fltProperties is null");
    }
    if (this.mapFilterProperties.containsKey(id)) {
      throw new ConfigurationException("The filter manager ID is already in use: " + id);
    }
    HashMap copy = (HashMap)this.mapFilterProperties.clone();
    copy.put(id, fltProperties);
    List managers = fltProperties.getManagerIDs();
    this.addFilterIdToMap(id, fltProperties.getType(), managers, copy);
    this.mapFilterProperties = copy;
  }

  /**
   * Updates the system properties of a filter. This includes the assignment to
   * repositories.
   *
   * @param id TBD: Description of the incoming method parameter
   * @param fltProperties TBD: Description of the incoming method parameter
   * @exception ConfigurationException If the specified identifier was not
   *      found.
   */
  /*
   * IdListDiff
   */
  public synchronized void updateFilter(String id, CmFilterProperties fltProperties)
    throws ConfigurationException {
    if (TRACE.beDebug()) {
      TRACE.debugT("updateFilter( id = " + id + ", fltProperties = " + fltProperties + " )");
    }
            
    if (id == null) {
      throw new IllegalArgumentException("id is null");
    }
    if (fltProperties == null) {
      throw new IllegalArgumentException("fltProperties is null");
    }
    if (!this.mapFilterProperties.containsKey(id)) {
      throw new ConfigurationException("The filter does not exist: " + id);
    }
    CmFilterProperties props = (CmFilterProperties)this.mapFilterProperties.get(id);
    if (props == null) {
      throw new ConfigurationException("Filter manager does not exist: " + id);
    }

    // Diff the repository assignments and call updateFilterAssignment() for each entry in the diff object.
    IdListDiff diff = this.diffIdList(props.getManagerIDs(), fltProperties.getManagerIDs());
    List added = diff.getAdded();
    for (int i = 0; i < added.size(); i++) {
      this.updateFilterAssignment((String)added.get(i), id, true);
    }
    List removed = diff.getRemoved();
    for (int i = 0; i < removed.size(); i++) {
      this.updateFilterAssignment((String)removed.get(i), id, false);
    }

    HashMap copy = (HashMap)this.mapFilterProperties.clone();
    copy.put(id, fltProperties);
    this.mapFilterProperties = copy;
  }

  /**
   * Assign the filter to the repository. This adds or removes the manager ID to
   * the list in the CmFilterProperties.
   *
   * @param repositoryID TBD: Description of the incoming method parameter
   * @param filterID TBD: Description of the incoming method parameter
   * @param addRemoveFlag TBD: Description of the incoming method parameter
   * @exception ConfigurationException If the filter is already assigned to the
   *      specified repository.
   */
  public synchronized void updateFilterAssignment(String repositoryID, String filterID, boolean addRemoveFlag)
    throws ConfigurationException {
    if (TRACE.beDebug()) {
      TRACE.debugT("updateFilter( id = " + repositoryID + ", filterID = " + filterID + " )");
    }

    if (filterID == null) {
      throw new IllegalArgumentException("filterID is null");
    }     
    if (repositoryID == null) {
      // It is a URI filter...
      if (!addRemoveFlag) {
        ArrayList copy = (ArrayList)this.listUriFilter.clone();       
        copy.remove(filterID);
        this.listUriFilter = copy;
      }
    }
    else {
      CmManagerProperties mgrProperties = (CmManagerProperties)this.mapManagerProperties.get(repositoryID);
      if (mgrProperties == null) {
        throw new ConfigurationException("The repository ID does not exist: " + repositoryID);
      }
      CmFilterProperties fltProperties = (CmFilterProperties)this.mapFilterProperties.get(filterID);
      if (fltProperties == null) {
        throw new ConfigurationException("The filter manager ID does not exist: " + filterID);
      }
      if (addRemoveFlag) {
        // Add
        ArrayList managers = new ArrayList();
        managers.add(repositoryID);
        this.addFilterIdToMap(filterID, fltProperties.getType(), managers, this.mapFilterProperties);
      }
      else {
        // Remove
        this.removeFilterIdFromMap(filterID, fltProperties.getType(), repositoryID, this.mapFilterProperties);
      }
    }
  }

  /**
   * Removes the filter.
   *
   * @param id TBD: Description of the incoming method parameter
   */
  public synchronized void removeFilter(String id) {
    if (TRACE.beDebug()) {
      TRACE.debugT("removeFilter( id = " + id + " )");
    }
        
    if (id == null) {
      throw new IllegalArgumentException("id is null");
    }
    CmFilterProperties fltProperties = (CmFilterProperties)mapFilterProperties.get(id);
    if (fltProperties == null) {
      return;
    }
    HashMap copy = (HashMap)this.mapFilterProperties.clone();
    // Remove all assignments from the maps
    List managerIDs = fltProperties.getManagerIDs();
    for (int i = 0; i < managerIDs.size(); i++) {
      try {
        this.updateFilterAssignment((String)managerIDs.get(i), id, false);
      }
      catch (ConfigurationException ex) {
        CmRegistry.TRACE.errorT("removeFilter(445)", LoggingFormatter.extractCallstack(ex));
      }
    }
    copy.remove(id);
    this.mapFilterProperties = copy;
  }
  // ---------------------------------------------------------------------------

  /**
   * Returns a collection of all repository IDs. The collection might be empty
   * but is not <code>null</code> .
   *
   * @return allRepositoryIDs
   */
  public Collection getAllRepositoryIDs() {
    return Collections.unmodifiableCollection(this.mapManagerProperties.keySet());
  }

  /**
   * Returns the ID of the repository for the specified prefix (or <code>null
   * </code>).
   *
   * @param prefix TBD: Description of the incoming method parameter
   * @return repositoryID
   */
  public String getRepositoryID(String prefix) {
    return (String)this.mapPrefixManager.get(prefix);
  }

  /**
   * Returns the service ID or <code>null</code> .
   *
   * @param repositoryID TBD: Description of the incoming method parameter
   * @param serviceType TBD: Description of the incoming method parameter
   * @return repositoryServiceID
   */
  public String getRepositoryServiceID(String repositoryID, String serviceType) {
    HashMap services = (HashMap)this.mapRepSrvToRepository.get(repositoryID);
    if( services == null ) return null;
    return (String)services.get(serviceType);
  }

  /**
   * Returns a list of service IDs for the specified repository ID. The returned
   * reference is <code>null</code> if the repository ID does not exist.
   *
   * @param repositoryID TBD: Description of the incoming method parameter
   * @return allRepositoryServiceIDs
   */
  public Collection getAllRepositoryServiceIDs(String repositoryID) {
    CmManagerProperties mgrProps = (CmManagerProperties)this.mapManagerProperties.get(repositoryID);
    if (mgrProps == null) {
      return null;
    }
    return mgrProps.getServiceIDs();
  }

  /**
   * Returns a list of filter manager IDs for the specified repository and
   * filter type. The filter IDs are sorted in priority order. The result list
   * might be empty but is not <code>null</code> .
   *
   * @param repositoryID The repository ID or <code>null</code> if the filter
   *      type is URI.
   * @param type The filter type: {@link CmFilterProperties.TYPE}
   * @return A list of filter IDs.
   */
  public List getFilterList(String repositoryID, int type) {
    List l = null;
    switch (type) {
      case CmFilterProperties.TYPE.CONTENT:
        l = (List)this.mapContentFilter.get(repositoryID);
        break;
      case CmFilterProperties.TYPE.NAMESPACE:
        l = (List)this.mapNamespaceFilter.get(repositoryID);
        break;
      case CmFilterProperties.TYPE.PROPERTY:
        l = (List)this.mapPropertyFilter.get(repositoryID);
        break;
      case CmFilterProperties.TYPE.URI:
        l = this.listUriFilter;
        break;
      default:
        throw new IllegalArgumentException("Unknown filter type");
    }
    if (l != null) {
      return Collections.unmodifiableList(l);
    }
    else {
      return Collections.EMPTY_LIST;
    }
  }

  /**
   * Returns the properties of the filter manager or <code>null</code> .
   *
   * @param id TBD: Description of the incoming method parameter
   * @return filterProperties
   */
  public CmFilterProperties getFilterProperties(String id) {
    return (CmFilterProperties)this.mapFilterProperties.get(id);
  }

  /**
   * Returns the properties of the global service or <code>null</code> .
   *
   * @param id TBD: Description of the incoming method parameter
   * @return serviceProperties
   */
  public CmServiceProperties getServiceProperties(String id) {
    return (CmServiceProperties)this.mapServiceProperties.get(id);
  }

  /**
   * Returns all global service IDs.
   * @return all global service IDs.
   */
  public Collection getAllServiceIDs() {
    return Collections.unmodifiableCollection(this.mapServiceProperties.keySet());
  }

  public CmManagerProperties getManagerProperties(String id) {
    return (CmManagerProperties)this.mapManagerProperties.get(id);
  }
  
  
  /**
   * Returns the properties of the repository service or <code>null</code> .
   *
   * @param id TBD: Description of the incoming method parameter
   * @return repositoryServiceProperties
   */
  public CmRepServiceProperties getRepositoryServiceProperties(String id) {
    return (CmRepServiceProperties)this.mapRepServiceProperties.get(id);
  }

  /**
   * Returns a colleciton of IConfigUri instances for a child component config
   * class name.
   *
   * @param childConfigClassName TBD: Description of the incoming method
   *      parameter
   * @return containerServiceConfigURIs
   */
  public Collection getContainerServiceConfigURIs(String childConfigClassName) {
    Collection c = (Collection)this.mapChildConfigService.get(childConfigClassName);
    if (c == null) {
      return Collections.EMPTY_LIST;
    }
    else {
      return Collections.unmodifiableCollection(c);
    }
  }

  public Collection getContainerManagerConfigURIs(String childConfigClassName) {
    Collection c = (Collection)this.mapChildConfigManager.get(childConfigClassName);
    if (c == null) {
      return Collections.EMPTY_LIST;
    }
    else {
      return Collections.unmodifiableCollection(c);
    }
  }
  
  // TODO: This is a workaround (see CmConfigurationProvider)
  public IConfigUri[] selectContainerServiceConfigURIs(String childConfigClassPath) {
    childConfigClassPath += '/';
    Iterator it = this.mapChildConfigService.keySet().iterator();
    while (it.hasNext()) {
      String key = (String)it.next();
      if (key.startsWith(childConfigClassPath)) {
        Collection c = (Collection)this.mapChildConfigService.get(key);
        return (IConfigUri[])c.toArray(new IConfigUri[c.size()]);
      }
    }
    return null;
  }

  // TODO: This is a workaround (see CmConfigurationProvider)
  public IConfigUri[] selectContainerManagerConfigURIs(String childConfigClassPath) {
    childConfigClassPath += '/';
    Iterator it = this.mapChildConfigManager.keySet().iterator();
    while (it.hasNext()) {
      String key = (String)it.next();
      if (key.startsWith(childConfigClassPath)) {
        Collection c = (Collection)this.mapChildConfigManager.get(key);
        return (IConfigUri[])c.toArray(new IConfigUri[c.size()]);
      }
    }
    return null;
  }
  
  // ---------------------------------------------------------------------------

  /**
   * Add the filter ID to the list of filters for all repositories and sort by
   * priority.
   *
   * @param filterID filterIdToMap to be added
   * @param type filterIdToMap to be added
   * @param managerIDs filterIdToMap to be added
   * @param mapProps filterIdToMap to be added
   * @exception ConfigurationException If the filter is already assigned to one
   *      of the repositories.
   */
  private void addFilterIdToMap(String filterID, int type, List managerIDs, HashMap mapProps)
    throws ConfigurationException {
    HashMap map = null;
    switch (type) {
      case CmFilterProperties.TYPE.CONTENT:
        map = this.mapContentFilter;
        break;
      case CmFilterProperties.TYPE.NAMESPACE:
        map = this.mapNamespaceFilter;
        break;
      case CmFilterProperties.TYPE.PROPERTY:
        map = mapPropertyFilter;
        break;
      case CmFilterProperties.TYPE.URI:
        if (this.listUriFilter.contains(filterID)) {
          throw new ConfigurationException("The filter manager ID does already exist: " + filterID);
        }
        ArrayList copy = (ArrayList)this.listUriFilter.clone();
        copy.add(filterID);
        Collections.sort(copy, new FilterPrioComparator(mapProps));
        this.listUriFilter = copy;
        return;
      default:
        throw new IllegalArgumentException("Unknown filter type");
    }
    HashMap mapCopy = (HashMap)map.clone();
    for (int i = 0; i < managerIDs.size(); i++) {
      String mgrID = (String)managerIDs.get(i);
      ArrayList filterIDs = (ArrayList)mapCopy.get(mgrID);
      if (filterIDs == null) {
        filterIDs = new ArrayList();
        filterIDs.add(filterID);
        mapCopy.put(mgrID, filterIDs);
      }
      else {
        if (filterIDs.contains(filterID)) {
          throw new ConfigurationException("The filter manager ID " + filterID + " is already assigned to manager " + mgrID);
        }
        ArrayList copy = (ArrayList)filterIDs.clone();
        copy.add(filterID);
        mapCopy.put(mgrID, copy);
        Collections.sort(copy, new FilterPrioComparator(mapProps));
      }
    }
    switch (type) {
      case CmFilterProperties.TYPE.CONTENT:
        this.mapContentFilter = mapCopy;
        break;
      case CmFilterProperties.TYPE.NAMESPACE:
        this.mapNamespaceFilter = mapCopy;
        break;
      case CmFilterProperties.TYPE.PROPERTY:
        mapPropertyFilter = mapCopy;
    }
  }

  private void removeFilterIdFromMap(String filterID, int type, String managerID, HashMap mapProps) {
    HashMap map = null;
    switch (type) {
      case CmFilterProperties.TYPE.CONTENT:
        map = this.mapContentFilter;
        break;
      case CmFilterProperties.TYPE.NAMESPACE:
        map = this.mapNamespaceFilter;
        break;
      case CmFilterProperties.TYPE.PROPERTY:
        map = mapPropertyFilter;
        break;
      case CmFilterProperties.TYPE.URI:
        ArrayList copy = (ArrayList)this.listUriFilter.clone();
        copy.remove(filterID);
        Collections.sort(copy, new FilterPrioComparator(mapProps));
        this.listUriFilter = copy;
        return;
      default:
        throw new IllegalArgumentException("Unknown filter type");
    }
    HashMap mapCopy = (HashMap)map.clone();
    ArrayList filterIDs = (ArrayList)mapCopy.get(managerID);
    if (filterIDs != null) {
      ArrayList copy = (ArrayList)filterIDs.clone();
      copy.remove(filterID);
      mapCopy.put(managerID, copy);
      Collections.sort(copy, new FilterPrioComparator(mapProps));
    }
    switch (type) {
      case CmFilterProperties.TYPE.CONTENT:
        this.mapContentFilter = mapCopy;
        break;
      case CmFilterProperties.TYPE.NAMESPACE:
        this.mapNamespaceFilter = mapCopy;
        break;
      case CmFilterProperties.TYPE.PROPERTY:
        mapPropertyFilter = mapCopy;
    }
  }


  /**
   * Diffs a list of Strings (IDs).
   *
   * @param oldList TBD: Description of the incoming method parameter
   * @param newList TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  private IdListDiff diffIdList(List oldList, List newList) {
    ArrayList added = new ArrayList();
    ArrayList removed = new ArrayList();

    for (int i = 0; i < oldList.size(); i++) {
      if (!newList.contains(oldList.get(i))) {
        removed.add(oldList.get(i));
      }
    }
    for (int i = 0; i < newList.size(); i++) {
      if (!oldList.contains(newList.get(i))) {
        added.add(newList.get(i));
      }
    }

    return new IdListDiff(added, removed);
  }

  /**
   * Compare prio of two filterIDs
   */
  private class FilterPrioComparator implements java.util.Comparator {
    private final Map mapFilterProperties;

    public FilterPrioComparator(Map mapFilterProperties) {
      this.mapFilterProperties = mapFilterProperties;
    }

    public int compare(Object o1, Object o2) {
      CmFilterProperties props1 = (CmFilterProperties)this.mapFilterProperties.get((String)o1);
      CmFilterProperties props2 = (CmFilterProperties)this.mapFilterProperties.get((String)o2);
      Integer prio1 = (Integer)props1.getPriority();
      Integer prio2 = (Integer)props2.getPriority();
      return prio1.compareTo(prio2);
    }
  }


  private void addServiceChildConfigMapping(IConfigUri configURI, String[] ccNames)
    throws ConfigurationException {
    for (int i = 0; i < ccNames.length; i++) {
      Collection serviceIDList = (Collection)this.mapChildConfigService.get(ccNames[i]);
      if (serviceIDList == null) {
        serviceIDList = new LinkedList();
        this.mapChildConfigService.put(ccNames[i], serviceIDList);
      }

      // Service ID already in list ?
      if (serviceIDList.contains(configURI)) {
        // TODO: Remove/Undo all mappings already added ? (Do changes for a cloned map)
        throw new ConfigurationException("The child config class " + ccNames[i] + " is already referenced by service" + (String)this.mapChildConfigService.get(ccNames[i]));
      }
      else {
        serviceIDList.add(configURI);
      }
    }
  }
  
  private void addManagerChildConfigMapping(IConfigUri configURI, String[] ccNames)
    throws ConfigurationException {
    for (int i = 0; i < ccNames.length; i++) {
      Collection idList = (Collection)this.mapChildConfigService.get(ccNames[i]);
      if (idList == null) {
        idList = new LinkedList();
        this.mapChildConfigManager.put(ccNames[i], idList);
      }

      // Service ID already in list ?
      if (idList.contains(configURI)) {
        // TODO: Remove/Undo all mappings already added ? (Do changes for a cloned map)
        throw new ConfigurationException("The child config class " + ccNames[i] + " is already referenced by service" + (String)this.mapChildConfigService.get(ccNames[i]));
      }
      else {
        idList.add(configURI);
      }
    }
  }
}
