/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.runtime;
import com.sapportals.wcm.crt.configuration.ConfigurationException;
import com.sapportals.wcm.repository.filter.*;

import com.sapportals.wcm.util.regex.PathPatternMatcher;
import java.util.*;

/**
 * A read-only object for the system properties of a filter manager.
 */
public final class CmFilterProperties {
  /**
   * TBD: Description of the class.
   */
  public static interface TYPE {
    public final static int CONTENT = 0;
    public final static int PROPERTY = 1;
    public final static int NAMESPACE = 2;
    public final static int URI = 3;
  }

  private Integer priority;
  private String prefix;
  private PathPatternMatcher mimePattern;
  private PathPatternMatcher extPattern;
  private PathPatternMatcher pathPattern;
  private PathPatternMatcher rTypePattern;
  private List listManagerIDs;
  private int type;

  // Cache instance for performance reasons (all filter managers a IThreadSafe!)
  private IFilterManager mgrInstance;

  public CmFilterProperties(
    Integer priority, String prefix,
    PathPatternMatcher mimePattern, PathPatternMatcher extPattern,
    PathPatternMatcher pathPattern, PathPatternMatcher rTypePattern,
    ArrayList managerIDs, int type
    ) {
    this.priority = priority;
    this.prefix = prefix;
    this.mimePattern = mimePattern;
    this.extPattern = extPattern;
    this.pathPattern = pathPattern;
    this.rTypePattern = rTypePattern;
    this.listManagerIDs = managerIDs;
    this.type = type;
  }

  public Integer getPriority() {
    return this.priority;
  }

  public String getPrefix() {
    return this.prefix;
  }

  public PathPatternMatcher getMimePattern() {
    return this.mimePattern;
  }

  public PathPatternMatcher getExtPattern() {
    return this.extPattern;
  }

  public PathPatternMatcher getPathPattern() {
    return this.pathPattern;
  }

  public PathPatternMatcher getResourceTypePattern() {
    return this.rTypePattern;
  }

  public int getType() {
    return this.type;
  }

  public List getManagerIDs() {
    return Collections.unmodifiableList(this.listManagerIDs);
  }


  public IFilterManager getFilterManagerInstance() {
    return this.mgrInstance;
  }

  public void setFilterManagerInstance(IFilterManager mgrInstance) {
    this.mgrInstance = mgrInstance;
  }

  public static int getFilterType(Class filterClass)
    throws ConfigurationException {
    if (IContentFilterManager.class.isAssignableFrom(filterClass)) {
      return CmFilterProperties.TYPE.CONTENT;
    }
    else if (IPropertyFilterManager.class.isAssignableFrom(filterClass)) {
      return CmFilterProperties.TYPE.PROPERTY;
    }
    else if (INamespaceFilterManager.class.isAssignableFrom(filterClass)) {
      return CmFilterProperties.TYPE.NAMESPACE;
    }
    else if (IUriFilterManager.class.isAssignableFrom(filterClass)) {
      return CmFilterProperties.TYPE.URI;
    }
    else if (IRidFilterManager.class.isAssignableFrom(filterClass)) {
      return CmFilterProperties.TYPE.URI;
    }
    else {
      throw new ConfigurationException("Filter class does not implement one of the required interfaces: " + filterClass);
    }
  }

  public String toString() {
    return "Filter properties: priority=" + this.priority + ", prefix=" + this.prefix + ", mime pattern="
       + this.mimePattern + ", extension pattern=" + this.extPattern + ", path pattern=" + this.pathPattern
       + ", managers=" + this.listManagerIDs + ", type=" + this.type;
  }
}
