/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.runtime;
import com.sap.tc.logging.Location;

import com.sapportals.wcm.*;
import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.repository.enum.NamespaceFilterMode;
import com.sapportals.wcm.repository.enum.PropertyFilterMode;
import com.sapportals.wcm.repository.filter.*;
import com.sapportals.wcm.repository.manager.*;
import com.sapportals.wcm.util.content.*;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.regex.*;
import com.sapportals.wcm.util.uri.RID;
import com.sapportals.wcm.util.uri.URI;
import java.io.*;
import java.util.*;

/**
 * Handles all filters. Used by resource and collection implementation. <p>
 *
 * Copyright (c) SAP AG 2002
 *
 * @author Markus Breitenfelder
 * @version $Id:$
 */
public class CmFilterHandler {

  private static Location log = Location.getLocation(com.sapportals.wcm.repository.runtime.CmFilterHandler.class);
  public static boolean KMC_SYS_PROP_FILTER =
    ("true".equals(System.getProperty("kmc_sys_prop_filter", "false"))) ? true : false;
  private final static String FF_NAMESPACE = IWcmConst.SAP_WCM_NAMESPACE + "/CmFilterHandler/";
  private final static String FLAG = "flag";
  private CmConfigurationProvider provider = null;

  private Map pNames = new HashMap(); // repository manager ID -> Set of IPropertyNames instances

  CmFilterHandler(CmConfigurationProvider provider) {
    this.provider = provider;
  }

  public IContent applyContentReadFilter(IRepositoryManager repMgr, IResource resource, IContent filterContent, boolean readOnly)
    throws ResourceException {
    return this.applyContentFilter(repMgr, resource, null, filterContent, false, readOnly);
  }

  public IContent applyContentWriteFilterOnUpdate(
    IRepositoryManager repMgr,
    IResource resource,
    IContent filterContent)
    throws ResourceException {
    return this.applyContentFilter(repMgr, resource, null, filterContent, true, false);
  }

  public IContent applyContentWriteFilterOnCreate(
    IRepositoryManager repMgr,
    String resourceURI,
    IContent filterContent)
    throws ResourceException {
    return this.applyContentFilter(repMgr, null, resourceURI, filterContent, true, false);
  }

  public IPropertyMap applyPropertyReadFilter(IRepositoryManager repMgr, IResource resource, IPropertyMap properties)
    throws ResourceException {
    IPropertyMap filterPropMap = properties;
    IMutablePropertyMap systemProps = null;
    boolean listCopy = false;

    if (!KMC_SYS_PROP_FILTER) { // filtering of system properties is not allowed
      IPropertyIterator iter = properties.iterator();
      while ((!listCopy) && (iter.hasNext())) {
        IProperty prop = iter.next();

        if (PropertyNameTool.checkIfSystemPropertyName(prop.getPropertyName())) {
          if (!listCopy) {
            systemProps = new MutablePropertyMap();
            filterPropMap = PropertyNameTool.copyPropertyMapWithoutSystemPropertyNames(properties, systemProps);
            listCopy = true;
          }
        }
      }
    }

    if (!listCopy) {
      return this.applyPropertyFilter(
        repMgr,
        resource,
        null,
        properties,
        null,
        false,
        PropertyFilterMode.ALL_PROPERTIES);
    }
    else {
      IMutablePropertyMap filteredPropMap =
        this
          .applyPropertyFilter(repMgr, resource, null, properties, null, false, PropertyFilterMode.ALL_PROPERTIES)
          .getMutable();

      // add again all system properties
      if (null != systemProps) {
        IPropertyIterator iter = systemProps.iterator();
        while (iter.hasNext()) {
          filteredPropMap.put(iter.next());
        }
      }
      return filteredPropMap.getImmutable();
    }
  }

  public IPropertyMap applyPropertyReadFilter(
    IRepositoryManager repMgr,
    IResource resource,
    IPropertyNameList propNameList,
    IPropertyMap properties)
    throws ResourceException {
    IPropertyNameList filterPropNameList = PropertyNameTool.getSmuggledListWithoutSystemPropertyNames(propNameList);

    if (filterPropNameList != null) {
      return this.applyPropertyFilter(
        repMgr,
        resource,
        null,
        properties,
        filterPropNameList,
        false,
        PropertyFilterMode.PROPERTY_LIST);
    }
    else {
      return properties;
    }

  }

  public IProperty applyPropertyReadFilter(
    IRepositoryManager repMgr,
    IResource resource,
    IProperty prop,
    IPropertyName propName)
    throws ResourceException {
    if (!KMC_SYS_PROP_FILTER) { // filtering of system properties is not allowed
      if (PropertyNameTool.checkIfSystemPropertyName(propName))
        return prop;
    }

    IMutablePropertyMap mutPropMap = new MutablePropertyMap();
    IPropertyNameList propNameList = new PropertyNameList();
    IPropertyMap propMap;
    if (prop != null) {
      mutPropMap.put(prop);
    }
    if (propName != null) {
      propNameList.add(propName);
    }
    propMap =
      this.applyPropertyFilter(
        repMgr,
        resource,
        null,
        mutPropMap,
        propNameList,
        false,
        PropertyFilterMode.SINGLE_PROPERTY);
    return propMap.get(propName);
  }

  public IPropertyMap applyPropertyWriteFilterOnUpdate(
    IRepositoryManager repMgr,
    IResource resource,
    IPropertyMap properties,
    PropertyFilterMode mode)
    throws ResourceException {

    IPropertyNameList propNameList = new PropertyNameList();
    if (properties != null) {
      IPropertyIterator it = properties.iterator();
      while (it.hasNext()) {
        propNameList.add(it.next().getPropertyName());
      }
    }
    return this.applyPropertyFilter(repMgr, resource, null, properties, propNameList, true, mode);
  }

  public IPropertyMap applyPropertyWriteFilterOnCreate(
    IRepositoryManager repMgr,
    String resourceURI,
    IPropertyMap properties,
    PropertyFilterMode mode)
    throws ResourceException {

    IPropertyNameList propNameList = new PropertyNameList();
    if (properties != null) {
      IPropertyIterator it = properties.iterator();
      while (it.hasNext()) {
        propNameList.add(it.next().getPropertyName());
      }
    }
    return this.applyPropertyFilter(repMgr, null, resourceURI, properties, propNameList, true, mode);
  }

  public IResourceList applyNamespaceReadFilter(
    IRepositoryManager repMgr,
    ICollection collection,
    IResourceList resources,
    NamespaceFilterMode mode)
    throws ResourceException {
    return applyNamespaceFilter(repMgr, collection, resources, false, mode);
  }

  public RID applyUriFilter(RID rid, IResourceContext context) throws ResourceException {
    if (rid == null) {
      return null;
    }
    if (CmRegistry.registry.getFilterList(null, CmFilterProperties.TYPE.URI).size() == 0) {
      return rid;
    }
    // Avoid calling this method again during filter processing
    String flag = context.getValue(FF_NAMESPACE + FLAG);
    if (flag != null && flag.length() > 0) {
      return rid;
    }
    else {
      context.setValue(FF_NAMESPACE + FLAG, "X");
    }
    // Apply all filters in priority order
    try {
      List filterIDList = CmRegistry.registry.getFilterList(null, CmFilterProperties.TYPE.URI);
      //boolean changed = false;
      RID filteredRID = rid;
      for (int i = 0; i < filterIDList.size(); i++) {
        String filterID = (String)filterIDList.get(i);
        CmFilterProperties filterProps = (CmFilterProperties)CmRegistry.registry.getFilterProperties(filterID);
        if (filterProps == null) {
          continue;
        }
        if (!this.prefixMatches(filterProps.getPrefix(), filteredRID)) {
          continue;
        }

        Object fm = this.getCachedFilterManager(filterID, filterProps);
        if (fm instanceof IUriFilterManager) {
          // Deprecated interfaces IUriFilterManager, IUriFilter
          IUriFilterManager fltInstance = (IUriFilterManager)fm;
          if (fltInstance != null) {
            URI tempURI = new URI(filteredRID.toString());
            IUriFilter filter = fltInstance.getFilter(tempURI, context);
            if (filter != null) {
              tempURI = filter.filter();
              if (CmFilterHandler.log.beDebug()) {
                CmFilterHandler.log.debugT(
                  "applyUriFilter(183)",
                  "Applied RID filter " + filterID + ": result=" + tempURI);
              }
              filteredRID = RID.getRID(tempURI.toString());
            }
          }
        }
        else {
          // The new IRidFilterManager, IRidFilter interfaces
          IRidFilterManager fltInstance = (IRidFilterManager)fm;
          if (fltInstance != null) {
            IRidFilter filter = fltInstance.getFilter(filteredRID, context);
            if (filter != null) {
              filteredRID = filter.filter();
              if (CmFilterHandler.log.beDebug()) {
                CmFilterHandler.log.debugT(
                  "applyUriFilter(197)",
                  "Applied RID filter " + filterID + ": result=" + filteredRID);
              }
            }
          }
        }
      } // for

      // Clear flag in resource context
      context.setValue(FF_NAMESPACE + FLAG, "");
      return filteredRID;
    }
    catch (ResourceException ex) {
      throw ex;
    }
    catch (Exception ex) {
      throw new ResourceException(ex);
    }
  }

  public int getNamespaceFilterCount(String repositoryManagerID) {
    if (repositoryManagerID == null) {
      throw new java.lang.NullPointerException("Parameter repositoryManagerID is null");
    }
    List l = (List)CmRegistry.registry.getFilterList(repositoryManagerID, CmFilterProperties.TYPE.NAMESPACE);
    if (l == null) {
      return 0;
    }
    else {
      return l.size();
    }
  }

  public int getPropertyFilterCount(String repositoryManagerID) {
    if (repositoryManagerID == null) {
      throw new java.lang.NullPointerException("Parameter repositoryManagerID is null");
    }
    List l = (List)CmRegistry.registry.getFilterList(repositoryManagerID, CmFilterProperties.TYPE.PROPERTY);
    if (l == null) {
      return 0;
    }
    else {
      return l.size();
    }
  }

  public Set getFilteredPropertyNames(String repID) {
    return (Set)this.pNames.get(repID);
  }

  /**
   * Apply matching rule for URI filter prefix: Empty prefix or slash matches
   * all URIs !
   */
  private boolean prefixMatches(String prefix, RID uri) {
    if (prefix == null || prefix.length() == 0 || prefix.equals("/")) {
      return true;
    }
    else {
      return this.getRidPrefix(uri).getPath().equals(prefix);
    }
  }

  /**
   * Workaround for RID.root() behaviour: "/abc" -> root = "/"
   * @return ridPrefix
   */
  private RID getRidPrefix(RID uri) {
    RID r = uri.root();
    if (r.isRoot() && !uri.isRoot()) {
      return uri;
    }
    else {
      return r;
    }
  }

  private IContent applyContentFilter(
    IRepositoryManager repMgr,
    IResource resource,
    String resourceURI,
    IContent content,
    boolean forWrite,
    boolean readOnly)
    throws ResourceException {
    if (content == null) {
      return null;
    }
    try {
      String mgrID = repMgr.getID();
      // Init. filter that provides the initial content
      IContentFilter filter = null;
      if (forWrite && resource == null) {
        filter = new NullContentFilter(new URI(resourceURI), repMgr, content);
      }
      else {
        filter = new NullContentFilter(resource, content);
      }
      IContentFilter dummy = filter;
      // Check resource context for "ContentFilterID" from query parameter
      String filterIDInContext = null;
      if (resource != null) {
        filterIDInContext = resource.getContext().getValue(CmAdapter.CONTEXT_CONTENTFILTERID);
      }
      if (filterIDInContext != null) {
        // In this case only one filter will be applied (no cascading) and no configuration
        // parameters must match (manager, path, etc.)
        IContentFilterManager fltMgr =
          (IContentFilterManager)CmAdapter.getInstance().getFilterManager(filterIDInContext);
        if (forWrite) {
          filter = fltMgr.getFilterForWrite(filter);
        }
        else {
          filter = fltMgr.getFilterForRead(filter);
        }
        if (filter == null) {
          throw new WcmException("Filter manager " + filterIDInContext + " returned null");
        }
      }
      else {
        // Get all filters IDs for this rep. manager
        List filterIDList = (List)CmRegistry.registry.getFilterList(mgrID, CmFilterProperties.TYPE.CONTENT);
        if (filterIDList == null || filterIDList.size() == 0) {
          return content;
        }
        // Get mimetype, extension and path of the resource
        String mimeType = content.getContentType();
        if (mimeType == null) {
          mimeType = "";
        }
        String uri = null;
        if (resource != null) {
          uri = resource.getRID().toString();
          // Remove prefix from uri - A filter can be registered for more than one manager
          String prefix = resource.getRepositoryManager().getPrefix();
          uri = uri.substring(prefix.length());
        }
        else {
          uri = resourceURI;
        }
        if (uri == null) {
          throw new ResourceException("Invalid filter call: resource and resourceURI parameter was null");
        }
        String ext = RID.getRID(uri).extension();
        // Build the chain of filter managers
        for (int i = 0; i < filterIDList.size(); i++) {
          String filterID = (String)filterIDList.get(i);
          CmFilterProperties filterProps = CmRegistry.registry.getFilterProperties(filterID);
          if (filterProps == null) {
            continue;
          }

          if (this.matchPatterns(filterProps, uri, mimeType, ext) && this.matchResourceType(filterProps, resource)) {
            IContentFilterManager fltMgr = (IContentFilterManager)this.getCachedFilterManager(filterID, filterProps);
            if (fltMgr != null) {
              IContentFilter tmp = null;     
              if (forWrite) {
                tmp = fltMgr.getFilterForWrite(filter);
              }
              else {
                tmp = fltMgr.getFilterForRead(filter);
              }
              if (tmp != null && readOnly && !(tmp instanceof IReadOnlyContentFilter)) {
                continue;
              } 
              else {
                filter = tmp;
              }              
            }
            else {
              CmFilterHandler.log.errorT("applyContentFilter(368)", "getFilterManager() returned null: " + filterID);
            }
            if (filter == null) {
              throw new WcmException("Filter manager " + filterID + " returned null");
            }
          }
        }
      }
      if (filter != null && filter != dummy) {
        // Return the filter: Filter process will start when the client calls an IContent method !
        return filter;
      }
      else {
        return content;
      }
    }
    catch (ResourceException ex) {
      throw ex;
    }
    catch (WcmException ex) {
      throw new ResourceException(ex);
    }
    catch (Exception ex) {
      this.handleRuntimeException(ex);
      return null;
    }
  }

  private IPropertyMap applyPropertyFilter(
    IRepositoryManager repMgr,
    IResource resource,
    String resourceURI,
    IPropertyMap properties,
    IPropertyNameList propNameList,
    boolean forWrite,
    PropertyFilterMode mode)
    throws ResourceException {
    if (properties == null) {
      return null;
    }
    try {
      String mgrID = repMgr.getID();
      // Get all filters IDs for this rep. manager
      List filterIDList = (List)CmRegistry.registry.getFilterList(mgrID, CmFilterProperties.TYPE.PROPERTY);
      if (filterIDList == null || filterIDList.size() == 0) {
        return properties;
      }
      if (this.pNames.get(mgrID) == null) {
        this.collectPropertyNamesForRead(mgrID, filterIDList);
      }

      // Get extension and path of the resource
      String uri = null;
      if (resource != null) {
        uri = resource.getRID().toString();
        // Remove prefix from uri - A filter can be registered for more than one manager
        String prefix = resource.getRepositoryManager().getPrefix();
        uri = uri.substring(prefix.length());
      }
      else {
        uri = resourceURI;
      }
      if (uri == null) {
        throw new ResourceException("Invalid filter call: resource and resourceURI parameter was null");
      }
      String ext = RID.getRID(uri).extension();
      IPropertyFilter nullFilter =
        new NullPropertyFilter(
          resource,
          new URI(resourceURI),
          repMgr,
          properties,
          propNameList,
          new Properties(),
          mode);
      IPropertyFilter filter = nullFilter;
      // Build the chain of filter managers
      for (int i = 0; i < filterIDList.size(); i++) {
        String filterID = (String)filterIDList.get(i);
        CmFilterProperties filterProps = CmRegistry.registry.getFilterProperties(filterID);
        if (filterProps == null) {
          continue;
        }

        if (this.matchPatterns(filterProps, uri, null, ext) && this.matchResourceType(filterProps, resource)) {
          IPropertyFilterManager fltMgr = (IPropertyFilterManager)this.getCachedFilterManager(filterID, filterProps);
          if (fltMgr != null) {
            if (forWrite) {
              filter = fltMgr.getFilterForWrite(filter);
            }
            else {
              filter = fltMgr.getFilterForRead(filter);
            }
          }
          else {
            CmFilterHandler.log.errorT("applyPropertyFilter(461)", "getFilterManager() returned null: " + filterID);
          }
          if (filter == null) {
            throw new WcmException("Filter manager " + filterID + " returned null");
          }
        }
      }

      if (filter != nullFilter && filter != null) {
        // Apply all filters by calling filter() on the last filter in the "chain"
        if (CmFilterHandler.log.beDebug()) {
          CmFilterHandler.log.debugT("applyPropertyFilter(472)", "Applying property filters: resource=" + uri);
        }
        return filter.filter();
      }
      else {
        return properties;
      }
    }
    catch (ResourceException ex) {
      throw ex;
    }
    catch (WcmException ex) {
      throw new ResourceException(ex);
    }
    catch (Exception ex) {
      this.handleRuntimeException(ex);
      return null;
    }
  }

  private IResourceList applyNamespaceFilter(
    IRepositoryManager repMgr,
    ICollection collection,
    IResourceList resources,
    boolean forWrite,
    NamespaceFilterMode mode)
    throws ResourceException {
    if (resources == null) {
      return null;
    }
    if (forWrite) {
      throw new ResourceException("Namespace write filter are not supported");
    }
    try {
      String mgrID = repMgr.getID();
      // Get all filters IDs for this rep. manager
      List filterIDList = (List)CmRegistry.registry.getFilterList(mgrID, CmFilterProperties.TYPE.NAMESPACE);
      if (filterIDList == null || filterIDList.size() == 0) {
        return resources;
      }
      // Get path of the collection
      String uri = null;
      String prefix = null;
      if (collection != null) {
        uri = collection.getRID().toString();
        prefix = collection.getRepositoryManager().getPrefix();
      }
      else {
        if (resources == null) {
          throw new ResourceException("Illegal namespace filter call: list is null.");
        }
        if (resources.size() == 0) {
          // Done. Adding resources is not possible.
          return resources;
        }
        uri = resources.get(0).getRID().toString();
        prefix = resources.get(0).getRepositoryManager().getPrefix();
      }
      // Remove prefix from uri - A filter can be registered for more than one manager
      uri = uri.substring(prefix.length());
      INamespaceFilter filter = null;
      IResourceList filteredResources = resources;
      filter = new NullNamespaceFilter(collection, resources);
      // Build chain of filter managers
      for (int i = 0; i < filterIDList.size(); i++) {
        String filterID = (String)filterIDList.get(i);
        CmFilterProperties filterProps = CmRegistry.registry.getFilterProperties(filterID);
        if (filterProps == null) {
          continue;
        }

        if (this.matchPatterns(filterProps, uri, null, null)) {
          INamespaceFilterManager fltMgr = (INamespaceFilterManager)this.getCachedFilterManager(filterID, filterProps);
          if (fltMgr != null) {
            filter = fltMgr.getFilterForRead(filter, mode);
          }
          else {
            CmFilterHandler.log.errorT("applyNamespaceFilter(559)", "getFilterManager() returned null: " + filterID);
          }
          if (filter == null) {
            throw new WcmException("Filter manager " + filterID + " returned null");
          }
        }
      }
      if (filter != null) {
        // Apply all filters by calling filter() on the last filter in the "chain"
        if (CmFilterHandler.log.beDebug()) {
          CmFilterHandler.log.debugT("applyNamespaceFilter(569)", "Applying namespace filters: resource=" + uri);
        }
        return filter.filter();
      }
      else {
        return resources;
      }
    }
    catch (ResourceException ex) {
      throw ex;
    }
    catch (WcmException ex) {
      throw new ResourceException(ex);
    }
    catch (Exception ex) {
      this.handleRuntimeException(ex);
      return null;
    }
  }

  /**
   * Pattern matching rules: mimetype and extension may be null but not URI.
   * empty pattern always matches.
   */
  private boolean matchPatterns(CmFilterProperties props, String uri, String mimetype, String ext) {
    PathPatternMatcher mimePattern = (PathPatternMatcher)props.getMimePattern();
    PathPatternMatcher extPattern = (PathPatternMatcher)props.getExtPattern();
    PathPatternMatcher pathPattern = (PathPatternMatcher)props.getPathPattern();

    // Empty pattern always matches
    boolean match = false;
    if (uri != null && ext != null && mimetype != null) {
      // Content filter
      match =
        ((mimePattern == null || mimePattern.matches(mimetype))
          && (extPattern == null || extPattern.matches(ext))
          && (pathPattern == null || pathPattern.matches(uri)));
    }
    else if (uri != null && mimetype == null && ext != null) {
      // Property filter
      match = ((extPattern == null || extPattern.matches(ext)) && (pathPattern == null || pathPattern.matches(uri)));
    }
    else if (uri != null && mimetype == null && ext == null) {
      // Namespace or URI filter
      match = (pathPattern == null || pathPattern.matches(uri));
    }
    else {
      // This should not happen
      throw new RuntimeException("Illegal pattern combination");
    }
    return match;
  }

  /**
   * Resourcetype matching rules (Content and Property filter only): Empty
   * parameter matches any resource. Case-insesitive string comparison.
   */
  private boolean matchResourceType(CmFilterProperties props, IResource resource) {
    PathPatternMatcher rtPattern = props.getResourceTypePattern();

    // Parameter not specified -> match any resource
    if (rtPattern == null) {
      return true;
    }

    if (resource == null) {
      // No resource - it is a content write filter
      // => Matching on resource type is not possible !
      return false;
    }

    IProperty p = null;
    try {
      // Avoid stack overflow if a property filter is used: Call manager directly
      IPropertyManager pm = ((AbstractRepositoryManager)resource.getRepositoryManager()).getPropertyManager(resource);
      if (pm == null)
        return false;
      p = pm.getProperty(PropertyName.createResourceType(), resource);
    }
    catch (ResourceException ex) {
      return false;
    }
    if (p == null) {
      return false;
    }

    String rt = p.getStringValue();
    if (rt == null) {
      return false;
    }

    return rtPattern.matches(rt);
  }

  // ---------------------------------------------------------------------------
  // Null Filter classes
  // ---------------------------------------------------------------------------

  /**
   * TBD: Description of the class.
   */
  public class NullContentFilter implements IContentFilter {
    private IContent m_content = null;
    private IResource m_res = null;
    private URI m_uri = null;
    private IRepositoryManager m_mgr = null;

    public NullContentFilter(IResource res, IContent content) {
      m_content = content;
      m_res = res;
    }

    public NullContentFilter(URI resourceURI, IRepositoryManager mgr, IContent content) {
      m_content = content;
      m_uri = resourceURI;
      m_mgr = mgr;
    }

    public IResource getResource() throws WcmException {
      return m_res;
    }

    public URI getURI() throws WcmException {
      return m_uri;
    }

    public IRepositoryManager getRepositoryManager() throws WcmException {
      return m_mgr;
    }

    public InputStream getInputStream() throws ContentException {
      return m_content.getInputStream();
    }

    public long getContentLength() throws ContentException {
      return m_content.getContentLength();
    }

    public String getContentType() throws ContentException {
      return m_content.getContentType();
    }

    public String getEncoding() throws ContentException {
      return m_content.getEncoding();
    }

    public void close() {
      m_content.close();
    }
  }

  /**
   * TBD: Description of the class.
   */
  public class NullPropertyFilter implements IPropertyFilter {
    private final IPropertyMap m_props;
    private final IResource m_res;
    private final URI m_uri;
    private final IRepositoryManager m_mgr;
    private final Properties m_context;
    private final IPropertyNameList m_propertyNames;
    private final PropertyFilterMode m_mode;

    public NullPropertyFilter(
      IResource res,
      URI resourceURI,
      IRepositoryManager mgr,
      IPropertyMap props,
      IPropertyNameList propNames,
      Properties context,
      PropertyFilterMode mode) {
      m_props = props;
      m_uri = resourceURI;
      m_mgr = mgr;
      m_res = res;
      m_context = context;
      m_mode = mode;
      if (null == propNames) {
        propNames = new PropertyNameList();
      }
      m_propertyNames = propNames;
    }

    public IResource getResource() {
      return m_res;
    }

    public URI getURI() {
      return m_uri;
    }

    public IRepositoryManager getRepositoryManager() throws WcmException {
      return m_mgr;
    }

    public IPropertyNameList getPropertyNameList() throws WcmException {
      return m_propertyNames;
    }

    public PropertyFilterMode getFilterMode() {
      return m_mode;
    }

    public Properties getFilterContext() {
      return m_context;
    }

    public IPropertyMap filter() throws ResourceException {
      return m_props;
    }
  }

  /**
   * TBD: Description of the class.
   */
  public class NullNamespaceFilter implements INamespaceFilter {
    private IResourceList m_list = null;
    private ICollection m_coll = null;

    public NullNamespaceFilter(ICollection coll, IResourceList list) {
      m_list = list;
      m_coll = coll;
    }

    public ICollection getCollection() {
      return m_coll;
    }

    public IResourceList filter() throws ResourceException {
      return m_list;
    }
  }

  private IFilterManager getCachedFilterManager(String filterID, CmFilterProperties props) throws Exception {
    // Use single properties object to cache the filter manager instance (IThreadSafe)
    // because several thousand lookup calls per request are too expensive
    IFilterManager mgr = props.getFilterManagerInstance();
    if (mgr == null) {
      mgr = (IFilterManager)CmAdapter.getInstance().getFilterManager(filterID);
      if (mgr != null) {
        props.setFilterManagerInstance(mgr);
      }
    }
    return mgr;
  }

  private void handleRuntimeException(Exception ex) throws ResourceException {
    log.errorT(
      "handleRuntimeException(837)",
      "Exception while applying filters: " + ex.getMessage() + ": " + LoggingFormatter.extractCallstack(ex));
    if (ex instanceof java.lang.reflect.InvocationTargetException) {
      java.lang.reflect.InvocationTargetException iex = (java.lang.reflect.InvocationTargetException)ex;
      log.errorT(
        "handleRuntimeException(840)",
        "Target exception" + ": " + LoggingFormatter.extractCallstack(iex.getTargetException()));
    }
    throw new ResourceException("Exception while applying filters: " + ex.getMessage());
  }

  private synchronized void collectPropertyNamesForRead(String repID, List filterIDs) {
    Set set = (Set)this.pNames.get(repID);
    if (set == null) {
      set = new HashSet();
      this.pNames.put(repID, set);
    }
    for (int i = 0, s = filterIDs.size(); i < s; i++) {
      IPropertyFilterManager fm = null;
      try {
        fm = (IPropertyFilterManager)CmAdapter.getInstance().getFilterManager((String)filterIDs.get(i));
      }
      catch (Exception ex) {
        CmFilterHandler.log.errorT(LoggingFormatter.extractCallstack(ex));
      }
      if (fm != null) {
        AbstractPropertyFilterManager am = (AbstractPropertyFilterManager)fm;
        IPropertyNameList l = am.getFilteredPropertyNamesForRead();
        if (l == null)
          continue;
        IPropertyNameListIterator it = l.listIterator();
        while (it.hasNext()) {
          IPropertyName n = it.next();
          if (CmFilterHandler.log.beDebug()) {
            CmFilterHandler.log.debugT("property filter manager " + am.getID() + " reported filtered property name: " + n);
          }
          set.add(n);
        }
      }
    }
  }
}
