/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.runtime;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;

import com.sap.tc.logging.Location;
import com.sapportals.config.event.IConfigEventListener;
import com.sapportals.config.fwk.CannotAccessConfigException;
import com.sapportals.config.fwk.ConfigException;
import com.sapportals.config.fwk.ConfigTypeException;
import com.sapportals.config.fwk.ConfigUri;
import com.sapportals.config.fwk.Configuration;
import com.sapportals.config.fwk.IConfigClass;
import com.sapportals.config.fwk.IConfigClientContext;
import com.sapportals.config.fwk.IConfigManager;
import com.sapportals.config.fwk.IConfigPlugin;
import com.sapportals.config.fwk.IConfigProperty;
import com.sapportals.config.fwk.IConfigUri;
import com.sapportals.config.fwk.IConfigUriResolver;
import com.sapportals.config.fwk.IConfigurable;
import com.sapportals.config.fwk.IMutableConfigurable;
import com.sapportals.config.fwk.InitialConfigException;
import com.sapportals.portal.security.usermanagement.UserManagementException;
import com.sapportals.wcm.crt.ComponentUri;
import com.sapportals.wcm.crt.component.ComponentState;
import com.sapportals.wcm.crt.component.IComponent;
import com.sapportals.wcm.crt.component.IContainer;
import com.sapportals.wcm.crt.component.ILifecycleInfo;
import com.sapportals.wcm.crt.component.IReconfigurable;
import com.sapportals.wcm.crt.configuration.ConfigurationException;
import com.sapportals.wcm.crt.configuration.DefaultConfiguration;
import com.sapportals.wcm.crt.configuration.IConfigConst;
import com.sapportals.wcm.crt.configuration.IConfiguration;
import com.sapportals.wcm.crt.configuration.IConfigurationEventListener;
import com.sapportals.wcm.crt.configuration.IConfigurationProvider;
import com.sapportals.wcm.repository.runtime.legacy.FilterProxy;
import com.sapportals.wcm.repository.runtime.legacy.RepositoryManagerProxy;
import com.sapportals.wcm.repository.runtime.legacy.RepositoryServiceProxy;
import com.sapportals.wcm.repository.runtime.legacy.ServiceProxy;
import com.sapportals.wcm.util.config.ConfigCrutch;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.regex.PathPatternMatcher;
import com.sapportals.wcm.util.regex.PatternSyntaxException;

/**
 * This configuration provider implementation converts the CM configuration data
 * and events to the objects needed by CRT. <p>
 *
 * Copyright (c) SAP AG 2002
 *
 * @author Markus Breitenfelder
 * @author Jens Kaiser
 * @version $Id:$
 */
public final class CmConfigurationProvider implements IConfigurationProvider, IConfigEventListener {

  /**
   * TBD: Description of the class.
   */
  public interface CONTAINER_URIS {
    // ATTENTION: whenever changing the key values below please change these values in the class
    // com.sapportals.wcm.crt.moni.ComponentRuntime.java, constants FILTERS_ID, RE_MANAGERS_ID,
    // REP_SERVICES_ID and SERVICES_ID, also. If you make a similar change for the system info
    // container, you have to change SYSTEM_INFO_ID in the above class. It is not possible to use
    // the below definitions in the above class because a cyclic dependency would result.
    // Contact heiko.kiessling@sap.com or frank.renkes@sap.com in case of questions.

    // stefan.eissing@greenbytes.de: make uris better readable and usable
    //ComponentUri REPMGR = ComponentUri.getRootUri(CmSystem.CRT_CLIENT_ID).appendKey(CmRepMgrContainer.class.getName());
    //ComponentUri REPSRV = ComponentUri.getRootUri(CmSystem.CRT_CLIENT_ID).appendKey(CmRepSrvContainer.class.getName());
    //ComponentUri SERVICE = ComponentUri.getRootUri(CmSystem.CRT_CLIENT_ID).appendKey(CmServiceContainer.class.getName());
    //ComponentUri FILTER = ComponentUri.getRootUri(CmSystem.CRT_CLIENT_ID).appendKey(CmRepFltContainer.class.getName());
    ComponentUri REPMGR = ComponentUri.getRootUri(CmSystem.CRT_CLIENT_ID).appendKey("repository_managers");
    ComponentUri REPSRV = ComponentUri.getRootUri(CmSystem.CRT_CLIENT_ID).appendKey("repository_services");
    ComponentUri SERVICE = ComponentUri.getRootUri(CmSystem.CRT_CLIENT_ID).appendKey("services");
    ComponentUri FILTER = ComponentUri.getRootUri(CmSystem.CRT_CLIENT_ID).appendKey("filters");
  }

  /**
   * TBD: Description of the class.
   */
  public interface CFG {
    String PROPERTY_PREFIX = "prefix";
    String ACTIVE_KEY = "active";
    String PROPERTY_CLASSNAME = "class";
    String CONFIG_CLASS = "configclass";
    String MANAGER_CLASSNAME = "RepositoryManager";
    String SERVICE_CLASSNAME = "Service";
    String REPSERVICE_CLASSNAME = "RepositoryService";
    String MANAGER_SERVICES_KEY = "services";
    String FILTER_CLASSNAME = "RepositoryFilter";
    String FILTERMGR_REPMGRS = "managers";
    String FILTERMGR_URIPREFIX = "prefix";
    String FILTERMGR_EXTENSIONS = "extensions";
    String FILTERMGR_MIMETYPES = "mimetypes";
    String FILTERMGR_RESOURCETYPE = "resourcetype";
    String FILTERMGR_PATH = "path";
    String FILTERMGR_PRIORITY = "priority";
    String SERVICE_CCLIST = "cclist";
  }

  /**
   * TBD: Description of the class.
   */
  public interface FILTER_PRIO {
    Integer MIN = new Integer(0);
    Integer MAX = new Integer(99);
  }

  /**
   * TBD: Description of the class.
   */
  public interface SERVICE {
    String NOTE_CLASS = "NotificatorService";
    String CRAWLER_CLASS = "CrawlerService";
    String INDEX_CLASS = "IndexmanagementService";
    String PIPE_CLASS = "PipelineService";
    String ATTACH_CLASS = "AttachmentRepositoryService";
    String COMMENT_CLASS = "CommentRepositoryService";
    String DISCUSSION_CLASS = "DiscussionRepositoryService";
    String FEEDBACK_CLASS = "FeedbackRepositoryService";
    String PERSNOTE_CLASS = "PersonalNoteRepositoryService";
    String SUBSCR_CLASS = "SubscriptionRepositoryService";
  }

  /**
   * TBD: Description of the class.
   */
  public interface WEB {
    String MANAGER_CLASS = "WebRepositoryManager";
    String SYSTEM_VIRTUAL_CLASS = "WebSystem";
    String SYSTEM_KEY_PREFIX = "system";
  }

  /**
   * TBD: Description of the class.
   */
  public interface CRAWLER {
    String VIRTUAL_CLASS = "Crawler";
  }

  /**
   * TBD: Description of the class.
   */
  public interface INDEX {
    String SERVICE_SERVICE_KEY_PREFIX = "service";
    String SERVICE_VIRTUAL_CLASS = "IndexService";
  }

  /**
   * TBD: Description of the class.
   */
  public interface PIPE {
    String TOOL_KEY_PREFIX = "tool";
    String TOOL_VIRTUAL_CLASS = "PipelineTool";
    String PROC_KEY_PREFIX = "processor";
    String PROC_VIRTUAL_CLASS = "PipelineProcessor";
    String FORM_KEY_PREFIX = "formatter";
    String FORM_VIRTUAL_CLASS = "PipelineFormatter";
  }

  public final static String VIRTUAL_ROOT_PREFIX = "/~virtualroot~";
  private static Location log = Location.getLocation(com.sapportals.wcm.repository.runtime.CmConfigurationProvider.class);
  private final static DefaultConfiguration EMPTY_COMPONENT_CONFIG = new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT_CONFIG, "");
  private IConfigurationEventListener listener;
  //  private CmRegistry registry;
  private ClassLoader loader;
  
  private final static String CFG_PLUGIN_CM_REPOSITORY_FILTERS = "/cm/repository_filters";  
  private final static String CFG_PLUGIN_CM_REPOSITORY_SERVICES = "/cm/repository_services";
  private final static String CFG_PLUGIN_CM_REPOSITORY_MANAGERS = "/cm/repository_managers";
  private final static String CFG_PLUGIN_CM_REPOSITORY_MANAGERS_WEB_SYSTEMS = "/cm/repository_managers/web_systems";
  private final static String CFG_PLUGIN_CM_SERVICES = "/cm/services";
  private final static String CFG_PLUGIN_CM_SERVICES_CRAWLERS = "/cm/services/crawlers";
  private final static String CFG_PLUGIN_CM_SERVICES_INDEX_SERVICES = "/cm/services/index_services";
  private final static String CFG_PLUGIN_CM_SERVICES_PIPELINE_ADDONS = "/cm/services/pipeline_addons";
  
  //--------------------------------------------------------------------------------------
  // IConfigurationProvider implementation
  //--------------------------------------------------------------------------------------

  public IConfiguration readConfiguration(ClassLoader cl) throws ConfigurationException {
    this.loader = cl;

		DefaultConfiguration cfg = new DefaultConfiguration(IConfigConst.ELEMENTS.APPLICATION, "");
		// Top level containers
		DefaultConfiguration cfgMgrContainer = new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT, "");
		cfgMgrContainer.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_CLASS, CmRepMgrContainer.class.getName());
		cfgMgrContainer.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_KEY, CONTAINER_URIS.REPMGR.getKey());
		cfgMgrContainer.addChild(EMPTY_COMPONENT_CONFIG);
		DefaultConfiguration cfgServiceContainer = new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT, "");
		cfgServiceContainer.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_CLASS, CmServiceContainer.class.getName());
		cfgServiceContainer.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_KEY, CONTAINER_URIS.SERVICE.getKey());
		cfgServiceContainer.addChild(EMPTY_COMPONENT_CONFIG);
		DefaultConfiguration cfgRepSrvContainer = new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT, "");
		cfgRepSrvContainer.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_CLASS, CmRepSrvContainer.class.getName());
		cfgRepSrvContainer.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_KEY, CONTAINER_URIS.REPSRV.getKey());
		cfgRepSrvContainer.addChild(EMPTY_COMPONENT_CONFIG);
		DefaultConfiguration cfgFilterContainer = new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT, "");
		cfgFilterContainer.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_CLASS, CmRepFltContainer.class.getName());
		cfgFilterContainer.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_KEY, CONTAINER_URIS.FILTER.getKey());
		cfgFilterContainer.addChild(EMPTY_COMPONENT_CONFIG);
		DefaultConfiguration cfgSystemInfo = new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT, "");
		cfgSystemInfo.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_CLASS, CmSystemInfo.class.getName());
		cfgSystemInfo.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_KEY, CmSystemInfo.class.getName());
		cfgSystemInfo.addChild(EMPTY_COMPONENT_CONFIG);
		cfg.addChild(cfgMgrContainer);
		cfg.addChild(cfgServiceContainer);
		cfg.addChild(cfgRepSrvContainer);
		cfg.addChild(cfgFilterContainer);
		cfg.addChild(cfgSystemInfo);
		convertRepositoryServiceConfig(cfgRepSrvContainer);
		convertRepositoryConfig(cfgMgrContainer);
		convertServiceConfig(cfgServiceContainer);
		convertFilterConfig(cfgFilterContainer);
		return cfg;
  }

  public void attachEventListener(IConfigurationEventListener listener) {
    this.listener = listener;
  }

  // --- config event listener ---

  public String getConfigListenerId() {
    return "cmlistener";
  }

  /**
   * Analyse the config framework event and map it to add/change/remove events
   * for CM components.
   *
   * @param e TBD: Description of the incoming method parameter
   */
  public void configEvent(com.sapportals.config.event.ConfigEvent e) {
    if (e == null) {
      CmConfigurationProvider.log.errorT("configEvent(224)", "configuration event = null");
      return;
    }

    try {
      CmConfigurationProvider.log.debugT("configEvent(233)", "received configuration event: " + e.toString());

      IConfigUriResolver configUriResolver = Configuration.getInstance().getConfigUriResolverInstance();

      if (e.getType() == com.sapportals.config.event.ConfigEvent.CONFIGURABLE_UPDATED) {
        com.sapportals.config.fwk.IConfigurable configurable =
          (com.sapportals.config.fwk.IConfigurable)configUriResolver.getItem(
            e.getSourceUri(),
            com.sapportals.config.fwk.IConfigurable.CONFIGURABLE);
        if (configurable == null) {
          return;
        }

        boolean ccs = false;
        IConfigUri[] containerServiceURIs = this.calculateContainerServiceURIs(configurable);
        if (containerServiceURIs != null && containerServiceURIs.length > 0) {
          for (int i = 0; i < containerServiceURIs.length; i++) {
            // Resolve config URI and send event for configurable
            com.sapportals.config.fwk.IConfigurable c =
              (com.sapportals.config.fwk.IConfigurable)configUriResolver.getItem(
                containerServiceURIs[i],
                com.sapportals.config.fwk.IConfigurable.CONFIGURABLE);
            if (c != null) {
              this.sendChangeEventOrRestart(c);
            }
          }
          ccs = true;
        }
        IConfigUri[] containerManagerURIs = this.calculateContainerManagerURIs(configurable);
        if (containerManagerURIs != null && containerManagerURIs.length > 0) {
          for (int i = 0; i < containerManagerURIs.length; i++) {
            // Resolve config URI and send event for configurable
            com.sapportals.config.fwk.IConfigurable c =
              (com.sapportals.config.fwk.IConfigurable)configUriResolver.getItem(
                containerManagerURIs[i],
                com.sapportals.config.fwk.IConfigurable.CONFIGURABLE);
            if (c != null) {
              this.sendChangeEventOrRestart(c);
            }
          }
          ccs = true;
        }

        if (!ccs) {
          this.sendChangeEventOrRestart(configurable);
        }
      }
      else if (e.getType() == com.sapportals.config.event.ConfigEvent.CONFIGURABLE_ADDED) {
        com.sapportals.config.fwk.IConfigurable configurable =
          (com.sapportals.config.fwk.IConfigurable)configUriResolver.getItem(
            e.getSourceUri(),
            com.sapportals.config.fwk.IConfigurable.CONFIGURABLE);
        if (configurable.getConfigClass().isSubClassOf(CmConfigurationProvider.CFG.REPSERVICE_CLASSNAME)) {
          this.logReconfigurationNotPossible("Adding repository service instances at runtime is not supported: " + e.getSourceUri());
        }
        if (configurable.getConfigClass().isSubClassOf(CmConfigurationProvider.CFG.SERVICE_CLASSNAME)) {
          this.logReconfigurationNotPossible("Adding service instances at runtime is not supported: " + e.getSourceUri());
        }

        boolean ccs = false;
        IConfigUri[] containerServiceURIs = this.calculateContainerServiceURIs(configurable);
        if (containerServiceURIs != null && containerServiceURIs.length > 0) {
          for (int i = 0; i < containerServiceURIs.length; i++) {
            // Resolve config URI and send event for configurable
            com.sapportals.config.fwk.IConfigurable c =
              (com.sapportals.config.fwk.IConfigurable)configUriResolver.getItem(
                containerServiceURIs[i],
                com.sapportals.config.fwk.IConfigurable.CONFIGURABLE);
            if (c != null) {
              this.sendChangeEventOrRestart(c);
            }
          }
          ccs = true;
        }
        IConfigUri[] containerManagerURIs = this.calculateContainerManagerURIs(configurable);
        if (containerManagerURIs != null && containerManagerURIs.length > 0) {
          for (int i = 0; i < containerManagerURIs.length; i++) {
            // Resolve config URI and send event for configurable
            com.sapportals.config.fwk.IConfigurable c =
              (com.sapportals.config.fwk.IConfigurable)configUriResolver.getItem(
                containerManagerURIs[i],
                com.sapportals.config.fwk.IConfigurable.CONFIGURABLE);
            if (c != null) {
              this.sendChangeEventOrRestart(c);
            }
          }
          ccs = true;
        }

        if (!ccs) {
          this.sendAddEvent(configurable);
        }
      }
      else if (e.getType() == com.sapportals.config.event.ConfigEvent.CONFIGURABLE_DELETED) {
        // Only removing child compontents is supported by now

        // TODO: The selectContainerServiceID() method is a hack !
        //
        // => The config event should provide the conig class of the deleted configurable !!
        //
        IConfigUri[] containerServiceURIs = CmRegistry.registry.selectContainerServiceConfigURIs(this.calculateConfigClassUri(e));
        if (containerServiceURIs != null && containerServiceURIs.length > 0) {
          // A child component has been removed, send change event for container service
          for (int i = 0; i < containerServiceURIs.length; i++) {
            // Resolve config URI and send event for configurable
            com.sapportals.config.fwk.IConfigurable c =
              (com.sapportals.config.fwk.IConfigurable)configUriResolver.getItem(
                containerServiceURIs[i],
                com.sapportals.config.fwk.IConfigurable.CONFIGURABLE);
            this.sendChangeEvent(c);
          }
        }

        IConfigUri[] containerManagerURIs = CmRegistry.registry.selectContainerManagerConfigURIs(this.calculateConfigClassUri(e));
        if (containerManagerURIs != null && containerManagerURIs.length > 0) {
          // A child component has been removed, send change event for container service
          for (int i = 0; i < containerManagerURIs.length; i++) {
            // Resolve config URI and send event for configurable
            com.sapportals.config.fwk.IConfigurable c =
              (com.sapportals.config.fwk.IConfigurable)configUriResolver.getItem(
                containerManagerURIs[i],
                com.sapportals.config.fwk.IConfigurable.CONFIGURABLE);
            this.sendChangeEvent(c);
          }
        }
      }
    }
    catch (Exception ex) {
      CmConfigurationProvider.log.errorT(
        "configEvent(309)",
        "Excpetion in configuration event handler for "
          + e.getSourceUri()
          + ": "
          + ex.getMessage()
          + ": "
          + LoggingFormatter.extractCallstack(ex));
    }
  }

  // ---------------------------------------------------------------------------

  void createFilterManager(Properties filterConfig) throws ConfigurationException {
    if (filterConfig == null) {
      throw new NullPointerException("parameter filterConfig is null");
    }
    IConfiguration cfg = this.convertRF(filterConfig, "");
    this.listener.notifyComponentConfigAdded(CmConfigurationProvider.CONTAINER_URIS.FILTER, cfg);
  }

  void attachFilterToRepository(String id, String managerID) throws Exception {
    CmRegistry.registry.updateFilterAssignment(managerID, id, true);
  }

  void detachFilterFromRepository(String id, String managerID) throws Exception {
    CmRegistry.registry.updateFilterAssignment(managerID, id, false);
  }

  // ---------------------------------------------------------------------------

  static ComponentUri buildRepositoryUri(String id) {
    return CmConfigurationProvider.CONTAINER_URIS.REPMGR.appendKey(id);
  }

  static ComponentUri buildRepServiceUri(String id) {
    return CmConfigurationProvider.CONTAINER_URIS.REPSRV.appendKey(id);
  }

  static ComponentUri buildServiceUri(String type) {
    return CmConfigurationProvider.CONTAINER_URIS.SERVICE.appendKey(type);
  }

  static ComponentUri buildFilterUri(String id) {
    return CmConfigurationProvider.CONTAINER_URIS.FILTER.appendKey(id);
  }

  /**
   * Convert and validate configuration data for repository managers, repository
   * services and filters.
   *
   * @param cfg TBD: Description of the incoming method parameter
   * @exception ConfigurationException Exception raised in failure situation
   */
  private void convertRepositoryConfig(DefaultConfiguration cfg) throws ConfigurationException {
    try {
      IConfigClientContext context = IConfigClientContext.createContext(ConfigCrutch.getConfigServiceUser());
      IConfigManager cfgm = Configuration.getInstance().getConfigManager(context);
      IConfigPlugin plugin = cfgm.getConfigPlugin(CFG_PLUGIN_CM_REPOSITORY_MANAGERS);
      if (plugin == null) {
        throw new RuntimeException("missing plugin " + CFG_PLUGIN_CM_REPOSITORY_MANAGERS);
      }

      IConfigurable[] configurables = plugin.getConfigurables();
      for (int i = 0; i < configurables.length; i++) {
        if (configurables[i].getConfigClass().isSubClassOf(CFG.MANAGER_CLASSNAME)) {
          if (this.isConfigurableActive(configurables[i])) {
            IConfiguration c = this.convertRM(configurables[i]);
            // convertRM() might return null if the repository system parameters are invalid
            // e.g. duplicate prefix
            if (c != null) {
              cfg.addChild(c);
            }
          }
        }
      }
    }
    catch (CannotAccessConfigException ex) {
      throw new ConfigurationException("Failed to access CM configuration", ex);
    }
    catch (UserManagementException x) {
      throw new ConfigurationException("Failed to access CM configuration", x);
    }
    catch (InitialConfigException x) {
      throw new ConfigurationException("Failed to access CM configuration", x);
    }
  }

  /**
   * Convert configurable(s) of repository managers to a component entry.
   * Returns null if the system configuration parameters are invalid - the
   * repository will not be available and configuration exception will be
   * logged.
   *
   * @param mgrConfigurable TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception ConfigurationException Exception raised in failure situation
   */
  private DefaultConfiguration convertRM(com.sapportals.config.fwk.IConfigurable mgrConfigurable) throws ConfigurationException {
    String id = mgrConfigurable.getIdValue();
    String className = mgrConfigurable.getPropertyValue(CmConfigurationProvider.CFG.PROPERTY_CLASSNAME);
    String location = mgrConfigurable.getUri().toString();
    String childConfigClassNames = mgrConfigurable.getPropertyValue(CFG.SERVICE_CCLIST);

    CmConfigurationProvider.log.debugT("convertRM(404)", "converting repository manager config: " + id);

    if (className == null) {
      this.throwMissingClassException(location, null);
    }
    String prefix = mgrConfigurable.getPropertyValue(CmConfigurationProvider.CFG.PROPERTY_PREFIX);

    if (prefix == null) {
      this.throwConfigurationException("missing parameter", location, CmConfigurationProvider.CFG.PROPERTY_PREFIX);
    }
    prefix = prefix.trim();

    // Trailing slashes not allowed
    if (prefix.endsWith("/") && prefix.length() > 1) {
      // CSN: 395968, no startup failure       
      //this.throwConfigurationException("prefix is invalid (no trailing slashes allowed):" + prefix, location, CFG.PROPERTY_PREFIX);
      CmConfigurationProvider.log.errorT("prefix <" + prefix + "> is invalid - trailing slashes are not allowed. The repository will not be available");
      return null;
    }
    if (!prefix.startsWith("/")) {
      //this.throwConfigurationException("prefix is invalid: " + prefix, location, CmConfigurationProvider.CFG.PROPERTY_PREFIX);
      prefix = "/" + prefix;
    }
    // Only single-part prefixes are allowed
    int pos = prefix.lastIndexOf("/");
    if (pos > 0) {
      // CSN: 395968, no startup failure
      //this.throwConfigurationException("prefix is invalid: " + prefix, location, CmConfigurationProvider.CFG.PROPERTY_PREFIX);    
      CmConfigurationProvider.log.errorT("prefix <" + prefix + "> is invalid - only single-part prefix strings are allowed. The repository will not be available");
      return null;      
    }
    if (prefix.equals("/")) {
      prefix = VIRTUAL_ROOT_PREFIX;
    }

    // Remove from registry first. This method is also called for "reload"
    CmRegistry.registry.removeRepository(id);

    String serviceIDs = mgrConfigurable.getPropertyValue(CmConfigurationProvider.CFG.MANAGER_SERVICES_KEY);
    ArrayList serviceList = null;
    if (serviceIDs != null && serviceIDs.length() > 0) {
      serviceList = parseValueList(serviceIDs, ",");
    }
    else {
      serviceList = new ArrayList();
    }
    DefaultConfiguration comp = new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT, location);

    comp.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_CLASS, className);
    comp.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_KEY, id);
    // load the manager class
    Class managerClass = null;
    try {
      managerClass = Class.forName(className, true, loader);
    }
    catch (Throwable x) {
      this.logClassLoadError(id, className, x);
      //this.throwMissingClassException(className, x);
      return null;
    }
    if (RepositoryManagerProxy.isLegacyClass(managerClass)) {
      // add a proxy while the manager is not conform to the component runtime
      comp.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_PROXY, RepositoryManagerProxy.class.getName());
    }

    String[] ccNames = null;
    if (childConfigClassNames != null && childConfigClassNames.length() > 0) {
      ccNames = (String[])parseValueList(childConfigClassNames, ",").toArray(new String[0]);
    }
    else {
      ccNames = new String[0];
    }

    DefaultConfiguration compConfig =
      new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT_CONFIG, location, mgrConfigurable.getProperties(true));
    comp.addChild(compConfig);

    // Services can have child component relationship to other configurables
    if (ccNames.length > 0) {
      this.convertChildConfigurables(ccNames, compConfig);
      if (log.beDebug()) {
        log.debugT("convertRM(568)", "Child config classes of service: " + className + " = " + ccNames);
      }
    }

    // Special web manager handling (don't ask!)
    if (WEB.MANAGER_CLASS.equals(mgrConfigurable.getConfigClass().getName())) {
      try {
        this.completeComponentProperties(
          compConfig,
          CFG_PLUGIN_CM_REPOSITORY_MANAGERS_WEB_SYSTEMS,
          WEB.SYSTEM_VIRTUAL_CLASS,
          WEB.SYSTEM_KEY_PREFIX);
      }
      catch (Exception ex) {
        throw new ConfigurationException("Exception completing web manager configuration: " + ex.getMessage(), ex);
      }
    }
    try {
      CmRegistry.registry.addRepository(id, new CmManagerProperties(prefix, mgrConfigurable.getUri(), serviceList, ccNames));
    }
    catch (InvalidPrefixConfigurationException ex) {
      CmConfigurationProvider.log.errorT("convertRM(481)", LoggingFormatter.extractCallstack(ex));
      return null;
    }

    comp.makeReadOnly();
    compConfig.makeReadOnly();
    return comp;
  }

  
  // ---------------------------------------------------------------------------

  /**
   * Convert and validate configuration data for (global) services
   *
   * @param cfg TBD: Description of the incoming method parameter
   * @exception ConfigurationException Exception raised in failure situation
   */
  private void convertServiceConfig(DefaultConfiguration cfg) throws ConfigurationException {
    try {
      IConfigClientContext context = IConfigClientContext.createContext(ConfigCrutch.getConfigServiceUser());
      IConfigManager cfgm = Configuration.getInstance().getConfigManager(context);
      IConfigPlugin plugin = cfgm.getConfigPlugin(CFG_PLUGIN_CM_SERVICES);
      if (plugin == null) {
        throw new RuntimeException("missing plugin " + CFG_PLUGIN_CM_SERVICES);
      }
      
      IConfigurable[] configurables = plugin.getConfigurables();
   
      for (int i = 0; i < configurables.length; i++) {
        if (configurables[i].getConfigClass().isSubClassOf(CFG.SERVICE_CLASSNAME)) {
          if (this.isConfigurableActive(configurables[i])) {
            IConfiguration c = this.convertGS(configurables[i]);
            if (c != null) {
              cfg.addChild(c);
            }
          }
        }
      }
    }
    catch (ConfigException ex) {
      throw new ConfigurationException("Failed to access CM configuration", ex);
    }
    catch(UserManagementException x) {
      throw new ConfigurationException("Failed to access CM configuration", x);
    }
  }

  /**
   * Convert config of one service configurable to one component entry on CRT
   * configuration.
   *
   * @param srvConfigurable TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception ConfigurationException Exception raised in failure situation
   */
  private DefaultConfiguration convertGS(com.sapportals.config.fwk.IConfigurable srvConfigurable) throws ConfigurationException {
    String id = srvConfigurable.getIdValue();
    String className = srvConfigurable.getPropertyValue(CFG.PROPERTY_CLASSNAME);
    String type = srvConfigurable.getConfigClass().getName();
    String childConfigClassNames = srvConfigurable.getPropertyValue(CFG.SERVICE_CCLIST);
    String location = srvConfigurable.getUri().toString();
    CmConfigurationProvider.log.debugT("convertGS(530)", "converting service config: " + id);
    DefaultConfiguration comp = new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT, location);
    comp.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_CLASS, className);
    // load the service class
    Class serviceClass = null;
    try {
      serviceClass = Class.forName(className, true, loader);
    }
    catch (Throwable x) {
      this.logClassLoadError(id, className, x);  
      //this.throwMissingClassException(className, x);
      return null; 
    }

    if (ServiceProxy.isLegacyClass(serviceClass)) {
      // creating a legacy service means to configure it.
      // Therefore the service's property "name" is used as a fallback
      // add a proxy while the service is not conform to the component runtime
      comp.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_PROXY, ServiceProxy.class.getName());
    }

    // The component ID is the service type !!!
    comp.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_KEY, type);

    String[] ccNames = null;
    if (childConfigClassNames != null && childConfigClassNames.length() > 0) {
      ccNames = (String[])parseValueList(childConfigClassNames, ",").toArray(new String[0]);
    }
    else {
      ccNames = new String[0];
    }
    DefaultConfiguration compConfig =
      new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT_CONFIG, location, srvConfigurable.getProperties(true));
    comp.addChild(compConfig);

    // Services can have child component relationship to other configurables
    if (ccNames.length > 0) {
      this.convertChildConfigurables(ccNames, compConfig);
      if (log.beDebug()) {
        log.debugT("convertGS(568)", "Child config classes of service: " + className + " = " + ccNames);
      }
    }

    CmRegistry.registry.addService(id, new CmServiceProperties(type, srvConfigurable.getUri(), ccNames));

    // (don't ask!)
    if (SERVICE.CRAWLER_CLASS.equals(srvConfigurable.getConfigClass().getName())) {
      try {
        completeComponentProperties(compConfig, CFG_PLUGIN_CM_SERVICES_CRAWLERS, CRAWLER.VIRTUAL_CLASS, null);
      }
      catch (Exception ex) {
        throw new ConfigurationException("Exception completing web manager configuration: " + ex.getMessage(), ex);
      }
    }
    if (SERVICE.INDEX_CLASS.equals(srvConfigurable.getConfigClass().getName())) {
      try {
        this.completeComponentProperties(
          compConfig,
          CFG_PLUGIN_CM_SERVICES_INDEX_SERVICES,
          INDEX.SERVICE_VIRTUAL_CLASS,
          INDEX.SERVICE_SERVICE_KEY_PREFIX);
      }
      catch (Exception ex) {
        throw new ConfigurationException("Exception completing web manager configuration: " + ex.getMessage(), ex);
      }
    }
    if (SERVICE.PIPE_CLASS.equals(srvConfigurable.getConfigClass().getName())) {
      try {
        completeComponentProperties(
          compConfig,
          CFG_PLUGIN_CM_SERVICES_PIPELINE_ADDONS,
          PIPE.TOOL_VIRTUAL_CLASS,
          PIPE.TOOL_KEY_PREFIX);
        completeComponentProperties(
          compConfig,
          CFG_PLUGIN_CM_SERVICES_PIPELINE_ADDONS,
          PIPE.PROC_VIRTUAL_CLASS,
          PIPE.PROC_KEY_PREFIX);
        completeComponentProperties(
          compConfig,
          CFG_PLUGIN_CM_SERVICES_PIPELINE_ADDONS,
          PIPE.FORM_VIRTUAL_CLASS,
          PIPE.FORM_KEY_PREFIX);
      }
      catch (Exception ex) {
        throw new ConfigurationException("Exception completing web manager configuration: " + ex.getMessage(), ex);
      }
    }
    comp.makeReadOnly();
    compConfig.makeReadOnly();
    return comp;
  }

  /**
   * Adds the child components configurations to the service's configuration.
   *
   * @param ccNames TBD: Description of the incoming method parameter
   * @param compConfig TBD: Description of the incoming method parameter
   * @exception ConfigurationException Exception raised in failure situation
   */
  private void convertChildConfigurables(String[] ccNames, DefaultConfiguration compConfig) throws ConfigurationException {
    for (int i = 0; i < ccNames.length; i++) {
      String ccName = ccNames[i];
      DefaultConfiguration childConfigParent = new DefaultConfiguration(ccName, compConfig.getLocation());
      compConfig.addChild(childConfigParent);
      if (ccName.endsWith("/")) {
        throw new ConfigurationException("Invalid config class reference: " + ccName);
      }
      int lastSlash = ccName.lastIndexOf("/");
      String pluginName = ccName.substring(0, lastSlash);
      String configClassName = ccName.substring(lastSlash + 1);
      IMutableConfigurable[] configurables = null;
      try {
        IConfigClientContext context = IConfigClientContext.createContext(
          ConfigCrutch.getConfigServiceUser());
        IConfigManager cfgManager = Configuration.getInstance().getConfigManager(context);
        IConfigPlugin plugin = cfgManager.getConfigPlugin(pluginName);
        
        configurables = plugin.getConfigurables(configClassName, true);
      }
      catch (Exception ex) {
        throw new ConfigurationException(ex);
      }
      if (configurables == null) {
        throw new ConfigurationException("Error in cclist: Config plugin/class not found: " + ccName);
      }
      for (int c = 0; c < configurables.length; c++) {
        String id = configurables[c].getIdValue();
        String cclass = configurables[c].getConfigClass().getName();
        DefaultConfiguration childConfig = new DefaultConfiguration(id, ccName, configurables[c].getProperties(true));
        childConfig.setAttribute(CFG.CONFIG_CLASS, cclass);
        childConfigParent.addChild(childConfig);
      }
    }
  }

  // ---------------------------------------------------------------------------

  /**
   * Convert and validate configuration data for (global) services
   *
   * @param cfg TBD: Description of the incoming method parameter
   * @exception ConfigurationException Exception raised in failure situation
   */
  private void convertRepositoryServiceConfig(DefaultConfiguration cfg) throws ConfigurationException {
    try {
      IConfigClientContext context = IConfigClientContext.createContext(
        ConfigCrutch.getConfigServiceUser());
      IConfigManager cfgManager = Configuration.getInstance().getConfigManager(context);
      IConfigPlugin plugin = cfgManager.getConfigPlugin(CFG_PLUGIN_CM_REPOSITORY_SERVICES);
      IConfigurable[] configurables = plugin.getConfigurables();

      for (int i = 0; i < configurables.length; i++) {
        if (configurables[i].getConfigClass().isSubClassOf(CFG.REPSERVICE_CLASSNAME)) {
          if (this.isConfigurableActive(configurables[i])) {
            IConfiguration c = this.convertRS(configurables[i]);
            if (c != null) {
              cfg.addChild(c);
            }
          }
        }
      }
    }
    catch (Exception ex) {
      throw new ConfigurationException("Failed to access CM configuration", ex);
    }
  }

  /**
   * Convert config of one repository service configurable to one component
   * entry on CRT configuration.
   *
   * @param srvConfigurable TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception ConfigurationException Exception raised in failure situation
   */
  private DefaultConfiguration convertRS(com.sapportals.config.fwk.IConfigurable srvConfigurable) throws ConfigurationException {
    String id = srvConfigurable.getIdValue();
    String className = srvConfigurable.getPropertyValue(CFG.PROPERTY_CLASSNAME);
    String type = srvConfigurable.getConfigClass().getName();
    String location = srvConfigurable.getUri().toString();
    String childConfigClassNames = srvConfigurable.getPropertyValue(CFG.SERVICE_CCLIST);

    CmConfigurationProvider.log.debugT("convertRS(692)", "converting repository service config: " + id);
    DefaultConfiguration comp = new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT, location);
    comp.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_CLASS, className);
    comp.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_KEY, id);
    // load the service class
    Class serviceClass = null;
    try {
      serviceClass = Class.forName(className, true, loader);
    }
    catch (Throwable x) {
      this.logClassLoadError(id, className, x);
      //this.throwMissingClassException(className, x);
      return null;
    }

    if (RepositoryServiceProxy.isLegacyClass(serviceClass)) {
      // add a proxy while the service is not conform to the component runtime
      comp.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_PROXY, RepositoryServiceProxy.class.getName());
    }

    comp.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_KEY, id);

    String[] ccNames = null;
    if (childConfigClassNames != null && childConfigClassNames.length() > 0) {
      ccNames = (String[])parseValueList(childConfigClassNames, ",").toArray(new String[0]);
    }
    else {
      ccNames = new String[0];
    }

    DefaultConfiguration compConfig =
      new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT_CONFIG, location, srvConfigurable.getProperties(true));
    comp.addChild(compConfig);

    // Get the list of repository manager IDs for this service an put them into config
    StringBuffer list = new StringBuffer();
    try {
      IConfigClientContext context = IConfigClientContext.createContext(
        ConfigCrutch.getConfigServiceUser());
      IConfigManager cfgManager = Configuration.getInstance().getConfigManager(context);
      IConfigPlugin plugin = cfgManager.getConfigPlugin(CFG_PLUGIN_CM_REPOSITORY_MANAGERS);
      IConfigurable[] configurables = plugin.getConfigurables();
            
      for (int managerIdx = 0; managerIdx < configurables.length; managerIdx++) {
        if (configurables[managerIdx].getConfigClass().isSubClassOf(CmConfigurationProvider.CFG.MANAGER_CLASSNAME)) {
          if (this.isConfigurableActive(configurables[managerIdx])) {
            Properties properties = getConfigurableProperties(configurables[managerIdx]);
            String managerID = configurables[managerIdx].getIdValue();
            String servicesString = properties.getProperty(CmConfigurationProvider.CFG.MANAGER_SERVICES_KEY);
            if (servicesString != null) {
              ArrayList services = parseValueList(servicesString, ",");
              for (int serviceIdx = 0; serviceIdx < services.size(); serviceIdx++) {
                if (((String)services.get(serviceIdx)).equals(id)) {
                  if (list.length() > 0) {
                    list.append(ICmRuntimeConst.LIST_SEPARATOR);
                  }
                  list.append(managerID);
                }
              }
            }
          }
        }
      }
    }
    catch (InitialConfigException ex) {
      throw new ConfigurationException("Failed to access CM configuration", ex);
    }
    catch (CannotAccessConfigException ex) {
      throw new ConfigurationException("Failed to access CM configuration", ex);
    }
    catch (UserManagementException ex) {
      throw new ConfigurationException("Failed to access CM configuration", ex);
    }
    compConfig.setAttribute(ICmRuntimeConst.CONFIG_ATTRIBUTE_NAME.MANAGER_LIST, list.toString());

    // Services can have child component relationship to other configurables
    if (ccNames.length > 0) {
      this.convertChildConfigurables(ccNames, compConfig);
      if (CmConfigurationProvider.log.beDebug()) {
        CmConfigurationProvider.log.debugT("convertRS(764)", "Child config classes of service: " + className + " = " + ccNames);
      }
    }

    CmRegistry.registry.addRepositoryService(id, new CmRepServiceProperties(type, srvConfigurable.getUri(), ccNames));

    // Some magic code follows - no idea what this is about (copied from old RepositoryServiceFactory)
    String configClassName = srvConfigurable.getConfigClass().getName();
    if (CmConfigurationProvider.SERVICE.ATTACH_CLASS.equals(configClassName)) {
      this.adjustRepositoryLink(compConfig);
    }
    else if (CmConfigurationProvider.SERVICE.COMMENT_CLASS.equals(configClassName)) {
      this.adjustRepositoryLink(compConfig);
    }
    else if (CmConfigurationProvider.SERVICE.DISCUSSION_CLASS.equals(configClassName)) {
      this.adjustRepositoryLink(compConfig);
    }
    else if (CmConfigurationProvider.SERVICE.FEEDBACK_CLASS.equals(configClassName)) {
      this.adjustRepositoryLink(compConfig);
    }
    else if (CmConfigurationProvider.SERVICE.PERSNOTE_CLASS.equals(configClassName)) {
      this.adjustRepositoryLink(compConfig);
    }
    else if (CmConfigurationProvider.SERVICE.SUBSCR_CLASS.equals(configClassName)) {
      this.adjustRepositoryLink(compConfig);
    }
    comp.makeReadOnly();
    compConfig.makeReadOnly();
    return comp;
  }

  // ---------------------------------------------------------------------------

  /**
   * Convert and validate configuration data for a filter manager
   *
   * @param cfg TBD: Description of the incoming method parameter
   * @exception ConfigurationException Exception raised in failure situation
   */
  private void convertFilterConfig(DefaultConfiguration cfg) throws ConfigurationException {
    try {
      IConfigClientContext context = IConfigClientContext.createContext(
        ConfigCrutch.getConfigServiceUser());
      IConfigManager cfgManager = Configuration.getInstance().getConfigManager(context);
      IConfigPlugin plugin = cfgManager.getConfigPlugin(CFG_PLUGIN_CM_REPOSITORY_FILTERS);
      IConfigurable[] configurables = plugin.getConfigurables();
            
      for (int i = 0; i < configurables.length; i++) {
        if (configurables[i].getConfigClass().isSubClassOf(CFG.FILTER_CLASSNAME)) {
          if (this.isConfigurableActive(configurables[i])) {
            Properties props = getConfigurableProperties(configurables[i]);
            IConfiguration c = this.convertRF(props, configurables[i].getUri().toString());
            if (c != null) { 
              cfg.addChild(c);
            }
          }
        }
      }
    }
    catch (Exception ex) {
      throw new ConfigurationException("Failed to access CM configuration", ex);
    }
  }

  private DefaultConfiguration convertRF(Properties fltProperties, String location) throws ConfigurationException {
    String id = fltProperties.getProperty(ICmRuntimeConst.CONFIG_ATTRIBUTE_NAME.ID);
    if (id == null) {
      this.throwConfigurationException(
        "Missing attribute: " + ICmRuntimeConst.CONFIG_ATTRIBUTE_NAME.ID,
        location,
        ICmRuntimeConst.CONFIG_ATTRIBUTE_NAME.ID);
    }
    CmConfigurationProvider.log.debugT("convertRF(830)", "converting repository filter config: " + id);
    String className = fltProperties.getProperty(CmConfigurationProvider.CFG.PROPERTY_CLASSNAME);
    if (className == null) {
      this.throwMissingClassException(location, null);
    }

    // Remove filter manager first. This method is also called for "reload"
    CmRegistry.registry.removeFilter(id);

    // Prefix: Only relevant for URI filter
    String prefix = fltProperties.getProperty(CmConfigurationProvider.CFG.FILTERMGR_URIPREFIX);
    // Priority
    String priority = fltProperties.getProperty(CmConfigurationProvider.CFG.FILTERMGR_PRIORITY);
    Integer prio;
    if (priority == null || priority.length() < 1) {
      prio = CmConfigurationProvider.FILTER_PRIO.MAX;
      // max value = lowest priority
    }
    else {
      prio = new Integer(Integer.parseInt(priority));
    }
    if (prio.compareTo(CmConfigurationProvider.FILTER_PRIO.MIN) < 0) {
      this.throwConfigurationException(
        "filter " + id + ": priority less than min.",
        location,
        CmConfigurationProvider.CFG.FILTERMGR_PRIORITY);
    }
    if (prio.compareTo(CmConfigurationProvider.FILTER_PRIO.MAX) > 0) {
      this.throwConfigurationException(
        "filter " + id + ": priority greater than max.",
        location,
        CmConfigurationProvider.CFG.FILTERMGR_PRIORITY);
    }
    // Compile patterns
    PathPatternMatcher mimePattern = null;
    PathPatternMatcher extPattern = null;
    PathPatternMatcher pathPattern = null;
    PathPatternMatcher rtPattern = null;
    try {
      String pattern = fltProperties.getProperty(CmConfigurationProvider.CFG.FILTERMGR_PATH);
      if (pattern != null && pattern.length() > 0) {
        pathPattern = new PathPatternMatcher(pattern);
      }
    }
    catch (PatternSyntaxException ex) {
      this.throwConfigurationException(
        "Error in pattern expression: " + ex.getMessage(),
        location,
        CmConfigurationProvider.CFG.FILTERMGR_PATH);
    }
    try {
      String pattern = fltProperties.getProperty(CmConfigurationProvider.CFG.FILTERMGR_EXTENSIONS);
      if (pattern != null && pattern.length() > 0) {
        extPattern = new PathPatternMatcher(pattern);
      }
    }
    catch (PatternSyntaxException ex) {
      this.throwConfigurationException(
        "Error in pattern expression: " + ex.getMessage(),
        location,
        CmConfigurationProvider.CFG.FILTERMGR_EXTENSIONS);
    }
    try {
      String pattern = fltProperties.getProperty(CmConfigurationProvider.CFG.FILTERMGR_MIMETYPES);
      if (pattern != null && pattern.length() > 0) {
        mimePattern = new PathPatternMatcher(pattern);
      }
    }
    catch (PatternSyntaxException ex) {
      this.throwConfigurationException(
        "Error in pattern expression: " + ex.getMessage(),
        location,
        CmConfigurationProvider.CFG.FILTERMGR_MIMETYPES);
    }
    try {
      String pattern = fltProperties.getProperty(CmConfigurationProvider.CFG.FILTERMGR_RESOURCETYPE);
      if (pattern != null && pattern.length() > 0) {
        rtPattern = new PathPatternMatcher(pattern);
      }
    }
    catch (PatternSyntaxException ex) {
      this.throwConfigurationException(
        "Error in pattern expression: " + ex.getMessage(),
        location,
        CmConfigurationProvider.CFG.FILTERMGR_RESOURCETYPE);
    }

    // List of repository IDs this filter is assigned to
    ArrayList managerIDs = null;
    String managers = fltProperties.getProperty(CmConfigurationProvider.CFG.FILTERMGR_REPMGRS);
    if (managers != null && managers.trim().length() > 0) {
      managerIDs = parseValueList(managers, ",");
    }
    else {
      managerIDs = new ArrayList();
    }
    DefaultConfiguration comp = new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT, location);
    DefaultConfiguration compConfig = new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT_CONFIG, location);
    comp.addChild(compConfig);
    comp.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_CLASS, className);
    comp.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_KEY, id);

    // load the filter class
    Class filterClass = null;
    try {
      filterClass = Class.forName(className, true, loader);
    }
    catch (Throwable x) {
      this.logClassLoadError(id, className, x);
      //this.throwMissingClassException(className, x);
      return null;
    }
    if (FilterProxy.isLegacyClass(filterClass)) {
      // add a proxy if the filter is not conform to the component runtime
      comp.setAttribute(IConfigConst.ATTRIBUTES.COMPONENT_PROXY, FilterProxy.class.getName());
    }

    // Convert properties to IConfiguration
    Enumeration en = fltProperties.propertyNames();
    while (en.hasMoreElements()) {
      String name = (String)en.nextElement();
      String value = fltProperties.getProperty(name);
      compConfig.setAttribute(name, value);
    }

    // Register filter with system properties
    try {
      int type = CmFilterProperties.getFilterType(filterClass);
      CmRegistry.registry.addFilter(
        id,
        new CmFilterProperties(prio, prefix, mimePattern, extPattern, pathPattern, rtPattern, managerIDs, type));
    }
    catch (Exception ex) {
      throw new ConfigurationException(ex.getMessage(), ex);
    }
    comp.makeReadOnly();
    compConfig.makeReadOnly();
    return comp;
  }

  // ---------------------------------------------------------------------------

  private void throwMissingClassException(String className, Throwable ex) throws ConfigurationException {
    String text = "Missing or wrong configuration property \"class\": " + className;
    if (ex != null) {
      text += ", " + ex.getClass().getName() + ": " + ex.getMessage() + ": ";
      text += LoggingFormatter.extractCallstack(ex);
    }
    log.fatalT(text);
    
    throw new ConfigurationException(text);
  }

  private void logClassLoadError(String id, String className, Throwable x) {
    log.errorT("Component <" + id + "> will not be available. Failed to load class <" + className + ">", LoggingFormatter.extractCallstack(x));
  }

  private void throwConfigurationException(String text, String location, String propName) throws ConfigurationException {
    throw new ConfigurationException(text + ", location=" + location + ", property=" + propName);
  }

  /**
   * Reconfigure or restart a component in the CRT
   *
   * @param configurable TBD: Description of the incoming method parameter
   */
  private void sendChangeEventOrRestart(com.sapportals.config.fwk.IConfigurable configurable) {
    String id = configurable.getIdValue();

    if (!(this.isRepManagerConfigurable(configurable)
      || this.isRepServiceConfigurable(configurable)
      || this.isServiceConfigurable(configurable)
      || this.isFilterConfigurable(configurable))) {
      return;
    }

    // Look at component state: If startup failed try to "reload"
    IComponent comp = null;
    boolean reload = false;
    try {
      if (this.isRepManagerConfigurable(configurable)) {
        comp = CmSystem.getInstance().getRepMgrContainer().getComponentManager().lookupComponent(id);
      }
      else if (this.isRepServiceConfigurable(configurable)) {
        comp = CmSystem.getInstance().getRepSrvContainer().getComponentManager().lookupComponent(id);
      }
      else if (this.isServiceConfigurable(configurable)) {
        comp = CmSystem.getInstance().getServiceContainer().getComponentManager().lookupComponent(configurable.getConfigClass().getName());
      }
      else if (this.isFilterConfigurable(configurable)) {
        comp = CmSystem.getInstance().getFilterContainer().getComponentManager().lookupComponent(id);
      }
    }
    catch (Exception ex) {
      log.errorT("sendChangeEventOrRestart(1006)", LoggingFormatter.extractCallstack(ex));
    }

    if (comp != null) {
      if (!(comp instanceof IReconfigurable)) {
        // Note: Component must implement the interface even for changing framework properties
        this.logReconfigurationNotPossible("Component is not reconfigurable: " + id);
        return;
      }
      if (comp instanceof ILifecycleInfo) {
        reload = ((ILifecycleInfo)comp).getState().equals(ComponentState.INIT_ERROR);
      }
    }
    else {
      this.logReconfigurationNotPossible("Component not fond: " + id);
      return;
    }

    if (reload) {
      try {
        this.listener.notifyComponentConfigRemoved(this.calculateComponentURI(configurable));

        ComponentUri containerUri = null;
        IConfiguration cfg = null;
        if (this.isRepManagerConfigurable(configurable)) {
          containerUri = CmConfigurationProvider.CONTAINER_URIS.REPMGR;
          cfg = this.convertRM(configurable);
        }
        else if (this.isRepServiceConfigurable(configurable)) {
          containerUri = CmConfigurationProvider.CONTAINER_URIS.REPSRV;
          cfg = this.convertRS(configurable);
        }
        else if (this.isServiceConfigurable(configurable)) {
          containerUri = CmConfigurationProvider.CONTAINER_URIS.SERVICE;
          cfg = this.convertGS(configurable);
        }
        else if (this.isFilterConfigurable(configurable)) {
          containerUri = CmConfigurationProvider.CONTAINER_URIS.FILTER;
          cfg = this.convertRF(getConfigurableProperties(configurable), configurable.getUri().toString());
        }

        if (cfg != null) {
          this.listener.notifyComponentConfigAdded(containerUri, cfg);
        }
      }
      catch (ConfigurationException ex) {
        log.errorT("sendChangeEventOrRestart(1052)", "Failed to restart component: " + id + ": " + LoggingFormatter.extractCallstack(ex));
      }
    }
    else {
      this.sendChangeEvent(configurable);
    }
  }

  private void sendChangeEvent(com.sapportals.config.fwk.IConfigurable configurable) {
    String id = configurable.getIdValue();
    String location = configurable.getUri().toString();

    DefaultConfiguration compConfig =
      new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT_CONFIG, location, configurable.getProperties(true));

    // Services can have child component relationships to other configurables
    if (this.isServiceConfigurable(configurable)) {
      CmServiceProperties sp = CmRegistry.registry.getServiceProperties(id);
      if (sp == null) {
        CmConfigurationProvider.log.infoT("Reconfiguration: Service does not exist: " + id);
        return;
      }
      try {
        this.convertChildConfigurables(sp.getChildConfigClassNames(), compConfig);
      }
      catch (ConfigurationException ex) {
        CmConfigurationProvider.log.errorT(LoggingFormatter.extractCallstack(ex));
      }
    }
    else if (this.isRepServiceConfigurable(configurable)) {
      CmRepServiceProperties sp = CmRegistry.registry.getRepositoryServiceProperties(id);
      if (sp == null) {
        CmConfigurationProvider.log.infoT("Reconfiguration: Repository Service does not exist: " + id);
        return;
      }
      try {
        this.convertChildConfigurables(sp.getChildConfigClassNames(), compConfig);
      }
      catch (ConfigurationException ex) {
        CmConfigurationProvider.log.errorT(LoggingFormatter.extractCallstack(ex));
      }
    }
    else if (this.isRepManagerConfigurable(configurable)) {
      CmManagerProperties mp = CmRegistry.registry.getManagerProperties(id);
      try {
        this.convertChildConfigurables(mp.getChildConfigClassNames(), compConfig);
      }
      catch (ConfigurationException ex) {
        CmConfigurationProvider.log.errorT(LoggingFormatter.extractCallstack(ex));
      }
    }

    this.listener.notifyComponentConfigChanged(this.calculateComponentURI(configurable), compConfig);
  }

  /**
   * Log message if reconfiguration is denied by framework (not component)
   *
   * @param details TBD: Description of the incoming method parameter
   */
  private void logReconfigurationNotPossible(String details) {
    CmConfigurationProvider.log.infoT("Reconfiguration not possible: " + details);
  }

  /**
   * Adds a new component to the CRT
   *
   * @param configurable TBD: Description of the incoming method parameter
   */
  private void sendAddEvent(com.sapportals.config.fwk.IConfigurable configurable) {
    DefaultConfiguration compCfg = null;
    ComponentUri containerUri = null;
    try {
      if (configurable.getConfigClass().isSubClassOf(CmConfigurationProvider.CFG.MANAGER_CLASSNAME)) {
        if (this.isConfigurableActive(configurable)) {
          compCfg = this.convertRM(configurable);
          containerUri = CmConfigurationProvider.CONTAINER_URIS.REPMGR;
        }
      }
      else if (configurable.getConfigClass().isSubClassOf(CmConfigurationProvider.CFG.REPSERVICE_CLASSNAME)) {
        if (this.isConfigurableActive(configurable)) {
          compCfg = this.convertRS(configurable);
          containerUri = CmConfigurationProvider.CONTAINER_URIS.REPSRV;
        }
      }
      else if (configurable.getConfigClass().isSubClassOf(CmConfigurationProvider.CFG.SERVICE_CLASSNAME)) {
        if (this.isConfigurableActive(configurable)) {
          compCfg = this.convertGS(configurable);
          containerUri = CmConfigurationProvider.CONTAINER_URIS.SERVICE;
        }
      }
      else if (configurable.getConfigClass().isSubClassOf(CmConfigurationProvider.CFG.FILTER_CLASSNAME)) {
        if (this.isConfigurableActive(configurable)) {
          compCfg = this.convertRF(getConfigurableProperties(configurable), configurable.getIdValue());
          containerUri = CmConfigurationProvider.CONTAINER_URIS.FILTER;
        }
      }
    }
    catch (ConfigurationException ex) {
      CmConfigurationProvider.log.errorT(
        "sendAddEvent(1144)",
        "Adding component failed: " + ex.getMessage() + ": " + LoggingFormatter.extractCallstack(ex));
    }
    catch (Exception ex) {
      CmConfigurationProvider.log.errorT(
        "sendAddEvent(1147)",
        "Adding component failed: " + ex.getMessage() + ": " + LoggingFormatter.extractCallstack(ex));
    }
    if (compCfg != null) {
      this.listener.notifyComponentConfigAdded(containerUri, compCfg);
    }
  }

  // ---------------------------------------------------------------------------

  /**
   * Helper method: Checks the "active" property of a configurable
   *
   * @param cfg TBD: Description of the incoming method parameter
   * @return configurableActive
   */
  private boolean isConfigurableActive(com.sapportals.config.fwk.IConfigurable cfg) {
    Boolean result = (Boolean)cfg.getPropertyValueAsObject(CmConfigurationProvider.CFG.ACTIVE_KEY);
    if (result == null) {
      return true;
    }
    return result.booleanValue();
  }

  /**
   * Copies some parameters of other configurables to this config (copied from
   * old ManagerFactory ...)
   *
   * @param config TBD: Description of the incoming method parameter
   * @param pluginName TBD: Description of the incoming method parameter
   * @param parentClass TBD: Description of the incoming method parameter
   * @param keyPrefix TBD: Description of the incoming method parameter
   * @exception Exception Exception raised in failure situation
   */
  private void completeComponentProperties(DefaultConfiguration config, String pluginName, String parentClass, String keyPrefix)
    throws Exception {
    IConfigClientContext context = IConfigClientContext.createContext(ConfigCrutch.getConfigServiceUser());
    IConfigManager cfg = Configuration.getInstance().getConfigManager(context);
    IConfigPlugin plugin = cfg.getConfigPlugin(pluginName);
    if (plugin == null) {
      throw new RuntimeException("missing plugin " + pluginName);
    }
    
    IConfigurable[] configurables = plugin.getConfigurables();      
    for (int i = 0; i < configurables.length; i++) {
      if (configurables[i].getConfigClass().isSubClassOf(parentClass)) {
        this.addConfigurableProperties(
          config,
          (keyPrefix == null ? "" : keyPrefix + '.') + configurables[i].getIdValue(),
          getConfigurableProperties(configurables[i]));
      }
    }
  }

  private static Properties getConfigurableProperties(IConfigurable configurable) {
    Properties properties = new Properties();
    Map map = configurable.getProperties(true);
    Iterator i = map.keySet().iterator();
    while (i.hasNext()) {
      Object key = i.next();
      Object value = map.get(key);
      if (value != null) {
        properties.setProperty(key.toString(), value.toString());
      }
    }
    return properties;
  }
  
  private void addConfigurableProperties(DefaultConfiguration config, String baseKey, Properties additional) {
    Enumeration keys = additional.keys();
    while (keys.hasMoreElements()) {
      String key = (String)keys.nextElement();
      config.setAttribute(baseKey + '.' + key, additional.getProperty(key));
    }
  }

  private void adjustRepositoryLink(DefaultConfiguration cfg) {
    final String REPOSITORY_KEY = "repository";
    String repository = cfg.getAttribute(REPOSITORY_KEY, null);
    if (repository != null) {
      cfg.setAttribute(REPOSITORY_KEY, '/' + repository);
    }
  }

  /**
   * Helper method: Tokenize a string with a separator char into a List
   *
   * @param values TBD: Description of the incoming method parameter
   * @param separator TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public static ArrayList parseValueList(String values, String separator) {
    ArrayList list = new ArrayList();
    if (values == null) {
      return list;
    }
    StringTokenizer st = new StringTokenizer(values, separator);
    while (st.hasMoreTokens()) {
      list.add(st.nextToken().trim());
    }
    return list;
  }

  private boolean isServiceConfigurable(com.sapportals.config.fwk.IConfigurable cfg) {
    return (cfg.getConfigClass().isSubClassOf(CmConfigurationProvider.CFG.SERVICE_CLASSNAME) && this.isConfigurableActive(cfg));
  }

  private boolean isRepManagerConfigurable(com.sapportals.config.fwk.IConfigurable cfg) {
    return (cfg.getConfigClass().isSubClassOf(CmConfigurationProvider.CFG.MANAGER_CLASSNAME) && this.isConfigurableActive(cfg));
  }

  private boolean isRepServiceConfigurable(com.sapportals.config.fwk.IConfigurable cfg) {
    return (cfg.getConfigClass().isSubClassOf(CmConfigurationProvider.CFG.REPSERVICE_CLASSNAME) && this.isConfigurableActive(cfg));
  }

  private boolean isFilterConfigurable(com.sapportals.config.fwk.IConfigurable cfg) {
    return (cfg.getConfigClass().isSubClassOf(CmConfigurationProvider.CFG.FILTER_CLASSNAME) && this.isConfigurableActive(cfg));
  }

  private ComponentUri calculateComponentURI(com.sapportals.config.fwk.IConfigurable cfg) {
    if (this.isRepManagerConfigurable(cfg)) {
      return CmConfigurationProvider.buildRepositoryUri(cfg.getIdValue());
    }
    else if (this.isRepServiceConfigurable(cfg)) {
      return CmConfigurationProvider.buildRepServiceUri(cfg.getIdValue());
    }
    else if (this.isServiceConfigurable(cfg)) {
      return CmConfigurationProvider.buildServiceUri(cfg.getConfigClass().getName());
    }
    else if (this.isFilterConfigurable(cfg)) {
      return CmConfigurationProvider.buildFilterUri(cfg.getIdValue());
    }
    else {
      throw new RuntimeException("Invalid configurable class");
    }
  }

  /**
   * Calculate the config URI of the config class from a delete event.
   *
   * @param e TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  private String calculateConfigClassUri(com.sapportals.config.event.ConfigEvent e) {
    String path = e.getSourceUri().getPath();
    int x = path.lastIndexOf(ConfigUri.PATH_SEPARATOR_CHAR);
    if (x == -1 || x == path.length() - 1) {
      CmConfigurationProvider.log.errorT("calculateConfigClassUri(1318)", "Strange config URI: " + path);
      return "";
    }
    else {
      return path.substring(0, x);
    }
  }

  private IConfigUri[] calculateContainerServiceURIs(com.sapportals.config.fwk.IConfigurable c) throws Exception {
    ArrayList result = new ArrayList();
    result.addAll(CmRegistry.registry.getContainerServiceConfigURIs(c.getConfigClass().getUri().getPath()));
    // cclist parameters can contain the names of abstract config classes
    IConfigClass abstractParent = this.getAbstractSuperConfigClass(c);
    if (abstractParent != null) {
      result.addAll(CmRegistry.registry.getContainerServiceConfigURIs(abstractParent.getUri().getPath()));
    }
    return (IConfigUri[])result.toArray(new IConfigUri[result.size()]);
  }

  private IConfigUri[] calculateContainerManagerURIs(com.sapportals.config.fwk.IConfigurable c) throws Exception {
    ArrayList result = new ArrayList();
    result.addAll(CmRegistry.registry.getContainerManagerConfigURIs(c.getConfigClass().getUri().getPath()));
    // cclist parameters can contain the names of abstract config classes
    IConfigClass abstractParent = this.getAbstractSuperConfigClass(c);
    if (abstractParent != null) {
      result.addAll(CmRegistry.registry.getContainerManagerConfigURIs(abstractParent.getUri().getPath()));
    }
    return (IConfigUri[])result.toArray(new IConfigUri[result.size()]);
  }

  private IConfigClass getAbstractSuperConfigClass(com.sapportals.config.fwk.IConfigurable c) throws Exception {
    IConfigClass cc = c.getConfigClass();
    while (cc != null && !cc.isAbstract()) {
      cc = cc.getParent();
    }
    return cc;
  }
  
}
