/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: CmAdapter.java,v 1.3 2004/02/25 13:53:41 jre Exp $
 */

package com.sapportals.wcm.repository.runtime;

import java.io.InputStream;
import java.util.*;

import com.sap.netweaver.bc.rf.mi.AbstractManager;
import com.sap.tc.logging.Location;
import com.sapportals.portal.security.usermanagement.IServiceUserFactory;
import com.sapportals.portal.security.usermanagement.IUser;
import com.sapportals.portal.security.usermanagement.UserManagementException;
import com.sapportals.wcm.IWcmConst;
import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.crt.component.ComponentException;
import com.sapportals.wcm.crt.component.ComponentState;
import com.sapportals.wcm.crt.component.IComponent;
import com.sapportals.wcm.crt.component.ILifecycleInfo;
import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.repository.batch.*;
import com.sapportals.wcm.repository.enum.LinkType;
import com.sapportals.wcm.repository.enum.NamespaceFilterMode;
import com.sapportals.wcm.repository.enum.SupportedOption;
import com.sapportals.wcm.repository.filter.IFilterFactory;
import com.sapportals.wcm.repository.filter.IFilterManager;
import com.sapportals.wcm.repository.manager.*;
import com.sapportals.wcm.repository.service.IRepositoryService;
import com.sapportals.wcm.repository.service.IRepositoryServiceFactory;
import com.sapportals.wcm.service.IService;
import com.sapportals.wcm.service.IServiceFactory;
import com.sapportals.wcm.util.cache.CacheException;
import com.sapportals.wcm.util.cache.CacheFactory;
import com.sapportals.wcm.util.cache.ICache;
import com.sapportals.wcm.util.cache.ICacheEntry;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.uri.*;
import com.sapportals.wcm.util.usermanagement.WPUMFactory;
import com.sap.netweaver.bc.rf.mi.IManager;
import com.sapportals.wcm.repository.wrapping.RidTransformer;

/**
 * Forwards all calls to the old factory interfaces to the CmSystem class.
 * Starts the CM system when a method is called for the first time. <p>
 *
 * Copyright (c) SAP AG 2002-2004
 * @author m.breitenfelder@sap.com
 * @version $Id: CmAdapter.java,v 1.3 2004/02/25 13:53:41 jre Exp $
 */
public class CmAdapter
	implements IResourceFactory, IManagerFactory, IRepositoryServiceFactory, IServiceFactory, IFilterFactory {

	private final static int SERVICE_USER_CACHE_TTL = 60 * 10;
	private final static String VIRTUAL_ROOT_INTERNAL = "/~virtualroot~";
	private static final String[] anonymousUserNames = new String[] { "anonymous" };
	private static final String[] serviceUserNames =
		new String[] {
			"cmadmin_service",
			"index_service",
			"subscription_service",
			"ice_service",
			"collaboration_service",
			"timebasedpublish_service",
			"notificator_service",
			"action_inbox_service" };
	private static final String[] roleNames =
		new String[] {
			"com.sapportals.km.km_admin",
			"portal_admin",
			"pcd:portal_content/com.sap.pct/administrator/super_admin/com.sap.portal.super_admin_role",
			"pcd:portal_content/com.sap.pct/administrator/system_admin/com.sap.portal.system_admin_role",
			"pcd:portal_content/com.sap.pct/administrator/content_admin/com.sap.portal.content_admin_role" };
	private final static String SERVICE_TYPE_MAP = "servicemap.properties";
	private final static String CONTEXT_NAMESPACE = "repository-framework";
	private final static String CONTEXT_FILTERID = "filterid";
	final static String CONTEXT_CONTENTFILTERID =
		IWcmConst.SAP_WCM_NAMESPACE + "/" + CONTEXT_NAMESPACE + "/" + CONTEXT_FILTERID;
	private final static String QUERY_FILTERID = "ContentFilterID";

	private final static int NORMAL = 0;
	private final static int SLASH = 1;
	private final static int CURRENT = 2;
	private final static int PARENT = 3;

	private final static Location log = Location.getLocation(CmAdapter.class);

	/**
	 * The set of resources we are currently (in this thread) performing lookups
	 * on. Used for loop detection.
	 */
	private final ThreadLocal visitedResources = new ThreadLocal() {
		protected Object initialValue() {
			return new HashSet(7);
		}
	};

	// Cache instance for service user objects
	private ICache serviceUserCache;
	// Map old service type names to new ones
	private HashMap serviceTypeMap;

	static CmAdapter instance;

	private CmAdapter() {
		try {
			this.serviceTypeMap = new HashMap();

			Properties typeMap = new Properties();
			InputStream stream = this.getClass().getResourceAsStream(SERVICE_TYPE_MAP);
			if (stream != null) {
				try {
					typeMap.load(stream);
				}
				finally {
					stream.close();
				}
			}
			else {
				log.errorT("Failed to locate or load " + SERVICE_TYPE_MAP);
			}

			Enumeration enum = typeMap.keys();
			while (enum.hasMoreElements()) {
				String key = (String) enum.nextElement();
				if (log.beDebug()) {
					log.debugT("Service type mapping: " + key + "=" + typeMap.getProperty(key));
				}
				this.serviceTypeMap.put(key, typeMap.getProperty(key));
			}
		}
		catch (Exception ex) {
			log.errorT("Failed to locate or load " + SERVICE_TYPE_MAP + ": " + LoggingFormatter.extractCallstack(ex));
		}
	}

	public static synchronized CmAdapter getInstance() throws Exception {
		if (CmAdapter.instance == null) {
			CmAdapter.instance = new CmAdapter();
		}
		return CmAdapter.instance;
	}

	public IResource getResource(URI uri, IResourceContext context)
		throws ResourceException, AuthorizationRequiredException {
		return getResourceImpl(RID.getRID(uri.toString()), context, false, true);
	}

	public IResource getResource(URI uri, IResourceContext context, boolean createCollections)
		throws ResourceException, AuthorizationRequiredException {
		return getResourceImpl(RID.getRID(uri.toString()), context, createCollections, true);

	}

	public IResource getResource(RID rid, IResourceContext context) throws ResourceException {
		return getResourceImpl(rid, context, false, true);
	}

	public IResource getResource(RID rid, IResourceContext context, boolean createCollections) throws ResourceException {
		return getResourceImpl(rid, context, createCollections, true);

	}

	public IResource getResource(RID rid, boolean resolveInternalLinks, IResourceContext context)
		throws ResourceException {

		return getResourceImpl(rid, context, false, resolveInternalLinks);

	}

	public Object getObject(RID rid, IResourceContext context, Class asClass)
		throws ResourceException, AuthorizationRequiredException {

		IResource res = getResource(rid, context);
		return res.as(asClass);

	}

	public IResourceList getResources(IUriList uriList, ResourceErrors errors, IResourceContext context)
		throws ResourceException {
		if (uriList == null) {
			return new ResourceList();
		}
		RidList ridList = new RidList();
		for (int i = 0; i < uriList.size(); i++) {
			ridList.add(RID.getRID(uriList.get(i).toString()));
		}
		return this.getResources(ridList, errors, context);
	}

	public IResourceList getResources(IRidList ridList, ResourceErrors errors, IResourceContext context)
		throws ResourceException {
		try {
			return getResourcesImpl(ridList, null, errors, null, context);
		}
		catch (WcmException x) {
			throw new ResourceException(x);
		}
	}

	public IResourceList getResources(
		IRidList ridList,
		ResourceErrors errors,
		String[] permissionNames,
		IResourceContext context)
		throws ResourceException {
		try {
			return getResourcesImpl(ridList, null, errors, permissionNames, context);
		}
		catch (WcmException x) {
			throw new ResourceException(x);
		}

	}

	public IResourceList getResources(
		IRidList ridList,
		Map propertyMaps,
		ResourceErrors errors,
		String[] permissionNames,
		IResourceContext context)
		throws ResourceException {
		try {
			return getResourcesImpl(ridList, propertyMaps, errors, permissionNames, context);
		}
		catch (WcmException x) {
			throw new ResourceException(x);
		}
	}

	public IResourceList getResources(
		IRidList ridList,
		Map propertyMaps,
		ResourceErrors errors,
		IResourceContext context)
		throws ResourceException {
		try {
			return getResourcesImpl(ridList, propertyMaps, errors, null, context);
		}
		catch (WcmException x) {
			throw new ResourceException(x);
		}
	}

	public boolean checkExistence(URI uri, IResourceContext context)
		throws ResourceException, AuthorizationRequiredException {

		return checkExistence(RID.getRID(uri.toString()), context);

	}

	public boolean checkExistence(RID rid, IResourceContext context)
		throws ResourceException, AuthorizationRequiredException {
		try {
			return this.findRepositoryManager(rid).checkExistence(rid, context);
		}
		catch (ResourceException x) {
			throw x;
		}
		catch (WcmException x) {
			throw new ResourceException(x);
		}
	}

	public IRepositoryServiceFactory getServiceFactory() throws ResourceException {
		return this;
	}

	public IManagerFactory getManagerFactory() throws ResourceException {
		return this;
	}

	public IFilterFactory getFilterFactory() throws ResourceException {
		return this;
	}

	public IResourceContext getServiceContext() throws ResourceException {
		try {
			CmSystem.getInstance();
		}
		catch (CmStartupException x) {
			log.fatalT(LoggingFormatter.extractCallstack(x));
		}
		return new ResourceContext(null);
	}

	public IResourceContext getServiceContext(String userID) throws ResourceException {
		try {
			CmSystem.getInstance();
		}
		catch (CmStartupException x) {
			log.fatalT(LoggingFormatter.extractCallstack(x));
		}
		return new ResourceContext(retrieveServiceUser(userID));
	}

	public Collection getAllRepositoryManagers() throws ResourceException {
    CmRepMgrContainer mc = null;
    try {
			mc = CmSystem.getInstance().getRepMgrContainer();
		}
		catch (ComponentException e) {
      throw new ResourceException(e);
		}
		catch (CmStartupException e) {
      throw new ResourceException(e);
		}
    if (mc == null) return Collections.EMPTY_LIST;
    
		ArrayList result = new ArrayList();
		String[] childKeys = mc.getComponentManager().listComponentKeys();
		for (int i = 0; i < childKeys.length; i++) {
			try {
				IComponent comp =
					mc.getComponentManager().lookupComponent(childKeys[i]);
				if (comp != null && this.isRunning(comp, childKeys[i])) {
					if (comp instanceof AbstractManager) {
						// new manager (>= SP6)
						result.add(new ManagerAdapter((AbstractManager) comp));
					}
					else {
						result.add(comp);
					}
				}
			}
			catch (ComponentException ex) {
				log.debugT(LoggingFormatter.extractCallstack(ex));
			}
		}
		return result;
	}

	public Collection getAllRepositoryManagers(String className) throws ResourceException {
		try {
			Collection result = this.getAllRepositoryManagers();
			Iterator it = result.iterator();
			while (it.hasNext()) {
				Object o = it.next();
				if (!o.getClass().getName().equals(className)) {
					it.remove();
				}
			}
			return result;
		}
		catch (WcmException ex) {
			throw new ResourceException(ex.getMessage(), ex);
		}
	}

	public IRepositoryManager getRepositoryManager(String id) throws ResourceException {
		try {
      CmRepMgrContainer mc = CmSystem.getInstance().getRepMgrContainer();
      if (mc == null) return null;
			IComponent comp = mc.getComponentManager().lookupComponent(id);
			if (comp != null && this.isRunning(comp, id)) {
				if (comp instanceof AbstractManager) {
					return new ManagerAdapter((AbstractManager) comp);
				}
				else {
					return (IRepositoryManager) comp;
				}
			}
			else {
				return null;
			}
		}
		catch (ComponentException ex) {
			throw new ResourceException(ex.getMessage(), ex);
		}
		catch (CmStartupException x) {
			throw new ResourceException(x);
		}
	}

	public IRepositoryManager getRepositoryManager(RID rid) throws ResourceException {
		try {
			RMAdapter rmAdapter = this.findRepositoryManager(rid);
			if (rmAdapter.isNew()) {
				return new ManagerAdapter(rmAdapter.getNewAbstract());
			}
			else {
				return rmAdapter.getOldAbstract();
			}
		}
		catch (WcmException ex) {
			throw new ResourceException(ex.getMessage(), ex);
		}
	}

	public Collection getAllRepositoryServices(IRepositoryManager repository) throws ResourceException {
		try {
			Collection result = new ArrayList();
			Collection services = CmRegistry.registry.getAllRepositoryServiceIDs(repository.getID());
			Iterator it = services.iterator();
			while (it.hasNext()) {
				result.add(this.getRepositoryService((String) it.next()));
			}
			return result;
		}
		catch (WcmException ex) {
			throw new ResourceException(ex.getMessage(), ex);
		}
	}

	public IRepositoryService getRepositoryService(IRepositoryManager repository, String serviceType)
		throws ResourceException {
		try {
			String serviceID = CmRegistry.registry.getRepositoryServiceID(repository.getID(), serviceType);

			// Maybe type name prior SP5 ?
			if (serviceID == null) {
				String typeName = (String) this.serviceTypeMap.get(serviceType);
				if (typeName != null) {
					serviceID = CmRegistry.registry.getRepositoryServiceID(repository.getID(), typeName);
				}
				if (log.beDebug()) {
					log.debugT("Service type mapping: old=" + serviceType + ", new=" + typeName);
				}
			}

			if (serviceID == null) {
				return null;
			}
			else {
				return this.getRepositoryService(serviceID);
			}
		}
		catch (WcmException ex) {
			throw new ResourceException(ex.getMessage(), ex);
		}
	}

	public boolean isRepositoryServiceAvailable(IRepositoryManager repository, String servicetype)
		throws ResourceException {
		try {
			return (getRepositoryService(repository, servicetype) != null);
		}
		catch (WcmException ex) {
			return false;
		}
	}

	public Collection getAllRepositoryServices(IResource resource) throws ResourceException {
		return this.getAllRepositoryServices(resource.getRepositoryManager());
	}

	public IRepositoryService getRepositoryService(IResource resource, String servicetype) throws ResourceException {
		return this.getRepositoryService(resource.getRepositoryManager(), servicetype);
	}

	public boolean isRepositoryServiceAvailable(IResource resource, String servicetype) throws ResourceException {
		return this.isRepositoryServiceAvailable(resource.getRepositoryManager(), servicetype);
	}

	public IRepositoryService getRepositoryService(String id) throws ResourceException {
		try {
      CmRepSrvContainer sc = CmSystem.getInstance().getRepSrvContainer();
      if (sc == null) return null;      
			IComponent comp = sc.getComponentManager().lookupComponent(id);
			if (comp != null && this.isRunning(comp, id)) {
				return (IRepositoryService) comp;
			}
			else {
				return null;
			}
		}
		catch (ComponentException ex) {
			throw new ResourceException(ex.getMessage(), ex);
		}
		catch (CmStartupException x) {
			throw new ResourceException(x);
		}
	}

	public IService getService(String serviceID) throws ResourceException {
		String typeName = serviceID;
		try {
      CmServiceContainer sc = CmSystem.getInstance().getServiceContainer();
      if (sc == null) return null;
			IComponent comp = sc.getComponentManager().lookupComponent(typeName);

			// Maybe type name prior SP5 ?
			if (comp == null) {
				typeName = (String) this.serviceTypeMap.get(serviceID);
				if (typeName != null) {
					comp = CmSystem.getInstance().getServiceContainer().getComponentManager().lookupComponent(typeName);
				}
				if (log.beDebug()) {
					log.debugT("Service type mapping: old=" + serviceID + ", new=" + typeName);
				}
			}

			if (comp != null && this.isRunning(comp, typeName)) {
				return (IService) comp;
			}
			else {
				return null;
			}
		}
		catch (IllegalStateException ex) {
			throw new ResourceException(ex.getMessage(), ex);
		}
		catch (ComponentException ex) {
			throw new ResourceException(ex.getMessage(), ex);
		}
		catch (CmStartupException x) {
			throw new ResourceException(x);
		}

	}

	public Collection getAllServices() throws ResourceException {
    CmServiceContainer sc = null;
    try {
			sc = CmSystem.getInstance().getServiceContainer();
		}
		catch (ComponentException e) {
      throw new ResourceException(e);
		}
		catch (CmStartupException e) {
      throw new ResourceException(e);
		}
    if (sc == null) return Collections.EMPTY_LIST;
        
		ArrayList result = new ArrayList();
		String[] childKeys = sc.getComponentManager().listComponentKeys();
		for (int i = 0; i < childKeys.length; i++) {
			try {
				IComponent comp =
					sc.getComponentManager().lookupComponent(childKeys[i]);
				if (comp != null && this.isRunning(comp, childKeys[i])) {
					result.add(comp);
				}
			}
			catch (ComponentException ex) {
				log.debugT(LoggingFormatter.extractCallstack(ex));
			}
		}
		return result;
	}

	public boolean isAvailable(String type) {
		try {
			return (getService(type) != null);
		}
		catch (WcmException ex) {
			return false;
		}
	}

	public void releaseService(IService service) throws WcmException {
		CmSystem.getInstance().releaseComponent((IComponent) service);
	}

	public void register(String filterID, String repMgrID) throws WcmException {
		CmSystem.getInstance().attachFilterToRepository(filterID, repMgrID);
	}

	public void register(String filterID, String repMgrID, Properties filterConfig) throws WcmException {
		if (repMgrID == null) {
			// Assume that the path contains a prefix in this case
			String path = (String) filterConfig.get(CmConfigurationProvider.CFG.FILTERMGR_PATH);
			if (path != null) {
				RMAdapter adapter = findRepositoryManager(RID.getRID(path));
				if (adapter != null) {
					repMgrID = adapter.getID();
					path = path.substring(adapter.getPrefix().length());
					filterConfig.setProperty(CmConfigurationProvider.CFG.FILTERMGR_PATH, path);
					filterConfig.setProperty(CmConfigurationProvider.CFG.FILTERMGR_REPMGRS, repMgrID);
				}
			}
		}
		if (repMgrID != null) {
			if (this.getRepositoryManager(repMgrID) == null) {
				throw new WcmException("Failed to register filter, unknown manager: " + repMgrID);
			}
			else {
				filterConfig.setProperty(CmConfigurationProvider.CFG.FILTERMGR_REPMGRS, repMgrID);
			}
		}
		filterConfig.setProperty("name", filterID);
		CmSystem.getInstance().createFilterManager(filterConfig);
	}

	public void unregister(String filterID, String repMgrID) throws WcmException {
		CmSystem.getInstance().detachFilterFromRepository(filterID, repMgrID);
	}

	public IResourceEventBroker getOverallResourceEventBroker() throws WcmException {
		return CmSystem.getInstance().getOverallResourceEventBroker();
	}

	public IResourceBatch getResourceBatch(IResourceList resourceList) {
		return new ResourceBatchImpl(resourceList);
	}

	private ResourceException makeStartupResourceException(CmStartupException ex) {
		return new ResourceException("CM Startup failed: " + ex.getMessage(), ex);
	}

	/**
	 * Gets a service user instance from usermanagement and caches it
	 *
	 * @param userID TBD: Description of the incoming method parameter
	 * @return TBD: Description of the outgoing return value
	 * @exception ResourceException Exception raised in failure situation
	 */
	private IUser retrieveServiceUser(String userID) throws ResourceException {
		try {
			if (this.serviceUserCache == null) {
				// create memory cache instance
				Properties props = new Properties();
				props.setProperty("type", "memory");
				props.setProperty("class", "com.sapportals.wcm.util.cache.memory.MemoryLRUCache");
				props.setProperty("capacity", "10");
				props.setProperty("maxsize", "0");
				props.setProperty("maxentrysize", "0");
				props.setProperty("defaulttimetolive", String.valueOf(SERVICE_USER_CACHE_TTL));
				props.setProperty("averageentrysize", "0");
				props.setProperty("singleton", "true");
				props.setProperty("autodelayexpiration", "false");
				CacheFactory cf = CacheFactory.getInstance();
				this.serviceUserCache = cf.getCache(CacheFactory.getUniqueCacheID(), props);
				if (this.serviceUserCache == null) {
					throw new ResourceException("getCache() returned null");
				}
			}
			IUser serviceUser = null;
			synchronized (this.serviceUserCache) {
				ICacheEntry entry = this.serviceUserCache.getEntry(userID);
				if (entry != null) {
					serviceUser = (IUser) entry.getObject();
				}
				if (serviceUser == null) {
					IServiceUserFactory suf = WPUMFactory.getServiceUserFactory();
					if (suf == null) {
						throw new ResourceException("WPUMFactory.getServiceUserFactory() returned null");
					}
					serviceUser = suf.getServiceUser(userID);
					if (serviceUser == null) {
						throw new ResourceException("getServiceUser() returned null for userID " + userID);
					}
					if (log.beDebug()) {
						log.debugT("retrieved service user: " + userID);
					}
					this.serviceUserCache.addEntry(userID, serviceUser);
				}
			}
			return serviceUser;
		}
		catch (UserManagementException ex) {
			throw new ResourceException("User management exception: " + ex.getMessage(), ex);
		}
		catch (CacheException ex) {
			throw new ResourceException("Cache exception: " + ex.getMessage(), ex);
		}
	}

	private void checkServiceUsers() {
		// Initialize security provider
		iaik.security.provider.IAIK.addAsProvider();

		// Create anonymous users
		try {
			com.sap.security.api.IUser user;
			com.sap.security.api.IUserAccount userAccount;
			com.sap.security.api.IUserFactory userFactory = com.sap.security.api.UMFactory.getUserFactory();
			com.sap.security.api.IUserAccountFactory userAccountFactory =
				com.sap.security.api.UMFactory.getUserAccountFactory();

			for (int i = 0; i < anonymousUserNames.length; i++) {
				// Create anonymous user
				try {
					try {
						user = userFactory.getUserByUniqueName(anonymousUserNames[i]);
					}
					catch (Throwable throwable) {
						user = null;
					}
					if (user == null) {
						try {
							log.infoT("Constructing new anonymous user " + anonymousUserNames[i]);
							com.sap.security.api.IUserMaint userMutable = userFactory.newUser(anonymousUserNames[i]);
							userMutable.commit();
							log.infoT("Anonymous user " + anonymousUserNames[i] + " constructed.");
						}
						catch (Throwable throwable) {
							log.errorT("Failed to construct anonymous user " + anonymousUserNames[i] + ": " + throwable);
							// LoggingFormatter.extractCallstack(throwable));
						}
					}
					else {
						log.debugT("Anonymous user " + anonymousUserNames[i] + " already exists.");
					}
					try {
						userAccount = userAccountFactory.getUserAccountByLogonId(anonymousUserNames[i]);
					}
					catch (Throwable throwable) {
						userAccount = null;
					}
					if (userAccount == null) {
						try {
							log.infoT("Constructing new anonymous user account " + anonymousUserNames[i]);
							com.sap.security.api.IUserAccount userAccountMutable =
								userAccountFactory.newUserAccount(anonymousUserNames[i], user.getUniqueID());
							userAccountMutable.commit();
							log.infoT("Anonymous user account " + anonymousUserNames[i] + " constructed.");
						}
						catch (Throwable throwable) {
							log.errorT("Failed to construct anonymous user account " + anonymousUserNames[i] + ": " + throwable);
							// LoggingFormatter.extractCallstack(throwable));
						}
					}
					else {
						log.debugT("Anonymous user account " + anonymousUserNames[i] + " already exists.");
					}

				}
				catch (Throwable throwable) {
					// Handle caught exception
					log.debugT("Caught exception creating anonymous user "
							+ anonymousUserNames[i]
							+ ": "
							+ LoggingFormatter.extractCallstack(throwable));
				}
			}
		}
		catch (Exception exception) {
			// Handle caught exception
			log.debugT("Caught exception creating anonymous users: " + LoggingFormatter.extractCallstack(exception));
		}

		// Create service users
		try {
			com.sap.security.api.IUser user;
			com.sap.security.api.IUserFactory userFactory = com.sap.security.api.UMFactory.getUserFactory();
			com.sap.security.api.srvUser.IServiceUserFactory serviceUserFactory =
				com.sap.security.api.UMFactory.getServiceUserFactory();
			for (int i = 0; i < serviceUserNames.length; i++) {
				// Create service user
				try {

					try {
						user = serviceUserFactory.getServiceUser(serviceUserNames[i]);
					}
					catch (Throwable throwable) {
						user = null;
					}
					if (user == null) {
						try {
							log.infoT("Constructing new service user " + serviceUserNames[i]);
							serviceUserFactory.createServiceUser(serviceUserNames[i]);
							log.infoT("Service user " + serviceUserNames[i] + " constructed.");
						}
						catch (Throwable throwable) {
							log.errorT("Failed to construct service user " + serviceUserNames[i] + ": " + throwable);
						}
					}
					else {
						log.debugT("Service user " + serviceUserNames[i] + " already exists.");
					}

				}
				catch (Throwable throwable) {
					// Handle caught exception
					log.debugT("Caught exception creating service user "
							+ serviceUserNames[i]
							+ ": "
							+ LoggingFormatter.extractCallstack(throwable));
				}
			}
		}
		catch (Exception exception) {
			// Handle caught exception
			log.debugT("Caught exception creating service users: " + LoggingFormatter.extractCallstack(exception));
		}

		// Create roles
		try {
			com.sap.security.api.IRole role;
			com.sap.security.api.IRoleFactory roleFactory = com.sap.security.api.UMFactory.getRoleFactory();
			for (int i = 0; i < roleNames.length; i++) {
				// Create role
				try {
					try {
						role = roleFactory.getRoleByUniqueName(roleNames[i]);
					}
					catch (Throwable throwable) {
						role = null;
					}
					if (role == null) {
						try {
							log.infoT("Constructing new role " + roleNames[i]);
							com.sap.security.api.IRole roleMutable = roleFactory.newRole(roleNames[i]);
							roleMutable.commit();
							log.infoT("Role " + roleNames[i] + " constructed.");
						}
						catch (Throwable throwable) {
							log.errorT("Failed to construct role " + roleNames[i] + ": " + throwable);
						}
					}
					else {
						log.debugT("Role " + roleNames[i] + " already exists.");
					}

				}
				catch (Throwable throwable) {
					// Handle caught exception
					log.debugT("Caught exception creating role " + roleNames[i] + ": " + LoggingFormatter.extractCallstack(throwable));
				}
			}
		}
		catch (Exception exception) {
			// Handle caught exception
			log.debugT("Caught exception creating roles: " + LoggingFormatter.extractCallstack(exception));
		}
	}

	private IResource getResourceImpl(
		RID rid,
		IResourceContext context,
		boolean createCollections,
		boolean resolveInternalLinks)
		throws ResourceException, AuthorizationRequiredException {      
    if (CmAdapter.log.beDebug()) {
      CmAdapter.log.debugT("entering getResourceImpl(), rid=" + rid.toString());
      CmAdapter.log.debugT(LoggingFormatter.extractCallstack(new Exception("getResourcesImpl() stacktrace")));
    }            
		if (context == null) {
			throw new ResourceException("No resource context");
		}
		if (rid == null) {
			return null;
		}

		// Query paramater "ContentFilterID"
		Properties query = rid.getQueryParameter();
		if (query != null && query.size() > 0) {
			String filterID = query.getProperty(QUERY_FILTERID);
			if (filterID != null && filterID.length() > 0) {
				context.setValue(CONTEXT_CONTENTFILTERID, filterID);
			}
		}

		rid = normalizePath(rid);

		// URI filter
		try {
			RID filteredURI = CmSystem.getInstance().getFilterHandler().applyUriFilter(rid, context);
			if (filteredURI == null) {
				throw new InvalidUriException("URI Filter(s) returned null");
			}

			IResource res = this.findManagerAndResource(filteredURI, context, createCollections, resolveInternalLinks);
			if (res == null) {
				return null;
			}

			// Namespace filter
			res = this.applyNamespaceReadFilter(res);
			if (res == null) {
				log.debugT("Resource removed by namespace filter: " + filteredURI);
				return null;
			}
			else {
				// Set filteredURI as access URI (e.g. /~alias~ will be removed and placeholder are resolved)
				setAccessRID(res, filteredURI);
				return res;
			}
		}
		catch (CmStartupException x) {
			return null;
		}
		catch (ResourceException x) {
			throw x;
		}
		catch (WcmException x) {
			throw new ResourceException(x);
		}
	}

	private IResourceList getResourcesImpl(
		IRidList ridList,
		Map propertyMaps,
		ResourceErrors errors,
		String[] permissionNames,
		IResourceContext context)
		throws WcmException, ResourceException {
		if (CmAdapter.log.beDebug()) {
			CmAdapter.log.debugT(
				"entering getResourcesImpl(), ridList="
					+ ridList.toString()
					+ ", propertyMaps="
					+ ((propertyMaps==null)?"null":propertyMaps.toString())
					+ ", permissionNames="
					+ permissionNames.toString());
		}     
		if (context == null) {
			throw new ResourceException("No resource context");
		}
		if (ridList == null) {
			throw new ResourceException("Parameter ridList is null");
		}
		// Squeeze all URIs through the URI filters
		IRidList filteredUriList = new RidList();
		RID rid = null;
		for (int i = 0; i < ridList.size(); i++) {
			rid = ridList.get(i);
			rid = CmSystem.getInstance().getFilterHandler().applyUriFilter(rid, context);
			if (rid != null) {
				filteredUriList.add(rid);
			}
		}
		ResourceList result = new ResourceList();
		// Maps repository manager (adapter) to a set of RIDs
		HashMap ridMap = new HashMap();
		// Separate the URIs of different managers
		RMAdapter rm = null;
		RidSet set = null;
		for (int i = 0; i < filteredUriList.size(); i++) {
			rid = filteredUriList.get(i);
			try {
				rm = this.findRepositoryManager(rid);
				if (rm != null) {
					set = (RidSet) ridMap.get(rm);
					if (set == null) {
						set = new RidSet();
						ridMap.put(rm, set);
					}
					set.add(rid);
				}
			}
			catch (InvalidUriException ex) {
				if (errors != null) {
					errors.append(ex);
				}
			}
		}
		// Call getResources for all repositories and collect result
		ResourceErrors newErrors = null;
		IResourceList resources = null;
		Iterator it = ridMap.keySet().iterator();
		String id = null;
		while (it.hasNext()) {
			rm = (RMAdapter) it.next();
			set = (RidSet) ridMap.get(rm);
			if (set != null) {
				if (errors != null) {
					newErrors = new ResourceErrors();
				}
				RidList l = new RidList(set.toArray());
				try {
					if (permissionNames == null) {
						resources = rm.getResources(l, propertyMaps, newErrors, null, context);
					}
					else {
						resources = rm.getResources(l, propertyMaps, newErrors, permissionNames, context);
					}
				}
				catch (ResourceException ex) {
          CmAdapter.log.errorT(LoggingFormatter.extractCallstack(ex));
					if (errors != null) {
						errors.append(ex);
					}
          continue;
				}
				if (errors != null && newErrors.getResourceException() != null) {
					errors.append(newErrors);
				}
        if (resources == null) {
          // Note: Although is not stated in the old and new MI javadoc that null is prohibited
          //       it should also be fixed by the RM implementation
          CmAdapter.log.errorT("The repository manager <" + rm.getID() + "> returned \"null\" for getResources()/lookup(List)");
          resources = new ResourceList();
        }        
				// map internal links to collections
				this.mapLink(resources, rm.getIRepositoryManagerForClients());
				// Calculate which resources do not exists in this repository and call findResource()
				RidSet notFound = new RidSet(set);
				IResource res = null;
				for (int i = 0; i < resources.size(); i++) {
					res = resources.get(i);
					rid = res.getRID();
					if (set.contains(rid)) {
						notFound.remove(rid);
					}
				}
				if (notFound.size() > 0) {
					IRidIterator nit = notFound.iterator();
					while (nit.hasNext()) {
						rid = nit.next();
						try {
							res = this.findResource(rm, rid, context, false, true);
							if (res != null) {
								boolean visible = true;
								if (permissionNames != null) {
									visible = this.checkPermission(res, permissionNames, context);
								}
								if (visible) {
									this.setAccessRID(res, rid);
									resources.add(res);
								}
							}
						}
						catch (ResourceException ex) {
							if (errors != null) {
								errors.append(ex);
							}
						}
					}
				}
				// Apply namespace read filters
				if (resources != null && resources.size() > 0) {
					resources = this.applyNamespaceReadFilter(rm.getIRepositoryManagerForClients(), resources);
					result.addAll(resources);
				}
			}
		}
		return result;
	}

	/**
	 * Apply namespace filter to a single resource
	 *
	 * @param resource TBD: Description of the incoming method parameter
	 * @return The filtered resource, can be null if the filter decides that the
	 *      resource is not visible
	 * @exception ResourceException Exception raised in failure situation
	 */
	private IResource applyNamespaceReadFilter(IResource resource) throws WcmException, ResourceException {
		IResourceList list = new ResourceList();
		list.add(resource);
		// collection = null -> Filter only the sinlgle resource in the list
		list =
			CmSystem.getInstance().getFilterHandler().applyNamespaceReadFilter(
				resource.getRepositoryManager(),
				null,
				list,
				NamespaceFilterMode.GET_RESOURCE);
		// If the resource is still there it must be the only resource in the list, because the collection was null
		if (list != null && list.size() == 1) {
			return list.get(0);
		}
		else {
			return null; // Resource is not visible
		}
	}

	/**
	 * Apply namespace filter to a list of resources
	 *
	 * @param rm TBD: Description of the incoming method parameter
	 * @param list TBD: Description of the incoming method parameter
	 * @return The filtered list
	 * @exception ResourceException Exception raised in failure situation
	 * @rm The repostiory manager instance
	 * @list The resource list to filter
	 */
	private IResourceList applyNamespaceReadFilter(IRepositoryManager rm, IResourceList list)
		throws WcmException, ResourceException {
		return CmSystem.getInstance().getFilterHandler().applyNamespaceReadFilter(
			rm,
			null,
			list,
			NamespaceFilterMode.GET_RESOURCE);
	}

	/**
	 * Path normalization (remove ".." and "."). Copied from ResourceUrl class.
	 *
	 * @param rid TBD: Description of the incoming method parameter
	 * @return TBD: Description of the outgoing return value
	 */
	private static RID normalizePath(RID rid) {
		StringBuffer sb = new StringBuffer(rid.getPath());

		int state = NORMAL;
		int i = 0;
		while (i < sb.length()) {
			char c = sb.charAt(i);
			switch (state) {
				case NORMAL :
					if (c == '/') {
						state = SLASH;
					}
					break;
				case SLASH :
					if (c == '.') {
						state = CURRENT;
					}
					else if (c == '/') {
						// We are at the end of "//", remove one "/"
						sb.deleteCharAt(i);
						--i;
					}
					else {
						state = NORMAL;
					}
					break;
				case CURRENT :
					if (c == '.') {
						state = PARENT;
					}
					else if (c == '/') {
						// we are at the end of "/./", remove it
						sb.delete(i - 2, i);
						i -= 2;
						state = SLASH;
					}
					else {
						state = NORMAL;
					}
					break;
				case PARENT :
					if (c == '/') {
						// We are at the end of "/../", remove it up to the previous "/x"
						// where 'x' is any character except '.'
						//
						int lastSlash = -1;
						boolean seenNonDot = false;
						forloop : for (int j = i - 4; j >= 0; --j) {
							switch (sb.charAt(j)) {
								case '/' :
									if (seenNonDot) {
										lastSlash = j;
									}
									break forloop;
								case '.' :
									break;
								default :
									seenNonDot = true;
									break;
							}
						}
						if (lastSlash >= 0) {
							sb.delete(lastSlash, i);
							i = lastSlash;
						}
						else {
							// We cannot go back any further, leave the .. as it is
						}
						state = SLASH;
					}
					else {
						state = NORMAL;
					}
					break;
				default :
					break;
			}
			++i;
		}

		// end of buffer reached
		switch (state) {
			case CURRENT :
				// buffer ends in "/.", remove "."
				sb.delete(i - 1, sb.length());
				break;
			case PARENT :
				// buffer ends in "/..", remove up to previous "/" or remove ".."
				int lastSlash = -1;
				for (int j = i - 4; j >= 0; --j) {
					if (sb.charAt(j) == '/') {
						lastSlash = j;
						break;
					}
				}
				if (lastSlash >= 0) {
					sb.delete(lastSlash + 1, sb.length());
				}
				else {
					sb.delete(i - 2, sb.length());
				}
				break;
			default :
				break;
		}

		return new RID(sb.toString(), rid.getQuery());
	}

	private IResource findManagerAndResource(
		RID rid,
		IResourceContext context,
		boolean createCollections,
		boolean resolveInternalLinks)
		throws WcmException, ResourceException, AuthorizationRequiredException {
		return this.findResource(this.findRepositoryManager(rid), rid, context, createCollections, resolveInternalLinks);
	}

	public IResource findResource(RMAdapter mgr, RID rid, IResourceContext context, boolean create, boolean resolve)
		throws WcmException, ResourceException, AuthorizationRequiredException {
		// Get the Set of RIDs we are already performing lookups for (in the context of this thread).
		// If the Set is empty, it means that we are called from a framework client ("outside").
		// If the Set is nonemoty, it means that we are called from a repository manager ("inside").
		//
		// If we encounter a lookup of a RID already seen, we're in a loop.
		//
		// For outside lookups we perform link examination, e.g. reporting links to collections as collection.
		//
		// Finally, we remove the RIDs from the Set again, so "outside" calls will leave an empty Set.
		//
		Set seen = (Set) visitedResources.get();
		boolean outsideCall = seen.isEmpty();
		IResource res = null;
		if (seen.contains(rid)) {
			if (log.beDebug()) {
				log.debugT("loop detected(" + rid + "), seen " + seen + ": " + LoggingFormatter.extractCallstack(new Exception()));
			}
			throw new LinkLoopException("findResource: already seen RID: " + rid + " in " + seen, rid);
		}
		try {
			seen.add(rid);
			if (outsideCall) {
				res = this.mapLink(mgr.getResource(rid, context), true, true);
			}
			else {
				res = mgr.getResource(rid, context);
			}
			if (res != null) {
				return res;
			}

			boolean followLinksToCollections = true;
			try {
				ISupportedOptionSet set = mgr.getSupportedOptions(null);
				followLinksToCollections = !set.isSupported(SupportedOption.RESOLVES_LINKS);
			}
			catch (Exception ex) {
				// Ignore: some repository manager do not support this call on a null-resource.
				// assume that links are not resolved internally then.
				log.debugT(LoggingFormatter.extractCallstack(ex));
			}
			if (create || followLinksToCollections) {
				// resource not known, might be child of a link resource
				List names = rid.split();
				int nsize = names.size() - 1; // we know already that the last one does not exist
				RID path = RID.getRID("/");
				RID unknownURI = rid;
				IResource knownResource = null;
				int index;
				for (index = 0; index < nsize; ++index) {
					path = path.add((String) names.get(index));
					try {
						res = this.mapLink(mgr.getResource(path, context), true, true);
					}
					catch (AccessDeniedException ex) {
						knownResource = null;
						break;
					}
					if (res == null) {
						break;
					}
					knownResource = res;
					unknownURI = unknownURI.removeRoot();
				}
				if (knownResource != null) {
					// If the name identifies a link to a collection try to find a new manager and the resource
					if (knownResource.getLinkType().equals(LinkType.INTERNAL) && knownResource.isCollection()) {
						if (!resolve) {
							throw new InternalLinkException(
								"is internal link to a collection",
								knownResource.getTargetURL().toString(),
								unknownURI.toString());
						}
						String targetURL = knownResource.getTargetURL().toString();
						return this.mapLink(
							this.findManagerAndResource(RID.getRID(targetURL).add(unknownURI), context, create, resolve),
							true,
							true);
					}
					else if (knownResource.getLinkType().equals(LinkType.EXTERNAL)) {
						throw new ExternalLinkException(
							"is external link",
							knownResource.getTargetURL().toString(),
							unknownURI.toString());
					}
					else if (create && knownResource.isCollection()) {
						// Must create all collections in the URI
						CollectionImpl createHere = (CollectionImpl) knownResource;
						while (index <= nsize) {
							String newName = (String) names.get(index);
							try {
								createHere = (CollectionImpl) createHere.createCollection(newName, null);
							}
							catch (NameAlreadyExistsException e) {
								// Hmm, resource created in the meantime ?
								RID crid = createHere.getRID().add(newName);
								IResource existing = mapLink(mgr.getResource(crid, context), true, false);
								if (existing == null || !existing.isCollection()) {
									// We give up
									throw e;
								}
								createHere = (CollectionImpl) existing;
							}
							index++;
						}
						return createHere;
					}
				}
			}
			return null;
		}
		finally {
			if (outsideCall) {
				seen.clear();
			}
			else {
				seen.remove(rid);
			}
		}
	}

	/**
	 * Eaxmine the resource if it is a link to a collection.
	 *
	 * @param resource to examine
	 * @param checkManager check if manager needs such examination
	 * @return original or mapped resource
	 * @exception ResourceException Exception raised in failure situation
	 */
	final static IResource mapLink(IResource resource, boolean checkManager) throws ResourceException {
		if (resource != null
			&& !resource.isCollection()
			&& LinkType.INTERNAL.equals(resource.getLinkType())
			&& !resource.isRevision()) {
			// Hmmm, could be a link to a collection. Does the manager resolve links?
			if (!checkManager || needsLinkMapping(resource.getRepositoryManager())) {
				IResource target = resource.getTargetResource();
				if (target != null && target instanceof ICollection) {
					return new CollectionLinkImpl(resource, (ICollection) target);
				}
			}
		}
		return resource;
	}

	/**
	 * Examines the resources in the list for links to collections. Replaces links
	 * to collections with collection resources.
	 *
	 * @param list of resources
	 * @param mgr repository manager resources come from
	 */
	final void mapLink(IResourceList list, IRepositoryManager mgr) {
    if (list.size() == 0) return;
		if (needsLinkMapping(mgr)) {
			Set seen = (Set) visitedResources.get();
			boolean cleanup = seen.isEmpty();
			if (cleanup) {
				seen.add("");
			}
			try {
				for (int i = 0, n = list.size(); i < n; ++i) {
					IResource resource = list.get(i);
					try {
						IResource mapped = mapLink(resource, false);
						if (mapped != resource) {
							list.set(i, mapped);
						}
					}
					catch (ResourceException e) {
						if (log.beDebug()) {
							log.debugT("mapping resource " + resource + ": " + LoggingFormatter.extractCallstack(e));
						}
					}
				}
			}
			finally {
				if (cleanup) {
					seen.clear();
				}
			}
		}
	}

	/**
	 * Determine if the manager needs links to collections mapped to collection
	 * resources.
	 *
	 * @param mgr to check
	 * @return if manager needs collection link handling
	 */
	private static boolean needsLinkMapping(IRepositoryManager mgr) {
		try {
			ISupportedOptionSet options = mgr.getSupportedOptions(null);
			return (options.isSupported(SupportedOption.LINKING) && !options.isSupported(SupportedOption.RESOLVES_LINKS));
		}
		catch (Exception e) {
			log.debugT("Some manager do not like null arguments here: " + LoggingFormatter.extractCallstack(e));
		}
		return true;
	}

	/**
	 * Eaxmine the resource if it is a link to a collection.
	 *
	 * @param resource to examine
	 * @param catchExceptions determines if Exceptions during lookup are thrown or
	 *      silently ignored
	 * @param checkManager TBD: Description of the incoming method parameter
	 * @return original or mapped resource
	 * @exception ResourceException Exception raised in failure situation
	 */
	private final IResource mapLink(IResource resource, boolean checkManager, boolean catchExceptions)
		throws ResourceException {
		if (catchExceptions) {
			try {
				return mapLink(resource, true);
			}
			catch (LinkLoopException e) {
				if (log.beInfo()) {
					log.infoT(
						"link loop("
							+ resource
							+ "): "
							+ e.getMessage()
							+ ": "
							+ LoggingFormatter.extractCallstack(new Exception()));
				}
				return resource;
			}
			catch (ResourceException e) {
				if (log.beDebug()) {
					log.debugT("mapping resource " + resource + ": " + LoggingFormatter.extractCallstack(e));
				}
				return resource;
			}
		}
		else {
			return mapLink(resource, true);
		}
	}

	/**
	 * Tries to find a repository manager for the given RID.
	 *
	 * @param rid The RID starting with a prefix
	 * @return The manager instance
	 * @exception ResourceException
	 */
	final RMAdapter findRepositoryManager(RID rid) throws WcmException, ResourceException {
		if (rid == null) {
			throw new InvalidUriException("Invalid RID: null");
		}
		if (log.beDebug()) {
			log.debugT("entering findRepositoryManager(): rid=" + rid);
		}
		String path = rid.getPath();
		String prefix = null;
		if (path.length() == 0) {
			// Empty RID is allowed => root
			path = "/";
		}
		if (path.length() == 1) {
			if (path.charAt(0) == '/') {
				prefix = path = VIRTUAL_ROOT_INTERNAL;
			}
			else {
				throw new InvalidUriException("Invalid RID: " + path + " must start with /", rid);
			}
		}
		else {
			if (path.charAt(0) != '/' || path.indexOf("\\") > -1 || path.indexOf("//") > -1) {
				throw new InvalidUriException("Invalid RID: " + path, rid);
			}
			// uri starts with '/', find root collection
			int index = path.indexOf('/', 1);
			if (index > 0) {
				prefix = path.substring(0, index);
			}
			else {
				prefix = path;
			}
		}
		String id = (String) CmRegistry.registry.getRepositoryID(prefix);
		if (id == null) {
			return this.handleUnknownManager(prefix, rid);
		}
		IComponent rm = null;
		try {
			rm = CmSystem.getInstance().getRepMgrContainer().getComponentManager().lookupComponent(id);
		}
		catch (ComponentException ex) {
			log.errorT(LoggingFormatter.extractCallstack(ex));
		}
		if (rm != null && this.isRunning(rm, id)) {
			return new RMAdapter(rm);
		}
		else {
			return this.handleUnknownManager(prefix, rid);
		}
	}

	/**
	 * Returns a mock manager for unit testing or throws InvalidUriException.
   * Note: The mock manager is not a CRT component !
	 */
	private RMAdapter handleUnknownManager(String prefix, RID rid) throws WcmException {
		if (CmSystem.testabilityMode) {
			Object rm = CmSystem.getInstance().getMockManager(prefix.substring(1));
			if (rm != null) {
				return new RMAdapter(rm);
			}
		}
		throw new InvalidUriException("Invalid RID: No repository manager found: " + prefix, rid);
	}

	/**
	 * Returns true if component state = RUNNING
	 */
	private boolean isRunning(IComponent comp, String key) {
		if (comp == null) {
			throw new NullPointerException("parameter comp is null");
		}
		if (comp instanceof ILifecycleInfo) {
			if (((ILifecycleInfo) comp).getState().equals(ComponentState.INIT_ERROR)) {
				log.debugT("Component state is INIT_ERROR: " + key);
				return false;
			}
			else {
				return true;
			}
		}
		else {
			// This should not happen
			log.errorT("Component does not implement ILifecycleInfo: " + comp.getClass().getName());
			return false;
		}
	}

	/**
	 * Called by filter handler
	 *
	 * @param id TBD: Description of the incoming method parameter
	 * @return filterManager
	 * @exception WcmException Exception raised in failure situation
	 */
	final IFilterManager getFilterManager(String id) throws WcmException {
		try {
      CmRepFltContainer fc = CmSystem.getInstance().getFilterContainer();
      if (fc == null) return null;
			IComponent comp = fc.getComponentManager().lookupComponent(id);
			if (comp != null && this.isRunning(comp, id)) {
				return (IFilterManager) comp;
			}
			else {
				return null;
			}
		}
		catch (ComponentException ex) {
			throw new WcmException(ex.getMessage(), ex);
		}
	}

	/**
	 * Call security manager to check for a list of permissions
	 */
	private boolean checkPermission(IResource res, String[] permissionNames, IResourceContext context)
		throws ResourceException {
		if (permissionNames.length == 0) {
			return true;
		}
		ISecurityManager sm = res.getRepositoryManager().getSecurityManager(res);
		if (sm == null) {
			return true;
		}
		boolean allowed = true;
		PermissionList permList = new PermissionList();
		for (int i = 0; i < permissionNames.length; i++) {
			permList.add(sm.getPermission(permissionNames[i]));
		}
		try {
			allowed = sm.isAllowed(res, context.getUser(), permList);
		}
		catch (NotSupportedException ex) {
			IPermissionListIterator it = permList.iterator();
			while (it.hasNext() && allowed) {
				IPermission perm = it.next();
				allowed = allowed && sm.isAllowed(res, context.getUser(), perm);
			}
		}
		return allowed;
	}

	private void setAccessRID(IResource resource, RID rid) throws ResourceException {
		if (resource instanceof ResourceImpl) {
			((ResourceImpl) resource).setAccessRID(rid);
		}
		else if (resource instanceof CollectionLinkImpl) {
			((CollectionLinkImpl) resource).setAccessRID(rid);
		}
	}
}