/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.runtime;

import com.sapportals.wcm.crt.component.*;
import com.sapportals.wcm.crt.configuration.*;

import java.util.*;

/**
 * Abstract base class for all CM framework containers. <p>
 *
 * Copyright (c) SAP AG 2002
 *
 * @author Markus Breitenfelder
 * @version $Id:$
 */
public class CmAbstractContainer implements IContainer, IThreadSafe,
  IComponentInfo, ILifecycleInfo {
  private IContainerManager containerManager;
  private ConfigurationException dummyException;

  public CmAbstractContainer() { }

  public IComponentManager getComponentManager() {
    return this.containerManager;
  }

  public final void contextualize(IContext context)
    throws ContextException {
    this.containerManager = context.getContainerManager();
    if (this.containerManager == null) {
      throw new ContextException("No container manager in context");
    }
  }

  // ---------------------------------------------------------------------------
  // interface IComponentInfo
  // ---------------------------------------------------------------------------

  public String getName() {
    return "";
  }

  public String getDescription(Locale locale) {
    return "";
  }

  public Properties getProperties() {
    return null;
  }

  // ---------------------------------------------------------------------------
  // interface ILifecycleInfo
  // ---------------------------------------------------------------------------

  public ComponentState getState() {
    // Combine state of all child components:
    // If not all components are running and config is OK create a fake state and exception so
    // that the CRT repository will return a WARN or ERROR mimetype (yellow or red bullet!).
    String[] childKeys = this.getComponentManager().listComponentKeys();
    int countNotRunning = 0;
    int countConfigException = 0;
    IComponent comp = null;
    for (int i = 0; i < childKeys.length; i++) {
      try {
        comp = this.getComponentManager().lookupComponent(childKeys[i]);
        ComponentState state = ((ILifecycleInfo)comp).getState();
        if (state != null && !state.equals(ComponentState.RUNNING)) {
          countNotRunning++;
        }
        if (((ILifecycleInfo)comp).getLastConfigurationException() != null) {
          countConfigException++;
        }
      }
      catch (Exception ex) {
        com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(
          com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(ex));
      }
      finally {
        if (comp != null) {
          this.getComponentManager().releaseComponent(comp);
        }
      }
    }
    this.dummyException = null;
    if (countNotRunning == childKeys.length) {
      return ComponentState.STOPPED;
    }
    else if (countNotRunning > 0 || countConfigException > 0) {
      this.dummyException = new ConfigurationException("");
    }
    return ComponentState.RUNNING;
  }

  public ConfigurationException getLastConfigurationException() {
    return this.dummyException;
  }

  public StartupException getStartupException() {
    return null;
  }

  public Date getCreationDate() {
    return null;
  }

  public Date getLastReconfigurationDate() {
    return null;
  }

  public Date getNextAutoRestartDate() {
    return null;
  }
}
