/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.manager;
import com.sap.tc.logging.Location;
import com.sapportals.portal.security.usermanagement.IUser;

import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.util.logging.LoggingFormatter;

/**
 * Checks correct permissions for operations on resources. <p>
 *
 * Copyright (c) SAP AG 2002
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: SecurityChecker.java,v 1.3 2002/09/05 16:19:43 sei Exp $
 */
public final class SecurityChecker implements ISecurityChecker {

  private static Location log = Location.getLocation(com.sapportals.wcm.repository.manager.SecurityChecker.class);

  public final static ISecurityChecker NO_CHECK = NullChecker.INSTANCE;

  public static ISecurityChecker getInstance(IRepositoryManager manager)
    throws ResourceException {
    ISecurityManager secmgr = (manager != null) ? manager.getSecurityManager(null) : null;
    if (secmgr == null) {
      return NullChecker.INSTANCE;
    }
    return new SecurityChecker(secmgr);
  }


  private final ISecurityManager mgr;
  private final IPermission p_create;
  private final IPermission p_delete;
  private final IPermission p_delete_node;
  private final IPermission p_list;
  private final IPermission p_read_content;
  private final IPermission p_read_node_properties;
  private final IPermission p_read_properties;
  private final IPermission p_write_content;
  private final IPermission p_write_node_properties;
  private final IPermission p_write_properties;

  private final IPermissionList l_read_all;
  private final IPermissionList l_read_and_list;
  private final IPermissionList l_write_all;
  private final IPermissionList l_create_write_node_properties;

  private SecurityChecker(ISecurityManager mgr) {
    this.mgr = mgr;

    this.p_create = this.mgr.getPermission(IPermission.PERMISSION_CREATE);
    this.p_delete = this.mgr.getPermission(IPermission.PERMISSION_DELETE);
    this.p_delete_node = this.mgr.getPermission(IPermission.PERMISSION_DELETE_NODE);
    this.p_list = this.mgr.getPermission(IPermission.PERMISSION_LIST);
    this.p_read_content = this.mgr.getPermission(IPermission.PERMISSION_READ_CONTENT);
    this.p_read_node_properties = this.mgr.getPermission(IPermission.PERMISSION_READ_NODE_PROPERTIES);
    this.p_read_properties = this.mgr.getPermission(IPermission.PERMISSION_READ_PROPERTIES);
    this.p_write_content = this.mgr.getPermission(IPermission.PERMISSION_WRITE_CONTENT);
    this.p_write_node_properties = this.mgr.getPermission(IPermission.PERMISSION_WRITE_NODE_PROPERTIES);
    this.p_write_properties = this.mgr.getPermission(IPermission.PERMISSION_WRITE_PROPERTIES);

    this.l_read_all = makeList(this.p_read_content, this.p_read_properties);
    this.l_write_all = makeList(this.p_write_content, this.p_write_properties);
    this.l_read_and_list = makeList(this.p_read_node_properties, this.p_list);
    this.l_create_write_node_properties = makeList(this.p_create, this.p_write_node_properties);
  }

  public void checkReadProperties(IResource resource)
    throws ResourceException, AccessDeniedException {
    if (resource.isCollection()) {
      check(resource, this.p_read_node_properties, IPermission.PERMISSION_READ_NODE_PROPERTIES);
    }
    else {
      check(resource, this.p_read_properties, IPermission.PERMISSION_READ_PROPERTIES);
    }
  }

  public boolean isNecessary() {
    return true;
  }

  public void checkDelete(IResource resource)
    throws ResourceException, AccessDeniedException {
    checkDelete(null, resource);
  }
  
  public void checkDelete(IResource parent, IResource resource)
    throws ResourceException, AccessDeniedException {

    if (parent == null) parent = resource.getParentCollection();
    if (resource.isCollection()) {
      check(resource, this.p_delete_node, IPermission.PERMISSION_DELETE_NODE);
      if (parent != null) {
        check(parent, this.p_write_node_properties, IPermission.PERMISSION_WRITE_NODE_PROPERTIES);
      }
    }
    else {
      check(resource, this.p_delete, IPermission.PERMISSION_DELETE);
      if (parent != null) {
        check(parent, this.p_write_node_properties, IPermission.PERMISSION_WRITE_NODE_PROPERTIES);
      }
    }
  }

  public void checkReadContent(IResource resource)
    throws ResourceException, AccessDeniedException {
    check(resource, this.p_read_content, IPermission.PERMISSION_READ_CONTENT);
  }

  public void checkReadAll(IResource resource)
    throws ResourceException, AccessDeniedException {
    if (resource.isCollection()) {
      check(resource, this.p_read_node_properties, IPermission.PERMISSION_READ_NODE_PROPERTIES);
    }
    else {
      check(resource, this.l_read_all, IPermission.PERMISSION_READ_PROPERTIES);
    }
  }

  public void checkListChildren(IResource resource)
    throws ResourceException, AccessDeniedException {
    check(resource, this.p_list, IPermission.PERMISSION_LIST);
  }

  public void checkCreateChild(IResource resource, IPosition position)
    throws ResourceException, AccessDeniedException {
    if (position != null) {
      check(resource, this.l_create_write_node_properties, IPermission.PERMISSION_CREATE);
    }
    else {
      check(resource, this.p_create, IPermission.PERMISSION_CREATE);
    }
  }

  public void checkRemoveChild(IResource resource)
    throws ResourceException, AccessDeniedException {
    if (resource.isCollection()) {
      check(resource, this.p_write_node_properties, IPermission.PERMISSION_WRITE_NODE_PROPERTIES);
    }
  }

  public void checkRenameChild(IResource resource)
    throws ResourceException, AccessDeniedException {
    if (resource.isCollection()) {
      check(resource, this.p_write_node_properties, IPermission.PERMISSION_WRITE_NODE_PROPERTIES);
    }
  }
    
  public void checkModifyProperties(IResource resource)
    throws ResourceException, AccessDeniedException {
    if (resource.isCollection()) {
      check(resource, this.p_write_node_properties, IPermission.PERMISSION_WRITE_NODE_PROPERTIES);
    }
    else {
      check(resource, this.p_write_properties, IPermission.PERMISSION_WRITE_PROPERTIES);
    }
  }

  public void checkModifyContent(IResource resource)
    throws ResourceException, AccessDeniedException {
    check(resource, this.p_write_content, IPermission.PERMISSION_WRITE_CONTENT);
  }

  public void checkModifyAll(IResource resource)
    throws ResourceException, AccessDeniedException {
    if (resource.isCollection()) {
      check(resource, this.p_write_node_properties, IPermission.PERMISSION_WRITE_NODE_PROPERTIES);
    }
    else {
      check(resource, this.l_write_all, IPermission.PERMISSION_WRITE_PROPERTIES);
    }
  }

  public void checkLock(IResource resource)
    throws ResourceException, AccessDeniedException {
    checkModifyProperties(resource);
  }

  public void checkCopy(IResource resource)
    throws ResourceException, AccessDeniedException {
    if (resource.isCollection()) {
      check(resource, this.l_read_and_list, IPermission.PERMISSION_READ_NODE_PROPERTIES);
    }
    else {
      checkReadAll(resource);
    }
  }

  //---------------------------------- private ----------------------------------------------

  private IPermissionList makeList(IPermission p1, IPermission p2) {
    if (p1 == null || p2 == null) {
      return null;
    }
    PermissionList l = new PermissionList();
    l.add(p1);
    l.add(p2);
    return l;
  }

  private void check(IResource resource, IPermission permission, String pname)
    throws ResourceException, AccessDeniedException {
    IUser user = resource.getContext().getUser();
    if (user == null) {
      throw new AccessDeniedException("No user in context", resource.getRID(), pname, null);
    }
    if (!user.isAuthenticated()) {
      throw new AccessDeniedException("User <" + user.getId() + "> is not authenticated",
        resource.getRID(), pname, user.getId());
    }

    if (permission == null) {
      if (log.beDebug()) {
        log.debugT("check(218)", "Security Manager: " + this.mgr + " returned no permission for " + pname + ", access denied");
      }
      throw new AccessDeniedException("Permission denied: RID=" + resource.getRID() + ", permission=" + pname
         + ", user=" + user.getId(),
        resource.getRID(), pname, user.getId());
    }

    if (!this.mgr.isAllowed(resource, user, permission)) {
      throw new AccessDeniedException("Permission denied: uri=" + resource.getRID()
         + ", permission=" + pname + ", user=" + user.getId(),
        resource.getRID(), pname, user.getId());
    }
  }

  private void check(IResource resource, IPermissionList list, String pname)
    throws ResourceException, AccessDeniedException {
    IUser user = resource.getContext().getUser();
    if (user == null) {
      throw new AccessDeniedException("No user in context", resource.getRID(), pname, null);
    }
    if (!user.isAuthenticated()) {
      throw new AccessDeniedException("User <" + user.getId() + "> is not authenticated",
        resource.getRID(), pname, user.getId());
    }

    if (list == null) {
      if (log.beDebug()) {
        log.debugT("check(245)", "Security Manager: " + this.mgr + " returned no permission for " + pname + ", access denied");
      }
      throw new AccessDeniedException("Permission denied: RID=" + resource.getRID() + ", permission=" + pname
         + ", user=" + user.getId(),
        resource.getRID(), pname, user.getId());
    }

    if (!this.mgr.isAllowed(resource, user, list)) {
      throw new AccessDeniedException("Permission denied: uri=" + resource.getRID()
         + ", permission=" + pname + ", user=" + user.getId(),
        resource.getRID(), pname, user.getId());
    }
  }

  /**
   * TBD: Description of the class.
   */
  private final static class NullChecker implements ISecurityChecker {

    private final static ISecurityChecker INSTANCE = new NullChecker();

    private NullChecker() { }

    public boolean isNecessary() {
      return false;
    }

    public void checkReadProperties(IResource resource) { }

    public void checkReadContent(IResource resource) { }

    public void checkReadAll(IResource resource) { }

    public void checkListChildren(IResource resource) { }

    public void checkCreateChild(IResource resource, IPosition position) { }

    public void checkModifyProperties(IResource resource) { }

    public void checkModifyContent(IResource resource) { }

    public void checkModifyAll(IResource resource) { }

    public void checkDelete(IResource resource) { }
    
    public void checkDelete(IResource parent, IResource resource) { }

    public void checkRemoveChild(IResource resource) { }

    public void checkRenameChild(IResource resource) { }

    public void checkLock(IResource resource) { }

    public void checkCopy(IResource resource) { }
  }
}
