/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.manager;
import com.sap.netweaver.bc.rf.mi.IManager;
import com.sap.netweaver.bc.rf.util.logging.LoggingFormatter;
import com.sap.tc.logging.Location;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.crt.component.*;
import com.sapportals.wcm.crt.configuration.*;
import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.RMAdapter;
import com.sapportals.wcm.repository.service.IRepositoryService;
import com.sapportals.wcm.util.events.*;

import java.util.*;

/**
 * An event broker for resource events that are sent by repositories and
 * services. <p>
 *
 * Copyright (c) SAP AG 2001 - 2002
 *
 * @author Markus Breitenfelder
 */
public class ResourceEventBroker
  extends TransactionalEventBroker
  implements IResourceEventBroker, IContainer, IStartable, IThreadSafe, IComponentInfo, ILifecycleInfo {

  private static Location log = Location.getLocation(ResourceEventBroker.class);

  /**
   * The Repository this broker belongs to
   */
  RMAdapter rmAdapter;

  /**
   * IDs of all services configured for the repository
   */
  private Collection allowedServiceIDs;

  private String key;
  private IContainerManager containerManager;
  private ComponentStateHandler stateHandler;

  private boolean sendGet = false;
  private boolean sendGetChildren = false;
  private boolean sendGetProperty = false;
  private boolean sendSetProperty = false;
  private boolean sendDeleteProperty = false;

  public ResourceEventBroker() {
    this.stateHandler = new ComponentStateHandler(this);
  }

  // ---------------------------------------------------------------------------
  // interface IContextualizable
  // ---------------------------------------------------------------------------

  public void contextualize(IContext context) throws ContextException {
    this.stateHandler.preContextualize();
    this.containerManager = context.getContainerManager();
    this.key = context.getKey();
    if (this.containerManager == null) {
      throw new ContextException("No container manager in context");
    }
    try {
      Object[] params = (Object[])context.getParameters();
      this.rmAdapter = new RMAdapter(params[0]);
      this.allowedServiceIDs = (Collection)params[1];
    }
    catch (Exception ex) {
      throw new ContextException("Invalid context parameters", ex);
    }

    this.stateHandler.postContextualize();
  }

  // ---------------------------------------------------------------------------
  // interface IStartable
  // ---------------------------------------------------------------------------

  public final void start() throws StartupException {
    this.stateHandler.preStart();

    // Register the repository
    try {
      if (this.rmAdapter.isNew()) {
        super.register(this.rmAdapter.getNewAbstract());
      }
      else {
        super.register(this.rmAdapter.getOldAbstract());
      }
    }
    catch (WcmException ex) {
      throw this.stateHandler.postStart(new StartupException("Failed to register: " + ex.getMessage(), ex));
    }
    this.stateHandler.postStart();
  }

  public final void stop() {
    this.stateHandler.preStop();

    // Unregister the repository
    try {
      if (this.rmAdapter.isNew()) {
        unregister(this.rmAdapter.getNewAbstract());
      }
      else {
        unregister(this.rmAdapter.getOldAbstract());
      }
      
      super.stop();
    }
    catch (WcmException x) {
      log.errorT("failed to stop the event broker: " + LoggingFormatter.extractCallstack(x));
    }

    this.stateHandler.postStop();
  }

  // ---------------------------------------------------------------------------
  // interface IContainer
  // ---------------------------------------------------------------------------

  public IComponentManager getComponentManager() {
    return this.containerManager;
  }

  // ---------------------------------------------------------------------------
  // interface IComponentInfo
  // ---------------------------------------------------------------------------

  public String getName() {
    return "resource event broker";
  }

  public String getDescription(Locale locale) {
    return null;
  }

  public Properties getProperties() {
    return null;
  }

  // ---------------------------------------------------------------------------
  // interface ILifecycleInfo
  // ---------------------------------------------------------------------------

  public final ComponentState getState() {
    return this.stateHandler.getState();
  }

  public ConfigurationException getLastConfigurationException() {
    return this.stateHandler.getLastConfigurationException();
  }

  public StartupException getStartupException() {
    return this.stateHandler.getStartupException();
  }

  public final Date getCreationDate() {
    return this.stateHandler.getCreationDate();
  }

  public Date getLastReconfigurationDate() {
    return this.stateHandler.getLastReconfigurationDate();
  }

  public Date getNextAutoRestartDate() {
    return this.stateHandler.getNextAutoRestartDate();
  }

  // ---------------------------------------------------------------------------

  /**
   * Collect all possible resource-specific events. Call the
   * getEvents(IResource) method of all registered IResourceEventSender
   * instances
   *
   * @param resource TBD: Description of the incoming method parameter
   * @return events
   */
  public IEventList getEvents(IResource resource) {
    IEventList list = new EventList();
    Iterator it = this.senderMap.values().iterator();
    while (it.hasNext()) {
      IEventSender sender = (IEventSender)it.next();
      if (sender instanceof IResourceEventSender) {
        IResourceEventSender res = (IResourceEventSender)sender;
        IEventList l = res.getEvents(resource);
        if (l != null) {
          list.addAll(l);
        }
      }
    }
    return list;
  }

  public void send(IEvent e, IEventSender s) throws WcmException {
    boolean isRunning = this.stateHandler.preService();
    try {
      if (!isRunning) {
        this.throwNotRunningException();
      }
      super.send(e, s);
    }
    finally {
      this.stateHandler.postService();
    }
  }

  /**
   * Overite register method: Check for valid classes or service ID of sender
   *
   * @param sender TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public void register(IEventSender sender) throws WcmException {
    boolean isRunning = this.stateHandler.preService();
    try {
      if (!isRunning) {
        this.throwNotRunningException();
      }

      if (!((sender instanceof IRepositoryManager)
        || (sender instanceof IManager)
        || (sender instanceof IRepositoryService))) {
        throw new WcmException("Failed to register. Wrong instance: " + ((Object)sender).toString());
      }

      if (sender instanceof IRepositoryService) {
        if (!this.allowedServiceIDs.contains(((IRepositoryService)sender).getID())) {
          log.warningT(
            "register(235)",
            "Repository Service "
              + sender
              + " has registered as event sender but is not configured for this repository: "
              + this.rmAdapter.getID());
          // throw new WcmException("Service" + ((IRepositoryService)sender).getID()
          //    + " is not allowed to register for events of repository " + this.mgr.getID());
        }
      }

      super.register(sender);
      log.debugT("register(242)", "Sender is registerd: " + ((Object)sender).toString());
    }
    finally {
      this.stateHandler.postService();
    }
  }

  public void register(IEventReceiver receiver, IEvent template) throws WcmException {
    boolean isRunning = this.stateHandler.preService();
    try {
      if (!isRunning) {
        this.throwNotRunningException();
      }

      this.sendGet |= ResourceEvent.GET_TEMPLATE.isLike(template);
      this.sendGetChildren |= ResourceEvent.GET_CHILDREN_TEMPLATE.isLike(template);
      this.sendGetProperty |= ResourceEvent.PROPERTY_GET_TEMPLATE.isLike(template);
      this.sendSetProperty |= ResourceEvent.PROPERTY_SET_TEMPLATE.isLike(template);
      this.sendDeleteProperty |= ResourceEvent.PROPERTY_DELETE_TEMPLATE.isLike(template);

      super.register(receiver, template);
    }
    finally {
      this.stateHandler.postService();
    }
  }

  public void register(IEventReceiver receiver, IEvent template, IEventMapper mapping) throws WcmException {
    boolean isRunning = this.stateHandler.preService();
    try {
      if (!isRunning) {
        this.throwNotRunningException();
      }

      if ((mapping != null) && (!(mapping instanceof IResourceEventMapper))) {
        throw new WcmException(
          "EventMapper "
            + mapping.getID()
            + " is not a ResourceEventMapper and not allowed for repository "
            + this.rmAdapter.getID());
      }

      this.sendGet |= ResourceEvent.GET_TEMPLATE.isLike(template);
      this.sendGetChildren |= ResourceEvent.GET_CHILDREN_TEMPLATE.isLike(template);
      this.sendGetProperty |= ResourceEvent.PROPERTY_GET_TEMPLATE.isLike(template);
      this.sendSetProperty |= ResourceEvent.PROPERTY_SET_TEMPLATE.isLike(template);
      this.sendDeleteProperty |= ResourceEvent.PROPERTY_DELETE_TEMPLATE.isLike(template);

      super.register(receiver, template, mapping);
    }
    finally {
      this.stateHandler.postService();
    }
  }

  public void register(IEventReceiver receiver, IEvent template, IEventMapper mapping, int priority, boolean async)
    throws WcmException {
    boolean isRunning = this.stateHandler.preService();
    try {
      if (!isRunning) {
        this.throwNotRunningException();
      }
      if (ResourceEventBroker.log.beWarning()) {
        if(   ( template == null )
           || ( template.getType() == ResourceEvent.ALL )
          ) {
          ResourceEventBroker.log.warningT(
            "register(277)",
            "Event receiver is registering to all events (null or type=ALL): " + receiver.getClass().getName());
        }
      }
      if ((mapping != null) && (!(mapping instanceof IResourceEventMapper))) {
        throw new WcmException(
          "EventMapper "
            + mapping.getID()
            + " is not a ResourceEventMapper and not allowed for repository "
            + this.rmAdapter.getID());
      }

      this.sendGet |= ResourceEvent.GET_TEMPLATE.isLike(template);
      this.sendGetChildren |= ResourceEvent.GET_CHILDREN_TEMPLATE.isLike(template);
      this.sendGetProperty |= ResourceEvent.PROPERTY_GET_TEMPLATE.isLike(template);
      this.sendSetProperty |= ResourceEvent.PROPERTY_SET_TEMPLATE.isLike(template);
      this.sendDeleteProperty |= ResourceEvent.PROPERTY_DELETE_TEMPLATE.isLike(template);

      super.register(receiver, template, mapping, priority, async);
    }
    finally {
      this.stateHandler.postService();
    }
  }

  public boolean mustSendResourceEventType(int type) {
    switch( type ) {
      case ResourceEvent.PROPERTY_GET: return this.sendGetProperty;
      case ResourceEvent.GET_CHILDREN: return this.sendGetChildren;
      case ResourceEvent.GET: return this.sendGet;
      case ResourceEvent.PROPERTY_SET: return this.sendSetProperty;
      case ResourceEvent.PROPERTY_DELETE: return this.sendDeleteProperty;
      default: return true;
    }
  }
  
  private void throwNotRunningException() throws WcmException {
    throw new WcmException("The broker is not started or shutting down (" + this.rmAdapter.getID());
  }
}
