/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.manager;
import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.util.content.*;

import com.sapportals.wcm.util.uri.RID;

/**
 * A versioning manager handles versioning operations of a specific repository
 *
 * @author SAP AG
 * @version $Revision: 1.11 $
 */
public interface IVersioningManager {

  /**
   * Check in a new version of the resource with changed content.
   *
   * @param newContent The new content. Can be null: the existing content will
   *      be used for the new version.
   * @param properties The new properties, can be null
   * @param ignorePropertyFailures Ignore exceptions if some or all properties
   *      could not be set/removed
   * @param expectedCheckInRID TBD: Description of the incoming method parameter
   * @param resource TBD: Description of the incoming method parameter
   * @return The new revision that was created
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public ICheckInInfo checkIn(IContent newContent, IPropertyMap properties,
    boolean ignorePropertyFailures, RID expectedCheckInRID, IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Checks whether a resource is currently checked out.
   *
   * @param resource The resource
   * @return Boolean value
   * @exception ResourceException
   */
  public boolean isCheckedOut(IResource resource)
    throws ResourceException;

  /**
   * Undo a previous check out operation.
   *
   * @param resource TBD: Description of the incoming method parameter
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public void undoCheckOut(IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Check out the resource for editing.
   *
   * @param resource The resource
   * @return ICheckOutInfo instance that contains the expected check-in RID
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public ICheckOutInfo checkOut(IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Get the version history for the resource.
   *
   * @param resource The resource
   * @return versionHistory
   * @exception ResourceException
   * @exception NotSupportedException
   * @returns The version history
   */
  public IVersionHistory getVersionHistory(IResource resource)
    throws ResourceException, NotSupportedException;

  /**
   * Returns a list of currently checked out resources. The manager
   * implementation must support this method only for revision resources.
   *
   * @param resource The resource
   * @return Resource list
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException
   */
  public IResourceList getCheckedOutResources(IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Enables or disables versioning for a resource.
   *
   * @param enable If versioning must be eabled or disabled
   * @param resource The resource
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException
   */
  public void enableVersioning(boolean enable, IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Returns <code>true</code> if versioning is enabled.
   *
   * @param resource TBD: Description of the incoming method parameter
   * @return <code>true</code> if versioning is enabled.
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException
   */
  public boolean isVersioned(IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException;
}
