/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.manager;
import com.sapportals.wcm.repository.AccessDeniedException;
import com.sapportals.wcm.repository.IPosition;

import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.ResourceException;

/**
 * Checks correct permissions for operations on resources. <p>
 *
 * Copyright (c) SAP AG 2002
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: ISecurityChecker.java,v 1.2 2002/09/05 14:19:56 sei Exp $
 */
public interface ISecurityChecker {

  /**
   * Return if checking of permissions is necessary. If this is false, then this
   * checker will never throw any AccessDeniedExceptions.
   *
   * @return if checking of permissions is necessary.
   */
  public boolean isNecessary();

  /**
   * Checks permissions for property access.
   *
   * @param resource to check on
   * @throws AccessDeniedException on insufficient permissions
   * @throws ResourceException on failure
   */
  public void checkReadProperties(IResource resource)
    throws ResourceException, AccessDeniedException;

  /**
   * Checks permissions for content access.
   *
   * @param resource to check on
   * @throws AccessDeniedException on insufficient permissions
   * @throws ResourceException on failure
   */
  public void checkReadContent(IResource resource)
    throws ResourceException, AccessDeniedException;

  /**
   * Checks permissions for content and property access.
   *
   * @param resource to check on
   * @throws AccessDeniedException on insufficient permissions
   * @throws ResourceException on failure
   */
  public void checkReadAll(IResource resource)
    throws ResourceException, AccessDeniedException;

  /**
   * Checks permissions for listing children.
   *
   * @param resource to check on
   * @throws AccessDeniedException on insufficient permissions
   * @throws ResourceException on failure
   */
  public void checkListChildren(IResource resource)
    throws ResourceException, AccessDeniedException;

  /**
   * Checks permissions for creating a child with optional position.
   *
   * @param resource to check on
   * @param position information or <code>null</code>
   * @throws AccessDeniedException on insufficient permissions
   * @throws ResourceException on failure
   */
  public void checkCreateChild(IResource resource, IPosition position)
    throws ResourceException, AccessDeniedException;

  /**
   * Checks permissions for removing a child.
   *
   * @param resource to check on
   * @throws AccessDeniedException on insufficient permissions
   * @throws ResourceException on failure
   */
  public void checkRemoveChild(IResource resource)
    throws ResourceException, AccessDeniedException;

  /**
   * Checks permissions for renaming a child.
   *
   * @param resource to check on
   * @throws AccessDeniedException on insufficient permissions
   * @throws ResourceException on failure
   */
  public void checkRenameChild(IResource resource)
    throws ResourceException, AccessDeniedException;

  /**
   * Checks permissions for property modification.
   *
   * @param resource to check on
   * @throws AccessDeniedException on insufficient permissions
   * @throws ResourceException on failure
   */
  public void checkModifyProperties(IResource resource)
    throws ResourceException, AccessDeniedException;

  /**
   * Checks permissions for content modification.
   *
   * @param resource to check on
   * @throws AccessDeniedException on insufficient permissions
   * @throws ResourceException on failure
   */
  public void checkModifyContent(IResource resource)
    throws ResourceException, AccessDeniedException;

  /**
   * Checks permissions for content and property modification.
   *
   * @param resource to check on
   * @throws AccessDeniedException on insufficient permissions
   * @throws ResourceException on failure
   */
  public void checkModifyAll(IResource resource)
    throws ResourceException, AccessDeniedException;

  /**
   * Checks permissions for delete of resource.
   *
   * @param resource to check on
   * @throws AccessDeniedException on insufficient permissions
   * @throws ResourceException on failure
   */
  public void checkDelete(IResource resource)
    throws ResourceException, AccessDeniedException;
  
  /**
   * Checks permissions for delete of resource.
   *
   * @param parentCollection the parent of the to be removed resource
   * @param resource to check on
   * @throws AccessDeniedException on insufficient permissions
   * @throws ResourceException on failure
   */
  public void checkDelete(IResource parentCollection, IResource resource)
    throws ResourceException, AccessDeniedException;  

  /**
   * Checks permissions for lock/unlock of resource.
   *
   * @param resource to check on
   * @throws AccessDeniedException on insufficient permissions
   * @throws ResourceException on failure
   */
  public void checkLock(IResource resource)
    throws ResourceException, AccessDeniedException;

  /**
   * Checks permissions for copying of resource.
   *
   * @param resource to check on
   * @throws AccessDeniedException on insufficient permissions
   * @throws ResourceException on failure
   */
  public void checkCopy(IResource resource)
    throws ResourceException, AccessDeniedException;

}
