/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.manager;

import java.util.List;

import com.sapportals.wcm.repository.*;

/**
 * The property manager implements properties operations for a specific
 * repository.
 * <p>
 * @author SAP AG 2001-2004
 * @version $Revision: 1.3 $
 */
public interface IPropertyManager {

  /**
   * Get a property.
   *
   * @param propName The property Name of the property to read
   * @param resource The resource
   * @return A reference to the property or null if the property was not found
   * @exception ResourceException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IProperty getProperty(IPropertyName propName, IResource resource)
    throws ResourceException, AccessDeniedException;

  /**
   * Returns a map with <code>IProperty</code> references for the resource.
   *
   * @param resource The resource
   * @return The property map
   * @exception ResourceException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IPropertyMap getProperties(IResource resource)
    throws ResourceException, AccessDeniedException;

  /**
   * Sets a property.
   *
   * @param prop The property to set
   * @param resource The resource
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public void setProperty(IProperty prop, IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Set all properties of the resource.
   *
   * @param props The map with the properties.
   * @param resource The resource
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   * @exception SetPropertiesException If some properties could not be set or
   *      removed
   */
  public void setProperties(IPropertyMap props, IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException, SetPropertiesException;

  /**
   * Change some properties of the resource.
   *
   * @param resource the resource
   * @param propChangeList change list (contains {@link IProperty} entries for properties to be set and {@link IPropertyName} entries for properties to be removed)
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   * @exception SetPropertiesException If some properties could not be set or
   *      removed
   */
  public void setProperties(List propChangeList, IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException, SetPropertiesException;

  /**
   * Removes the property.
   *
   * @param propName The property name of the property to delete
   * @param resource The resource
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public void deleteProperty(IPropertyName propName, IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException;
}

