/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.manager;

import com.sapportals.wcm.IWcmConst;
import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.repository.enum.*;
import com.sapportals.wcm.util.content.*;
import com.sapportals.wcm.util.uri.IRidList;
import com.sapportals.wcm.util.uri.RID;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collection;
import java.util.List;

/**
 * A namespace manager implements the namespace operations for a specific
 * repository.
 *
 * @author SAP AG
 * @version $Revision:$
 */
public interface INamespaceManager {

  /**
   * Rename the resource
   *
   * @param newName The new name
   * @param resource The resource
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public void rename(String newName, IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Delete a resource or collection from the repository.
   *
   * @param resource A resource object
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public void delete(IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Delete a collection and all its children from the repository.
   *
   * @param collection A collection object
   * @return A list RIDs of all deleted resources. This is optional (can be
   *      null) if the manager implementation does not supports deep delete.
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IRidList delete(ICollection collection)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Get the members of a collection.
   *
   * @param resource TBD: Description of the incoming method parameter
   * @return The list of resources
   * @exception ResourceException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResourceList getChildren(ICollection resource)
    throws ResourceException, AccessDeniedException;

  /**
   * Creates a new sub collection in a collection that supports ordering.
   *
   * @param name The name of the new collection
   * @param position The position in the collection
   * @param properties The properties to set for the new collection
   * @param ignorePropertyFailures Ignore exception if some or all properties
   *      could not be set/removed
   * @param collection The collection
   * @return The new collection
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public ICollection createCollection(String name, IPosition position, IPropertyMap properties,
    boolean ignorePropertyFailures, ICollection collection)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Creates a new resource in an ordered collection.
   *
   * @param name The name of the resource
   * @param position The position in the ordered collection
   * @param properties A map of properties for the new resource
   * @param ignorePropertyFailures Do not throw a exception if some or all of
   *      the properties could not be set
   * @param collection The collection
   * @param content TBD: Description of the incoming method parameter
   * @return The new resource object
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResource createResource(String name, IPosition position,
    IPropertyMap properties, IContent content, boolean ignorePropertyFailures, ICollection collection)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Creates a initial resource that is empty and locked ("lock null resource")
   *
   * @param name The name of the resource
   * @param lockProperties The properties of the lock
   * @param collection TBD: Description of the incoming method parameter
   * @return The lock info
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public ILockInfo createResource(String name, ILockProperties lockProperties, ICollection collection)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Returns the order type of a collection
   *
   * @param collection TBD: Description of the incoming method parameter
   * @return The order type
   * @exception ResourceException
   */
  public OrderType getOrderType(ICollection collection)
    throws ResourceException;

  /**
   * Sets the order type of a collection
   *
   * @param orderType orderType to be set
   * @param collection orderType to be set
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public void setOrderType(OrderType orderType, ICollection collection)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Change the position of some or all resources in an ordered collection.
   *
   * @param list A list that contains <code>IPositioning</code> instances that
   *      specify how the resources in the collection should be positioned.
   * @param collection The collection
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public void reorder(IReorderList list, ICollection collection)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Creates a new link in a collection that supports ordering.
   *
   * @param name The name of the resource
   * @param targetURL The link target
   * @param linkType The link type
   * @param position The positioning information
   * @param properties An IPropertyMap with all properties to set for the new
   *      resource
   * @param ignorePropertyFailures
   * @param collection The collection
   * @return The new resource
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResource createLink(String name, String targetURL, LinkType linkType, IPosition position,
    IPropertyMap properties, boolean ignorePropertyFailures, ICollection collection)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Change the position of a resource in an ordered collection
   *
   * @param resource The resources
   * @param position TBD: Description of the incoming method parameter
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public void changePosition(IPosition position, IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Change a link's URI/URL and type
   *
   * @param targetURL The new target URL
   * @param linkType The new link type
   * @param resource TBD: Description of the incoming method parameter
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public void updateLink(String targetURL, LinkType linkType, IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Copy the resource to another collection within this repository
   *
   * @param destinationRID The RID of the destination
   * @param param The copy parameter object, can be null
   * @param resource TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResource copy(RID destinationRID, ICopyParameter param, IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Copy the resource to a destination collection within this repository that
   * supports ordered children
   *
   * @param destinationRID TheRID of the destination
   * @param position The positioning information
   * @param param The copy parameter object, can be null
   * @param resource TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResource copy(RID destinationRID, IPosition position, ICopyParameter param, IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Move the resource to another collection within this reposiory
   *
   * @param destinationRID The RID of the destination.
   * @param param The copy parameter object, can be null
   * @param resource TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResource move(RID destinationRID, ICopyParameter param, IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException;

  /**
   * Move the resource to a destination collection within this repository that
   * supports ordered children.
   *
   * @param destinationRID The RID of the destination.
   * @param position The positioning information
   * @param param The copy parameter object, can be null
   * @param resource TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception ResourceException
   * @exception NotSupportedException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResource move(RID destinationRID, IPosition position, ICopyParameter param, IResource resource)
    throws ResourceException, NotSupportedException, AccessDeniedException;
}

