/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.manager;
import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.util.content.IContent;
import com.sapportals.wcm.util.name.IName;

import com.sapportals.wcm.util.uri.IRidSet;
import com.sapportals.wcm.util.uri.RID;

import java.util.*;

/**
 * Extended versioning manager interface. <p>
 *
 * Copyright (c) SAP AG 2002-2003
 *
 * @author stefan.eissing@greenbytes.de
 * @author julian.reschke@greenbytes.de
 * @version $Revision: 1.9 $
 */
public interface IExtendedVersioningManager extends IVersioningManager {

  /**
   * Adds a label to the given version resource. Must fail if the label is
   * already present on another version in the same version history.
   *
   * @param labelName the label to set
   * @param version the version resource
   * @exception ResourceException Exception raised in failure situation
   */
  public void addLabel(String labelName, IResource version)
    throws ResourceException;

  /**
   * Checks in the given resource and creates a new version in the version
   * history of the resource. The resource must be version controlled (a vcr).
   * The new version will be a successor of the base version of the vcr. If the
   * base version already has a successor, checking would create a fork in the
   * version history. This will only succeed of forkOk is set. If keepCheckedOut
   * is set, the resource will be checked out again immediatly.
   *
   * @param newContent for resource and created version
   * @param properties for resource and created version
   * @param ignorePropertyFailures continue if not all properities could be set
   * @param expectedCheckInRID RID which created version should have
   * @param forkOk allow to create new forks in version history of resource
   * @param keepCheckedOut resource will be checked out again
   * @param resource to check in <p>
   *
   *      Extension of the IVersioningManager.checkIn method.
   * @return TBD: Description of the outgoing return value
   * @exception ResourceException Exception raised in failure situation
   */
  public ICheckInInfo2 checkIn(IContent newContent,
    IPropertyMap properties, boolean ignorePropertyFailures,
    RID expectedCheckInRID, boolean forkOk,
    boolean keepCheckedOut, IResource resource)
    throws ResourceException;

  /**
   * Check out a resource which is either a version or a version controlled
   * resource. <p>
   *
   * When a version resource is checked out, a new working resource is created
   * in the repository and the returned ICheckOutInfo2 will have the RID of the
   * working resource. <p>
   *
   * The same happens when the resource is a VCR and applyToVersion is set. The
   * VCR must be checked in for this to work. <p>
   *
   * With applyToVersion set to false, resource must be a checked in VCR and
   * will be checked out in place. The resource itself will have status checked
   * out. <p>
   *
   * If a subsequent checkIn of the checked out resource would create a fork in
   * the version history, the forkOk parameter must be true otherwise the method
   * must fail.
   *
   * @param applyToVersion checkout version of resource
   * @param forkOk allow subsequent forks in version history upon check in
   * @param resource version or VCR to check out
   * @return checkout information with expected uri and possibly RID of created
   *      working resource
   * @exception ResourceException Exception raised in failure situation
   */
  public ICheckOutInfo2 checkOut(boolean applyToVersion, boolean forkOk, IResource resource)
    throws ResourceException;

  /**
   * mba TODO: javadoc is not accurate Get the RID of the version controlled
   * resource from which the given working resource was created.
   *
   * @param workingResource TBD: Description of the incoming method parameter
   * @return the RID of the version controlled resource from which the working
   *      resource was created. Otherwise <code>null</code> .
   * @exception ResourceException Exception raised in failure situation
   */
  public RID getAutoUpdateVCR(IResource workingResource)
    throws ResourceException;

  /**
   * Get IName of autoversioning mode of the version controlled resource.
   *
   * @param vcr to retrieve autoversioning mode for
   * @return IName of current autoversioning mode (or <code>null</code> when no
   *      autoversioning present)
   * @exception ResourceException Exception raised in failure situation
   */
  public IName getAutoVersioningMode(IResource vcr)
    throws ResourceException;

  /**
   * Get the version resource from which the given version controlled resource
   * is checked out.
   *
   * @param vcr the version controlled resource
   * @return version resource from which vcr is checked out
   * @exception ResourceException Exception raised in failure situation
   */
  public IResource getCheckedOutResource(IResource vcr)
    throws ResourceException;

  /**
   * Get the version resource to which the given version controlled resource is
   * checked in.
   *
   * @param vcr the version controlled resource
   * @return version resource to which vcr is checked in
   * @exception ResourceException Exception raised in failure situation
   */
  public IResource getCheckedInResource(IResource vcr)
    throws ResourceException;

  /**
   * Get the version resource ID to which the given version controlled resource
   * is checked in.
   *
   * @param vcr the version controlled resource
   * @return RID of version resource to which vcr is checked in
   * @exception ResourceException Exception raised in failure situation
   */
  public RID getCheckedInRID(IResource vcr)
    throws ResourceException;

  /**
   * Get RID of the version resource from which the given version controlled
   * resource is checked out.
   *
   * @param vcr the version controlled resource
   * @return RID of the version resource from which vcr is checked out
   * @exception ResourceException Exception raised in failure situation
   */
  public RID getCheckedOutRID(IResource vcr)
    throws ResourceException;

  /**
   * mba TODO: rename Get the set of RID of resources which are checked out from
   * the given version
   *
   * @param version from which resources are checked out
   * @return set of current checkouts of this version
   * @exception ResourceException Exception raised in failure situation
   */
  public IRidSet getCheckedOutRIDs(IResource version)
    throws ResourceException;

  /**
   * @param version to get behaviour for
   * @return a set of INames describing the checkout-in behaviour for this
   *      resource (when checked-out).
   * @exception ResourceException Exception raised in failure situation
   * @see IVersionResource#getCheckInForkBehaviour()
   */
  public Set getCheckInForkBehaviour(IResource version)
    throws ResourceException;

  /**
   * @param version to get behaviour for
   * @return a set of INames describing the checkout-in behaviour for this
   *      resource (when checked-out).
   * @exception ResourceException Exception raised in failure situation
   * @see IVersionResource#getCheckOutForkBehaviour()
   */
  public Set getCheckOutForkBehaviour(IResource version)
    throws ResourceException;

  /**
   * For version controlled collections, retrieve the set of collection member
   * names which are eclipsed by non-version-controlled members.
   *
   * @param vc version controlled collection
   * @return list of the internal member names of all version-controlled
   *      resources that are currently eclipsed by a non-version-controlled
   *      internal member of the collection (see <a
   *      href="http://greenbytes.de/tech/webdav/rfc3253.html#PROPERTY_eclipsed-set">
   *      RFC3253, section 14.1.1</a> ).
   * @exception ResourceException Exception raised in failure situation
   */
  public Set getEclipsedSet(IResource vc)
    throws ResourceException;

  /**
   * Get the labels defined on the given version resource.
   *
   * @param resource version resource
   * @return set of labels present on the version resource(empty list when no
   *      labels present).
   * @exception ResourceException Exception raised in failure situation
   */
  public Set getLabelSet(IResource resource)
    throws ResourceException;

  /**
   * Get the list of version resources which are immediate predecessors in the
   * version history to the given version resource, or the list of immediate
   * predecessors of the version that would be the result of checking in this
   * version controlled resource.
   *
   * @param resource to retrieve predecessors for
   * @return list of immediate predecessor resources or empty list if none
   *      defined
   * @exception ResourceException Exception raised in failure situation
   * @see <a href="http://www.webdav.org/deltav/protocol/rfc3253.html#rfc.section.3.3">
   *      RFC3253, section 3.3.2 and 3.4.1</a>
   */
  public IResourceList getPredecessorResources(IResource resource)
    throws ResourceException;

  /**
   * Get the list of RIDs of version resources which are immediate predecessors
   * in the version history to the given version resource, or the list of
   * immediate predecessors of the version that would be the result of checking
   * in this version controlled resource.
   *
   * @param resource to retrieve predecessors for
   * @return list of RIDs of immediate predecessor resources or empty list if
   *      none defined
   * @exception ResourceException Exception raised in failure situation
   * @see <a href="http://www.webdav.org/deltav/protocol/rfc3253.html#rfc.section.3.3">
   *      RFC3253, section 3.3.2 and 3.4.1</a>
   */
  public IRidSet getPredecessorRIDs(IResource resource)
    throws ResourceException;

  /**
   * For a given version history resource, report the version resource which is
   * the root of the history.
   *
   * @param vhr version history resource
   * @return root version for given version history
   * @exception ResourceException Exception raised in failure situation
   * @see <a href="http://www.webdav.org/deltav/protocol/rfc3253.html#rfc.section.5.1.2">
   *      RFC3253, section 5.1.2</a>
   */
  public IResource getRootVersionResource(IResource vhr)
    throws ResourceException;

  /**
   * For a given version history resource, report the version resource RID which
   * is the root of the history.
   *
   * @param vhr version history resource
   * @return root version RID for given version history
   * @exception ResourceException Exception raised in failure situation
   * @see <a href="http://www.webdav.org/deltav/protocol/rfc3253.html#rfc.section.5.1.2">
   *      RFC3253, section 5.1.2</a>
   */
  public RID getRootVersionRID(IResource vhr)
    throws ResourceException;

  /**
   * Get the list of version resources which are immediate successors in the
   * version history to the given version resource
   *
   * @param resource version to retrieve successors for
   * @return list of immediate successor resources or empty list if none defined
   * @exception ResourceException Exception raised in failure situation
   */
  public IResourceList getSuccessorResources(IResource resource)
    throws ResourceException;

  /**
   * Get the list of RIDs of version resources which are immediate successors in
   * the version history to the given version resource
   *
   * @param resource version to retrieve successors for
   * @return list of RIDs of immediate successor resources or empty list if none
   *      defined
   * @exception ResourceException Exception raised in failure situation
   */
  public IRidSet getSuccessorRIDs(IResource resource)
    throws ResourceException;

  /**
   * For collection versions, return a mapping of internal member names to RIDs
   * to the version history resources (for an ordered collection, this will be a
   * SortedMap, for non-collections, return <code>null</code> ).
   *
   * @param collectionVersion TBD: Description of the incoming method parameter
   * @return the mapping
   * @throws ResourceException
   */
  public Map getVersionControlledBindingSet(IResource collectionVersion)
    throws ResourceException;

  /**
   * Determine the RIDs of all version controlled resources which depend on the
   * same version history. If scope is non-null, report only version controlled
   * resources which are contained in the scope.
   *
   * @param scope of RIDs to report or <code>null</code> to report all RIDs
   * @param vhr version history resource to report vcr RIDs for
   * @return set of RIDs of version controlled resources based on the given
   *      version history
   * @exception ResourceException Exception raised in failure situation
   * @see <a href="http://www.webdav.org/deltav/protocol/rfc3253.html#rfc.section.5.2.3">
   *      RFC3253, section 5.2.3</a>
   */
  public IRidSet getVersionControlledResourceRIDs(RID scope, IResource vhr)
    throws ResourceException;

  /**
   * Returns a set of collections that are used to store version history
   * resources for the given resource. The collections can be at the top of a
   * whole tree in which version histories are stored. Several collections with
   * version histories can be represented by their common ancestor collection.
   *
   * @param resource to determine collection set for
   * @return a set of collections containing version histories
   * @exception ResourceException Exception raised in failure situation
   * @see <a href="http://www.webdav.org/deltav/protocol/rfc3253.html#rfc.section.5.2.4">
   *      RFC 3253, section 5.2.4</a>
   */
  public IRidSet getVersionHistoryCollectionSet(IResource resource)
    throws ResourceException;

  /**
   * Get the version history where the given version resource belongs to.
   *
   * @param version resource
   * @return version history resource for version
   * @exception ResourceException Exception raised in failure situation
   */
  public IResource getVersionHistoryResource(IResource version)
    throws ResourceException;

  /**
   * Get the version history RID where the given version resource belongs to.
   *
   * @param resource TBD: Description of the incoming method parameter
   * @return version history resource RID for version
   * @exception ResourceException Exception raised in failure situation
   */
  public RID getVersionHistoryRID(IResource resource)
    throws ResourceException;

  /**
   * Given a version history, report the version resource (or <code>null</code>
   * if non exists) which is tagged by the given label.
   *
   * @param label to look for
   * @param resource version history to find version in
   * @return version with label or <code>null</code> if not defined
   * @exception ResourceException Exception raised in failure situation
   */
  public IResource getVersionResourceByLabel(String label, IResource resource)
    throws ResourceException;

  /**
   * Given a version history, report the version RID (or <code>null</code> if
   * non exists) which is tagged by the given label.
   *
   * @param label to look for
   * @param resource version history to find version in
   * @return version RID with label or <code>null</code> if not defined
   * @exception ResourceException Exception raised in failure situation
   */
  public RID getVersionRIDByLabel(String label, IResource resource)
    throws ResourceException;

  /**
   * Return the set of versions in the given version history
   *
   * @param vhr the version history resource
   * @return the set of versions
   * @exception ResourceException Exception raised in failure situation
   */
  public IResourceList getVersionResourceSet(IResource vhr)
    throws ResourceException;

  /**
   * Return the set of version RIDs in the given version history
   *
   * @param vhr the version history resource
   * @return the set of version RIDs
   * @exception ResourceException Exception raised in failure situation
   */
  public IRidSet getVersionRIDs(IResource vhr)
    throws ResourceException;

  /**
   * Return the set of collection RIDs where workspaces can be created. This set
   * may vary depending on the resource given, e.g. different parts of the
   * repository namespace may have different workspace collection sets.
   *
   * @param resource to report collection set for
   * @return workspace collection set
   * @exception ResourceException Exception raised in failure situation
   */
  public IRidSet getWorkspaceCollectionSet(IResource resource)
    throws ResourceException;

  /**
   * Remove the given label from the version resource if present.
   *
   * @param labelName name of label
   * @param resource the version resource to remove label from
   * @throws ResourceException if label is not present
   */
  public void removeLabel(String labelName, IResource resource)
    throws ResourceException;

  /**
   * Tag the given version resource with the given label. If the label is
   * already present on another version in the same version history, the label
   * is removed from that version.
   *
   * @param labelName to tag version with
   * @param resource version resource to tag label on
   * @exception ResourceException Exception raised in failure situation
   */
  public void setLabel(String labelName, IResource resource)
    throws ResourceException;

  /**
   * Update content and dead properties of the version controlled resource with
   * the content and properties of the given version. The version controlled
   * resource will be checked in with the given version afterwards.
   *
   * @param version to update vcr with
   * @param vcr version controlled resource to update
   * @return RIDs of all modified resources
   * @exception ResourceException Exception raised in failure situation
   */
  public IRidSet updateFromVersion(IResource version, IResource vcr)
    throws ResourceException;

  /**
   * Update content and dead properties of the version controlled resource with
   * the content and properties of the given version. The version controlled
   * resource will be checked in with the given version afterwards.
   *
   * @param version RID to update vcr with
   * @param vcr version controlled resource to update
   * @return RIDs of all modified resources
   * @exception ResourceException Exception raised in failure situation
   */
  public IRidSet updateFromVersion(RID version, IResource vcr)
    throws ResourceException;

  // version controller interfaces

  /**
   * Enables child version controlling for this collection -- if a new
   * non-collection child is created it will be put under version control
   * immediately and automatically. As a side effect, enables version control
   * for all existing children that are versionable.
   *
   * @param changedResources when not <code>null</code> , will hold all modified
   *      resources afterwards
   * @param resource collection to enable auto versioning in
   * @exception ResourceException Exception raised in failure situation
   */
  public void enableAutoChildVersioning(IResourceList changedResources, IResource resource)
    throws ResourceException;

  /**
   * Disables child version controlling for this collection.
   *
   * @param disableChildVersioning if <code>true</code> , version control will
   *      be disabled for all direct children of this resource. Whether the
   *      version histories and all revisions will be deleted is determined by
   *      the default of the repository manager.
   * @param resource collection to disable auto versioning in
   * @exception ResourceException Exception raised in failure situation
   */
  public void disableAutoChildVersioning(boolean disableChildVersioning, IResource resource)
    throws ResourceException;

  /**
   * Returns for a given collection if auto versioning of children is enabled.
   *
   * @param resource the collection to ask
   * @return if auto versioning of children is enabled
   * @exception ResourceException Exception raised in failure situation
   */
  public boolean isAutoChildVersioningEnabled(IResource resource)
    throws ResourceException;

  // workspace

  /**
   * Return the RID of the workspace resource to which the given resource
   * belongs.
   *
   * @param resource to determine workspace for
   * @return workspace RID or <code>null</code> if not defined
   * @exception ResourceException Exception raised in failure situation
   */
  public RID getWorkspaceRID(IResource resource)
    throws ResourceException;

  /**
   * Returns a set containing all checked-out resources that belong to the
   * workspace identified by the resource.
   *
   * @param workspace the workspace resource
   * @return the checked-out resources contained in this workspace
   * @exception ResourceException Exception raised in failure situation
   */
  public IRidSet getCheckOutSet(ICollection workspace)
    throws ResourceException;

  /**
   * If called on a workspace controlled collection, a new version controlled
   * resource based on the given revision is created
   *
   * @param baseRevisionRid the rid of the new vcr's base revision
   * @param name member name in collection
   * @param position position where to create the vcr (or <code>null</code> when
   *      unspecified)
   * @param wspControlledCollection the workspace controlled collection to
   *      create the vcr in
   * @return version controlled resource
   * @exception ResourceException Exception raised in failure situation
   * @throws NotSupportedException if called on a resource that is not a sibling
   *      of a workspace resource <br>
   *      - if the resource identified by baseRevisionRid doesn't exist or is no
   *      version resource
   */
  public IResource createVersionControlledResource(RID baseRevisionRid, String name,
    IPosition position, ICollection wspControlledCollection)
    throws ResourceException;

  /**
   * Creates a new workspace resource.
   *
   * @param name member name in collection
   * @param position position where to create the link (or <code>null</code>
   *      when unspecified)
   * @param collection to create new workspace in
   * @return workspace resource
   * @exception ResourceException Exception raised in failure situation
   * @throws NotSupportedException if called on a resource that is not a
   *      workspace collection
   */
  public ICollection createWorkspace(String name, IPosition position, ICollection collection)
    throws ResourceException;
}
