/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.manager;
import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.repository.enum.LinkType;

import com.sapportals.wcm.util.uri.*;

/**
 * Extended namespace manager interface. <p>
 *
 * Copyright (c) SAP AG 2002-2003
 *
 * @author stefan.eissing@greenbytes.de
 * @author julian.reschke@greenbytes.de
 * @version $Revision: 1.8 $
 */
public interface IExtendedNamespaceManager extends INamespaceManager {

  /**
   * Get the target RID of an internal link
   *
   * @param internalLink the internal link resource
   * @return RID of target resource
   * @exception ResourceException Exception raised in failure situation
   */
  public RID getTargetRID(IResource internalLink)
    throws ResourceException;

  /**
   * Set the target RID of an internal link
   *
   * @param target RID of new target
   * @param internalLink internal link resource
   * @exception ResourceException Exception raised in failure situation
   */
  public void setTargetRID(RID target, IResource internalLink)
    throws ResourceException;

  /**
   * Get the target IUri of an external link resource
   *
   * @param externalLink the external link resource
   * @return the IUri of the target
   * @exception ResourceException Exception raised in failure situation
   */
  public IUri getTargetUri(IResource externalLink)
    throws ResourceException;

  /**
   * Set the target of an external link.
   *
   * @param target the target IUri
   * @param externalLink the external link resource
   * @exception ResourceException Exception raised in failure situation
   */
  public void setTargetUri(IUri target, IResource externalLink)
    throws ResourceException;

  /**
   * Determine of internal link is of type dynamic
   *
   * @param link to report type on
   * @return if link is dynamic internal link
   * @exception ResourceException Exception raised in failure situation
   */
  public boolean isDynamicInternalLink(IResource link)
    throws ResourceException;

  /**
   * Determine if the link is permanent (as opposed to temporary).
   *
   * @param link resource to report on
   * @return if link is permanent
   * @exception ResourceException Exception raised in failure situation
   */
  public boolean isPermanentLink(IResource link)
    throws ResourceException;

  /**
   * Creates an external link, compatible to {@link LinkType#EXTERNAL}
   *
   * @param name member name in collection
   * @param isPermanent whether the redirection is temporary (continue using the
   *      RID of the link) or permanent (use of the RID of the link discouraged)
   * @param position position where to create the link (or <code>null</code>
   *      when unspecified)
   * @param properties initial property map for link resource
   * @param ignorePropertyFailures set to <code>true</code> to ignore errors
   *      while initializing the property map
   * @param collection to create link in
   * @param absoluteTargetUri TBD: Description of the incoming method parameter
   * @return link resource
   * @exception ResourceException Exception raised in failure situation
   */
  public IResource createExternalLink(String name, IUri absoluteTargetUri, boolean isPermanent,
    IPosition position, IPropertyMap properties, boolean ignorePropertyFailures,
    ICollection collection)
    throws ResourceException;

  /**
   * Creates an internal link that automatically follows the target resource and
   * disappears when the link target disappears. Note that this type of link can
   * only be created when the link target already exists. <p>
   *
   * <em> Note that a relative RID will be resolved upon creation of the link
   * resource. </em>
   *
   * @param name member name in collection
   * @param targetRid link target
   * @param position position where to create the link (or <code>null</code>
   *      when unspecified)
   * @param properties initial property map for link resource
   * @param ignorePropertyFailures set to <code>true</code> to ignore errors
   *      while initializing the property map
   * @param collection to create link in
   * @return link resource
   * @exception ResourceException Exception raised in failure situation
   */
  public IResource createInternalDynamicLink(String name, RID targetRid, IPosition position,
    IPropertyMap properties, boolean ignorePropertyFailures, ICollection collection)
    throws ResourceException;

  /**
   * Creates an internal link, compatible to {@link LinkType#INTERNAL}
   *
   * @param name member name in collection
   * @param isPermanent whether the redirection is temporary (continue using the
   *      RID of the link) or permanent (use of the RID of the link discouraged)
   * @param position position where to create the link (or <code>null</code>
   *      when unspecified)
   * @param properties initial property map for link resource
   * @param ignorePropertyFailures set to <code>true</code> to ignore errors
   *      while initializing the property map
   * @param collection to create link in
   * @param targetRid TBD: Description of the incoming method parameter
   * @return link resource
   * @exception ResourceException Exception raised in failure situation
   */
  public IResource createInternalStaticLink(String name, RID targetRid, boolean isPermanent,
    IPosition position, IPropertyMap properties, boolean ignorePropertyFailures,
    ICollection collection)
    throws ResourceException;

  /**
   * Return set of all RIDs (bindings) under which the given resource is known.
   *
   * @param resource to report bindings for
   * @return set of all binding RIDs
   * @exception ResourceException Exception raised in failure situation
   */
  public IRidSet getBindings(IResource resource)
    throws ResourceException;

  /**
   * Return a globally unique identifier for this resource
   *
   * @param resource to report id on
   * @return a globally unique identifier for this resource
   * @exception ResourceException Exception raised in failure situation
   */
  public IUri getGloballyUniqueId(IResource resource)
    throws ResourceException;

  /**
   * Return whether this resource allows additional bindings (this may be
   * disallowed for specific resource types)
   *
   * @param resource to report on
   * @return whether this resource allows additional bindings (this may be
   *      disallowed for specific resource types)
   * @exception ResourceException Exception raised in failure situation
   */
  public boolean supportsAdditionalBinds(IResource resource)
    throws ResourceException;

  /**
   * Modify the collection to contain a new binding to an existing resource.
   *
   * @param memberName internal collection member to be created
   * @param position position of new member (or <code>null</code> when not
   *      specified)
   * @param targetRid RID of resource to which the new member shall bind
   * @param overwrite when <code>true</code> , overwrite existing members
   * @param collection collection to be modified
   * @return whether an internal member was overwritten or not
   * @exception ResourceException Exception raised in failure situation
   */
  public boolean createBind(String memberName, IPosition position, RID targetRid, boolean overwrite, ICollection collection)
    throws ResourceException;

  /**
   * Modify the collection to contain a new binding to an existing resource.
   *
   * @param memberName internal collection member to be created
   * @param position position of new member (or <code>null</code> when not
   *      specified)
   * @param overwrite when <code>true</code> , overwrite existing members
   * @param collection collection to be modified
   * @param targetResource TBD: Description of the incoming method parameter
   * @return whether an internal member was overwritten or not
   * @exception ResourceException Exception raised in failure situation
   */
  public boolean createBind(String memberName, IPosition position, IResource targetResource, boolean overwrite, ICollection collection)
    throws ResourceException;

  /**
   * Returns a filtered list of the resources in the collection.
   *
   * @param visible true: only visible resource (system property "hidden" is
   *      false), false: only hidden resources, null: not relevant
   * @param collections true: only collections, false: only resources which are
   *      not collections, null: note relevant
   * @param links true: only links, false: only non-link resources, null: not
   *      relevant
   * @param collection TBD: Description of the incoming method parameter
   * @return A list of resources
   * @exception ResourceException
   * @exception AccessDeniedException If the resource is locked, readonly or
   *      authorization failed
   */
  public IResourceList getChildren(Boolean visible, Boolean collections, Boolean links, ICollection collection)
    throws ResourceException, AccessDeniedException;

  /**
   * Returns information about the collection's ordering, where the following
   * URIs are reserved: <p>
   *
   * "DAV:unordered": unordered collection<br>
   * "DAV:custom": client-maintained ordering, no additional information about
   * the ordering semantics available <p>
   *
   * Any other URI identifies an ordering for which additional information
   * semantics MAY be available at the given URI
   *
   * @param collection the collection for which to get the ordering type
   * @return the ordering type
   * @exception ResourceException Exception raised in failure situation
   */
  public IUri getOrderingType(IResource collection)
    throws ResourceException;

  /**
   * Sets the collection's ordering type. For a definition of predefined URIs,
   * see
   *
   * @param collection the collection for which to set the ordering type
   * @param orderingType orderingType to be set
   * @exception ResourceException Exception raised in failure situation
   * @link getOrderingType
   * @throws NotSupportedException when the repository does not persist specific
   *      ordering types
   */
  public void setOrderingType(IUri orderingType, ICollection collection)
    throws ResourceException;
}

