/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.manager;

import com.sap.tc.logging.Location;

import com.sapportals.portal.security.usermanagement.IUser;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.repository.AccessDeniedException;
import com.sapportals.wcm.repository.Content;
import com.sapportals.wcm.repository.IOErrorException;
import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.ResourceException;
import com.sapportals.wcm.util.content.*;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.uri.RID;
import com.sapportals.wcm.util.urlcontentaccess.*;

/**
 * Abstract base class for repository sub managers.
 */
public abstract class AbstractRepositorySubManager {

  private static Location log = Location.getLocation(com.sapportals.wcm.repository.manager.AbstractRepositorySubManager.class);

  protected final AbstractRepositoryManager repositoryManager;

  protected AbstractRepositorySubManager(IRepositoryManager manager) {
    this.repositoryManager = (AbstractRepositoryManager)manager;
  }

  /**
   * Generate an event for a resource and send it.
   *
   * @param type The type for the resource as defined in <class>ResourceEvent
   *      </class>
   * @param resource The resource associated with that event
   * @param param TBD: Description of the incoming method parameter
   * @deprecated as of NW04.
   */
  protected final void sendEvent(int type, IResource resource, Object param) {
    if (resource != null) {
      IResourceEvent event = new ResourceEvent(resource, type, null, param);
      try {
        resource.getRepositoryManager().getEventBroker()
          .send(event, (AbstractRepositoryManager)this.repositoryManager);
      }
      catch (WcmException ex) {
        log.errorT("sendEvent(63)", "Failed to send event " + event.getDescription() + ": " + ex.getMessage() + ": " + LoggingFormatter.extractCallstack(ex));
      }
    }
  }

  /**
   * Generate an event for a resource and send it.
   *
   * @param type The type for the resource as defined in <class>ResourceEvent
   *      </class>
   * @param resource The resource associated with that event
   * @param correlationId TBD: Description of the incoming method parameter
   * @param param TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  protected final IResourceEvent sendEvent(IResource resource, int type, String correlationId, Object param) {
    if (resource != null) {
      IResourceEvent event = new ResourceEvent(resource, type, correlationId, param);
      try {
        resource.getRepositoryManager().getEventBroker()
          .send(event, (AbstractRepositoryManager)this.repositoryManager);
        return event;
      }
      catch (WcmException ex) {
        log.errorT("sendEvent(87)", "Failed to send event " + event.getDescription() + ": " + ex.getMessage() + ": " + LoggingFormatter.extractCallstack(ex));
      }
    }
    return null;
  }

  /**
   * Get the content for a external URL
   *
   * @param url TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception ResourceException Exception raised in failure situation
   */
  protected final IContent readExternalContent(String url)
    throws ResourceException {
    try {
      IURLContent c = URLContentAccess.getInstance().readContent(url, true);
      return new Content(c.getInputStream(), c.getContentType(), c.getContentLength());
    }
    catch (URLContentAccessException ex) {
      throw new IOErrorException(ex.getMessage(), ex, (RID)null);
    }
  }

  /**
   * Check if the user ID in the resource context has the permission
   *
   * @param res The resource the user wants to access
   * @param permissionName TBD: Description of the incoming method parameter
   * @exception ResourceException
   * @exception AccessDeniedException If the user in the context does not have
   *      the permission If there is no user ID in the context
   * @see com.sapportals.wcm.repository.manager.IPermission
   */
  protected final void checkAuthorization(IResource res, String permissionName)
    throws ResourceException, AccessDeniedException {
    ISecurityManager sm = this.repositoryManager.getSecurityManager(res);

    if (sm == null) {
      // No security mgr, access granted
      return;
    }

    IPermission perm = sm.getPermission(permissionName);
    if (perm == null) {
      if (log.beDebug()) {
        log.debugT("checkAuthorization(133)", "Security Manager: " + sm + " returned no permission for name " + permissionName + ", access denied");
      }
      throw new AccessDeniedException("Permission denied: uri=" + res.getRID().toString() + ", permission=" + permissionName + ", user=" + res.getContext().getUser().getId(),
        res.getRID(), permissionName, res.getContext().getUser().getId());
    }

    // Throw AccessDeniedException if user in context not authorized or not authenticated
    IUser user = res.getContext().getUser();
    if (user != null) {
      if (!user.isAuthenticated()) {
        throw new AccessDeniedException("User <" + user.getId() + "> is not authenticated", res.getRID(),
          permissionName, res.getContext().getUser().getId());
      }
      if (!sm.isAllowed(res, res.getContext().getUser(), perm)) {
        throw new AccessDeniedException("Permission denied: uri=" + res.getRID().toString() + ", permission=" + permissionName + ", user=" + res.getContext().getUser().getId(),
          res.getRID(), permissionName, res.getContext().getUser().getId());
      }
    }
    else {
      throw new AccessDeniedException("No user in context", res.getRID(), permissionName, null);
    }
  }

  /**
   * Will be called by the framework after the repository manager has beed
   * started. Can optionally be overwritten in sub manager implementation. This
   * default implementation does nothing.
   *
   * @exception WcmException Exception raised in failure situation
   */
  protected void startUp()
    throws WcmException { }

  /**
   * Will be called by the framework prior to shutting down the repository
   * manager. Can optionally be overwritten in sub manager implementation. This
   * default implementation does nothing.
   */
  protected void shutDown() { }

  /**
   * Return ISecurityChecker for this manager. Is only valid when all repository
   * submanagers have been initialized.
   *
   * @return security checker
   * @exception ResourceException Exception raised in failure situation
   */
  protected final ISecurityChecker getSecurity()
    throws ResourceException {
    return this.repositoryManager.getSecurity();
  }
}

