/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.manager;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;

import com.sap.tc.logging.Location;
import com.sapportals.config.fwk.Configuration;
import com.sapportals.config.fwk.IConfigClientContext;
import com.sapportals.config.fwk.IConfigManager;
import com.sapportals.config.fwk.IConfigPlugin;
import com.sapportals.config.fwk.IConfigurable;
import com.sapportals.wcm.IFrameworkTransaction;
import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.crt.CrtClassLoaderRegistry;
import com.sapportals.wcm.crt.component.ComponentState;
import com.sapportals.wcm.crt.component.ComponentStateHandler;
import com.sapportals.wcm.crt.component.ContextException;
import com.sapportals.wcm.crt.component.IComponentInfo;
import com.sapportals.wcm.crt.component.IComponentManager;
import com.sapportals.wcm.crt.component.IContainer;
import com.sapportals.wcm.crt.component.IContainerManager;
import com.sapportals.wcm.crt.component.IContext;
import com.sapportals.wcm.crt.component.ILifecycleInfo;
import com.sapportals.wcm.crt.component.IStartable;
import com.sapportals.wcm.crt.component.IThreadSafe;
import com.sapportals.wcm.crt.component.StartupException;
import com.sapportals.wcm.crt.configuration.ConfigurationException;
import com.sapportals.wcm.crt.configuration.DefaultConfiguration;
import com.sapportals.wcm.crt.configuration.IConfigConst;
import com.sapportals.wcm.crt.configuration.IConfiguration;
import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.IResourceContext;
import com.sapportals.wcm.repository.IResourceList;
import com.sapportals.wcm.repository.NameInfo;
import com.sapportals.wcm.repository.ResourceErrors;
import com.sapportals.wcm.repository.ResourceException;
import com.sapportals.wcm.repository.ResourceFactory;
import com.sapportals.wcm.repository.ResourceImpl;
import com.sapportals.wcm.repository.ResourceList;
import com.sapportals.wcm.repository.VirtualRootNamespaceContentPropertyManager;
import com.sapportals.wcm.repository.VirtualRootRepositoryManager;
import com.sapportals.wcm.repository.runtime.ICmRuntimeConst;
import com.sapportals.wcm.repository.service.IRepositoryService;
import com.sapportals.wcm.util.config.ConfigCrutch;
import com.sapportals.wcm.util.events.EventList;
import com.sapportals.wcm.util.events.IEventList;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.uri.IRidList;
import com.sapportals.wcm.util.uri.RID;

/**
 * Base class for repository manager implementations. <p>
 *
 * Copyright (c) SAP AG 2001 - 2002
 *
 * @author Markus Breitenfelder
 * @version $Id: //javabas/com.sapportals.wcm/50_COR/src/java/repository/framework/core/com/sapportals/wcm/repository/manager/AbstractRepositoryManager.java#53
 *      $
 */
public abstract class AbstractRepositoryManager implements IRepositoryManager, IResourceEventSender, IThreadSafe, IContainer,
  com.sapportals.wcm.crt.component.IConfigurable, IStartable, IComponentInfo, ILifecycleInfo {

  /**
   * TBD: Description of the class.
   */
  private interface CFG {
    String PREFIX = "prefix";
    String EVENTS = "sendevents";
    String SERVICES = "services";
    String NAMESPACE_MANAGER = "namespacemgr.class";
    String PROPERTY_MANAGER = "propertymgr.class";
    String LOCK_MANAGER = "lockmgr.class";
    String VERSIONING_MANAGER = "versioningmgr.class";
    String CONTENT_MANAGER = "contentmgr.class";
    String SECURITY_MANAGER = "securitymgr.ref";
    String PROPSEARCH_MANAGER = "propertysearchmgr.class";
  }


  private final static String CFG_SEC_MGR_CLASS = "class";
  protected final ComponentStateHandler stateHandler = new ComponentStateHandler(this);
  private static Location log = Location.getLocation(com.sapportals.wcm.repository.manager.AbstractRepositoryManager.class);

  private final static String[] NOT_RECONFIGURABLE = {
    AbstractRepositoryManager.CFG.PREFIX,
    AbstractRepositoryManager.CFG.EVENTS,
    AbstractRepositoryManager.CFG.SERVICES,
    AbstractRepositoryManager.CFG.NAMESPACE_MANAGER,
    AbstractRepositoryManager.CFG.PROPERTY_MANAGER,
    AbstractRepositoryManager.CFG.LOCK_MANAGER,
    AbstractRepositoryManager.CFG.VERSIONING_MANAGER,
    AbstractRepositoryManager.CFG.CONTENT_MANAGER,
    AbstractRepositoryManager.CFG.PROPSEARCH_MANAGER
    };

  private final static String EVENTBROKER_CRTKEY = "eventbroker";
  private final static String FILTER_CRTKEY = "filter";
  private final static String SERVICES_CRTKEY = "services";
    
  private final static EventList resourceEvents = new EventList();
  private final static EventList collectionEvents = new EventList();

  private final static String CFG_PLUGIN_CM_REPOSITORY_MANAGERS_SECURITY_MANAGERS = "/cm/repository_managers/security_managers";

  /**
   * A "empty" name info object. It's constructor will take the Framework
   * defaults
   */
  private final static NameInfo g_nameInfo = new NameInfo(
    new char[]{},
    new char[]{}, -1, -1);

  /**
   * This manager's configuration
   */
  protected IConfiguration config;
  
  /**
   * This manager's configuration converted to a Properties object. This will be
   * null unless the manager implementation calls the conversion method in
   * startUp !!
   */
  protected Properties properties;
  protected String prefix;
  protected String prefix_withSlash;
  private INamespaceManager namespaceManager;
  private IPropertyManager propertyManager;
  private ILockManager lockManager;
  private IVersioningManager versioningManager;
  private IContentManager contentManager;
  private ISecurityManager securityManager;
  private IPropertySearchManager propSearchManager;
  private String id;
  private String description;

  /**
   * Value of the configuration paramete "sendevents"
   */
  private boolean eventsEnabled;

  /**
   * Reference to the event broker
   */
  private IResourceEventBroker eventBroker;

  /**
   * The container manager of CRT
   */
  private IContainerManager containerManager;

  /**
   * The CRT key of this component
   */
  private String key;

  /**
   * Flag: startup was called
   */
  private boolean started;
  private ISecurityChecker securityChecker = SecurityChecker.NO_CHECK;
  private static HashMap securityManagerMap;// contains Strings

  static {
    AbstractRepositoryManager.resourceEvents.add(ResourceEvent.CHECKIN_TEMPLATE);
    AbstractRepositoryManager.resourceEvents.add(ResourceEvent.CHECKOUT_TEMPLATE);
    AbstractRepositoryManager.resourceEvents.add(ResourceEvent.COPY_TEMPLATE);
    AbstractRepositoryManager.resourceEvents.add(ResourceEvent.DELETE_TEMPLATE);
    AbstractRepositoryManager.resourceEvents.add(ResourceEvent.GET_TEMPLATE);
    AbstractRepositoryManager.resourceEvents.add(ResourceEvent.LOCK_TEMPLATE);
    AbstractRepositoryManager.resourceEvents.add(ResourceEvent.MOVE_TEMPLATE);
    AbstractRepositoryManager.resourceEvents.add(ResourceEvent.PROPERTY_DELETE_TEMPLATE);
    AbstractRepositoryManager.resourceEvents.add(ResourceEvent.PROPERTY_SET_TEMPLATE);
    AbstractRepositoryManager.resourceEvents.add(ResourceEvent.PROPERTY_GET_TEMPLATE);
    AbstractRepositoryManager.resourceEvents.add(ResourceEvent.RENAME_TEMPLATE);
    AbstractRepositoryManager.resourceEvents.add(ResourceEvent.SET_TEMPLATE);
    AbstractRepositoryManager.resourceEvents.add(ResourceEvent.UNDO_CHECKOUT_TEMPLATE);
    AbstractRepositoryManager.resourceEvents.add(ResourceEvent.UNLOCK_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.CHECKIN_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.CHECKOUT_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.COPY_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.DELETE_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.GET_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.LOCK_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.MOVE_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.PROPERTY_DELETE_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.PROPERTY_SET_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.PROPERTY_GET_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.RENAME_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.SET_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.UNDO_CHECKOUT_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.UNLOCK_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.CREATE_CHILD_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.CREATE_COLLECTION_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.CREATE_LINK_TEMPLATE);
    AbstractRepositoryManager.collectionEvents.add(ResourceEvent.GET_CHILDREN_TEMPLATE);
  }

  /**
   * The default constructor
   */
  public AbstractRepositoryManager() { }

  /**
   * @param properties Properties containing configuration data.
   * @deprecated as of NW04. Use the default constructor instead
   */
  public AbstractRepositoryManager(Properties properties) {
    this.properties = (Properties)properties.clone();
  }

  protected void registerSecurityManager(String name, String manager) {
    synchronized (AbstractRepositoryManager.class) {
      AbstractRepositoryManager.securityManagerMap.put(name, manager);
    }
  }

  /**
   * ResourceImpl will call this method to check if a given resource event type
   * must be sent at all.
   * @param type the type of ResourceEvent to send
   * @return true, if the given ResourceEvent type has to be sent,
   *         false if no receiver registered for this ResourceEvent type
   */
  public final boolean mustSendResourceEventType(int type) {
    return this.eventBroker.mustSendResourceEventType(type);
  }
  
  /**
   * Sends an event to all registered receivers.
   * @param type The event type, e.g. ResourceEvent.DELETE
   * @param resource The resource object which will be attached to the event. Should not be <code>null</code> - otherwise no event will be generated.
   * @param param Some arbitrary parameter which will be attached to the event.
   * @deprecated as of NW04. Use the sendEvent() method with the additional correlationId parameter.
   */
  public final void sendEvent(int type, IResource resource, Object param) {
    if (resource != null) {
      IResourceEvent event = new ResourceEvent(resource, type, null, param);
      try {
        this.eventBroker.send(event, this);
      }
      catch (WcmException ex) {
        AbstractRepositoryManager.log.errorT("Failed to send event " + event.getDescription() + ": " + LoggingFormatter.extractCallstack(ex));
      }
    }
  }

  /**
   * Sends an event to all registered receivers. 
   * @param type The event type, e.g. ResourceEvent.DELETE
   * @param resource The resource object which will be attached to the event. Should not be <code>null</code> - otherwise no event will be generated.
   * @param correlationId An arbitrary identifier value that can be used by event receivers to correlate multible events which
   * have the same ID (e.g. pre and post events). Can be <code>null</code>.
   * @param param Some arbitrary parameter which will be attached to the event.
   */
  public final IResourceEvent sendEvent(IResource resource, int type, String correlationId, Object param) {
    if (resource != null) {
      IResourceEvent event = new ResourceEvent(resource, type, correlationId, param);
      try {
        this.eventBroker.send(event, this);
        return event;
      }
      catch (WcmException ex) {
        AbstractRepositoryManager.log.errorT("Failed to send event " + event.getDescription() + ": " + LoggingFormatter.extractCallstack(ex));
      }
    }
    return null;
  }

  /**
   * ResourceImpl will call this method to send the standard resource events
   *
   * @param resource TBD: Description of the incoming method parameter
   * @param type TBD: Description of the incoming method parameter
   * @param correlationId TBD: Description of the incoming method parameter
   * @param param TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public final boolean sendEvent(IFrameworkTransaction ft, IResource resource, int type, String correlationId, Object param) {
    boolean ok = true;
    if (resource != null) {
      IResourceEvent event = new ResourceEvent(resource, type, correlationId, param);
      try {
        ok = this.eventBroker.send(ft, event, this);        
      }
      catch (WcmException ex) {
        ok = false;
        AbstractRepositoryManager.log.errorT("Failed to send event " + event.getDescription() + ": " + LoggingFormatter.extractCallstack(ex));
      }
    }
    return ok;
  }
  
  /**
   * ResourceImpl will call this method to send the standard resource events
   *
   * @param resource TBD: Description of the incoming method parameter
   * @param type TBD: Description of the incoming method parameter
   * @param correlationId TBD: Description of the incoming method parameter
   * @param param TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public final boolean sendEvent(IFrameworkTransaction ft, int type, IResource resource, Object param) {
    boolean ok = true;
    if (resource != null) {
      IResourceEvent event = new ResourceEvent(resource, type, null, param);
      try {
        ok = this.eventBroker.send(ft, event, this);
      }
      catch (WcmException ex) {
        ok = false;
        AbstractRepositoryManager.log.errorT("Failed to send event " + event.getDescription() + ": " + LoggingFormatter.extractCallstack(ex));
      }
    }
    return ok;
  }

  /**
   * Returns true if standard resource events should be sent by the framework.
   * Will be called by ResourceImpl.
   * @return true if standard resource events should be sent by the framework.
   */
  public final boolean eventsEnabled() {
    return this.eventsEnabled;
  }

  // ---------------------------------------------------------------------------
  // interface IRepositoryManager
  // ---------------------------------------------------------------------------

  public final Collection getAllServices()
    throws ResourceException {
    return ResourceFactory.getInstance().getServiceFactory().getAllRepositoryServices(this);
  }

  public final IRepositoryService getService(String type)
    throws ResourceException {
    return ResourceFactory.getInstance().getServiceFactory().getRepositoryService(this, type);
  }

  public final boolean isServiceAvailable(String type)
    throws ResourceException {
    return ResourceFactory.getInstance().getServiceFactory().isRepositoryServiceAvailable(this, type);
  }

  public final IResourceEventBroker getEventBroker() {
    return this.eventBroker;
  }

  public final String getPrefix() {
    return this.prefix;
  }

  public final Properties getConfig() {
    if (this.properties != null) {
      return (Properties)this.properties.clone();
    }
    else {
      throw new RuntimeException("Configuration has not been converted to Properties");
    }
  }

  public final IConfiguration getConfiguration() {
    return this.config;
    // configuration object is in readonly-mode !
  }

  // ---------------------------------------------------------------------------
  // interface IConfigurable
  // ---------------------------------------------------------------------------

	public final void configure(IConfiguration config) throws ConfigurationException {
		this.stateHandler.preConfigure();
		this.config = config;
		try {
			this.id = config.getAttribute(ICmRuntimeConst.CONFIG_ATTRIBUTE_NAME.ID);
			this.prefix = config.getAttribute(AbstractRepositoryManager.CFG.PREFIX);
			this.prefix_withSlash = this.prefix + '/';
			this.description = config.getAttribute(ICmRuntimeConst.CONFIG_ATTRIBUTE_NAME.DESCRIPTION, null);
			String value = config.getAttribute(AbstractRepositoryManager.CFG.EVENTS, "false");
			this.eventsEnabled = new Boolean(value).booleanValue();

			synchronized (AbstractRepositoryManager.class) {
				if (AbstractRepositoryManager.securityManagerMap == null) {
					AbstractRepositoryManager.securityManagerMap = new HashMap();
					try {
						// register security managers
						IConfigClientContext context = IConfigClientContext.createContext(ConfigCrutch.getConfigServiceUser());
						IConfigManager cfg = Configuration.getInstance().getConfigManager(context);
						IConfigPlugin plugin = cfg.getConfigPlugin(CFG_PLUGIN_CM_REPOSITORY_MANAGERS_SECURITY_MANAGERS);
						if (plugin == null) {
							throw new RuntimeException("missing plugin " + CFG_PLUGIN_CM_REPOSITORY_MANAGERS_SECURITY_MANAGERS);
						}

						IConfigurable[] configurables = plugin.getConfigurables();
						for (int i = 0; i < configurables.length; i++) {
							AbstractRepositoryManager.securityManagerMap.put(
								configurables[i].getIdValue(),
								configurables[i].getPropertyValue(CFG_SEC_MGR_CLASS));
						}
					}
					catch (Exception e) {
						throw new ConfigurationException(e);
					}
				}
			}
		}
		catch (ConfigurationException x) {
			throw stateHandler.postConfigure(x);
		}
		this.stateHandler.postConfigure();
	}

  // ---------------------------------------------------------------------------
  // interface IStartable
  // ---------------------------------------------------------------------------

  public final void start()
    throws StartupException {
    this.stateHandler.preStart();

    // Instantiate a event broker as child component and register this repository as sender
    if (!this.started) {
      String crtKey = AbstractRepositoryManager.EVENTBROKER_CRTKEY;
      try {
        IConfiguration cfg = new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT_CONFIG);
        this.containerManager.addChild(crtKey, ResourceEventBroker.class.getName(), null, cfg,
          new Object[]{this, parseValueList(this.config.getAttribute("services", ""), ",")});
        this.eventBroker = (IResourceEventBroker)this.containerManager.lookupComponent(crtKey);
      }
      catch (Exception ex) {
        AbstractRepositoryManager.log.errorT("start(347)", LoggingFormatter.extractCallstack(ex));
        throw this.stateHandler.postStart(new StartupException("Failed to create event broker: " + ex.getMessage(), ex));
      }

      crtKey = AbstractRepositoryManager.FILTER_CRTKEY;
      try {
        IConfiguration cfg = new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT_CONFIG);
        this.containerManager.addChild(crtKey, RepFilterContainer.class.getName(), null, cfg, this.id);
      }
      catch (Exception ex) {
        AbstractRepositoryManager.log.errorT("start(347)", LoggingFormatter.extractCallstack(ex));
        throw this.stateHandler.postStart(new StartupException("Failed to create filter collection: " + ex.getMessage(), ex));
      }

      // runtime collection of repository services
      crtKey = AbstractRepositoryManager.SERVICES_CRTKEY;
      try {
        IConfiguration cfg = new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT_CONFIG);
        this.containerManager.addChild(crtKey, RepServiceContainer.class.getName(), null, cfg, this.id);
      }
      catch (Exception ex) {
        AbstractRepositoryManager.log.errorT("start(347)", LoggingFormatter.extractCallstack(ex));
      }

      // Create an instance for each sub-manager class and initialize them with a reference
      // to this manager.
      try {
        String cn = this.config.getAttribute(AbstractRepositoryManager.CFG.NAMESPACE_MANAGER, null);
        if (cn != null) {
          this.assignSubManager(this.createSubManager(cn));
        }
        cn = this.config.getAttribute(AbstractRepositoryManager.CFG.CONTENT_MANAGER, null);
        if (cn != null) {
          this.assignSubManager(this.createSubManager(cn));
        }
        cn = this.config.getAttribute(AbstractRepositoryManager.CFG.PROPERTY_MANAGER, null);
        if (cn != null) {
          this.assignSubManager(this.createSubManager(cn));
        }
        cn = this.config.getAttribute(AbstractRepositoryManager.CFG.LOCK_MANAGER, null);
        if (cn != null) {
          this.assignSubManager(this.createSubManager(cn));
        }
        cn = this.config.getAttribute(AbstractRepositoryManager.CFG.VERSIONING_MANAGER, null);
        if (cn != null) {
          this.assignSubManager(this.createSubManager(cn));
        }
        cn = this.config.getAttribute(AbstractRepositoryManager.CFG.SECURITY_MANAGER, null);
        if (cn != null) {
          cn = (String)AbstractRepositoryManager.securityManagerMap.get(cn);
          if (cn == null) {
            throw new StartupException("Unknown security manager: " + cn);
          }
          this.assignSubManager(this.createSubManager(cn));
        }
        cn = this.config.getAttribute(AbstractRepositoryManager.CFG.PROPSEARCH_MANAGER, null);
        if (cn != null) {
          this.assignSubManager(this.createSubManager(cn));
        }
      }
      catch (Exception ex) {
        AbstractRepositoryManager.log.errorT(LoggingFormatter.extractCallstack(ex));
        throw this.stateHandler.postStart(new StartupException("Failed to init sub-manager: " + ex.getMessage(), ex));
      }
    }// if (!this.started)

    this.started = true;
    try {
      this.startUpImpl();
    }
    catch (ConfigurationException ex) {
      StartupException x = this.stateHandler.postStart(ex);
      this.logStartUpTime();
      throw x;
    }
    catch (StartupException ex) {
      StartupException x = this.stateHandler.postStart(ex);
      this.logStartUpTime();
      throw x;
    }
    catch (Exception ex) {
      StartupException x = this.stateHandler.postStart(new StartupException(this.getID() + ": runtime exception during start up: " + ex.getMessage(), ex));
      this.logStartUpTime();
      throw x;
    }

    // Now call startUp() methof of sub-managers
    // startup of this manager must fail if one sub-manager fails.
    // (sub-managers are NOT components and are not visible in the CRT).
    if (!this.getClass().equals(VirtualRootRepositoryManager.class)) {// Yes I know, this is a HACK !
      try {
        if (this.securityManager != null) {
          ((AbstractRepositorySubManager)this.securityManager).startUp();
        }
        // Now that a (possible) security Manager is started, initialize the security checker
        try {
          this.securityChecker = SecurityChecker.getInstance(this);
        }
        catch (ResourceException e) {
          AbstractRepositoryManager.log.errorT("start(426)", LoggingFormatter.extractCallstack(e));
          throw new StartupException("Failed to prepare security checker: " + e.getMessage(), e);
        }
        if (this.namespaceManager != null) {
          ((AbstractRepositorySubManager)this.namespaceManager).startUp();
        }
        if (this.contentManager != null) {
          ((AbstractRepositorySubManager)this.contentManager).startUp();
        }
        if (this.propertyManager != null) {
          ((AbstractRepositorySubManager)this.propertyManager).startUp();
        }
        if (this.lockManager != null) {
          ((AbstractRepositorySubManager)this.lockManager).startUp();
        }
        if (this.versioningManager != null) {
          ((AbstractRepositorySubManager)this.versioningManager).startUp();
        }
        if (this.propSearchManager != null) {
          ((AbstractRepositorySubManager)this.propSearchManager).startUp();
        }
      }
      catch (WcmException ex) {
        StartupException x = this.stateHandler.postStart(
          new StartupException("Exception during start up of sub-manager: " + ex.getMessage(), ex));
        this.logStartUpTime();
        throw x;
      }
      catch (Exception ex) {
        StartupException x = this.stateHandler.postStart(
          new StartupException("Runtime exception during startup of sub-manager: " + ex.getMessage(), ex));
        this.logStartUpTime();
        throw x;
      }
    }

    this.stateHandler.postStart();
    this.logStartUpTime();
  }

  public final void stop() {
    this.stateHandler.preStop();   
    
		if (!(this instanceof VirtualRootRepositoryManager)) {// Yes this is a HACK !
			try {
				if (this.securityManager != null) {
					((AbstractRepositorySubManager)this.securityManager).shutDown();
				}
			}
			catch (Exception e) {
				AbstractRepositoryManager.log.errorT("stop", LoggingFormatter.extractCallstack(e));
			}				
			
			try {		
  			if (this.namespaceManager != null) {
					((AbstractRepositorySubManager)this.namespaceManager).shutDown();
				}
			}
			catch (Exception e) {
				AbstractRepositoryManager.log.errorT("stop", LoggingFormatter.extractCallstack(e));
			}			
			try {
				if (this.contentManager != null) {
					((AbstractRepositorySubManager)this.contentManager).shutDown();
				}
			}
			catch (Exception e) {
				AbstractRepositoryManager.log.errorT("stop", LoggingFormatter.extractCallstack(e));
			}			
			try {
				if (this.propertyManager != null) {
					((AbstractRepositorySubManager)this.propertyManager).shutDown();
				}
			}
			catch (Exception e) {
				AbstractRepositoryManager.log.errorT("stop", LoggingFormatter.extractCallstack(e));
			}							
			try {
				if (this.lockManager != null) {
					((AbstractRepositorySubManager)this.lockManager).shutDown();
				}
			}
			catch (Exception e) {
				AbstractRepositoryManager.log.errorT("stop", LoggingFormatter.extractCallstack(e));
			}							
			try {
				if (this.versioningManager != null) {
					((AbstractRepositorySubManager)this.versioningManager).shutDown();
				}
			}
			catch (Exception e) {
				AbstractRepositoryManager.log.errorT("stop", LoggingFormatter.extractCallstack(e));
			}							
			try {
				if (this.propSearchManager != null) {
					((AbstractRepositorySubManager)this.propSearchManager).shutDown();
				}
			}
			catch (Exception e) {
				AbstractRepositoryManager.log.errorT("stop", LoggingFormatter.extractCallstack(e));
			}			
		}
		    
		this.shutDownImpl();
		
		this.stateHandler.postStop();
  }

  // ---------------------------------------------------------------------------
  // interface ILifecycleInfo
  // ---------------------------------------------------------------------------

  public ComponentState getState() {
      return this.stateHandler.getState();
  }

  public final Date getCreationDate() {
    return this.stateHandler.getCreationDate();
  }

  /**
   * Helper method for conversion of configuration data into properties.
   *
   * @param config to be converted to properties
   * @return all attributes of config as properties
   */
  protected final Properties convertConfigToProperties(IConfiguration config) {
    return convertConfigToProperties(config, null);
  }

  /**
   * Helper method for conversion of configuration data into properties.
   *
   * @param config to be converted to properties
   * @param prefix to prepend to all attribute names
   * @return all attributes of config as properties
   */
  protected final Properties convertConfigToProperties(IConfiguration config, String prefix) {
    Properties props = new Properties();
    String[] attributeNames = config.getAttributeNames();
    for (int i = 0; i < attributeNames.length; i++) {
      String name = attributeNames[i];
      String value = config.getAttribute(name, null);
      if (value != null) {
        if (prefix != null) {
          name = prefix + name;
        }
        props.setProperty(name, value);
      }
    }
    return props;
  }

	/**
	 * Helper method for reconfigure() implementations. Throws a configuration
	 * exception if any of the specified attributes or any of the non-reconfigurable framework attributes (e.g. prefix) has changed.
	 * @param notReconfigurable An array of names of attributes that cannot be canged wihout restarting the component/server.
	 * @param newConfig The new configuration data received as argument of the reconfigure() method.
	 * @exception ConfigurationException
	 */
	protected final void checkNotReconfigurableAttributes(String[] notReconfigurable, IConfiguration newConfig)
		throws ConfigurationException {
		this.checkNotReconfigurableAttributeNames(AbstractRepositoryManager.NOT_RECONFIGURABLE, newConfig);
		this.checkNotReconfigurableAttributeNames(notReconfigurable, newConfig);
	}

  public boolean isRoot(String uri) {
    return (uri.equals(this.prefix) || uri.equals(this.prefix_withSlash));
  }

  // ---------------------------------------------------------------------------
  // interface IResourceEventSender
  // ---------------------------------------------------------------------------

  public IEventList getEvents() {
    if (eventsEnabled()) {
      return AbstractRepositoryManager.collectionEvents;
    }
    else {
      return null;
    }
  }

  public IEventList getEvents(IResource resource) {
    if (!eventsEnabled()) {
      return null;
    }
    if (resource.isCollection()) {
      return AbstractRepositoryManager.collectionEvents;
    }
    else {
      return AbstractRepositoryManager.resourceEvents;
    }
  }

  public INamespaceManager getNamespaceManager(IResource resource)
    throws ResourceException {
    return this.namespaceManager;
  }

  public IPropertyManager getPropertyManager(IResource resource)
    throws ResourceException {
    return this.propertyManager;
  }

  public ILockManager getLockManager(IResource resource)
    throws ResourceException {
    return this.lockManager;
  }

  public IVersioningManager getVersioningManager(IResource resource)
    throws ResourceException {
    return this.versioningManager;
  }

  public IContentManager getContentManager(IResource resource)
    throws ResourceException {
    return this.contentManager;
  }

  public ISecurityManager getSecurityManager(IResource resource)
    throws ResourceException {
    return this.securityManager;
  }

  public IPropertySearchManager getPropertySearchManager(IResource resource)
    throws ResourceException {
    return this.propSearchManager;
  }

  public String getID() {
    return this.id;
  }

  public String getDescription() {
    return this.description;
  }

  // ---------------------------------------------------------------------------
  // interface IContextualizable
  // ---------------------------------------------------------------------------

  public void contextualize(IContext context)
    throws ContextException {
    this.stateHandler.preContextualize();
    this.containerManager = context.getContainerManager();
    this.key = context.getKey();
    if (this.containerManager == null) {
      throw new ContextException("No container manager in context");
    }
    this.stateHandler.postContextualize();
  }

  // ---------------------------------------------------------------------------
  // interface IContainer
  // ---------------------------------------------------------------------------

  public IComponentManager getComponentManager() {
    return this.containerManager;
  }

  // ---------------------------------------------------------------------------
  // interface IComponentInfo
  // ---------------------------------------------------------------------------

  public String getName() {
    return this.id;
  }

  public String getDescription(Locale locale) {
    // Default implementation returns description in configuration data
    return this.getDescription();
  }

  public Properties getProperties() {
    Properties props = this.convertConfigToProperties(this.config, "config-");
    props.setProperty("type", "RepositoryManager");
    return props;
  }

  public ConfigurationException getLastConfigurationException() {
    return this.stateHandler.getLastConfigurationException();
  }

  public StartupException getStartupException() {
    return this.stateHandler.getStartupException();
  }

  public Date getLastReconfigurationDate() {
    return this.stateHandler.getLastReconfigurationDate();
  }

  public Date getNextAutoRestartDate() {
    return this.stateHandler.getNextAutoRestartDate();
  }

  /**
   * A repository manager must implement this method to provide a resource for a
   * RID and context. Must return null if the resource does not exist.
   * @param rid The resource identifier.
   * @param context The resource context containing the user object.
   * @return A resource instance.
   * @exception ResourceException Exception raised in failure situation
   */
  public abstract ResourceImpl getResource(RID rid, IResourceContext context)
    throws ResourceException;

  /**
   * Base version: Calls getResource for each RID. The manager should overwrite
   * this with a more efficient implementation if possible.
   *
   * @param ridList A RID list
   * @param errors The errors
   * @param context The resource context
   * @return The list of valid resource. Some
   * @exception ResourceException If an error occured while executing the
   *      method.
   */
  public ResourceList getResources(IRidList ridList, ResourceErrors errors, IResourceContext context)
    throws ResourceException {
    return getResourcesDefault(ridList, errors, null, context);
  }

  /**
   * Base version: Calls getResource for each RID and removes all resources
   * without the specified permission. The manager should overwrite this with a
   * more efficient implementation if possible.
   *
   * @param ridList A RID list
   * @param errors The errors
   * @param permissionNames The permission names
   * @param context The resource context
   * @return The list of valid resource. Some
   * @exception ResourceException If an error occured while executing the
   *      method.
   */
  public ResourceList getResources(IRidList ridList, ResourceErrors errors, String[] permissionNames, IResourceContext context)
    throws ResourceException {
    return getResourcesDefault(ridList, errors, permissionNames, context);
  }

  /**
   * Base version: Calls getResource for each RID. The manager should overwrite
   * this with a more efficient implementation if possible. The map of {@link IPropertyMap}s 
   * need not hold a property map for each and every RID.
   *
   * @param ridList A RID list
   * @param propertyMaps a map of {@link IPropertyMap}s for RIDs
   * @param errors The errors
   * @param context The resource context
   * @return The list of valid resource. Some
   * @exception ResourceException If an error occured while executing the
   *      method.
   */
  public ResourceList getResources(IRidList ridList, Map propertyMaps, ResourceErrors errors, IResourceContext context)
    throws ResourceException {
    return getResourcesDefault(ridList, errors, null, context);
  }

  /**
   * Base version: Calls getResource for each RID. The manager should overwrite
   * this with a more efficient implementation if possible. The map of {@link
   * IPropertyMap}s need not hold a property map for each and every RID.
   *
   * @param ridList A RID list
   * @param propertyMaps a map of {@link IPropertyMap}s for RIDs
   * @param errors The errors
   * @param permissionNames The permission names
   * @param context The resource context
   * @return The list of valid resource. Some
   * @exception ResourceException If an error occured while executing the
   *      method.
   */
  public ResourceList getResources(IRidList ridList, Map propertyMaps, ResourceErrors errors, String[] permissionNames,
    IResourceContext context)
    throws ResourceException {
    return getResourcesDefault(ridList, errors, permissionNames, context);
  }

  /**
   * Returns an object that contains information about the attributes of
   * resource names supported by this repository manager. This base class
   * implementation returns the WMC Framework's restrictions on resource names.
   * An repository manager implementation should overwrite it. It should not
   * include characters that are already forbidden by the framwork: '?', '/',
   * '\', '<' and '>'
   *
   * @return The nameInfo value
   */
  public NameInfo getNameInfo() {
    return g_nameInfo;
  }
  // ---------------------------------------------------------------------------

  /**
   * The implementation must overwrite this method if it has initial startup
   * handling to do (acquire resources, open connections to remote systems,
   * etc). Accoring to the contract of the {@link
   * com.sapportals.wcm.crt.component.IStartable} interface of the CRT this
   * method will be called only once. It will be called on systen start up or
   * when a new instance of this component is added to the persistence
   * configuration.
   *
   * @exception ConfigurationException Exception raised in failure situation
   * @exception StartupException Exception raised in failure situation
   */
  protected void startUpImpl()
    throws ConfigurationException, StartupException {
    try {
      this.startUp();
    }
    catch (WcmException x) {
      throw new StartupException(x);
    }
  }

  /**
   * The implementation must overwrite this method if it has to release
   * resources acquired during shutdown. Accoring to the contract of the {@link
   * com.sapportals.wcm.crt.component.IStartable} interface of the CRT this
   * method will be called only once. It will be called when the component is
   * removed from the persistent configuration or when the system shuts down.
   */
  protected void shutDownImpl() {				
    this.shutDown();
  }

  /**
   * The implementation must overwrite this method if it has to release
   * resources acquired during shutdown.
   *
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of NW04. Use {@link #startUpImpl()} instead.
   */
  protected void startUp()
    throws WcmException {
    AbstractRepositoryManager.log.infoT("The service implementation does not overwrite the startUpImpl() method: " + this.id);
  }

  /**
   * @deprecated as of NW04. Use {@link #shutDownImpl()} instead.
   */
  protected void shutDown() { }

  // ---------------------------------------------------------------------------

  /**
   * Removes resources without the required permission(s) from the list
   *
   * @param list TBD: Description of the incoming method parameter
   * @param permissionNames TBD: Description of the incoming method parameter
   * @param context TBD: Description of the incoming method parameter
   * @exception ResourceException Exception raised in failure situation
   */
  protected final void removeResourcesWithoutPermission(IResourceList list, String[] permissionNames, IResourceContext context)
    throws ResourceException {
    com.sapportals.wcm.repository.CollectionImpl
      .removeResourcesWithoutPermission(this.securityManager, list, permissionNames, context);
  }

  /**
   * Return ISecurityChecker for this manager. Is valid on startup of repository
   * manager as well as startup of sub managers.
   *
   * @return security checker
   * @exception ResourceException Exception raised in failure situation
   */
  protected ISecurityChecker getSecurity()
    throws ResourceException {
    return this.securityChecker;
  }

  private void checkNotReconfigurableAttributeNames(String[] names, IConfiguration newConfig)
    throws ConfigurationException {
    for (int i = 0; i < names.length; i++) {
      String name = names[i];
      String newValue = newConfig.getAttribute(name, null);
      String oldValue = this.config.getAttribute(name, null);
      boolean changed = (newValue == null && oldValue != null || newValue != null && oldValue == null)
         || (newValue != null && oldValue != null && !newValue.equals(oldValue));
      if (changed) {
        throw new ConfigurationException("Property cannot be changed at runtime: " + name);
      }
    }
  }

  /**
   * Base implementation of getResources()
   *
   * @param ridList TBD: Description of the incoming method parameter
   * @param errors TBD: Description of the incoming method parameter
   * @param permissionNames TBD: Description of the incoming method parameter
   * @param context TBD: Description of the incoming method parameter
   * @return resourcesDefault
   * @exception ResourceException Exception raised in failure situation
   */
  private ResourceList getResourcesDefault(IRidList ridList, ResourceErrors errors, String[] permissionNames, IResourceContext context)
    throws ResourceException {
    ResourceList list = new ResourceList();
    if (ridList == null) {
      return list;
    }
    IResource res = null;
    for (int i = 0; i < ridList.size(); i++) {
      try {
        RID rid = ridList.get(i);
        res = null;
        res = getResource(rid, context);
        if (res != null) {
          list.add(res);
        }
      }
      catch (ResourceException ex) {
        if (errors != null) {
          errors.append(ex);
        }
      }
    }
    if (permissionNames == null) {
      return list;
    }
    else {
      this.removeResourcesWithoutPermission(list, permissionNames, context);
      return list;
    }
  }

	private Object createSubManager(String className) throws Exception {
		Object sm = null;
		try {
			java.lang.reflect.Constructor ctor =
				CrtClassLoaderRegistry.forName(className).getConstructor(new Class[] { IRepositoryManager.class });
			sm = ctor.newInstance(new Object[] { this });
		}
		catch (Exception ex) {
			throw new Exception(
				"Exception loading and instanciating sub-manager: "
					+ className
					+ " ("
					+ ex.getClass().getName()
					+ ": "
					+ ex.getMessage()
					+ ")");
		}
		if (!(sm instanceof AbstractRepositorySubManager)) {
			if (!(sm instanceof VirtualRootNamespaceContentPropertyManager)) {
				throw new Exception("Sub-manager class is not an instance of AbstractRepositorySubManager: " + className);
			}
		}
		return sm;
	}

  /**
   * Assign procedure: A sub-manager class can implement multible sub-manager
   * interfaces.
   *
   * @param o TBD: Description of the incoming method parameter
   */
  private void assignSubManager(Object o) {
    if (this.namespaceManager == null && o instanceof INamespaceManager) {
      this.namespaceManager = (INamespaceManager)o;
    }
    if (this.contentManager == null && o instanceof IContentManager) {
      this.contentManager = (IContentManager)o;
    }
    if (this.propertyManager == null && o instanceof IPropertyManager) {
      this.propertyManager = (IPropertyManager)o;
    }
    if (this.lockManager == null && o instanceof ILockManager) {
      this.lockManager = (ILockManager)o;
    }
    if (this.versioningManager == null && o instanceof IVersioningManager) {
      this.versioningManager = (IVersioningManager)o;
    }
    if (this.securityManager == null && o instanceof ISecurityManager) {
      this.securityManager = (ISecurityManager)o;
    }
    if (this.propSearchManager == null && o instanceof IPropertySearchManager) {
      this.propSearchManager = (IPropertySearchManager)o;
    }
  }
  
	/**
	 * Must be called by the RM implementation in its reconfigure() method if reconfiguration of security managers at runtime must be supported.
	 * @param newConfig The new configuration data (paramater of the reconfigure() method).
	 * @return true if the security manager was changed (removed, added or different sub manager), false otherwise.
	 * @exception ConfigurationException If the new sub manager can not be started.
	 *                                   If the sub manager implements other sub manager interfaces than ISecurityManager.
	 *                                   If the sub manager does not implement ISecurityManager.
	 *                                   If the sub manager does not extend AbstractRepositorySubManager.
	 */
	protected final boolean reconfigureSecurityManager(IConfiguration newConfig) throws ConfigurationException {
		String id = newConfig.getAttribute(AbstractRepositoryManager.CFG.SECURITY_MANAGER, null);
		String smClassName = (String) AbstractRepositoryManager.securityManagerMap.get(id);

		if (smClassName == null) {
			// Security manager removed
			try {
        if (this.securityManager != null) {
  				((AbstractRepositorySubManager) this.securityManager).shutDown();
        }
				// Note: SM implementation must ensure that clients which still hold a reference
				// to this instance will receive a apropriate exception.
			}
			catch (RuntimeException ex) {
				AbstractRepositoryManager.log.errorT(LoggingFormatter.extractCallstack(ex));
			}
			this.securityManager = null;
			// Update sec. checker (will assign a NullChecker instance if no security manager is present)
			try {
				this.securityChecker = SecurityChecker.getInstance(this);
			}
			catch (ResourceException e) {
				AbstractRepositoryManager.log.errorT(LoggingFormatter.extractCallstack(e));
			}
			AbstractRepositoryManager.log.infoT("security manager removed for repository: " + this.getID());
			return true;
		}
		else {
			String oldId = this.config.getAttribute(AbstractRepositoryManager.CFG.SECURITY_MANAGER, null);
			if (oldId != null && oldId.equals(id)) {
				AbstractRepositoryManager.log.debugT("security manager not changed. old and new id are equal: " + id);
				return false;
			}

			Object smInstance = null;
			try {
				smInstance = this.createSubManager(smClassName);
			}
			catch (Exception ex) {
				throw new ConfigurationException(ex.getMessage(), ex);
			}

			if (!(smInstance instanceof ISecurityManager)) {
				throw new ConfigurationException("class does not implement ISecurityManager: " + smClassName);
			}
			if (smInstance instanceof INamespaceManager
				|| smInstance instanceof IContentManager
				|| smInstance instanceof IPropertyManager
				|| smInstance instanceof ILockManager
				|| smInstance instanceof IVersioningManager
				|| smInstance instanceof IPropertySearchManager
				|| smInstance instanceof IExtendedNamespaceManager
				|| smInstance instanceof IExtendedVersioningManager) {
				throw new ConfigurationException(
					"class must not implement other sub manager interfaces other than ISecurityManager: " + smClassName);
			}

			try {
				((AbstractRepositorySubManager) smInstance).startUp();
			}
			catch (WcmException ex) {
				throw new ConfigurationException("Failed to start security manager: " + ex.getMessage(), ex);
			}
			catch (RuntimeException ex) {
				AbstractRepositoryManager.log.errorT(LoggingFormatter.extractCallstack(ex));
				throw new ConfigurationException("caught runtime exception in startUp() method of security manager: " + id);
			}

			// Note: Until now an existing SM instance remains up and running in case an exception is thrown in the code above
			boolean changed = (this.securityManager != null);
			if (changed) {
				// Security manager changed: Shut down current instance
				try {
					((AbstractRepositorySubManager) this.securityManager).shutDown();
				}
				catch (RuntimeException ex) {
					AbstractRepositoryManager.log.errorT(LoggingFormatter.extractCallstack(ex));
				}
			}

			this.securityManager = (ISecurityManager) smInstance;
			try {
				this.securityChecker = SecurityChecker.getInstance(this);
			}
			catch (ResourceException e) {
				AbstractRepositoryManager.log.errorT(LoggingFormatter.extractCallstack(e));
			}

			// @todo This is relevant for security audit logs ??
			AbstractRepositoryManager.log.infoT(
				"security manager "
					+ (changed ? "changed" : "set")
					+ " for repository: "
					+ this.getID()
					+ ", old ID = "
					+ (oldId == null ? "none" : oldId)
					+ ", new ID = "
					+ (id == null ? "none" : id));
			return true;
		}
	}


  /**
   * Helper method: Tokenize a string with a separator char into a List
   *
   * @param values TBD: Description of the incoming method parameter
   * @param separator TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  private List parseValueList(String values, String separator) {
    List list = new LinkedList();
    if (values == null) {
      return list;
    }
    StringTokenizer st = new StringTokenizer(values, separator);
    while (st.hasMoreTokens()) {
      list.add(st.nextToken().trim());
    }
    return list;
  }

  private void logStartUpTime() {
    long time = this.stateHandler.getConfigurationTime() + this.stateHandler.getStartTime();
    double sec = (double)time / 1000.0;
    AbstractRepositoryManager.log.debugT("logStartUpTime(1029)", "Startup of repository manager " + this.getID()
       + " took " + sec + " seconds");
  }
 
  public String toString() {
    return this.getID();
  }

  public final boolean equals(Object other) {
    if (other == null) {
      return false;
    }       
    if (!(other instanceof AbstractRepositoryManager)) {
      return false;
    }
    return ((AbstractRepositoryManager)other).getID().equals(this.getID());
  }
  
  public final int hashCode() {
    return this.getID().hashCode();   
  }  
}
