/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;
import com.sap.netweaver.bc.rf.mi.IManager;
import com.sap.tc.logging.Location;
import com.sapportals.config.fwk.Configuration;
import com.sapportals.config.fwk.IConfigClientContext;
import com.sapportals.config.fwk.IConfigManager;
import com.sapportals.config.fwk.IConfigPlugin;
import com.sapportals.config.fwk.IConfigProperty;
import com.sapportals.config.fwk.IConfigurable;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.crt.component.*;

import com.sapportals.wcm.crt.configuration.*;
import com.sapportals.wcm.crt.component.IComponentListener;
import com.sapportals.wcm.repository.enum.SupportedOption;
import com.sapportals.wcm.repository.manager.*;
import com.sapportals.wcm.repository.service.*;
import com.sapportals.wcm.util.config.ConfigCrutch;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.uri.RID;
import com.sapportals.wcm.util.uri.URI;

import java.util.*;

/**
 * Repository manager that creats a virtual root folder named "/". The children
 * are the root collection of all repositories.
 */
public class VirtualRootRepositoryManager extends AbstractRepositoryManager implements IComponentListener {

  private static Location log = Location.getLocation(com.sapportals.wcm.repository.VirtualRootRepositoryManager.class);
  private ISupportedOptionSet options = new SupportedOptionSet();

  private final static String CONFIG_EXCLUDE_FLAG = "hidden";
  private final static String CFG_PLUGIN_CM_REPOSITORY_MANAGERS = "/cm/repository_managers";
  
  private Set excludeList = new HashSet();

  public VirtualRootRepositoryManager() {
    this.options.add(SupportedOption.GET_CONTENT);
    this.options.add(SupportedOption.GET_PROPERTIES);
  }

  public void notify(ComponentEvent ev) {
  	// for all event types:
  	if (IRepositoryManager.class.isAssignableFrom(ev.getComponentClass()) ||
  	    IManager.class.isAssignableFrom(ev.getComponentClass())) {
  	  try {
    		this.reloadRMConfig();
  	  }
  	  catch (Exception ex) {
				VirtualRootRepositoryManager.log.errorT("notify(63)", "Failed to reload RM configuration:" + LoggingFormatter.extractCallstack(ex));
  	  }
  	}
  }

  private void reloadRMConfig() throws Exception {
  	synchronized(this.excludeList) {
  	  this.excludeList.clear();
      
      IConfigClientContext context = IConfigClientContext.createContext(
        ConfigCrutch.getConfigServiceUser());
      IConfigManager cfgManager = Configuration.getInstance().getConfigManager(context);
      IConfigPlugin plugin = cfgManager.getConfigPlugin(CFG_PLUGIN_CM_REPOSITORY_MANAGERS);
  	  IConfigurable[] configurables = plugin.getConfigurables();
      
	      for (int i = 0; i < configurables.length; i++) {
	  	  if (configurables[i].getConfigClass().isSubClassOf("RepositoryManager")) {
					if (this.isConfigurableActive(configurables[i])) {
					  String idValue = configurables[i].getIdValue();
					  if (!idValue.equals(this.getID())) {
						  IConfigProperty p = configurables[i].getProperty(VirtualRootRepositoryManager.CONFIG_EXCLUDE_FLAG);
					  	if (p == null) {
						    VirtualRootRepositoryManager.log.errorT("startUpImpl(80)", "Missing config property \"hidden\" in repository: " + idValue);
				  		}
				  		else {
			  		  	Boolean hiddenValue = (Boolean)p.getValueAsObject();
				    		if (hiddenValue == null) {
					    		VirtualRootRepositoryManager.log.errorT("startUpImpl(85)", "Missing value of config property \"hidden\" in repository: " + idValue);
			  	  		}
				    		else if (hiddenValue.booleanValue()) {
				    			this.excludeList.add(idValue);
				    		}
			  			}
		  	  	}
	  			}
		  	}
  		}
  	}  	
  }

  protected void startUpImpl()
    throws ConfigurationException, StartupException {
    try {
      this.reloadRMConfig();
    }
    catch (Exception ex) {
      throw new StartupException(ex.getMessage(), ex);
    }
  }

  /**
   * Helper method: Checks the "active" property of a configurable
   *
   * @param cfg TBD: Description of the incoming method parameter
   * @return configurableActive
   */
  private boolean isConfigurableActive(com.sapportals.config.fwk.IConfigurable cfg) {
    IConfigProperty activeProperty = cfg.getProperty("active");
    if (activeProperty == null) {
      return true;
    }
    try {
      return (((Boolean)activeProperty.getValueAsObject()).booleanValue());
    }
    catch (Exception ex) {
      return false;
    }
  }

  public ResourceImpl getResource(RID rid, IResourceContext context)
    throws ResourceException {
    if (rid.isRoot()) {
      return new CollectionImpl(rid, this, context);
    }
    else {
      return null;
    }
  }

  public ISupportedOptionSet getSupportedOptions(IResource resource) {
    return this.options;
  }

  public NameInfo getNameInfo() {
    return null;
  }

  public INamespaceManager getNamespaceManager(IResource resource)
    throws ResourceException {
    return new VirtualRootNamespaceContentPropertyManager(this);
  }

  public IPropertyManager getPropertyManager(IResource resource)
    throws ResourceException {
    return new VirtualRootNamespaceContentPropertyManager(this);
  }

  public ILockManager getLockManager(IResource resource)
    throws ResourceException {
    return null;
  }

  public IVersioningManager getVersioningManager(IResource resource)
    throws ResourceException {
    return null;
  }

  public IContentManager getContentManager(IResource resource)
    throws ResourceException {
    return new VirtualRootNamespaceContentPropertyManager(this);
  }

  public ISecurityManager getSecurityManager(IResource resource)
    throws ResourceException {
    return null;
  }

  public IPropertySearchManager getPropertySearchManager(IResource resource)
    throws ResourceException {
    return null;
  }


  /**
   * Returns a set of repository manager IDs. IDs can be added or removed in
   * Abstract(Repository)Manager class when the exclude-flag of repositories is
   * modified via configuration.
   *
   * @return a set of repository managers IDs (String).
   */
  public Set getExcludeList() {
    return this.excludeList;
  }


  /**
   * Helper method: Tokenize a string with a separator char into a List
   *
   * @param values TBD: Description of the incoming method parameter
   * @param separator TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  private final static List parseValueList(String values, String separator) {
    List list = new LinkedList();
    if (values == null) {
      return list;
    }
    StringTokenizer st = new StringTokenizer(values, separator);
    while (st.hasMoreTokens()) {
      list.add(st.nextToken().trim());
    }
    return list;
  }
}
