/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import com.sapportals.wcm.IFrameworkTransaction;
import com.sapportals.wcm.repository.manager.IExtendedVersioningManager;
import com.sapportals.wcm.util.uri.IRidSet;
import com.sapportals.wcm.util.uri.RID;

import java.util.Set;

/**
 * Version resource object implementation <p>
 *
 * Copyright (c) SAP AG 2002-2003
 *
 * @author julian.reschke@greenbytes.de
 * @version $Id: VersionResourceImpl.java,v 1.3 2003/02/17 14:24:04 jre Exp $
 */

public final class VersionResourceImpl implements IVersionResource {

  private final IResource resource;
  private final IExtendedVersioningManager vmgr;

  public VersionResourceImpl(IResource resource, IExtendedVersioningManager vmgr)
    throws ResourceException {
    this.resource = resource;
    this.vmgr = vmgr;
  }

  public IResourceList getCheckedOutResources()
    throws ResourceException {
    return this.vmgr.getCheckedOutResources(this.resource);
  }

  public ICheckOutInfo2 checkOut(boolean forkOk)
    throws ResourceException {
    IFrameworkTransaction ft = FrameworkTransaction.required();
    try {
      ICheckOutInfo2 info = this.vmgr.checkOut(false, forkOk, this.resource);
      ft.commit();
      ft = null;
      return info;
    }
    finally {
      if (ft != null) ft.rollback();
    }
  }

  public IRidSet getCheckedOutRIDs()
    throws ResourceException {
    return this.vmgr.getCheckedOutRIDs(this.resource);
  }

  public Set getCheckInForkBehaviour()
    throws ResourceException {
    return this.vmgr.getCheckInForkBehaviour(this.resource);
  }

  public Set getCheckOutForkBehaviour()
    throws ResourceException {
    return this.vmgr.getCheckOutForkBehaviour(this.resource);
  }

  public IResourceList getPredecessorResources()
    throws ResourceException {
    return this.vmgr.getPredecessorResources(this.resource);
  }

  public IRidSet getPredecessorRIDs()
    throws ResourceException {
    return this.vmgr.getPredecessorRIDs(this.resource);
  }

  public IResource getVersionHistoryResource()
    throws ResourceException {
    return this.vmgr.getVersionHistoryResource(this.resource);
  }

  public RID getVersionHistoryRID()
    throws ResourceException {
    return this.vmgr.getVersionHistoryRID(this.resource);
  }

  public IResourceList getSuccessorResources()
    throws ResourceException {
    return this.vmgr.getSuccessorResources(this.resource);
  }

  public IRidSet getSuccessorRIDs()
    throws ResourceException {
    return this.vmgr.getSuccessorRIDs(this.resource);
  }


  public void addLabel(String labelName)
    throws ResourceException {
    IFrameworkTransaction ft = FrameworkTransaction.required();
    try {
      this.vmgr.addLabel(labelName, this.resource);
      ft.commit();
      ft = null;
    }
    finally {
      if (ft != null) ft.rollback();
    }
  }

  public void setLabel(String labelName)
    throws ResourceException {
    IFrameworkTransaction ft = FrameworkTransaction.required();
    try {
      this.vmgr.setLabel(labelName, this.resource);
      ft.commit();
      ft = null;
    }
    finally {
      if (ft != null) ft.rollback();
    }
  }

  public void removeLabel(String labelName)
    throws ResourceException {
    IFrameworkTransaction ft = FrameworkTransaction.required();
    try {
      this.vmgr.removeLabel(labelName, this.resource);
      ft.commit();
      ft = null;
    }
    finally {
      if (ft != null) ft.rollback();
    }
  }

  public Set getLabelSet()
    throws ResourceException {
    return this.vmgr.getLabelSet(this.resource);
  }
}
