/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import com.sapportals.wcm.IFrameworkTransaction;
import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.repository.enum.*;
import com.sapportals.wcm.repository.manager.*;

import com.sapportals.wcm.util.content.IContent;
import com.sapportals.wcm.util.name.*;
import com.sapportals.wcm.util.uri.*;

import java.util.*;

/**
 * Version controlled resource object impl. <p>
 *
 * Copyright 2002 (c) SAP AG
 *
 * @author julian.reschke@greenbytes.de
 * @version $Id: VersionControlledResourceImpl.java,v 1.4 2002/09/30 10:56:45
 *      mba Exp $
 */

public final class VersionControlledResourceImpl implements IVersionControlledResource {

  private final IResource resource;
  private final IExtendedVersioningManager vmgr;

  public VersionControlledResourceImpl(IResource resource, IExtendedVersioningManager vmgr)
    throws ResourceException {
    this.resource = resource;
    this.vmgr = vmgr;
  }

  public ICheckInInfo2 checkIn(IContent newContent, IPropertyMap properties,
    boolean ignorePropertyFailures, RID expectedCheckInRID,
    boolean forkOk, boolean keepCheckedOut)
    throws ResourceException {
    
    IFrameworkTransaction ft = FrameworkTransaction.required();
    try {
      ICheckInInfo2 info = this.vmgr.checkIn(newContent, properties, ignorePropertyFailures,
        expectedCheckInRID, forkOk, keepCheckedOut, this.resource);
      ft.commit();
      ft = null;
      return info;
    }
    finally {
      if (ft != null) ft.rollback();
    }
  }

  public ICheckOutInfo2 checkOut(boolean applyToVersion, boolean forkOk)
    throws ResourceException {
    
    ICheckOutInfo2 result = null;
    IRepositoryManager mgr = this.resource.getRepositoryManager();
    IFrameworkTransaction ft = FrameworkTransaction.required();
    try {
      result = this.vmgr.checkOut(applyToVersion, forkOk, this.resource);
      boolean doCommit = true;
      if (mgr instanceof AbstractRepositoryManager) {
        AbstractRepositoryManager amgr = (AbstractRepositoryManager)mgr;
        if (amgr.eventsEnabled()) {
          doCommit = doCommit && amgr.sendEvent(ft, this.resource, ResourceEvent.CHECKOUT, null, null);
        }
      }
      if (doCommit) {
        ft.commit();
        ft = null;
      }
    }
    finally {
      if (ft != null) ft.rollback();
    }
    if (mgr instanceof AbstractRepositoryManager) {
      AbstractRepositoryManager amgr = (AbstractRepositoryManager)mgr;
      if (amgr.eventsEnabled()) {
        amgr.sendEvent(this.resource, ResourceEvent.CHECKOUT, null, null);
      }
    }
    return result;
  }

  public IName getAutoVersioningMode()
    throws ResourceException {
    return this.vmgr.getAutoVersioningMode(this.resource);
  }

  public IResource getCheckedInResource()
    throws ResourceException {
    return this.vmgr.getCheckedInResource(this.resource);
  }

  public RID getCheckedInRID()
    throws ResourceException {
    return this.vmgr.getCheckedInRID(this.resource);
  }

  public IResource getCheckedOutResource()
    throws ResourceException {
    return this.vmgr.getCheckedOutResource(this.resource);
  }

  public RID getCheckedOutRID()
    throws ResourceException {
    return this.vmgr.getCheckedOutRID(this.resource);
  }

  public Set getCheckInForkBehaviour()
    throws ResourceException {
    return this.vmgr.getCheckInForkBehaviour(this.resource);
  }

  public Set getCheckOutForkBehaviour()
    throws ResourceException {
    return this.vmgr.getCheckOutForkBehaviour(this.resource);
  }

  public Set getEclipsedSet()
    throws ResourceException {
    return this.vmgr.getEclipsedSet(this.resource);
  }

  public IResourceList getPredecessorResources()
    throws ResourceException {
    return this.vmgr.getPredecessorResources(this.resource);
  }

  public IRidSet getPredecessorRIDs()
    throws ResourceException {
    return this.vmgr.getPredecessorRIDs(this.resource);
  }

  public IResource getVersionHistoryResource()
    throws ResourceException {
    return this.vmgr.getVersionHistoryResource(this.resource);
  }

  public RID getVersionHistoryRID()
    throws ResourceException {
    return this.vmgr.getVersionHistoryRID(this.resource);
  }

  public IRidSet updateFromVersion(IResource version)
    throws ResourceException {
    IFrameworkTransaction ft = FrameworkTransaction.required();
    try {
      IRidSet set = this.vmgr.updateFromVersion(version, this.resource);
      ft.commit();
      ft = null;
      return set;
    }
    finally {
      if (ft != null) ft.rollback();
    }
  }

  public IRidSet updateFromVersion(RID version)
    throws ResourceException {
    IFrameworkTransaction ft = FrameworkTransaction.required();
    try {
      IRidSet set = this.vmgr.updateFromVersion(version, this.resource);
      ft.commit();
      ft = null;
      return set;
    }
    finally {
      if (ft != null) ft.rollback();
    }
  }
}
