/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;
import java.util.*;

import com.sap.netweaver.bc.rf.common.IResourceHandle;
import com.sap.netweaver.bc.rf.common.namespace.ILinkDescriptor;
import com.sap.netweaver.bc.rf.mi.AbstractManager;
import com.sap.tc.logging.Location;
import com.sapportals.portal.security.usermanagement.*;
import com.sapportals.wcm.IFrameworkTransaction;
import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.WcmObject;
import com.sapportals.wcm.repository.enum.*;
import com.sapportals.wcm.repository.manager.*;
import com.sapportals.wcm.repository.runtime.CmStartupException;
import com.sapportals.wcm.repository.runtime.CmSystem;
import com.sapportals.wcm.repository.so.ObjectFactoryRegistry;
import com.sapportals.wcm.repository.wrapping.RidTransformer;
import com.sapportals.wcm.repository.wrapping.exception.ExceptionTransformer;
import com.sapportals.wcm.util.content.IContent;
import com.sapportals.wcm.util.events.IEventList;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.systemconfig.ISystemPrincipal;
import com.sapportals.wcm.util.systemconfig.ISystemPrincipalList;
import com.sapportals.wcm.util.systemconfig.ISystemPrincipalListIterator;
import com.sapportals.wcm.util.systemconfig.SystemConfig;
import com.sapportals.wcm.util.uri.*;
import com.sapportals.wcm.util.usermanagement.WPUMFactory;

/**
 * This is the framework implementation of the IResource interface. It contains
 * the semantics for dealing with resources of all kinds of repository managers.
 * It delegates all low-level repository operations to a repository manager
 * implementation. <p>
 *
 * This class, and the CollectionImpl sub-class, perform the following framework
 * tasks: <br>
 * Apply content, property and namespace filters <br>
 * Handle copy and move operations between different repositories <br>
 * Sending of events for all basic operations (optional per repository manager)
 * <p>
 *
 * Copyright (c) SAP AG 2001-2004
 * @author m.breitenfelder@sap.com
 * @author frank.renkes@sap.com
 * @version $Revision: 1.22 $
 */
public class ResourceImpl extends WcmObject implements IResource, ISortItem {

  protected static boolean KM_PERF_MON = (System.getProperty("km_perf_monitor") != null) ? true : false;

  private static boolean KMC_SYS_PROP_FILTER =
    ("true".equals(System.getProperty("kmc_sys_prop_filter", "false"))) ? true : false;

  // package-private access for ResourcePropertyCacheHandler (and other XXXImpl classes...)
  HashMap cachedPrefilledProperties = new HashMap() ;

  private IResource targetResourceCache = null;
  private long targetResourceSet = 0;
  static long TIME_IN_MS_TO_CACHE_TARGETRESOURCE = 60000;

  protected final static IContent EMPTY_CONTENT =
    new Content(new java.io.ByteArrayInputStream(new byte[0]), "application/octet-stream", 0L);

  /**
   * The repository manager that manages this resource
   */
  protected final RMAdapter rmAdapter;

  /**
   * Contained ContentImpl object
   */
  protected IContent content;

  /**
   * Contained ContentImpl object for unfiltered content
   */
  protected IContent unfilteredContent;

  /**
   * Object is deleted, accept no further call to this object
   */
  protected boolean deleted;

  /**
   * is this a root collection
   */
  protected boolean isRootCollection;

  /**
   * The revision ID
   */
  protected String revID = "";

  // ---- Events ----

  protected boolean initEvents;
  protected boolean mustSendEvents;

  /**
   * >= SP6: The handle of new manager API
   */
  private final IResourceHandle handle;

  /**
   * Reference to the context
   */
  private IResourceContext context;

  /**
   * The RID of the resource
   */
  private RID rid;

  /**
   * Name of the resource
   */
  private String name;

  /**
   * If the resource represents a link
   */
  private LinkType linkType = LinkType.NONE;

  /**
   * Link target
   */
  private URL targetURL;

  /**
   * The access URI
   */
  private RID accessRID;

  /**
   * is this a revision resource?
   */
  private boolean isRevision;

  private IPropertyMap sortProps;

  // ---- Timestamp ----

  private final Timestamp ts;

  /**
   * default for copy() and move() parameter
   */
  protected final static CopyParameter DEFAULT_COPY_PARAMETER = new CopyParameter();

  private final static Location log = Location.getLocation(com.sapportals.wcm.repository.ResourceImpl.class);

  /**
   * The set of resources we are currently (in this thread) performing lookups
   * on. Used for loop detection.
   */
  private final static ThreadLocal VisitedResources = new ThreadLocal() {

    protected Object initialValue() {
      return new HashSet(7);
    }
  };

  private static IProperty P_enhancedCollectionTrue = null;
  private static IProperty P_enhancedCollectionFalse = null;

  // ----------------- deprecated constructors !! ---------------------

  /**
   * Constructs a new resource.
   *
   * @deprecated as of NW04. Use the constructor that takes just a revisionID parameter, no
   *      revisionRID
   */
  public ResourceImpl(
    RID rid,
    String targetURL,
    LinkType linkType,
    String revisionID,
    RID revisionRID,
    IRepositoryManager repositoryManager,
    IResourceContext context)
    throws ResourceException {

    this.ts = new Timestamp();
    this.rid = revisionRID != null ? revisionRID : rid;
    this.targetURL = targetURL == null ? null : new URL(targetURL);
    this.linkType = linkType;
    this.isRevision = (revisionRID != null);
    this.revID = revisionID;
    this.rmAdapter = new RMAdapter(repositoryManager);
    this.context = context;
    this.handle = null;
    this.isRootCollection = this.rmAdapter.getOldAbstract().isRoot(this.rid.toString());

    // Modify URI: No trailing slashes for directories except root collection
    if (!this.rid.equals("/")) {
      this.rid = this.rid.removeTrailingSlash();
    }

    // Extract the name part of the URI
    this.name = this.rid.name().toString();
    //    if (ResourceImpl.log.isDebugEnabled()) {
    //      if (this.name.equals("/")) {
    //        ResourceImpl.log.debug("Name = \"/\":" + this.rid);
    //      }
    //    }
  }

  /**
   * Constructs a new resource
   *
   * @deprecated as of NW04. Use the constructor that just takes a revisionID parameter, no
   *      revisionRID
   */
  public ResourceImpl(
    RID rid,
    String targetURL,
    LinkType linkType,
    RID revisionRID,
    IRepositoryManager repositoryManager,
    IResourceContext context)
    throws ResourceException {
    this(rid, targetURL, linkType, null, revisionRID, repositoryManager, context);
  }

  // ---------------------- new constructors !! --------------------------

  /**
   * Constructs a new resource.
   *
   * @param rid The resource identifier of the resource, must not be <code>null
   *      </code>.
   * @param repositoryManager The repository manager of the resource, must not
   *      be <code>null</code> .
   * @param context The resource context, must not be <code>null</code> .
   * @exception ResourceException Description of Exception
   */
  public ResourceImpl(RID rid, IRepositoryManager repositoryManager, IResourceContext context)
    throws ResourceException {
    this(rid, null, null, LinkType.NONE, repositoryManager, context);
  }

  /**
   * Constructs a new resource which will be a link and/or a revision.
   *
   * @param rid The resource identifier of the resource, must not be <code>null
   *      </code>.
   * @param revisionID Specified the revision identified (version number) if the
   *      resource is a revision, <code>null</code> otherwise
   * @param targetURL Specifies the target if the resource is a link, <code>null
   *      </code> otherwise.
   * @param linkType Specifies the link type ({@link
   *      com.sapportals.wcm.repository.enum.LinkType}). If the link type is
   *      "external" then the target must not be a collection.
   * @param repositoryManager The repository manager of the resource, must not
   *      be <code>null</code> .
   * @param context The resource context, must not be <code>null</code> .
   * @exception ResourceException If required parameters are not specified.
   */
  public ResourceImpl(
    RID rid,
    String revisionID,
    String targetURL,
    LinkType linkType,
    IRepositoryManager repositoryManager,
    IResourceContext context)
    throws ResourceException {
    if (rid == null) {
      throw new ResourceException("parameter rid is null");
    }
    if (repositoryManager == null) {
      throw new ResourceException("parameter repositoryManager is null");
    }
    if (context == null) {
      throw new ResourceException("parameter context is null");
    }

    this.ts = new Timestamp();
    this.rid = rid;
    this.targetURL = targetURL == null ? null : new URL(targetURL);
    this.linkType = linkType;
    this.isRevision = (revisionID != null);
    this.revID = revisionID;
    this.rmAdapter = new RMAdapter(repositoryManager);
    this.context = context;
    this.handle = null;
    this.isRootCollection = this.rmAdapter.getOldAbstract().isRoot(this.rid.toString());

    // Modify URI: No trailing slashes for directories except root collection
    if (!this.rid.equals("/")) {
      this.rid = this.rid.removeTrailingSlash();
    }

    // Extract the name part of the URI
    this.name = this.rid.name().toString();
  }

  /**
   * Framework-internal constructor
   *
   * @param handle Description of Parameter
   * @param manager Description of Parameter
   * @param context Description of Parameter
   * @exception ResourceException Exception raised in failure situation
   */
  ResourceImpl(IResourceHandle handle, AbstractManager manager, IResourceContext context) throws ResourceException {
    if (handle == null) {
      throw new java.lang.NullPointerException("parameter handle is null");
    }
    if (manager == null) {
      throw new java.lang.NullPointerException("parameter manager is null");
    }
    if (context == null) {
      throw new java.lang.NullPointerException("parameter context is null");
    }

    this.ts = new Timestamp();
    this.handle = handle;
    this.rid = RidTransformer.transform(handle.getRid());
    this.rmAdapter = new RMAdapter(manager);

    ILinkDescriptor ld = null;
    try {
      ld = manager.getNamespaceManager().getLinkDescriptor(handle);
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException ex) {
      throw ExceptionTransformer.transform(ex);
    }
    if (ld != null) {
      if (ld.getLinkType().equals(com.sap.netweaver.bc.rf.common.namespace.LinkType.INTERNAL_STATIC)) {
        this.linkType = LinkType.INTERNAL;
        this.targetURL = new URL(ld.getTargetRid().toString());
      }
      else if (ld.getLinkType().equals(com.sap.netweaver.bc.rf.common.namespace.LinkType.EXTERNAL_STATIC)) {
        this.linkType = LinkType.EXTERNAL;
        this.targetURL = new URL(ld.getTargetUri().toString());
      }
      else if (ld.getLinkType().equals(com.sap.netweaver.bc.rf.common.namespace.LinkType.REPOSITORY_DYNAMIC)) {
        this.linkType = LinkType.INTERNAL;
        this.targetURL = new URL(ld.getTargetRid().toString());
      }
    }

    // TODO: Init. revision flag !
    //manager.getBasicVersioningManager().isRevision() ???
    //this.isRevision =

    this.context = context;

    // Is root collection ?
    String path = this.rid.getPath();
    this.isRootCollection = (path.equals(manager.getRidPrefix()) || path.equals(manager.getRidPrefix() + '/'));

    // No trailing slashes for directories except root collection
    if (!this.rid.equals("/")) {
      this.rid = this.rid.removeTrailingSlash();
    }

    this.name = this.rid.name().toString();
  }

  public final void setProperties(IPropertyMap props) throws ResourceException {

    this.checkDeleted();
    if (this.isRevision()) {
      throw new RevisionResourceException("resource is revision", this.rid);
    }

    ResourcePropertyCacheHandler.clearCachedPrefilledProperties(this);

    IProperty enhancedColl = props.get(PropertyName.createEnhancedCollection());
    ResourceException addexc = null;
    if (enhancedColl != null) {
      IMutablePropertyMap mp = props.getMutable();
      mp.remove(PropertyName.createEnhancedCollection());
      props = mp;
      try {
        if (!enhancedColl.getType().equals(PropertyType.BOOLEAN)) {
          addexc = new PropertyTypeMismatchException(this.rid, PropertyName.createEnhancedCollection());
        }
        else {
          if (enhancedColl.getBooleanValue()) {
            internalEnableChildAutoVersioning();
          }
          else {
            internalDisableChildAutoVersioning();
          }
        }
      }
      catch (ResourceException ex) {
        addexc = ex;
      }
    }

    IPropertyMap filteredProps = this.applyPropertyWriteFilterForUpdate(props, PropertyFilterMode.PROPERTY_LIST);

    if (filteredProps == null) return;
    
    if (this.mustSendEvents()) {
      IPropertyMap before = this.internalGetProperties();
      
      IFrameworkTransaction ft = FrameworkTransaction.required();      
      try{        
        try {
          this.internalSetProperties(filteredProps);        
        }
        catch (SetPropertiesException ex) {
          if (this.sendEventsForPropertyChanges(ft, null, ex, props)) {
            ft.commit();
            ft = null;
          }
          this.sendEventsForPropertyChanges(null, ex, props);
          throw ex;
        }
        
        if (this.sendEventsForPropertyChanges(ft, null, before, filteredProps)) {
          ft.commit();        
          ft = null;
        }
      }
      finally{
        if (ft != null) ft.rollback();
      }
                  
      this.sendEventsForPropertyChanges(null, before, filteredProps);
    }
    else {      
      try {
        this.internalSetProperties(filteredProps);
      }
      catch (SetPropertiesException ex) {
        if (addexc != null) {
          ex.propChangeResults.add(new PropertyChangeResult(PropertyName.createEnhancedCollection(), null, addexc));
        }
      }      
    }        
  }

  public void setProperties(List propChangeList)
    throws ResourceException, NotSupportedException, AccessDeniedException, SetPropertiesException {
    checkDeleted();
    if (isRevision()) {
      throw new RevisionResourceException("resource is revision", this.rid);
    }

    ResourcePropertyCacheHandler.clearCachedPrefilledProperties(this);

    List filteredList = new ArrayList(propChangeList);
    IMutablePropertyMap map = new MutablePropertyMap();
    for (int i = 0, s = filteredList.size(); i < s; i++) {
      Object o = filteredList.get(i);
      if (o instanceof IProperty) {
        map.put((IProperty)o);
      }
    }
    IPropertyMap filteredMap = this.applyPropertyWriteFilterForUpdate(map, PropertyFilterMode.PROPERTY_LIST);
    IPropertyIterator newIt = filteredMap.iterator();
    while (newIt.hasNext()) {
      IProperty newProp = newIt.next();
      if (!filteredList.contains(newProp)) {
        // The filter has added the property
        filteredList.add(newProp);
      }
    }

    ResourceException addexc = null;
    
    IFrameworkTransaction ft = FrameworkTransaction.required();      
    try {        
      for (Iterator itr = filteredList.listIterator(); itr.hasNext();) {
        Object o = itr.next();
        if (o instanceof IProperty && ((IProperty)o).getPropertyName().equals(PropertyName.createEnhancedCollection())) {
          IProperty prop = (IProperty)o;
          if (!prop.getType().equals(PropertyType.BOOLEAN)) {
            addexc = new PropertyTypeMismatchException(this.rid, PropertyName.createEnhancedCollection());
          }
          else {
            try {
              if (prop.getBooleanValue()) {
                this.internalEnableChildAutoVersioning();
              }
              else {
                this.internalDisableChildAutoVersioning();
              }
            }
            catch (ResourceException ex) {
              addexc = ex;
            }
          }
        }
        else if (o instanceof IPropertyName && ((IPropertyName)o).equals(PropertyName.createEnhancedCollection())) {
          try {
            this.internalDisableChildAutoVersioning();
          }
          catch (ResourceException ex) {
            addexc = ex;
          }
        }
      }
  
      if (filteredList.size() > 0) {
        if (this.mustSendEvents()) {
          try {
            this.internalSetProperties(filteredList);
          }
          catch (SetPropertiesException ex) {
            if (addexc != null) {
              ex.getPropertyChangeResults().add(
                new PropertyChangeResult(PropertyName.createEnhancedCollection(), null, addexc));
            }
            
            if (this.sendEventsForPropertyChanges(ft, null, ex, propChangeList)) {
              ft.commit();
              ft = null;
            }
            this.sendEventsForPropertyChanges(null, ex, propChangeList);
            throw ex;
          }
          
          if (this.sendEventsForPropertyChanges(ft, (String) null, filteredList)) {
            ft.commit();        
            ft = null;
          }
         
          this.sendEventsForPropertyChanges((String) null, filteredList);
        }
        else {
          try {
            this.internalSetProperties(filteredList);
          }
          catch (SetPropertiesException ex) {
            if (addexc != null) {
              ex.getPropertyChangeResults().add(
                new PropertyChangeResult(PropertyName.createEnhancedCollection(), null, addexc));
            }
            ft.commit();        
            ft = null;
            throw ex;
          }
          ft.commit();        
          ft = null;
        }
      }
    }
    finally{
      if (ft != null) ft.rollback();
    }

    if (addexc != null) {
      PropertyChangeResultCollection pcr = new PropertyChangeResultCollection();
      pcr.add(new PropertyChangeResult(PropertyName.createEnhancedCollection(), null, addexc));
      throw new SetPropertiesException(pcr);
    }
  }

  public final void setProperty(IProperty prop)
    throws ResourceException, NotSupportedException, AccessDeniedException {

    this.checkDeleted();
    if (this.isRevision()) {
      throw new RevisionResourceException("resource is revision", this.rid);
    }

    ResourcePropertyCacheHandler.clearCachedPrefilledProperties(this);

    IMutablePropertyMap props = new MutablePropertyMap();
    props.put(prop);
    IPropertyMap filteredProps = this.applyPropertyWriteFilterForUpdate(props, PropertyFilterMode.SINGLE_PROPERTY);
    /*
     * old: (problem is: what if a write filter creates two properties for one? behavior will then differ from setProperties(List)
     * IProperty filteredProp = this.applyPropertyWriteFilterForUpdate(prop, PropertyFilterMode.SINGLE_PROPERTY);
     * if (filteredProp == null) {
     * return;
     * }
     */
    if (filteredProps.size() == 0) {
      // filter removed property
      // todo: remove the property? was not done before, so what?
    }
    else if (filteredProps.size() == 1) {
      // old behavior: proceed as usual
      prop = filteredProps.iterator().next();
      if (prop.getPropertyName().equals(PropertyName.createEnhancedCollection())) {
        if (!prop.getType().equals(PropertyType.BOOLEAN)) {
          throw new PropertyTypeMismatchException(this.rid, PropertyName.createEnhancedCollection());
        }
        else {
          if (prop.getBooleanValue()) {
            this.internalEnableChildAutoVersioning();
          }
          else {
            this.internalDisableChildAutoVersioning();
          }
        }
      }
      else {
        IFrameworkTransaction ft = FrameworkTransaction.required();      
        try{        
          this.internalSetProperty(prop);
          ResourcePropertyCacheHandler.setPrefilledProperty(this, prop);

          if (this.sendEvent(ft, ResourceEvent.PROPERTY_SET, null, prop)) {
            ft.commit();
          }
        } finally{
          if (ft != null) ft.rollback();
        }
      }
    }
    else {
      // new behaviour: like for setProperties(Map)
      IProperty enhancedColl = props.get(PropertyName.createEnhancedCollection());      
      if (enhancedColl != null) {
        IMutablePropertyMap mp = props.getMutable();
        mp.remove(PropertyName.createEnhancedCollection());
        props = mp;
        if (!enhancedColl.getType().equals(PropertyType.BOOLEAN)) {
          throw new PropertyTypeMismatchException(this.rid, PropertyName.createEnhancedCollection());
        }
        else {
          if (enhancedColl.getBooleanValue()) {
            internalEnableChildAutoVersioning();
          }
          else {
            internalDisableChildAutoVersioning();
          }
        }
      }
      if (this.mustSendEvents()) {
        IPropertyMap before = this.internalGetProperties();
        
        IFrameworkTransaction ft = FrameworkTransaction.required();      
        try{        
          try {                    
            this.internalSetProperties(filteredProps);          
          }
          catch (SetPropertiesException ex) {
            if (this.sendEventsForPropertyChanges(ft, null, ex, props)) {
              ft.commit();        
              ft = null;
            }
            this.sendEventsForPropertyChanges(null, ex, props);
            throw ex;
          }
          
          if (this.sendEventsForPropertyChanges(ft, null, before, filteredProps)) {
            ft.commit();        
            ft = null;
          }
        }
        finally{
          if (ft != null) ft.rollback();
        }
                               
        this.sendEventsForPropertyChanges(null, before, filteredProps);
      }
      else {
        this.internalSetProperties(filteredProps);
      }
    }

  }

  public final void setTargetURL(URL url) throws ResourceException, AccessDeniedException {
    targetResourceCache = null;
    this.checkDeleted();
    if (this.isRevision()) {
      throw new RevisionResourceException("resource is revision", this.rid);
    }
    if (!isLink()) {
      throw new ResourceException("resource is not a link");
    }
    if (this.linkType.equals(LinkType.INTERNAL)) {
      this.checkLinkTarget(RID.getRID(url.toString()), this.getRID());
    }

    IFrameworkTransaction ft = FrameworkTransaction.required();      
    try{        
      this.internalSetTargetURL(url);
      if (this.sendEvent(ft, ResourceEvent.UPDATE_LINK, null)) {
        ft.commit();        
        ft = null;
      }
      this.targetURL = url;
    }
    finally{
      if (ft != null) ft.rollback();
    }       
    
    this.sendEvent(ResourceEvent.UPDATE_LINK, null);
  }

  public final void setLinkType(LinkType linkType) throws ResourceException, AccessDeniedException {
    this.checkDeleted();
    if (this.isRevision()) {
      throw new RevisionResourceException("resource is revision", this.rid);
    }
    if (!this.isLink()) {
      throw new ResourceException("resource is not a link");
    }
    if (linkType.equals(LinkType.NONE)) {
      throw new InvalidArgumentException("cannot change link type to NONE");
    }
    
    IFrameworkTransaction ft = FrameworkTransaction.required();      
    try{        
      this.internalSetLinkType(linkType);
      if (this.sendEvent(ft, ResourceEvent.UPDATE_LINK, null)) {
        ft.commit();        
        ft = null;
      }
      this.linkType = linkType;
      ResourcePropertyCacheHandler.removeCachedPrefilledProperties(this, PropertyName.createLinkType());
    }
    finally{
      if (ft != null) ft.rollback();
    }        
    
    this.sendEvent(ResourceEvent.UPDATE_LINK, null);
  }

  public final void setAsCurrentVersion() throws ResourceException, NotSupportedException, AccessDeniedException {
    if (!isRevision()) {
      throw new AccessDeniedException("resource is not a version", this.rid, null, null);
    }
    this.internalSetAsCurrentVersion();
  }

  /**
   * @param accessURI The new AccessURI value
   * @exception ResourceException Description of Exception
   * @deprecated as of NW04.
   */
  public void setAccessURI(URI accessURI) throws ResourceException {
    this.accessRID = RID.getRID(accessURI.toString());
  }

  public void setAccessRID(RID accessRID) throws ResourceException {
    this.accessRID = accessRID;
  }

  public final IResourceHandle getHandle() {
    return this.handle;
  }

  //////////////////////////////////////////////////////////////////////////////
  // IResource interface
  //////////////////////////////////////////////////////////////////////////////

  public final IEventList getSupportedEvents() throws ResourceException {
    return getRepositoryManager().getEventBroker().getEvents(this);
  }

  public ISupportedOptionSet getSupportedOptions() {
    return this.rmAdapter.getSupportedOptions(this);
  }

  public final IResourceContext getContext() {
    return this.context;
  }

  public IRepositoryManager getRepositoryManager() {
    return this.rmAdapter.getIRepositoryManagerForClients();
  }

  /**
   * deprecated
   *
   * @return The URI value
   * @exception ResourceException Description of Exception
   */
  public final URI getURI() throws ResourceException {
    return new URI(this.rid.toString());
  }

  public final URI getAccessURI() throws ResourceException {
    if (this.accessRID != null) {
      return new URI(this.accessRID.toString());
    }
    else {
      return getURI();
    }
  }

  public RID getRID() throws ResourceException {
    return this.rid;
  }

  public RID getAccessRID() throws ResourceException {
    if (this.accessRID != null) {
      return accessRID;
    }
    else {
      return this.rid;
    }
  }

  public final String getName() throws ResourceException {
    return this.name;
  }

  public final Date getCreationDate() {
    if (this.deleted) {
      return null;
    }
    try {
      IProperty prop = this.getSystemProperty(PropertyName.createCreationDate());
      return (prop != null) ? prop.getDateValue() : null;
    }
    catch (ResourceException ex) {
      return null;
    }
  }

  public final Date getLastModified() {
    if (this.deleted) {
      return null;
    }
    try {
      IProperty prop = this.getSystemProperty(PropertyName.createLastModified());
      return (prop != null) ? prop.getDateValue() : null;
    }
    catch (ResourceException ex) {
      return null;
    }
  }

  public final String getDescription() {
    if (this.deleted) {
      return "";
    }
    try {
      IProperty prop = this.getSystemProperty(PropertyName.createDescription());
      return (prop != null) ? prop.getStringValue() : "";
    }
    catch (ResourceException ex) {
      return "";
    }
  }

  public final String getResourceType() {
    try {
      IProperty prop = this.getProperty(PropertyName.createResourceType());
      return (null != prop) ? prop.getStringValue() : "";
    }
    catch (ResourceException ex) {
      return "";
    }
  }

  public final String getDisplayName() {
    if (this.deleted) {
      return "";
    }
    IProperty prop = this.getDisplayNameProperty();
    return (null != prop) ? prop.getStringValue() : "";
  }

  public final String getDisplayName(boolean orNameIfNull) {
    if (this.deleted) {
      return "";
    }

    String displayName = this.getDisplayName();

    if (true == orNameIfNull) {
      if (null == displayName || 0 == displayName.length()) {
        try {
          return this.getName();
        }
        catch (ResourceException ex) {
          return "";
        }
      }
    }
    return displayName;
  }

  public String getCreatedBy() {
    if (this.deleted) {
      return "";
    }
    try {
      IProperty prop = this.getSystemProperty(PropertyName.createCreatedBy());
      return (prop != null) ? prop.getStringValue() : "";
    }
    catch (ResourceException ex) {
      return "";
    }
  }

  public String getLastModifiedBy() {
    if (this.deleted) {
      return "";
    }
    try {
      IProperty prop = this.getSystemProperty(PropertyName.createLastModifiedBy());
      return (prop != null) ? prop.getStringValue() : "";
    }
    catch (ResourceException ex) {
      return "";
    }
  }

  public final String getETag() {
    if (this.deleted) {
      return "";
    }
    try {
      IProperty prop = this.getSystemProperty(PropertyName.createETag());
      return (prop != null) ? prop.getStringValue() : "";
    }
    catch (ResourceException ex) {
      return null;
    }
  }

  public String getLanguage() {
    return null;
  }

  public final boolean isReadOnly() {
    if (this.deleted) {
      return false;
    }
    try {
      IProperty prop = this.getSystemProperty(PropertyName.createReadOnly());
      return (prop != null) ? prop.getBooleanValue() : false;
    }
    catch (ResourceException ex) {
      return false;
    }
  }

  public final boolean isHidden() {
    if (this.deleted) {
      return false;
    }
    try {
      IProperty prop = this.getSystemProperty(PropertyName.createHidden());
      return (prop != null) ? prop.getBooleanValue() : false;
    }
    catch (ResourceException ex) {
      return false;
    }
  }

  public boolean isCollection() {
    return false;
  }

  public final ICollection getParentCollection() throws ResourceException {
    this.checkDeleted();
    // Is it the root collection ?
    if (this.getRID().equals(getRepositoryManager().getPrefix())
      || this.getRID().equals(getRepositoryManager().getPrefix() + "/")) {
      return null;
    }
    return this.internalGetParentCollection();
  }

  public final IContent getContent() throws ResourceException, AccessDeniedException {
    return getContent(false);
  }

  /**
   * @param handleExternalLink TBD: Description of the incoming method parameter
   * @return content
   * @exception ResourceException Exception raised in failure situation
   * @exception AccessDeniedException Exception raised in failure situation
   * @deprecated as of EP 5.0 SP5 (for boolean parameter = true) <p>
   *
   *      This method is deprecated because it opens a potential security hole:
   *      The URL of an external link can have different schemas (e.g. "file:")
   *      which are all accessed with the security context of the servlet
   *      engine. This could be (ab)used to access files/resources on the local
   *      server and other systems without the security context of the logged in
   *      user. <p>
   *
   *      <b>No longer supported as of SP6: Throws a NotSupportedException if
   *      parameter is <code>true</code> </b>
   */
  public final IContent getContent(boolean handleExternalLink) throws ResourceException, AccessDeniedException {
    if (handleExternalLink) {
      log.errorT(
        "getContent(1010)",
        "getting content with external link resolution -- check caller!"
          + LoggingFormatter.extractCallstack(new Exception()));
    }
    this.checkDeleted();
    return this.internalGetContent(handleExternalLink);
  }

  public final IContent getUnfilteredContent() throws ResourceException, AccessDeniedException {
    return getUnfilteredContent(false);
  }

  public final IContent getUnfilteredContent(boolean handleExternalLink)
    throws ResourceException, AccessDeniedException {
    // JRE: resolving external links from within the servlet is a horrible
    // security hole. Log it for now.
    if (handleExternalLink) {
      log.errorT(
        "getUnfilteredContent(1039)",
        "getting content with external link resolution -- check caller!"
          + LoggingFormatter.extractCallstack(new Exception()));
    }
    this.checkDeleted();
    return this.internalGetUnfilteredContent(handleExternalLink);
  }

  public final IPropertyMap getProperties() throws ResourceException, NotSupportedException, AccessDeniedException {
    this.checkDeleted();

    IPropertyMap props = null;
    props = this.internalGetProperties();

    props = this.applyPropertyReadFilter(this, props);
    ResourcePropertyCacheHandler.setCachedPrefilledProperties(this, props);
    this.sendEvent(ResourceEvent.PROPERTY_GET, null, props);
    return props;
  }

  public final IPropertyMap getProperties(IPropertyNameList propNameList)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.checkDeleted();

    if (propNameList == null) {
      return new PropertyMap();
    }

    IMutablePropertyMap cachedProps = new MutablePropertyMap();
    IPropertyNameList uncachedNameList = null;
    IPropertyMap uncachedProps = new MutablePropertyMap();

    if (propNameList.contains(PropertyName.createCollection())) {
      IProperty prop = new Property(PropertyName.createCollection(), new Boolean(this.isCollection()));
      cachedProps.put(prop);
      ResourcePropertyCacheHandler.setPrefilledProperty(this, prop);
    }
    if (propNameList.contains(PropertyName.createLinkType())) {
      IProperty prop = new Property(PropertyName.createLinkType(), LinkType.linkType2String(this.getLinkType()));
      cachedProps.put(prop);
      ResourcePropertyCacheHandler.setPrefilledProperty(this, prop);
    }
    if (propNameList.contains(PropertyName.createExtension())) {
      IProperty prop = new Property(PropertyName.createExtension(), this.getRID().extension());
      cachedProps.put(prop);
      ResourcePropertyCacheHandler.setPrefilledProperty(this, prop);
    }

    // search for properties in cache
    boolean isEmpty ;
    synchronized ( this.cachedPrefilledProperties ) {
      isEmpty = this.cachedPrefilledProperties.isEmpty() ;
    }
    if (!isEmpty) {
      uncachedNameList = new PropertyNameList();
      for (int i = 0; i < propNameList.size(); i++) {
        ResourcePropertyCacheHandler.PrefilledProperty p =
          ResourcePropertyCacheHandler.getCachedPrefilledProperty(this, propNameList.get(i));
        if (p != null && p.property != null) {
          cachedProps.put(p.property);
        }
        else {
          uncachedNameList.add(propNameList.get(i));
        }        
      }
    }
    else {
      uncachedNameList = propNameList;
    }

    // get properties not found in cache
    if (uncachedNameList.size() > 0) {
      uncachedProps =
        this.internalGetPropertiesExtended(
          PropertyNameTool.removeFilteredPropertyNames(this.getRepositoryManager().getID(), uncachedNameList));

      // filter all uncached properties
      uncachedProps = this.applyPropertyReadFilter(uncachedNameList, uncachedProps).getMutable();

      // set not found properties to cache list as null-property
      for (int i = 0; i < uncachedNameList.size(); i++) {
        if (!uncachedProps.containsProperty(uncachedNameList.get(i))) {
          ResourcePropertyCacheHandler.setCachedPrefilledProperty(this, uncachedNameList.get(i), null);
        }
      }

      ResourcePropertyCacheHandler.setCachedPrefilledProperties(this, uncachedProps);
      this.sendEvent(ResourceEvent.PROPERTY_GET, null, uncachedProps);

      // copy uncached props to property list
      IPropertyIterator iter = uncachedProps.iterator();
      while (iter.hasNext()) {
        cachedProps.put(iter.next());
      }
    }

    return cachedProps;

  }

  private final IPropertyMap internalGetPropertiesExtended(IPropertyNameList propNameList) throws ResourceException {
    IExtendedPropertyManager pm = this.getExtendedPropertyManager(false);
    if (pm != null) {
      return this.internalGetProperties(propNameList);
    }
    else {
      IPropertyMap props = new MutablePropertyMap(propNameList.size());

      for (int i = 0, s = propNameList.size(); i < s; i++) {
        IPropertyName propName = propNameList.get(i);
        try {
          IProperty prop = this.internalGetProperty(propName);
          if (prop != null) {
            ((IMutablePropertyMap)props).put(prop);
          }
        }
        catch (ResourceException ex) {
          ((IMutablePropertyMap)props).addException(propName, ex);
        }
      }
      return props;
    }
  }

  private final IProperty getSystemProperty(IPropertyName propName)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.checkDeleted();
    ResourcePropertyCacheHandler.PrefilledProperty p =
      ResourcePropertyCacheHandler.getCachedPrefilledProperty(this, propName);
    if (p != null) {
      return p.property;
    }

    IProperty property = this.internalGetProperty(propName);

    if (ResourceImpl.KMC_SYS_PROP_FILTER) {
      property = this.applyPropertyReadFilter(property, propName);
    }

    ResourcePropertyCacheHandler.setCachedPrefilledProperty(this, propName, property);

    if (property != null) {
      this.sendEvent(ResourceEvent.PROPERTY_GET, property);
    }

    return property;
  }

  public final IProperty getProperty(IPropertyName propName)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.checkDeleted();

    ResourcePropertyCacheHandler.PrefilledProperty p =
      ResourcePropertyCacheHandler.getCachedPrefilledProperty(this, propName);
    if (p != null) {
      return p.property;
    }

    IProperty property = null;
    if (propName.equals(PropertyName.createCollection())) {
      property = new Property(PropertyName.createCollection(), new Boolean(this.isCollection()));
    }
    else if (propName.equals(PropertyName.createLinkType())) {
      property = new Property(PropertyName.createLinkType(), LinkType.linkType2String(this.getLinkType()));
    }
    else if (propName.equals(PropertyName.createExtension())) {
      property = new Property(PropertyName.createExtension(), this.getRID().extension());
    }
    else {
      if (!PropertyNameTool.checkFilteredPropertyName(this.getRepositoryManager().getID(), propName)) {
        property = this.internalGetProperty(propName);
      }
      property = this.applyPropertyReadFilter(property, propName);
    }

    ResourcePropertyCacheHandler.setCachedPrefilledProperty(this, propName, property);

    if (property == null)
      return null;

    this.sendEvent(ResourceEvent.PROPERTY_GET, property);
    return property;
  }

  public IProperty getInheritedProperty(IPropertyName propName)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.checkDeleted();
    IProperty p = this.applyPropertyReadFilter(this.internalGetInheritedProperty(propName), propName);
    this.sendEvent(ResourceEvent.PROPERTY_GET, null, p);
    return p;
  }

  public ILockInfoCollection getLocks() throws ResourceException, NotSupportedException, AccessDeniedException {
    this.checkDeleted();
    IFrameworkTransaction ft = FrameworkTransaction.required();
    try {
      ILockInfoCollection lockInfo = this.internalGetLocks();
      ft.commit();
      ft = null;
      return lockInfo;
    }
    finally {
      if (ft != null) ft.rollback();
    }
  }

  public final ILockInfo getLockByToken(String lockToken) throws ResourceException, NotSupportedException {
    checkDeleted();
    ILockInfo lockInfo = internalGetLockByToken(lockToken);
    return lockInfo;
  }

  public final boolean isLocked() throws ResourceException {
    checkDeleted();
    if (isRevision()) {
      return false;
    }

    try {
      return internalIsLocked();
    }
    catch (NotSupportedException ex) {
      return false;
    }
  }

  public final boolean isLockedByMe() throws ResourceException {
    checkDeleted();
    if (isRevision()) {
      return false;
    }

    try {
      return internalIsLockedByMe();
    }
    catch (NotSupportedException ex) {
      return false;
    }
  }

  public final URL getTargetURL() throws ResourceException {
    this.checkDeleted();
    return this.internalGetTargetURL();
  }

  public final IResource getTargetResource() throws ResourceException {
    this.checkDeleted();
    return this.targetResource();
  }

  public final LinkType getLinkType() throws ResourceException {
    this.checkDeleted();
    return this.internalGetLinkType();
  }

  public final boolean isVersioned() throws ResourceException {
    this.checkDeleted();

    if (this.isRevision()) {
      // revision resources are *versions*, but not *versioned*. To be versioned
      // means that the resource can be checked-in/checked-out. (jre)
      return false;
    }

    return this.internalIsVersioned();
  }

  public final boolean isRevision() {
    return this.isRevision;
  }

  public final boolean isCheckedOut() throws ResourceException {
    return this.internalIsCheckedOut();
  }

  public final IVersionHistory getVersionHistory() throws ResourceException, NotSupportedException {
    return this.applyNamespaceReadFilterForHistory(internalGetVersionHistory());
  }

  public final String getRevisionID() {
    return this.revID;
  }

  public final IResourceList getCheckedOutResources()
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.internalGetCheckedOutResources();
  }

  public boolean isA(Class classtest) {
    if (ITimestampedResource.class.isAssignableFrom(classtest)) {
      return true;
    }
    else if (IResource.class.isAssignableFrom(classtest)) {
      return true;
    }
    else if (ICollection.class.isAssignableFrom(classtest)) {
      return this instanceof ICollection;
    }
    else {
      try {
        return ObjectFactoryRegistry.getInstance().isA(classtest, this);
      }
      catch (Throwable ex) {
        log.errorT(ex.getMessage() + " - " + LoggingFormatter.extractCallstack(ex));
        return false;
      }
    }
  }

  public final void rename(String newName) throws ResourceException, NotSupportedException, AccessDeniedException {
    this.checkDeleted();
    if (this.isRevision()) {
      throw new RevisionResourceException("resource is revision", this.rid);
    }
    if (newName == null) {
      return;
    }
    this.checkNewName(newName);

    if (this.isCollection()) {
      // Test for root collection - cannot be renamed
      if (this.getRID().equals(getRepositoryManager().getPrefix())
        || this.getRID().equals(getRepositoryManager().getPrefix() + "/")) {
        throw new ResourceException("Cannot rename the root collection");
      }
    }

    ResourcePropertyCacheHandler.clearCachedPrefilledProperties(this);

    ResourceEvent initialEvent = (ResourceEvent)this.sendEvent(ResourceEvent.PRE_RENAME, null, newName);

    RID oldRID = getRID();

    IFrameworkTransaction ft = FrameworkTransaction.required();      
    try{
      if (this.sendEvent(ft, ResourceEvent.PRE_RENAME, null, newName)) {
        this.internalRename(newName);
        this.rid = this.rid.removeName().add(RID.getRID(newName));
        this.name = newName;
        if (this.sendEvent(ft, ResourceEvent.RENAME, initialEvent != null ? initialEvent.getUniqueId() : null, oldRID.toString())) {
          ft.commit();
          ft = null;
        }
      }
    }
    finally{
      if (ft != null) ft.rollback();
    }           

    this.sendEvent(ResourceEvent.RENAME, initialEvent != null ? initialEvent.getUniqueId() : null, oldRID.toString());
  }

  public final void delete() throws ResourceException, NotSupportedException, AccessDeniedException {
    this.checkDeleted();

    if (this.isCollection() && !this.isLink()) {
      ResourceEvent initialEvent = (ResourceEvent)sendEvent(ResourceEvent.PRE_DELETE, null, null);

      IRidList deletedURIs = null;
 
      IFrameworkTransaction ft = FrameworkTransaction.required();      
      try{        
        if (sendEvent(ft, ResourceEvent.PRE_DELETE, null, null)) {
          deletedURIs = internalDeleteCollection();
          if (sendEvent(ft, ResourceEvent.DELETE, initialEvent != null ? initialEvent.getUniqueId() : null, deletedURIs)) {
            ft.commit();
            ft = null;
            this.deleted = true;
          }
        }
      }
      catch (IncompleteOperationException e) {
        if (sendEvent(ft, ResourceEvent.DELETE, initialEvent != null ? initialEvent.getUniqueId() : null, e.getSuccessfulOperations())) {
          ft.commit();
          ft = null;          
        }          
        sendEvent(ResourceEvent.DELETE, initialEvent != null ? initialEvent.getUniqueId() : null, e.getSuccessfulOperations());
        throw e;
      }
      finally{
        if (ft != null) ft.rollback();
      }

      sendEvent(ResourceEvent.DELETE, initialEvent != null ? initialEvent.getUniqueId() : null, deletedURIs);
    }
    else {
      ResourceEvent initialEvent = (ResourceEvent)this.sendEvent(ResourceEvent.PRE_DELETE, null, null);

      IFrameworkTransaction ft = FrameworkTransaction.required();      
      try{        
        this.internalDelete();
        if (this.sendEvent(ft, ResourceEvent.DELETE, initialEvent != null ? initialEvent.getUniqueId() : null, null)) {
          ft.commit();
          ft = null;
        }        
      }
      finally{
        if (ft != null) ft.rollback();
      }

      // Close content object
      if (this.content != null) {
        this.content.close();
      }
      if (this.unfilteredContent != null) {
        this.unfilteredContent.close();
      }
      this.deleted = true;

      this.sendEvent(ResourceEvent.DELETE, initialEvent != null ? initialEvent.getUniqueId() : null, null);
    }
  }

  public final void updateContent(IContent newContent)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.checkDeleted();
    if (this.isRevision()) {
      throw new RevisionResourceException("resource is revision", this.rid);
    }
    if (newContent == null) {
      newContent = ResourceImpl.EMPTY_CONTENT;
    }
    newContent = this.applyContentWriteFilterForUpdate(newContent);
        
    IFrameworkTransaction ft = FrameworkTransaction.required();      
    try{        
      this.internalUpdateContent(newContent);
      if (this.sendEvent(ft, ResourceEvent.SET, null, null)) {
        ft.commit();
        ft = null;
      }      
    }
    finally{
      if (ft != null) ft.rollback();
    }
    
    this.sendEvent(ResourceEvent.SET, null, null);

    // Throw away the content object because it holds the content received from the manager
    if (this.content != null) {
      this.content.close();
      this.content = null;
    }
    if (this.unfilteredContent != null) {
      this.unfilteredContent.close();
      this.unfilteredContent = null;
    }
  }

  public final void update(IContent newContent, IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException, SetPropertiesException {
    
    checkDeleted();
    if (isRevision()) {
      throw new RevisionResourceException("resource is revision", this.rid);
    }
    if (mustSendEvents()) {
      IFrameworkTransaction ft = FrameworkTransaction.required();      
      try{        
        try {
          internalUpdate(
            applyContentWriteFilterForUpdate(newContent),
            applyPropertyWriteFilterForUpdate(properties, PropertyFilterMode.PROPERTY_LIST));
        }
        catch (SetPropertiesException ex) {
          if (sendEventsForPropertyChanges(ft, null, ex, properties)) {
            ft.commit();
            ft = null;
          }
          sendEventsForPropertyChanges(null, ex, properties);
          throw ex;
        }
        
        if (sendEvent(ft, ResourceEvent.SET, null, null)) {
          ft.commit();
          ft = null;
        }
      }
      finally{
        if (ft != null) ft.rollback();
      }
    }
    else {
      if (content == null) {
        content = ResourceImpl.EMPTY_CONTENT;
      }
      IFrameworkTransaction ft = new FrameworkTransaction();      
      try{        
        internalUpdate(
            applyContentWriteFilterForUpdate(newContent),
            applyPropertyWriteFilterForUpdate(properties, PropertyFilterMode.PROPERTY_LIST));
      }
      finally{
        if (ft != null) ft.rollback();
      }
    }

    // Throw away the content object because it holds the content received from the manager
    if (this.content != null) {
      this.content.close();
      this.content = null;
    }
    if (this.unfilteredContent != null) {
      this.unfilteredContent.close();
      this.unfilteredContent = null;
    }
  }

  public IResource copy(URI destinationURI, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.copy(RID.getRID(destinationURI.toString()), null, param);
  }

  public IResource copy(URI destinationURI, IPosition position, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.copy(RID.getRID(destinationURI.toString()), position, param);
  }

  public IResource move(URI destinationURI, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.move(RID.getRID(destinationURI.toString()), null, param);
  }

  public IResource move(URI destinationURI, IPosition position, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.move(RID.getRID(destinationURI.toString()), position, param);
  }

  public IResource copy(RID destinationURI, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.copy(destinationURI, null, param);
  }

  public IResource copy(RID destinationURI, IPosition position, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.checkDeleted();
    if (destinationURI == null) {
      return null;
    }
    if (!destinationURI.isAbsolute()) {
      throw new InvalidArgumentException("destinationURI must be absolute");
    }
    if (param == null) {
      param = DEFAULT_COPY_PARAMETER;
    }
    this.checkNewName(destinationURI.name().toString());

    IResource newRes = null;
    
    IFrameworkTransaction ft = FrameworkTransaction.required();
    try{        
      newRes = this.internalCopy(destinationURI, position, param, false);
      if (newRes != null && this.sendEvent(ft, ResourceEvent.COPY, null, newRes)) {      
        ft.commit();
        ft = null;        
      }
    }
    finally{
      if (ft != null) ft.rollback();
    }

    if (newRes != null) {
      this.sendEvent(ResourceEvent.COPY, null, newRes);
    }
    return newRes;
  }

  public IResource move(RID destinationURI, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return move(destinationURI, null, param);
  }

  public IResource move(RID destinationRID, IPosition position, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.checkDeleted();
    if (destinationRID == null) {
      return null;
    }
    if (!destinationRID.isAbsolute()) {
      throw new InvalidArgumentException("destinationURI must be absolute");
    }

    ResourcePropertyCacheHandler.clearCachedPrefilledProperties(this);

    // If source and destination are the same change position in ordered collection
    if (destinationRID.equals(this.getRID())) {
      this.internalChangePosision(position);
      return null;
    }

    if (param == null) {
      param = DEFAULT_COPY_PARAMETER;
    }
    this.checkNewName(destinationRID.name().toString());

    ResourceEvent initialEvent = (ResourceEvent)this.sendEvent(ResourceEvent.PRE_MOVE, null, destinationRID.getPath());

    IResource movedRes = null;
    
    //TODO: Prbs with srvlt-cm2cmfs: 03 CR MOVE hangs
    IFrameworkTransaction ft = FrameworkTransaction.required();      
    try{
      if (this.sendEvent(ft, ResourceEvent.PRE_MOVE, null, destinationRID.getPath())) {
        movedRes = this.internalMove(destinationRID, position, param);
        // Close content object
        if (this.content != null) {
          this.content.close();
        }
        if (this.unfilteredContent != null) {
          this.unfilteredContent.close();
        }      
        this.deleted = true;
    
        if (movedRes != null) {
          if (this.sendEvent(ft, ResourceEvent.MOVE, initialEvent != null ? initialEvent.getUniqueId() : null, movedRes)) {
            ft.commit();        
            ft = null;        
          }
        }
      }
    }
    finally{
      if (ft != null) ft.rollback();
    }

    if (movedRes != null) {
      this.sendEvent(ResourceEvent.MOVE, initialEvent != null ? initialEvent.getUniqueId() : null, movedRes);
    }
    return movedRes;
  }

  public final void deleteProperty(IPropertyName propName)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.checkDeleted();
    if (this.isRevision()) {
      throw new RevisionResourceException("resource is revision", this.rid);
    }

    ResourcePropertyCacheHandler.clearCachedPrefilledProperties(this);

    if (propName.equals(PropertyName.createEnhancedCollection())) {
      IFrameworkTransaction ft = FrameworkTransaction.required();      
      try{        
        this.internalDisableChildAutoVersioning();
        ft.commit();
        ft = null;
      }
      finally{
        if (ft != null) ft.rollback();
      }
    }
    else {
      IFrameworkTransaction ft = FrameworkTransaction.required();      
      try{        
        this.internalDeleteProperty(propName);
        if (this.sendEvent(ft, ResourceEvent.PROPERTY_DELETE, null, propName)) {
          ft.commit();
          ft = null;
        }
      }
      finally{
        if (ft != null) ft.rollback();
      }
      this.sendEvent(ResourceEvent.PROPERTY_DELETE, null, propName);
    }
  }

  public final ILockInfo lock() throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.lock(
      new LockProperties(LockType.WRITE, LockScope.EXCLUSIVE, LockDepth.SHALLOW, LockInfo.LOCK_TIMEOUT_INFINITE));
  }

  public final ILockInfo lock(ILockProperties lockProperties)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.checkDeleted();
    ILockInfo lock = null;
    
    IFrameworkTransaction ft = FrameworkTransaction.required();      
    try{        
      lock = this.internalLock(lockProperties);
      if (this.sendEvent(ft, ResourceEvent.LOCK, null, lock)) {
        ft.commit();        
        ft = null;
      }
    }
    finally{
      if (ft != null) ft.rollback();
    }
    
    this.sendEvent(ResourceEvent.LOCK, null, lock);
    return lock;
  }

  public final void unlock(ILockInfo lockInfo) throws ResourceException, NotSupportedException, AccessDeniedException {
    this.checkDeleted();

    // Check if user is authenticated
    IUser user = this.context.getUser();
    if (user == null) {
      throw new AccessDeniedException("No user in context", this.rid, null, null);
    }
    if (!user.isAuthenticated()) {
      throw new AccessDeniedException("User <" + user.getId() + "> is not authenticated", this.rid, null, user.getId());
    }

    // Check if user is the lock owner OR a "lock breaker"
    boolean canUnlock = false;
    try {
      canUnlock = this.isLockedByMe() || this.isLockBreaker(WPUMFactory.getUserFactory().getUser(user.getId()));
    }
    catch (UserManagementException ex) {
      log.errorT("unlock(1576)", ex.getMessage() + LoggingFormatter.extractCallstack(ex));
    }
    catch (WcmException ex) {
      log.errorT("unlock(1579)", ex.getMessage() + LoggingFormatter.extractCallstack(ex));
    }

    if (!canUnlock) {
      throw new LockedException("Resource is locked: " + this.rid, this.rid, null);
    }
    
    IFrameworkTransaction ft = FrameworkTransaction.required();      
    try{        
      this.internalUnlock(lockInfo);
      if (this.sendEvent(ft, ResourceEvent.UNLOCK, null, lockInfo)) {
        ft.commit();        
        ft = null;
      }
    }
    finally{
      if (ft != null) ft.rollback();
    }    
    
    this.sendEvent(ResourceEvent.UNLOCK, null, lockInfo);
  }

  public final void refreshLock(ILockInfo lockInfo)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.checkDeleted();
    IFrameworkTransaction ft = FrameworkTransaction.required();
    try {
      this.internalRefreshLock(lockInfo);
      ft.commit();
      ft = null;
    }
    finally {
      if (ft != null) ft.rollback();
    }
  }

  // ---------------- Versioning -----------------

  public final void enableVersioning(boolean enable)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.checkDeleted();
    if (this.isRevision()) {
      throw new RevisionResourceException("resource is revision", this.rid);
    }

    if (enable && this.isVersioned()) {
      return;
    }
    if (!enable && !this.isVersioned()) {
      return;
    }

    IFrameworkTransaction ft = FrameworkTransaction.required();      
    try{        
      this.internalEnableVersioning(enable);
      if (this.sendEvent(ft, ResourceEvent.ENABLE_VERSIONING, null, null)) {
        ft.commit();        
        ft = null;
      }
    }
    finally{
      if (ft != null) ft.rollback();
    }

    this.sendEvent(ResourceEvent.ENABLE_VERSIONING, null, null);
  }

  public ICheckOutInfo checkOut() throws ResourceException, NotSupportedException, AccessDeniedException {
    if (this.isRevision()) {
      throw new RevisionResourceException("resource is revision", this.rid);
    }
    ICheckOutInfo info = null;
    
    IFrameworkTransaction ft = FrameworkTransaction.required();      
    try{        
      info = this.internalCheckOut();
      if (this.sendEvent(ft, ResourceEvent.CHECKOUT, null, null)) {
        ft.commit();        
        ft = null;
      }
    }
    finally{
      if (ft != null) ft.rollback();
    }

    this.sendEvent(ResourceEvent.CHECKOUT, null, null);
    return info;
  }

  public final void undoCheckOut() throws ResourceException, NotSupportedException, AccessDeniedException {
    if (this.isRevision()) {
      throw new RevisionResourceException("resource is revision", this.rid);
    }
    
    IFrameworkTransaction ft = FrameworkTransaction.required();      
    try{        
      this.internalUndoCheckOut();
      if (this.sendEvent(ft, ResourceEvent.UNDO_CHECKOUT, null, null)) {
        ft.commit();        
        ft = null;
      }
    }
    finally{
      if (ft != null) ft.rollback();
    }

    this.sendEvent(ResourceEvent.UNDO_CHECKOUT, null, null);
  }

  public final ICheckInInfo checkIn(IContent newContent, IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return checkIn(newContent, properties, false);
  }

  public final ICheckInInfo checkIn(IContent newContent, IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return checkIn(newContent, properties, ignorePropertyFailures, (RID) null);
  }

  public final ICheckInInfo checkIn(
    IContent newContent,
    IPropertyMap properties,
    boolean ignorePropertyFailures,
    URI expectedCheckInURI)
    throws ResourceException, NotSupportedException, AccessDeniedException, ExpectedCheckInRIDException {
    return checkIn(
      newContent,
      properties,
      ignorePropertyFailures,
      expectedCheckInURI != null ? RID.getRID(expectedCheckInURI.toString()) : null);
  }

  public final ICheckInInfo checkIn(
    IContent newContent,
    IPropertyMap properties,
    boolean ignorePropertyFailures,
    RID expectedCheckInRID)
    throws ResourceException, NotSupportedException, AccessDeniedException, ExpectedCheckInRIDException {
    if (this.isRevision()) {
      throw new RevisionResourceException("resource is revision", this.rid);
    }
    
    ResourcePropertyCacheHandler.clearCachedPrefilledProperties(this);
    
    IContent filteredContent = null;
    if (newContent != null) {
      filteredContent = this.applyContentWriteFilterForUpdate(newContent);
    }
    IPropertyMap filteredProps = null;
    if (properties != null) {
      filteredProps = this.applyPropertyWriteFilterForUpdate(properties, PropertyFilterMode.PROPERTY_LIST);
    }

    ICheckInInfo checkInInfo = null;

    if (mustSendEvents()) {
      // Send pre-event
      ResourceEvent initialEvent = (ResourceEvent)this.sendEvent(ResourceEvent.PRE_CHECKIN, null, null);

      IPropertyMap before = null;
      IFrameworkTransaction ft = FrameworkTransaction.required();
      try {
        if (this.sendEvent(ft, ResourceEvent.PRE_CHECKIN, null, null)) {
          // Capture properties before check-in
          before = this.internalGetProperties();
          
          try {
            checkInInfo = this.internalCheckIn(filteredContent, filteredProps, ignorePropertyFailures, expectedCheckInRID);
          }
          catch (SetPropertiesException ex) {
            if (this.sendEventsForPropertyChanges(ft, initialEvent != null ? initialEvent.getUniqueId() : null, ex, properties)) {
              ft.commit();
              ft = null;
            }
            this.sendEventsForPropertyChanges(initialEvent != null ? initialEvent.getUniqueId() : null, ex, properties);          
            throw ex;
          }
    
          // Send transactional post-events
          boolean doCommit = true;
          doCommit = doCommit && this.sendEventsForPropertyChanges(ft, 
            initialEvent != null ? initialEvent.getUniqueId() : null,
            before,
            filteredProps);
          if (newContent != null) {
            doCommit = doCommit && this.sendEvent(ft, ResourceEvent.SET, initialEvent != null ? initialEvent.getUniqueId() : null, null);
          }
          doCommit = doCommit && this.sendEvent(ft, 
            ResourceEvent.CHECKIN,
            initialEvent != null ? initialEvent.getUniqueId() : null,
            checkInInfo.getRevisionRID());
          if (doCommit) {
            ft.commit();
            ft = null;
          }
        }
      }
      finally {
        if (ft != null) ft = null;
      }
      
      // Send non-transactional post-events
      this.sendEventsForPropertyChanges(
        initialEvent != null ? initialEvent.getUniqueId() : null,
        before,
        filteredProps);
      if (newContent != null) {
        this.sendEvent(ResourceEvent.SET, initialEvent != null ? initialEvent.getUniqueId() : null, null);
      }
      this.sendEvent(
        ResourceEvent.CHECKIN,
        initialEvent != null ? initialEvent.getUniqueId() : null,
        checkInInfo.getRevisionRID());
      
    }
    else {
      checkInInfo = this.internalCheckIn(filteredContent, filteredProps, ignorePropertyFailures, expectedCheckInRID);
    }

    return checkInInfo;
  }

  public Collection listTypes() {
    Collection types = new ArrayList();
    types.add(ITimestampedResource.class);
    types.add(IResource.class);
    if (this.isCollection()) {
      types.add(ICollection.class);
    }
    return types;
  }

  public Object as(Class classtest) throws ResourceException, NotSupportedException, AccessDeniedException {
    if (ITimestampedResource.class.isAssignableFrom(classtest)) {
      return this.ts;
    }
    else if (IResource.class.isAssignableFrom(classtest)) {
      return this;
    }
    else if (ICollection.class.isAssignableFrom(classtest)) {
      return this instanceof ICollection ? this : null;
    }
    else {
      try {
        return ObjectFactoryRegistry.getInstance().as(classtest, this);
      }
      catch (Throwable ex) {
        log.errorT(ex.getMessage() + " - " + LoggingFormatter.extractCallstack(ex));
        return null;
      }
    }
  }

  //////////////////////////////////////////////////////////////////////////////
  // public
  //////////////////////////////////////////////////////////////////////////////

  public boolean equals(Object o) {
    if (o == null) {
      return false;
    }
    if (!(o instanceof IResource)) {
      return false;
    }
    try {
      return (((IResource)o).getRID().equals(getRID()));
    }
    catch (ResourceException ex) {
      return false;
    }
  }

  public int hashCode() {
    try {
      return getRID().hashCode();
    }
    catch (ResourceException ex) {
      return this.rid.hashCode();
    }
  }

  public String toString() {
    return this.rid.toString();
  }

  public final IResourceList search(IQueryExpression query, int depth, int maxResults, boolean includeRevisions)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    checkDeleted();

    IResourceList result = internalSearch(query, depth, maxResults, includeRevisions);
    if (result != null && result.size() > 0) {
      result = applyNamespaceReadFilterForSearch(result);
    }
    return result;
  }

  //////////////////////////////////////////////////////////////////////////////
  // private / protected
  //////////////////////////////////////////////////////////////////////////////

  /**
   * Returns the namespace manager for this resource.
   *
   * @param required throw NotSupportedException if the sub-manager is not
   *      available.
   * @return the namespace manager for this resource.
   * @exception NotSupportedException Description of Exception
   * @exception ResourceException Description of Exception
   */
  protected final INamespaceManager getNamespaceManager(boolean required)
    throws NotSupportedException, ResourceException {
    INamespaceManager nsm = this.rmAdapter.getOldAbstract().getNamespaceManager(this);
    if (required == true && nsm == null) {
      throw new NotSupportedException();
    }
    return nsm;
  }

  /**
   * Returns the namespace manager for this resource.
   *
   * @return the namespace manager for this resource.
   */
  protected final INamespaceManager getNamespaceManager() {
    try {
      return this.rmAdapter.getOldAbstract().getNamespaceManager(this);
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Returns the ext. namespace manager for this resource.
   *
   * @param required throw NotSupportedException if the sub-manager is not
   *      available.
   * @return the ext. namespace manager for this resource.
   * @exception NotSupportedException Description of Exception
   * @exception ResourceException Description of Exception
   */
  protected final IExtendedNamespaceManager getExtendedNamespaceManager(boolean required)
    throws NotSupportedException, ResourceException {
    IExtendedNamespaceManager ensm = this.getExtendedNamespaceManager();
    if (required == true && ensm == null) {
      throw new NotSupportedException();
    }
    return ensm;
  }

  /**
   * Returns the ext. namespace manager for this resource.
   *
   * @return the ext. namespace manager for this resource.
   */
  protected final IExtendedNamespaceManager getExtendedNamespaceManager() {
    if (this.rmAdapter.isNew()) {
      return null;
    }
    try {
      Object o = this.rmAdapter.getOldAbstract().getNamespaceManager(this);
      if (o instanceof IExtendedNamespaceManager) {
        return (IExtendedNamespaceManager)o;
      }
      else {
        return null;
      }
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Returns the lock manager for this resource.
   *
   * @param required throw NotSupportedException if the sub-manager is not
   *      available.
   * @return the lock manager for this resource.
   * @exception NotSupportedException Description of Exception
   * @exception ResourceException Description of Exception
   */
  protected final ILockManager getLockManager(boolean required) throws NotSupportedException, ResourceException {
    ILockManager lm = this.rmAdapter.getOldAbstract().getLockManager(this);
    if (required == true && lm == null) {
      throw new NotSupportedException();
    }
    return lm;
  }

  /**
   * Returns the property manager for this resource.
   *
   * @param required throw NotSupportedException if the sub-manager is not
   *      available.
   * @return the property manager for this resource.
   * @exception NotSupportedException Description of Exception
   * @exception ResourceException Description of Exception
   */
  protected final IPropertyManager getPropertyManager(boolean required)
    throws NotSupportedException, ResourceException {
    IPropertyManager pm = this.rmAdapter.getOldAbstract().getPropertyManager(this);
    if (required == true && pm == null) {
      throw new NotSupportedException();
    }
    return pm;
  }

  /**
   * Returns the property manager for this resource.
   *
   * @return the property manager for this resource.
   */
  protected final IPropertyManager getPropertyManager() {
    try {
      return this.rmAdapter.getOldAbstract().getPropertyManager(this);
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Returns the ext. property manager for this resource.
   *
   * @param required throw NotSupportedException if the sub-manager is not
   *      available.
   * @return the ext. property manager for this resource.
   * @exception NotSupportedException Description of Exception
   * @exception ResourceException Description of Exception
   */
  protected final IExtendedPropertyManager getExtendedPropertyManager(boolean required)
    throws NotSupportedException, ResourceException {
    IExtendedPropertyManager ensm = this.getExtendedPropertyManager();
    if (required == true && ensm == null) {
      throw new NotSupportedException();
    }
    return ensm;
  }

  /**
   * Returns the ext. property manager for this resource.
   *
   * @return the ext. property manager for this resource.
   */
  protected final IExtendedPropertyManager getExtendedPropertyManager() {
    if (this.rmAdapter.isNew()) {
      return null;
    }
    try {
      Object o = this.rmAdapter.getOldAbstract().getPropertyManager(this);
      if (o instanceof IExtendedPropertyManager) {
        return (IExtendedPropertyManager)o;
      }
      else {
        return null;
      }
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Returns the versioning manager for this resource.
   *
   * @param required throw NotSupportedException if the sub-manager is not
   *      available.
   * @return the versioning manager for this resource.
   * @exception NotSupportedException Description of Exception
   * @exception ResourceException Description of Exception
   */
  protected final IVersioningManager getVersioningManager(boolean required)
    throws NotSupportedException, ResourceException {
    IVersioningManager vm = this.rmAdapter.getOldAbstract().getVersioningManager(this);
    if (required == true && vm == null) {
      throw new NotSupportedException();
    }
    return vm;
  }

  /**
   * Returns the versioning manager for this resource.
   *
   * @return the versioning manager for this resource.
   */
  protected final IVersioningManager getVersioningManager() {
    try {
      return this.rmAdapter.getOldAbstract().getVersioningManager(this);
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Returns the security manager for this resource.
   *
   * @param required throw NotSupportedException if the sub-manager is not
   *      available.
   * @return the security manager for this resource.
   * @exception NotSupportedException Description of Exception
   * @exception ResourceException Description of Exception
   */
  protected final ISecurityManager getSecurityManager(boolean required)
    throws NotSupportedException, ResourceException {
    ISecurityManager sm = this.rmAdapter.getOldAbstract().getSecurityManager(this);
    if (required == true && sm == null) {
      throw new NotSupportedException();
    }
    return sm;
  }

  /**
   * Returns the security manager for this resource.
   *
   * @return the security manager for this resource.
   */
  protected final ISecurityManager getSecurityManager() {
    try {
      return this.rmAdapter.getOldAbstract().getSecurityManager(this);
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Returns the content manager for this resource.
   *
   * @param required throw NotSupportedException if the sub-manager is not
   *      available.
   * @return the content manager for this resource.
   * @exception NotSupportedException Description of Exception
   * @exception ResourceException Description of Exception
   */
  protected final IContentManager getContentManager(boolean required) throws NotSupportedException, ResourceException {
    IContentManager cm = this.rmAdapter.getOldAbstract().getContentManager(this);
    if (required == true && cm == null) {
      throw new NotSupportedException();
    }
    return cm;
  }

  /**
   * Returns the content manager for this resource.
   *
   * @return the content manager for this resource.
   */
  protected final IContentManager getContentManager() {
    try {
      return this.rmAdapter.getOldAbstract().getContentManager(this);
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Returns the search manager for this resource.
   *
   * @param required throw NotSupportedException if the sub-manager is not
   *      available.
   * @return the search manager for this resource.
   * @exception NotSupportedException Description of Exception
   * @exception ResourceException Description of Exception
   */
  protected final IPropertySearchManager getPropertySearchManager(boolean required)
    throws NotSupportedException, ResourceException {
    IPropertySearchManager cm = this.rmAdapter.getOldAbstract().getPropertySearchManager(this);
    if (required == true && cm == null) {
      throw new NotSupportedException();
    }
    return cm;
  }

  /**
   * Returns the search manager for this resource.
   *
   * @return the search manager for this resource.
   */
  protected final IPropertySearchManager getPropertySearchManager() {
    try {
      return this.rmAdapter.getOldAbstract().getPropertySearchManager(this);
    }
    catch (Exception e) {
      return null;
    }
  }

  protected boolean isLink() throws ResourceException {
    if (this.linkType == null || this.targetURL == null) {
      return false;
    }
    else {
      return (!this.linkType.equals(LinkType.NONE));
    }
  }

  protected boolean isExternalLink() throws ResourceException {
    if (this.linkType == null || this.targetURL == null) {
      return false;
    }
    else {
      return (this.linkType.equals(LinkType.EXTERNAL));
    }
  }

  protected boolean isInternalLink() throws ResourceException {
    if (this.linkType == null || this.targetURL == null) {
      return false;
    }
    else {
      return (this.linkType.equals(LinkType.INTERNAL));
    }
  }

  protected void internalRename(String newName)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.getNamespaceManager(true).rename(newName, this);
  }

  protected ICollection internalGetParentCollection() throws ResourceException {
    IResource parent = ResourceFactory.getInstance().getResource(this.rid.removeName(), this.context);
    if (parent != null && parent instanceof ICollection) {
      return (ICollection)parent;
    }
    else {
      return null;
    }
  }

  protected IContent internalGetContent(boolean handleExternalLink) throws ResourceException, AccessDeniedException {
    if (handleExternalLink) {
      throw new NotSupportedException();
    }
    if (this.content == null) {
      this.content = new ContentImpl(this, true);
    }
    return this.content;
  }

  protected IContent internalGetUnfilteredContent(boolean handleExternalLink)
    throws ResourceException, AccessDeniedException {
    if (handleExternalLink) {
      throw new NotSupportedException();
    }
    if (this.unfilteredContent == null) {
      this.unfilteredContent = new ContentImpl(this, false);
    }
    return this.unfilteredContent;
  }

  protected IResource internalCopyManager(RID destinationRID, IPosition position, ICopyParameter param)
    throws ResourceException {
    return this.getNamespaceManager(true).copy(destinationRID, position, param, this);
  }

  protected IResource internalCopy(RID destinationRID, IPosition position, ICopyParameter param, boolean isMove)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    IResource destinationToOverwrite = null;

    try {
      destinationToOverwrite = ResourceFactory.getInstance().getResource(destinationRID, this.context);
    }
    catch (ResourceException ex) {
      ResourceImpl.log.debugT(LoggingFormatter.extractCallstack(ex));
    }

    // Are the source and destination the same resource ?
    if (destinationRID.equals(getRID())) {
      throw new NameAlreadyExistsException(destinationRID);
    }

    // check parent relation
    if (destinationRID.isAncestorOf(getRID())) {
      // special case: if the destination exists and is not a collection, proceed. It may
      // be an update of a version controlled resource from a version in one of the
      // legacy repositories.
      if (destinationToOverwrite == null || destinationToOverwrite.isCollection()) {
        throw new InvalidTargetException("can't overwrite parent of source", destinationRID);
      }
    }

    // Get destination collection
    RID destCollectionRID = destinationRID.removeName();
    String destCollectionName = destinationRID.name().toString();
    ICollection destColl = null;
    IResource res = ResourceFactory.getInstance().getResource(destCollectionRID, getContext());
    if (res == null) {
      throw new InvalidTargetException("Destination collection does not exist:" + destCollectionRID, destinationRID);
    }
    // The target collection might be a link
    if (!res.getLinkType().equals(LinkType.NONE)) {
      res = res.getTargetResource();
    }
    if (res == null) {
      throw new InvalidTargetException("Destination collection does not exist:" + destCollectionRID, destCollectionRID);
    }
    if (res.isCollection()) {
      destColl = (CollectionImpl)res;
    }
    else {
      throw new InvalidTargetException("Not a collection: " + res.getRID(), res.getRID());
    }
    IRepositoryManager destinationRepMgr = destColl.getRepositoryManager();

    // Calculate supported options
    IRepositoryManager mgr = this.getRepositoryManager();
    ISupportedOptionSet supportedOptions = mgr.getSupportedOptions(this);
    boolean supportsInternalCopy = supportedOptions.isSupported(SupportedOption.INTERNAL_COPY);
    boolean supportsCopyMoveExternal = supportedOptions.isSupported(SupportedOption.COPY_MOVE_EXTERNAL);
    boolean supportsSetProperties =
      destinationRepMgr.getSupportedOptions(null).isSupported(SupportedOption.SET_PROPERTIES);

    // If target resource is within the same repository find out if the manager supports the copy method
    if (destinationRepMgr.getID().equals(mgr.getID())) {
      if (supportsInternalCopy) {
        try {
          // Note: No CREATE_XXX event if the copy in repository internal
          return this.internalCopyManager(destColl.getRID().add(destinationRID.name()), position, param);
        }
        catch (NotSupportedException ex) {
          // if this failed because the target existed and was a folder,
          // deep-delete isn't supported, and the source wasn't a collection,
          // retry once after cleaning it up
          IResource tmp;
          if (!this.isCollection()
            && null != (tmp = ResourceFactory.getInstance().getResource(destinationRID, this.context))
            && tmp.isCollection()) {
            log.debugT("internalCopy(2248)", "cleaning up target collection before copy/overwrite");
            tmp.delete();
            return this.internalCopyManager(destColl.getRID().add(destinationRID.name()), position, param);
          }
          else {
            throw ex;
          }
        }
      }
    }
    else if (supportsCopyMoveExternal) {      
      try {
        ResourceImpl newRes =
          (ResourceImpl)this.internalCopyManager(destColl.getRID().add(destinationRID.name()), position, param);
        if (newRes == null) {
          // The repository manager misbehaved: Must throw an exception if it is unclear whether a resource was created or not because
          // we need the resource for the event and during a "move" the source will be deleted after the "copy" !
          throw new ResourceException("Repository manager returned null in copy() method: " + this.getRepositoryManager().getID());
        }          
        this.sendCreateEvent(newRes);
        return newRes;
      }
      catch (NotSupportedException ex) {
        // The RM has decided it can not handle this external copy operation: continue
      }
    }

    // Test if destination resource exists
    if (destinationToOverwrite != null) {
      if (!param.getOverwrite()) {
        throw new NameAlreadyExistsException("Name already exists", destinationRID);
      }
      else {
        // get the destination resource. DO NOT delete it (that would kill
        // it's version history) (unless it is a collection, which will go
        // away when overwritten by a non-collection)
        if (isMove || (destinationToOverwrite.isCollection())) {
          // move is different from COPY here!
          destinationToOverwrite.delete();
          destinationToOverwrite = null;
        }
      }
    }

    // Get this resource's properties if the target supports custom properties
    IPropertyMap properties = null;
    if (supportsSetProperties) {
      properties = this.internalGetProperties();
    }

    // Create new resource with content and properties, position may be null
    // Use internal create methods: must use unfiltered content and properties, no events will be sent
    if (isLink()) {
      if (destinationToOverwrite != null) {
        destinationToOverwrite.delete();
      }
      ResourceImpl newRes = null;
      if (destColl instanceof CollectionImpl) {
        newRes =
          (ResourceImpl) ((CollectionImpl)destColl).internalCreateLink(
            destCollectionName,
            getTargetURL(),
            getLinkType(),
            position,
            properties,
            param.getIgnorePropertyFailures());
      }
      else if (destColl instanceof CollectionLinkImpl) {
        newRes =
          (ResourceImpl) ((CollectionLinkImpl)destColl).internalCreateLink(
            destCollectionName,
            getTargetURL(),
            getLinkType(),
            position,
            properties,
            param.getIgnorePropertyFailures());
      }
      else {
        newRes =
          (ResourceImpl)destColl.createLink(
            destCollectionName,
            getTargetURL(),
            getLinkType(),
            position,
            properties,
            param.getIgnorePropertyFailures());
      }
      if (newRes == null) {
        throw new ResourceException("Repository manager returned null in createXXX() method: " + this.getRepositoryManager().getID());
      }               
      this.sendCreateEvent(newRes);
      return newRes;
    }
    else {
      if (destinationToOverwrite != null) {
        if (properties == null || properties.isEmpty()) {
          destinationToOverwrite.updateContent(getUnfilteredContent());
        }
        else {
          try {
            if (param.getIgnorePropertyFailures()) {
              // workaround: remove read-only properties if they exist on the
              // destination resource (need better API, jre)

              IPropertyMap existingProps = destinationToOverwrite.getProperties();
              IPropertyIterator it = properties.iterator();
              while (it.hasNext()) {
                IProperty p = it.next();
                IProperty dp = existingProps.get(p.getPropertyName());
                if (dp != null && dp.getPropertyDef().isReadonly()) {
                  log.debugT(
                    "internalCopy(2331)",
                    "not overwriting " + dp.getPropertyName() + " on " + destinationToOverwrite.getRID());
                  it.remove();
                }
              }
            }

            try {
              destinationToOverwrite.update(getUnfilteredContent(), properties);
            }
            catch (NotSupportedException nex) {
              destinationToOverwrite.updateContent(getUnfilteredContent());
              destinationToOverwrite.setProperties(properties);
            }
          }
          catch (SetPropertiesException ex) {
            if (!param.getIgnorePropertyFailures()) {
              throw ex;
            }
          }
        }
        if (position != null) {
          IReorderList orderlist = new ReorderList();
          orderlist.add(new Positioning(getRID().name().toString(), position));
          destColl.reorder(orderlist);
        }
        return destinationToOverwrite;
      }
      else {
        ResourceImpl newRes = null;
        if (destColl instanceof CollectionImpl) {
          newRes =
            (ResourceImpl) ((CollectionImpl)destColl).internalCreateResource(
              destCollectionName,
              position,
              properties,
              getUnfilteredContent(),
              param.getIgnorePropertyFailures());
        }
        else if (destColl instanceof CollectionLinkImpl) {
          newRes =
            (ResourceImpl) ((CollectionLinkImpl)destColl).internalCreateResource(
              destCollectionName,
              position,
              properties,
              getUnfilteredContent(),
              param.getIgnorePropertyFailures());
        }
        else {
          newRes =
            (ResourceImpl)destColl.createResource(
              destCollectionName,
              position,
              properties,
              getUnfilteredContent(),
              param.getIgnorePropertyFailures());
        }
        if (newRes == null) {
          throw new ResourceException("Repository manager returned null in createXXX() method: " + this.getRepositoryManager().getID());
        }                  
        this.sendCreateEvent(newRes);
        return newRes;
      }
    }
  }

  protected IResource internalMoveManager(RID destinationRID, IPosition position, ICopyParameter param)
    throws ResourceException {
    return this.getNamespaceManager(true).move(destinationRID, position, param, this);
  }

  protected IResource internalMove(RID destinationURI, IPosition position, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    // If source and destination are the same change position in ordered collection
    if (destinationURI.equals(getRID())) {
      if (position != null) {
        this.getNamespaceManager(true).changePosition(position, this);
      }
      return this;
    }

    // If target resource is within the same repository find out if the manager supports the move method
    RID destCollectionURI = destinationURI.removeName();
    ICollection destColl = null;
    IResource res = ResourceFactory.getInstance().getResource(destCollectionURI, getContext());
    if (res == null) {
      throw new InvalidTargetException("Destination collection does not exist:" + destCollectionURI, destCollectionURI);
    }
    if (!res.getLinkType().equals(LinkType.NONE)) {
      res = res.getTargetResource();
    }
    if (res == null) {
      throw new InvalidTargetException("Destination collection does not exist:" + destCollectionURI, destCollectionURI);
    }
    if (res.isCollection()) {
      destColl = (ICollection)res;
    }
    else {
      throw new InvalidTargetException("Not a collection: " + res.getRID(), res.getRID());
    }
    if (destColl.getRepositoryManager().getID().equals(getRepositoryManager().getID())) {
      if (this.getRepositoryManager().getSupportedOptions(this).isSupported(SupportedOption.INTERNAL_MOVE)) {
        return this.internalMoveManager(destColl.getRID().add(destinationURI.name()), position, param);
      }
    }
    else if (this.getRepositoryManager().getSupportedOptions(this).isSupported(SupportedOption.COPY_MOVE_EXTERNAL)) {      
      try {
        return this.internalMoveManager(destColl.getRID().add(destinationURI.name()), position, param);
      }
      catch (NotSupportedException ex) {
        // The RM has decided it can not handle this external copy operation: continue
      }
    }

    // Default move

    // jre: can't use the default method (copy/delete) for version controlled
    // resources and revisions
    log.debugT("internalMove(2432)", "internalMove: " + this.getRID() + " " + this.isVersioned());

    if (this.isRevision() || this.isVersioned() || this.isA(IVersionHistoryResource.class)) {
      throw new NotSupportedException("cannot move this type of resource using default method", this.getRID());
    }

    IResource newRes = this.internalCopy(destinationURI, position, param, true);
    this.internalDelete();
    return newRes;
  }

  protected void internalDeleteProperty(IPropertyName propName)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    ResourcePropertyCacheHandler.removeCachedPrefilledProperties(this, propName);
    this.getPropertyManager(true).deleteProperty(propName, this);
  }

  protected ILockInfo internalLock(ILockProperties lockProperties)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.getLockManager(true).lock(lockProperties, this);
  }

  protected void internalUnlock(ILockInfo lockInfo)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.getLockManager(true).unlock(lockInfo, this);
  }

  protected void internalRefreshLock(ILockInfo lockInfo)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.getLockManager(true).refreshLock(lockInfo, this);
  }

  protected ILockInfo internalGetLockByToken(String lockToken) throws ResourceException, NotSupportedException {
    List list = this.getLockManager(true).getLocks(this);
    if (list == null) {
      throw new LockNotFoundException("Lock not found: " + lockToken + " (no list returned)", this.rid);
    }

    ListIterator it = list.listIterator();
    while (it.hasNext()) {
      ILockInfo li = (ILockInfo)it.next();
      if (li.getLockToken().equals(lockToken)) {
        return li;
      }
    }

    throw new LockNotFoundException("Lock not found: " + lockToken, this.rid);
  }

  protected ILockInfoCollection internalGetLocks()
    throws ResourceException, NotSupportedException, AccessDeniedException {
    List list = this.getLockManager(true).getLocks(this);
    if (list == null) {
      return new LockInfoCollection();
    }
    else {
      return new LockInfoCollection(list);
    }
  }

  protected boolean internalIsLocked() throws ResourceException {
    ILockManager lockMgr = getLockManager(false);
    if (lockMgr == null) {
      return false;
    }
    List locks = lockMgr.getLocks(this);
    if (locks != null) {
      return (locks.size() > 0);
    }
    else {
      return false;
    }
  }

  protected boolean internalIsLockedByMe() throws ResourceException {
    ILockManager lockManager = getLockManager(false);
    if (null == lockManager) {
      return false;
    }

    List list = lockManager.getLocks(this);
    ListIterator it = list.listIterator();
    boolean lockedByMe = false;
    while (it.hasNext()) {
      ILockInfo li = (ILockInfo)it.next();
      String owner = li.getOwner();
      String userid = null;
      IUser user = this.context.getUser();
      if (user != null) {
        userid = user.getId();
      }
      else {
        throw new ResourceException("No user ID in context", this.rid);
      }

      if (owner != null && userid != null) {
        if (owner.equals(userid)) {
          lockedByMe = true;
          break;
        }
      }
    }
    return lockedByMe;
  }

  protected void internalSetTargetURL(URL url) throws ResourceException, AccessDeniedException {
    getNamespaceManager().updateLink(url.toString(), null, this);
  }

  protected void internalSetLinkType(LinkType linkType) throws ResourceException, AccessDeniedException {
    getNamespaceManager().updateLink(null, linkType, this);
  }

  protected boolean internalIsCheckedOut() throws ResourceException {
    IVersioningManager vmgr = getVersioningManager(false);
    return (vmgr != null) ? vmgr.isCheckedOut(this) : false;
  }

  protected ICheckOutInfo internalCheckOut() throws ResourceException, NotSupportedException, AccessDeniedException {
    return getVersioningManager(false).checkOut(this);
  }

  protected void internalUndoCheckOut() throws ResourceException, NotSupportedException, AccessDeniedException {
    getVersioningManager(true).undoCheckOut(this);
    if (this.isLink()) {
      IResource res = this.rmAdapter.getOldAbstract().getResource(this.getRID(), this.context);
      this.targetURL = res.getTargetURL();
      this.linkType = res.getLinkType();
    }
  }

  protected ICheckInInfo internalCheckIn(
    IContent newContent,
    IPropertyMap properties,
    boolean ignorePropertyFailures,
    RID expectedCheckInRID)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    ResourcePropertyCacheHandler.setCachedPrefilledProperties(this, null);
    return getVersioningManager(true).checkIn(newContent, properties, ignorePropertyFailures, expectedCheckInRID, this);
  }

  protected IVersionHistory internalGetVersionHistory() throws ResourceException, NotSupportedException {
    return getVersioningManager(true).getVersionHistory(this);
  }

  protected void internalSetAsCurrentVersion() throws ResourceException, NotSupportedException, AccessDeniedException {

    // we support two cases:
    // 1. the repository supports version history resources
    // 2. the repository has a specific format for version URIs (backw. compatibility with WCM repository)

    RID versionControlledResourceRID = null;
    IResource versionControlledResource = null;
    IVersionResource version = (IVersionResource)this.as(IVersionResource.class);

    if (version != null) {
      IVersionHistoryResource versionHistory =
        (IVersionHistoryResource)version.getVersionHistoryResource().as(IVersionHistoryResource.class);

      if (versionHistory != null) {
        IRidSet VCRs = versionHistory.getVersionControlledResourceRIDs();
        if (VCRs.size() != 1) {
          throw new NotSupportedException("there is more than one version controlled resource for this version.");
        }
        versionControlledResourceRID = VCRs.iterator().next();
        versionControlledResource =
          ResourceFactory.getInstance().getResource(versionControlledResourceRID, getContext());
        if (versionControlledResource == null) {
          throw new ResourceNotFoundException(versionControlledResourceRID);
        }
      }
    }
    else {
      versionControlledResourceRID = this.getRID().parent().parent();
      versionControlledResource = ResourceFactory.getInstance().getResource(versionControlledResourceRID, getContext());

      if (versionControlledResource == null) {
        throw new ResourceNotFoundException(versionControlledResourceRID);
      }

      IVersionHistory vh = versionControlledResource.getVersionHistory();
      if (vh == null || !vh.containsResource(this)) {
        throw new ResourceException("version controlled resource not known");
      }
    }

    if (!versionControlledResource.isCheckedOut()) {
      versionControlledResource.checkOut();
    }

    // copy version to version controlled resource, update DAV:comment property, checkin
    this.internalCopy(versionControlledResourceRID, null, new CopyParameter(true, true, false, true, true), false);
    versionControlledResource.setProperty(
      new Property(new PropertyName("DAV:", "comment"), "update from version " + this.getName()));
    versionControlledResource.checkIn(null, null);
  }

  protected IResourceList internalGetCheckedOutResources()
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return getVersioningManager(true).getCheckedOutResources(this);
  }

  protected void internalEnableVersioning(boolean enable)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    getVersioningManager(true).enableVersioning(enable, this);
  }

  protected boolean internalIsVersioned() throws ResourceException {
    IVersioningManager vm = getVersioningManager(false);
    if (vm == null) {
      return false;
    }
    return vm.isVersioned(this);
  }

  protected IPropertyMap internalGetProperties()
    throws ResourceException, NotSupportedException, AccessDeniedException {
    IPropertyMap properties = this.getPropertyManager(true).getProperties(this);
    if (properties == null) {
      properties = new PropertyMap();
    }
    return properties;
  }

  protected IPropertyMap internalGetProperties(IPropertyNameList propNameList)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    IPropertyManager propManager = getPropertyManager(true);
    IMutablePropertyMap props = new MutablePropertyMap();
    IPropertyNameList pNameList = new PropertyNameList();

    for (int i = 0; i < propNameList.size(); i++) {
      ResourcePropertyCacheHandler.PrefilledProperty p =
        ResourcePropertyCacheHandler.getCachedPrefilledProperty(this, propNameList.get(i));
      if (p != null && p.property != null) {
        props.put(p.property);
      }
      else {
        pNameList.add(propNameList.get(i));
      }
    }

    IPropertyMap uncachedProps = new MutablePropertyMap();

    if (pNameList.size() > 0) {
      if (propManager instanceof IExtendedPropertyManager) {
        uncachedProps = ((IExtendedPropertyManager)getPropertyManager(true)).getProperties(pNameList, this);
      }
      else {
        IMutablePropertyMap sprops = new MutablePropertyMap();
        for (int i = 0; i < pNameList.size(); i++) {
          IProperty prop = getPropertyManager(true).getProperty(pNameList.get(i), this);
          if (null != prop)
            sprops.put(prop);
        }
        uncachedProps = sprops.getImmutable();
      }
      ResourcePropertyCacheHandler.setCachedPrefilledProperties(this, uncachedProps);

      IPropertyIterator iter = uncachedProps.iterator();
      while (iter.hasNext()) {
        props.put(iter.next());
      }
    }

    return props;
  }

  protected IProperty internalGetProperty(IPropertyName propName)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.getPropertyManager(true).getProperty(propName, this);
  }

  protected IProperty internalGetInheritedProperty(IPropertyName propName)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    IResource res = this;
    IProperty prop = null;
    boolean found = false;
    do {
      prop = res.getProperty(propName);
      if (prop == null) {
        res = res.getParentCollection(); // returns null if res is root collection
      }
      else {
        found = true;
      }
    }
    while (!found && res != null);

    if (!found) {
      return null;
    }
    else {
      return prop;
    }
  }

  protected void internalSetProperties(IPropertyMap props)
    throws ResourceException, NotSupportedException, AccessDeniedException, SetPropertiesException {
    getPropertyManager(true).setProperties(props, this);
  }

  protected void internalSetProperties(List propChangeList)
    throws ResourceException, NotSupportedException, AccessDeniedException, SetPropertiesException {
    this.getPropertyManager(true).setProperties(propChangeList, this);
  }

  protected void internalSetProperty(IProperty prop)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    // Must be called because this method is used internally e.g. to set
    // enhanced coll. property
    ResourcePropertyCacheHandler.removeCachedPrefilledProperties(this, prop.getPropertyName());
    this.getPropertyManager(true).setProperty(prop, this);
  }

  protected void internalUpdateContent(IContent newContent)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    ResourcePropertyCacheHandler.clearCachedPrefilledProperties(this);    
    this.getContentManager(true).setContent(newContent, this);
  }

  protected void internalUpdate(IContent newContent, IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException, SetPropertiesException {
    ResourcePropertyCacheHandler.clearCachedPrefilledProperties(this);    
    this.getContentManager(true).setContentAndProperties(newContent, properties, this);
  }

  protected void internalDelete() throws ResourceException, NotSupportedException, AccessDeniedException {
    this.getNamespaceManager(true).delete(this);
  }

  protected IRidList internalDeleteCollection()
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return null;
  }

  protected URL internalGetTargetURL() throws ResourceException {
    return (this.linkType.equals(LinkType.NONE)) ? null : this.targetURL;
  }

  protected LinkType internalGetLinkType() throws ResourceException {
    return this.linkType;
  }

  protected void internalChangePosision(IPosition pos) throws ResourceException {
    this.getNamespaceManager(true).changePosition(pos, this);
  }

  //////////////////////////////////////////////////////////////////////////////

  /**
   * Apply the write filter to the content
   *
   * @param content Description of Parameter
   * @return Description of the Returned Value
   * @exception ResourceException Description of Exception
   */
  protected IContent applyContentWriteFilterForUpdate(IContent content) throws ResourceException {
    try {
      return CmSystem.getInstance().getFilterHandler().applyContentWriteFilterOnUpdate(
        this.rmAdapter.getIRepositoryManagerForClients(),
        this,
        content);
    }
    catch (CmStartupException ex) {
      throw new ResourceException(ex);
    }
  }

  /**
   * Apply read filter to properties
   *
   * @param resource Description of Parameter
   * @param properties Description of Parameter
   * @return Description of the Returned Value
   * @exception ResourceException Description of Exception
   */
  protected IPropertyMap applyPropertyReadFilter(IResource resource, IPropertyMap properties)
    throws ResourceException {
    try {
      return CmSystem.getInstance().getFilterHandler().applyPropertyReadFilter(
        this.rmAdapter.getIRepositoryManagerForClients(),
        resource,
        properties);
    }
    catch (CmStartupException ex) {
      throw new ResourceException(ex);
    }
  }

  /**
   * Apply read filter to a property
   *
   * @param property Description of Parameter
   * @param propName Description of Parameter
   * @return Description of the Returned Value
   * @exception ResourceException Description of Exception
   */
  protected IProperty applyPropertyReadFilter(IProperty property, IPropertyName propName) throws ResourceException {
    try {
      return CmSystem.getInstance().getFilterHandler().applyPropertyReadFilter(
        this.rmAdapter.getIRepositoryManagerForClients(),
        this,
        property,
        propName);
    }
    catch (CmStartupException ex) {
      throw new ResourceException(ex);
    }
  }

  /**
   * Apply read filter to a property name list
   *
   * @param nameList TBD: Description of the incoming method parameter
   * @param properties TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception ResourceException Exception raised in failure situation
   */
  protected IPropertyMap applyPropertyReadFilter(IPropertyNameList nameList, IPropertyMap properties)
    throws ResourceException {
    try {
      return CmSystem.getInstance().getFilterHandler().applyPropertyReadFilter(
        this.rmAdapter.getIRepositoryManagerForClients(),
        this,
        nameList,
        properties);
    }
    catch (CmStartupException ex) {
      throw new ResourceException(ex);
    }
  }

  /**
   * Apply write filter to properties
   *
   * @param properties Description of Parameter
   * @param mode Description of Parameter
   * @return Description of the Returned Value
   * @exception ResourceException Description of Exception
   */
  protected IPropertyMap applyPropertyWriteFilterForUpdate(IPropertyMap properties, PropertyFilterMode mode)
    throws ResourceException {
    try {
      return CmSystem.getInstance().getFilterHandler().applyPropertyWriteFilterOnUpdate(
        this.rmAdapter.getIRepositoryManagerForClients(),
        this,
        properties,
        mode);
    }
    catch (CmStartupException ex) {
      throw new ResourceException(ex);
    }
  }

  /**
   * Apply write filter to property
   *
   * @param property Description of Parameter
   * @param mode Description of Parameter
   * @return Description of the Returned Value
   * @exception ResourceException Description of Exception
   */
  protected IProperty applyPropertyWriteFilterForUpdate(IProperty property, PropertyFilterMode mode)
    throws ResourceException {
    IMutablePropertyMap map = new MutablePropertyMap();
    map.put(property);
    try {
      ResourcePropertyCacheHandler.removeCachedPrefilledProperties(this, map);
      IPropertyMap filteredMap =
        CmSystem.getInstance().getFilterHandler().applyPropertyWriteFilterOnUpdate(
          this.rmAdapter.getIRepositoryManagerForClients(),
          this,
          map,
          mode);
      return filteredMap.get(property.getPropertyName());
    }
    catch (CmStartupException ex) {
      throw new ResourceException(ex);
    }
  }

  /**
   * Apply namespace filter for version history
   *
   * @param resources Description of Parameter
   * @return Description of the Returned Value
   * @exception ResourceException Description of Exception
   */
  protected IVersionHistory applyNamespaceReadFilterForHistory(IVersionHistory resources) throws ResourceException {
    try {
      IResourceList l =
        CmSystem.getInstance().getFilterHandler().applyNamespaceReadFilter(
          getRepositoryManager(),
          null,
          resources,
          NamespaceFilterMode.GET_VERSIONHISTORY);
      if (!(l instanceof IVersionHistory)) {
        throw new ResourceException("Namespace filter did not return an instance of IVersionHistory");
      }
      return (IVersionHistory)l;
    }
    catch (CmStartupException ex) {
      throw new ResourceException(ex);
    }
  }

  protected void checkDeleted() throws ResourceException {
    if (this.deleted) {
      throw new ResourceNotFoundException("Resource is deleted", this.getRID());
    }
  }

  /**
   * Check link target for loops. Will also find a loop if the baseRID resource
   * does not yet exist (before create). This meas that the new link would cause
   * a loop (it is possible to create links to non-existing target resource).
   *
   * @param targetRID The link target to check
   * @param baseRID The RID of this link
   * @return target of link or <code>null</code> if it does not exist
   * @exception ResourceException Description of Exception
   */
  protected final IResource checkLinkTarget(RID targetRID, RID baseRID) throws ResourceException {
    IResource target = null;
    RID current = baseRID;
    RID next = targetRID;
    Set seen = null;
    for (boolean follow = true; follow;) {
      if (!next.isAbsolute()) {
        next = current.resolveRelativeRID(next.getPath());
      }

      follow = false;
      try {
        if (seen != null && seen.contains(next) || next.equals(baseRID)) {
          throw new LinkLoopException("following link " + baseRID + " leads to infinite loop", baseRID, next);
        }
        target = ResourceFactory.getInstance().getResource(next, this.context);
        if (target == null || !target.getLinkType().equals(LinkType.INTERNAL)) {
          break;
        }

        follow = true;
        // we follow the link. memorize the RIDs we have already seen
        current = next;
        next = RID.getRID(target.getTargetURL().toString());
        if (seen == null) {
          seen = new HashSet(7);
        }
        seen.add(current);
      }
      catch (LinkLoopException e) {
        throw e;
      }
      catch (ResourceException ex) {
        // sei: is this a good idea?
        ResourceImpl.log.debugT(LoggingFormatter.extractCallstack(ex));
      }
    }

    return target;
  }

  protected void internalEnableChildAutoVersioning() throws ResourceException {

    if (!this.isCollection() || !LinkType.NONE.equals(this.getLinkType())) {
      throw new NotSupportedException("only supported on collections");
    }

    IVersionController vct = (IVersionController)this.as(IVersionController.class);
    if (vct != null) {
      IResourceList rl = mustSendEvents() ? new ResourceList() : null;
      vct.enable(rl);
      if (mustSendEvents()) {
        for (IResourceListIterator it = rl.listIterator(); it.hasNext();) {
          this.rmAdapter.getOldAbstract().sendEvent(it.next(), ResourceEvent.ENABLE_VERSIONING, null, null);
        }
      }
    }
    else {
      ICollection coll = (ICollection)this;
      IRidSet alreadyEnabled = new RidSet();
      try {
        for (IResourceListIterator it = coll.getChildren().listIterator(); it.hasNext();) {
          IResource res = it.next();
          if (res.isVersioned() && !(res.isCollection() && LinkType.NONE.equals(res.getLinkType()))) {
            alreadyEnabled.add(res.getRID());
          }
        }
      }
      catch (ResourceException ex) {
        log.debugT(
          "internalEnableChildAutoVersioning(3011)",
          "getting currently version-controlled members" + LoggingFormatter.extractCallstack(ex));
      }

      internalSetProperty(P_enhancedCollectionTrue);

      if (mustSendEvents()) {
        try {
          for (IResourceListIterator it = coll.getChildren().listIterator(); it.hasNext();) {
            IResource res = it.next();
            if (res.isVersioned() && !(res.isCollection() && LinkType.NONE.equals(res.getLinkType()))) {
              if (!alreadyEnabled.contains(res.getRID())) {
                this.rmAdapter.getOldAbstract().sendEvent(res, ResourceEvent.ENABLE_VERSIONING, null, null);
              }
            }
          }
        }
        catch (ResourceException ex) {
          log.debugT(
            "internalEnableChildAutoVersioning(3028)",
            "getting currently version-controlled members" + LoggingFormatter.extractCallstack(ex));
        }
      }
    }
  }

  protected void internalDisableChildAutoVersioning() throws ResourceException {

    if (!this.isCollection() && !LinkType.NONE.equals(this.getLinkType())) {
      throw new NotSupportedException("only supported on collections");
    }

    IVersionController vct = (IVersionController)this.as(IVersionController.class);
    if (vct != null) {
      vct.disable(true);
    }
    else {
      internalSetProperty(P_enhancedCollectionFalse);
    }
  }

  protected IResourceList internalSearch(IQueryExpression query, int depth, int maxResults, boolean includeRevision)
    throws ResourceException, NotSupportedException, AccessDeniedException {

    IResourceList result = getPropertySearchManager(true).execute(query, this, depth, maxResults, includeRevision);
    ResourceImpl.mapLink(result, this.getRepositoryManager());
    return result;
  }

  /**
   * Apply namespace filter for search method
   *
   * @param resources Description of Parameter
   * @return Description of the Returned Value
   * @exception ResourceException Description of Exception
   */
  protected IResourceList applyNamespaceReadFilterForSearch(IResourceList resources) throws ResourceException {
    try {
      return CmSystem.getInstance().getFilterHandler().applyNamespaceReadFilter(
        getRepositoryManager(),
        null,
        resources,
        NamespaceFilterMode.SEARCH);
    }
    catch (CmStartupException ex) {
      throw new ResourceException(ex);
    }
  }

  /**
   * Check new resource names (create, rename, ...) for characters forbidden or
   * reserved by the framework or other restrictions
   *
   * @param name Description of Parameter
   * @exception InvalidNameException Description of Exception
   */
  protected void checkNewName(String name) throws InvalidNameException {
    NameInfo ni = this.rmAdapter.getNameInfo();
    if (ni != null) {
      ni.checkName(name);
    }
  }

  // --------------- Event handling -------------------

  protected boolean sendEvent(IFrameworkTransaction ft, int type, Object param) {
    return this.sendEvent(ft, type, null, param);
  }
  
  protected void sendEvent(int type, Object param) {
    this.sendEvent(type, null, param);
  }

  protected boolean sendEvent(IFrameworkTransaction ft, int type, String correlationId, Object param) {
    if (this.mustSendEvents()) {
      if (this.rmAdapter.mustSendResourceEventType(type)) {
        return this.rmAdapter.sendEvent(ft, this, type, correlationId, param);
      }
    }
    return true;
  }
  
  protected IResourceEvent sendEvent(int type, String correlationId, Object param) {
    if (this.mustSendEvents()) {
      if (this.rmAdapter.mustSendResourceEventType(type)) {
        return this.rmAdapter.sendEvent(this, type, correlationId, param);
      }
    }
    return null;
  }

  /**
   * Sends a CREATE_XXX event depending on the type of resource.
   *
   * @param newRes
   * @throws ResourceException
   */
  protected void sendCreateEvent(ResourceImpl newRes) throws ResourceException {
    int ev = 0;
    if (newRes.getLinkType().equals(LinkType.NONE)) {
      if (newRes.isCollection()) {
        ev = ResourceEvent.CREATE_COLLECTION;
      }
      else {
        ev = ResourceEvent.CREATE_CHILD;
      }
    }
    else {
      ev = ResourceEvent.CREATE_LINK;
    }
    newRes.sendEvent(ev, null, newRes);
  }

  /**
   * Send PROPERTY_SET of PROPERTY_DELETE for properties that where set or
   * deleted successfully
   *
   * @param ex Description of Parameter
   * @param props Description of Parameter
   */
  protected final void sendEventsForPropertyChanges(SetPropertiesException ex, IPropertyMap props) {
    sendEventsForPropertyChanges(null, ex, props);
  }

  /**
   * Send PROPERTY_SET of PROPERTY_DELETE for properties that where set or
   * deleted successfully
   *
   * @param ex Description of Parameter
   * @param props Description of Parameter
   * @param correlationId TBD: Description of the incoming method parameter
   */
  protected final void sendEventsForPropertyChanges(
      String correlationId,
      SetPropertiesException ex,
      IPropertyMap props) {
    sendEventsForPropertyChanges(null, correlationId, ex, props);
  }
    
  protected final boolean sendEventsForPropertyChanges(
      IFrameworkTransaction ft,
      String correlationId,
      SetPropertiesException ex,
      IPropertyMap props) {
    if (!mustSendEvents()) {
      return true;
    }
    if ((this.rmAdapter.mustSendResourceEventType(ResourceEvent.PROPERTY_SET))
        || (this.rmAdapter.mustSendResourceEventType(ResourceEvent.PROPERTY_DELETE))) {
      PropertyChangeResultCollection results = ex.getPropertyChangeResults();
      if (results == null) {
        return true;
      }
      PropertyChangeResultIterator it = results.iterator();
      while (it.hasNext()) {
        PropertyChangeResult result = it.next();
        if (result.isOK()) {
          IPropertyName name = result.getPropertyName();
          if (name != null) {
            ResourceEvent initialEvent = null;
            if (props.containsProperty(name)) {              
              if (ft == null)
                initialEvent = (ResourceEvent)sendEvent(ResourceEvent.PROPERTY_SET, correlationId, props.get(name));
              else {
                initialEvent = new ResourceEvent(this, ResourceEvent.PROPERTY_SET, correlationId, props.get(name));
                if (!sendEvent(ft, ResourceEvent.PROPERTY_SET, correlationId, props.get(name))) return false;
              }
              if (correlationId == null) {
                correlationId = initialEvent != null ? initialEvent.getUniqueId() : null;
              }
            }
            else {
              if (ft == null)
                initialEvent = (ResourceEvent)sendEvent(ResourceEvent.PROPERTY_DELETE, correlationId, name);
              else {
                initialEvent = new ResourceEvent(this, ResourceEvent.PROPERTY_DELETE, correlationId, name);
                if (!sendEvent(ft, ResourceEvent.PROPERTY_DELETE, correlationId, name)) return false;
              }
              if (correlationId == null) {
                correlationId = initialEvent != null ? initialEvent.getUniqueId() : null;
              }
            }
          }
        }
      }
    }
    return true;
  }
    
  protected final void sendEventsForPropertyChanges(IPropertyMap beforeMap, IPropertyMap map) {
    sendEventsForPropertyChanges(null, beforeMap, map);
  }

  protected final void sendEventsForPropertyChanges(String correlationId, IPropertyMap beforeMap, IPropertyMap map) {
    sendEventsForPropertyChanges(null, correlationId, beforeMap, map);
  }
  
  protected final boolean sendEventsForPropertyChanges(IFrameworkTransaction ft, String correlationId, IPropertyMap beforeMap, IPropertyMap map) {
    if (!mustSendEvents()) {
      return true;
    }
    if (map == null) {
      return true;
    }
    if ((this.rmAdapter.mustSendResourceEventType(ResourceEvent.PROPERTY_SET))
      || (this.rmAdapter.mustSendResourceEventType(ResourceEvent.PROPERTY_DELETE))) {
      IPropertyIterator it = null;
      if (beforeMap != null) {
        it = beforeMap.iterator();
        while (it.hasNext()) {
          IProperty p = it.next();
          if (!map.containsProperty(p)) {
            ResourceEvent initialEvent = null;
            if (ft == null)
              initialEvent = (ResourceEvent)this.sendEvent(ResourceEvent.PROPERTY_DELETE, correlationId, p);
            else {
              initialEvent = new ResourceEvent(this, ResourceEvent.PROPERTY_DELETE, correlationId, p);
              if (!this.sendEvent(ft, ResourceEvent.PROPERTY_DELETE, correlationId, p)) return false;
            }
            if (correlationId == null) {
              correlationId = initialEvent != null ? initialEvent.getUniqueId() : null;
            }
          }
        }
      }
      it = map.iterator();
      while (it.hasNext()) {
        ResourceEvent initialEvent = null;
        IProperty prop = it.next();
        if (ft == null)
          initialEvent = (ResourceEvent)sendEvent(ResourceEvent.PROPERTY_SET, correlationId, prop);
        else {
          initialEvent = new ResourceEvent(this, ResourceEvent.PROPERTY_SET, correlationId, prop);
          if (!this.sendEvent(ft, ResourceEvent.PROPERTY_SET, correlationId, prop)) return false;
        }
        if (correlationId == null) {
          correlationId = initialEvent != null ? initialEvent.getUniqueId() : null;
        }
      }
    }
    return true;
  }

  protected final void sendEventsForPropertyChanges(List changes) {
    sendEventsForPropertyChanges((String) null, changes);
  }

  protected final void sendEventsForPropertyChanges(String correlationId, List changes) {
    sendEventsForPropertyChanges(null, correlationId, changes);
  }
  
  protected boolean sendEventsForPropertyChanges(IFrameworkTransaction ft, String correlationId, List changes) {
    if (!mustSendEvents()) {
      return true;
    }
    if ((this.rmAdapter.mustSendResourceEventType(ResourceEvent.PROPERTY_SET))
      || (this.rmAdapter.mustSendResourceEventType(ResourceEvent.PROPERTY_DELETE))) {
      Iterator it = changes.listIterator();
      while (it.hasNext()) {
        Object o = it.next();
        ResourceEvent initialEvent = null;
        if (o instanceof IProperty) {          
          if (ft == null)
            initialEvent = (ResourceEvent)this.sendEvent(ResourceEvent.PROPERTY_SET, correlationId, o);
          else {
            initialEvent = new ResourceEvent(this, ResourceEvent.PROPERTY_SET, correlationId, o);
            if (!this.sendEvent(ft, ResourceEvent.PROPERTY_SET, correlationId, o)) return false;
          }
          if (correlationId == null) {
            correlationId = initialEvent != null ? initialEvent.getUniqueId() : null;
          }
        }
        else if (o instanceof IPropertyName) {
          if (ft == null)
            initialEvent = (ResourceEvent)this.sendEvent(ResourceEvent.PROPERTY_DELETE, correlationId, o);
          else {
            initialEvent = new ResourceEvent(this, ResourceEvent.PROPERTY_DELETE, correlationId, o);
            if (!this.sendEvent(ft, ResourceEvent.PROPERTY_DELETE, correlationId, o)) return false;
          }
          if (correlationId == null) {
            correlationId = initialEvent != null ? initialEvent.getUniqueId() : null;
          }
        }
      }
    }
    return true;
  }

  /**
   * Send PROPERTY_SET of PROPERTY_DELETE for properties that where set or
   * deleted successfully
   *
   * @param ex Description of Parameter
   * @param propChangeList Description of Parameter
   */
  protected final void sendEventsForPropertyChanges(SetPropertiesException ex, List propChangeList) {
    sendEventsForPropertyChanges(null, ex, propChangeList);
  }

  /**
   * Send PROPERTY_SET of PROPERTY_DELETE for properties that where set or
   * deleted successfully
   *
   * @param ex Description of Parameter
   * @param propChangeList Description of Parameter
   * @param correlationId TBD: Description of the incoming method parameter
   */
  protected final void sendEventsForPropertyChanges(
      String correlationId,
      SetPropertiesException ex,
      List propChangeList) {
    sendEventsForPropertyChanges(null, correlationId, ex, propChangeList);
  }
    
  protected final boolean sendEventsForPropertyChanges(
      IFrameworkTransaction ft,
      String correlationId,
      SetPropertiesException ex,
      List propChangeList) {
    if (!mustSendEvents()) {
      return true;
    }
    if ((this.rmAdapter.mustSendResourceEventType(ResourceEvent.PROPERTY_SET))
        || (this.rmAdapter.mustSendResourceEventType(ResourceEvent.PROPERTY_DELETE))) {
      PropertyChangeResultCollection results = ex.getPropertyChangeResults();
      if (results == null) {
        return true;
      }
      PropertyChangeResultIterator it = results.iterator();
      while (it.hasNext()) {
        PropertyChangeResult result = it.next();
        if (result.isOK()) {
          IPropertyName name = result.getPropertyName();
          Integer index = result.getPropertyChangeIndex();
          if (index != null) {
            if (name != null && index.intValue() > 0) {
              Object o = propChangeList.get(index.intValue());
              if (o != null) {
                ResourceEvent initialEvent = null;
                if (o instanceof IPropertyName) {
                  if (ft == null)
                    initialEvent = (ResourceEvent)sendEvent(ResourceEvent.PROPERTY_DELETE, correlationId, o);
                  else {
                    initialEvent = new ResourceEvent( this, ResourceEvent.PROPERTY_DELETE, correlationId, o);
                    if (!sendEvent(ft, ResourceEvent.PROPERTY_DELETE, correlationId, o)) return false;
                  }
                  if (correlationId == null) {
                    correlationId = initialEvent != null ? initialEvent.getUniqueId() : null;
                  }
                }
                else if (o instanceof IProperty) {
                  if (ft == null)
                    initialEvent = (ResourceEvent)sendEvent(ResourceEvent.PROPERTY_SET, correlationId, o);
                  else {
                    initialEvent = new ResourceEvent(this, ResourceEvent.PROPERTY_SET, correlationId, o);
                    if (!sendEvent(ft, ResourceEvent.PROPERTY_SET, correlationId, o)) return false;
                  }
                  if (correlationId == null) {
                    correlationId = initialEvent != null ? initialEvent.getUniqueId() : null;
                  }
                }
              }
            }
          }
        }
      }
    }
    return true;
  }

  private final IProperty getDisplayNameProperty() {
    if (this.deleted) {
      return null;
    }
    try {
      return this.getProperty(PropertyName.createDisplayname());
    }
    catch (ResourceException ex) {
      return null;
    }
  }

  private boolean isLockBreaker(IUMPrincipal principal) throws WcmException {
    try {
      ISystemPrincipalList systemPrincipals = SystemConfig.getInstance().getSystemPrincipals();
      ISystemPrincipalListIterator it = systemPrincipals.iterator();
      while (it != null && it.hasNext()) {
        ISystemPrincipal systemPrincipal = it.next();
        if (this.containsPrincipal(systemPrincipal.getPrincipal(), principal)) {
          if (systemPrincipal.isLockBreaker()) {
            return true;
          }
        }
      }
    }
    catch (WcmException e) {
      log.errorT("isLockBreaker(3348)", LoggingFormatter.extractCallstack(e));
    }
    catch (Exception e) {
      log.errorT("isLockBreaker(3351)", LoggingFormatter.extractCallstack(e));
    }
    return false;
  }

  /**
   * Get the resource for a link's target URL
   *
   * @return Description of the Returned Value
   * @exception ResourceException Description of Exception
   */
  private IResource targetResource() throws ResourceException {
    if (isInternalLink()) {
      if ((this.targetResourceCache == null)
        || ((System.currentTimeMillis() - this.targetResourceSet) > TIME_IN_MS_TO_CACHE_TARGETRESOURCE)) {
        this.targetResourceCache = checkLinkTarget(RID.getRID(this.targetURL.toString()), this.rid);
        this.targetResourceSet = System.currentTimeMillis();
      }
      return targetResourceCache;
    }
    return null;
  }

  /*
  * allow subclasses to set the target resource cache
  */
  protected void setTargetResourceCache(IResource targetResource) {
    this.targetResourceCache = targetResource;
    this.targetResourceSet = System.currentTimeMillis();
  }

  /**
   * Ask the manager if standard resource events are switched on/off in its
   * configuration
   *
   * @return Description of the Returned Value
   */
  private boolean mustSendEvents() {
    return this.rmAdapter.eventsEnabled();
  }

  private boolean containsPrincipal(IUMPrincipal p1, IUMPrincipal p2) {
    if (p1.equals(p2)) {
      return true;
    }
    switch (p2.getType()) {
      case IUMPrincipal.IUSER :
        if ((p1.getType() == IUMPrincipal.IGROUP && ((IGroup)p1).containsUser(p2.getId()))
          || (p1.getType() == IUMPrincipal.IROLE && ((IRole)p1).containsUser(p2.getId()))) {
          return true;
        }
        break;
      case IUMPrincipal.IGROUP :
        if ((p1.getType() == IUMPrincipal.IGROUP && ((IGroup)p1).containsGroup(p2.getId()))
          || (p1.getType() == IUMPrincipal.IROLE && ((IRole)p1).containsGroup(p2.getId()))) {
          return true;
        }
        break;
    }
    return false;
  }

  /**
   * Eaxmine the resource if it is a link to a collection.
   *
   * @param resource to examine
   * @param checkManager check if manager needs such examination
   * @return original or mapped resource
   * @exception ResourceException Description of Exception
   */
  final static IResource mapLink(IResource resource, boolean checkManager) throws ResourceException {
    if (resource != null
      && !resource.isCollection()
      && LinkType.INTERNAL.equals(resource.getLinkType())
      && !resource.isRevision()) {
      // Hmmm, could be a link to a collection. Does the manager resolve links?
      //
      if (!checkManager || needsLinkMapping(resource.getRepositoryManager())) {
        IResource target = resource.getTargetResource();
        if (target != null && target instanceof ICollection) {
          return new CollectionLinkImpl(resource, target);
        }
      }
    }
    return resource;
  }

  /**
   * Examines the resources in the list for links to collections. Replaces links
   * to collections with collection resources.
   *
   * @param list of resources
   * @param mgr repository manager resources come from
   */
  final static void mapLink(IResourceList list, IRepositoryManager mgr) {
    if (!(list instanceof ICompletedResourceList) && (needsLinkMapping(mgr))) {
      Set seen = (Set)VisitedResources.get();
      boolean cleanup = seen.isEmpty();
      if (cleanup) {
        seen.add("");
      }

      try {
        for (int i = 0, n = list.size(); i < n; ++i) {
          IResource resource = list.get(i);
          try {
            IResource mapped = mapLink(resource, false);
            if (mapped != resource) {
              list.set(i, mapped);
            }
          }
          catch (ResourceException e) {
            if (log.beDebug()) {
              log.debugT("mapLink(3465)", "mapping resource " + resource + LoggingFormatter.extractCallstack(e));
            }
          }
        }
      }
      finally {
        if (cleanup) {
          seen.clear();
        }
      }
    }
  }

  /**
   * Determine if the manager needs links to collections mapped to collection
   * resources.
   *
   * @param mgr to check
   * @return if manager needs collection link handling
   */
  private static boolean needsLinkMapping(IRepositoryManager mgr) {
    try {
      ISupportedOptionSet options = mgr.getSupportedOptions(null);
      return (options.isSupported(SupportedOption.LINKING) && !options.isSupported(SupportedOption.RESOLVES_LINKS));
    }
    catch (Exception e) {
      log.debugT(
        "needsLinkMapping(3492)",
        "Some manager do not like null arguments here: " + LoggingFormatter.extractCallstack(e));
    }
    return true;
  }

  /**
   * Description of the Class
   */
  public final class Timestamp implements ITimestampedResource {
    private final long ts;

    public Timestamp() {
      this.ts = System.currentTimeMillis();
    }

    public long getTimestamp() {
      return this.ts;
    }
  }

  static {
    try {
      P_enhancedCollectionTrue = new Property(PropertyName.createEnhancedCollection(), Boolean.TRUE);
      P_enhancedCollectionFalse = new Property(PropertyName.createEnhancedCollection(), Boolean.FALSE);
    }
    catch (ResourceException ex) {
      throw new java.lang.RuntimeException(ex.getMessage());
    }
  }

  /**
   * Internal package-private access to the RM adapter instance.
   */
  RMAdapter getRepositoryManagerAdapter() {
    return this.rmAdapter;
  }

  public IPropertyMap getSortProperties() {
    return this.sortProps;
  }

  public void setSortProperties(IPropertyMap props) {
    this.sortProps = props;
  }
}
