/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.repository.batch.*;
import com.sapportals.wcm.repository.manager.*;
import com.sapportals.wcm.repository.runtime.CmFilterHandler;
import com.sap.tc.logging.Location;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import java.util.*;

/**
 * Can be used to call batch operations for a list of resources of one or more repositories.
 */
public final class ResourceBatchImpl implements IResourceBatch {

  private final static Location LOG = Location.getLocation(ResourceBatchImpl.class);

  private IResourceList list;

  /**
   * Constructs a new batch instance. The resource list might contain resources
   * of different repositories. The resource list MUST be the result of a previous call of
   * one of the following methods on the same thread: ICollection.getChildren() or IResourceFactory.getResources().
   * @param resourceList A non-empty resource list.
   * @exception NullPointerException If the parameter is null.
   */
  public ResourceBatchImpl(IResourceList resourceList) {
    if (resourceList == null) {
      throw new NullPointerException("parameter resourceList is null");
    }
    this.list = resourceList;
  }

  /**
   * Returns a map containing the requested properties which are available for the resources in this resource batch.
   * The resulting map will contain resource identifiers ({@link com.sapportals.wcm.util.uri.RID}) as keys and
   * {@link com.sapportals.wcm.repository.IPropertyMap} instances as values.
   * If a resource does not exist no entry will exist in the map for the RID (a resource might no longer exist
   * because it was deleted by another user/thread since ICollection.getChildren() or IResourceFactory.getResources() was called).
   * If a ResourceException occurred for some resources the exceptions will be contained in the (optional) ResourceErrors object.
   * If the ResourceErrors collection contains an exception for a particular RID then this RID will not
   * be contained in the map.
   * If none of the properties exist for a resource the RID is not contained in the map.
   * If none of the properties exist for all of the specified resources (or the resource list is empty)
   * the return value is <code>null</code>.
   * If a property for a specified name does not exist it will not be contained in the IPropertyMap.
   * If a ResourceException is caused by one of the properties it will be appended to
   * the IPropertyMap instance and can be accessed with {@link IPropertyMap#getException(IPropertyName)} or
   * {@link IPropertyMap#getExceptions()}.
   * @param propNameList A list of property names. Must not be <code>null</code>.
   * @param errors Will be used to collect all exceptions which occurred when trying to access the properties
   * of a resource. Can be <code>null</code>.
   * @return a map containing the requested properties which are available for the resources in this resource batch.
   * Can be <code>null</code>.
   * @exception ResourceException
   */
  public Map getProperties(IPropertyNameList propNameList, ResourceErrors errors) throws ResourceException {
    if (this.list.size() == 0) {
      return null;
    }
    // loop over resource list and separate resources of different repositories
    Map rmListMap = new HashMap();
    boolean multipleRep = false;
    for (int i = 0, s = this.list.size(); i < s; i++) {
      IResource r = this.list.get(i);
      String rmID = r.getRepositoryManager().getID();
      IResourceList l = (IResourceList)rmListMap.get(rmID);
      if (l == null) {
        l = new ResourceList();
        rmListMap.put(rmID, l);
      }
      // CollectionLinkImpl wrappers must be replaced because repositories do not know about them and evtl. want
      // to cast the resources
      if (r instanceof CollectionLinkImpl) {
        r = ((CollectionLinkImpl)r).getLinkResourceImpl();
      }
      l.add(r);
    }

    HashMap result = null;
    Set s = rmListMap.entrySet();
    Map rmResult = null;
    Iterator it = s.iterator();
    while (it.hasNext()) {
      try {        
        rmResult = this.getPropertiesFromRM((IResourceList)((Map.Entry)it.next()).getValue(), propNameList, errors);
      }
      catch (ResourceException ex) {
        LOG.errorT(LoggingFormatter.extractCallstack(ex));
        if (errors != null) {
          errors.append(ex);
        }
      }
      if (rmResult != null) {
        if (result == null) {
          result = new HashMap();
        }
        result.putAll(rmResult);
      }
    }
    return result;
  }

  /**
   * Call the managers method and handle default/fallback, filters and events.
   *
   * TODO: ??? Property Caching ???
   */
  private Map getPropertiesFromRM(IResourceList resourceList, IPropertyNameList propNameList, ResourceErrors errors)
    throws ResourceException {
    RMAdapter rma = this.resourceImpl(resourceList.get(0)).getRepositoryManagerAdapter();
    if (rma.isNew()) {
      throw new NotSupportedException("The RF does not yet support this call for repositories using the new MI");
    }
    IPropertyManager pm = rma.getOldAbstract().getPropertyManager(resourceList.get(0));
    Map result = null;
    if (pm instanceof IExtendedPropertyManager2) {
      IPropertyNameList pnl = PropertyNameTool.removeFilteredPropertyNames(
        this.list.get(0).getRepositoryManager().getID(), propNameList);
      result = ((IExtendedPropertyManager2)pm).getProperties(resourceList, pnl, errors);
      // apply property filters and eventing
      if (result != null) {
        for (int i = 0, s = resourceList.size(); i < s; i++) {
          IResource r = resourceList.get(i);
          IPropertyMap props = (IPropertyMap)result.get(r.getRID());
          if (props != null) {
            result.put(r.getRID(), this.resourceImpl(r).applyPropertyReadFilter(propNameList, props));
            this.resourceImpl(r).sendEvent(ResourceEvent.PROPERTY_GET, null, props);
          }
        }
      }
    }
    else {
      result = null;
      IResourceListIterator it = resourceList.listIterator();
      while (it.hasNext()) {
        IResource r = it.next();
        IPropertyMap props = null;
        try {
          props = r.getProperties(propNameList);
        }
        catch (ResourceException ex) {
          if (errors != null) {
            errors.append(ex);
          }
        }
        if (props != null && props.size() > 0) {
          if (result == null) {
            result = new HashMap(resourceList.size());
          }
          result.put(r.getRID(), props);
        }
      }
    }

    return result;
  }

  private ResourceImpl resourceImpl(IResource r) {
    if (r instanceof CollectionLinkImpl) {
      return ((CollectionLinkImpl)r).getLinkResourceImpl();
    }
    else {
      return (ResourceImpl)r;
    }
  }

}