/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;
import com.sap.netweaver.bc.rf.common.*;
import com.sap.netweaver.bc.rf.mi.*;
import com.sap.netweaver.bc.rf.util.logging.*;

import com.sapportals.wcm.repository.manager.*;
import com.sapportals.wcm.repository.runtime.*;
import com.sapportals.wcm.repository.runtime.legacy.*;
import com.sapportals.wcm.repository.wrapping.*;
import com.sapportals.wcm.repository.wrapping.exception.*;
import com.sapportals.wcm.repository.wrapping.property.*;
import com.sap.netweaver.bc.rf.common.context.*;
import com.sap.netweaver.bc.rf.util.context.ContextUnavailableException;
import com.sapportals.wcm.util.uri.*;
import com.sapportals.wcm.IFrameworkTransaction;
import com.sapportals.wcm.WcmException;

import java.util.*;

/**
 * Framework internal access to repository managers.
 */
public final class RMAdapter {

	private AbstractManager newMgr;
	private AbstractRepositoryManager oldMgr;

	public RMAdapter(Object manager) {
		if (manager instanceof AbstractRepositoryManager) {
			this.oldMgr = (AbstractRepositoryManager) manager;
		}
		else if (manager instanceof AbstractManager) {
			this.newMgr = (AbstractManager) manager;
		}
		else {
			throw new java.lang.ClassCastException("invalid manager class: " + manager.getClass().getName());
		}
	}

	public boolean isNew() {
		return (this.newMgr != null);
	}

	public AbstractRepositoryManager getOldAbstract() {
		if (this.oldMgr == null) {
			throw new java.lang.NullPointerException();
		}
		return this.oldMgr;
	}

	public AbstractManager getNewAbstract() {
		if (this.newMgr == null) {
			throw new java.lang.NullPointerException();
		}
		return this.newMgr;
	}

	public IRepositoryManager getIRepositoryManagerForClients() {
		if (this.newMgr != null) {
			return new ManagerAdapter(this.newMgr);
		}
		else {
			return this.oldMgr;
		}
	}

	public String getPrefix() {
		if (this.newMgr != null) {
			return this.newMgr.getRidPrefix();
		}
		else {
			return this.oldMgr.getPrefix();
		}
	}

	public String getID() {
		if (this.newMgr != null) {
			return this.newMgr.getId();
		}
		else {
			return this.oldMgr.getID();
		}
	}

	public ISupportedOptionSet getSupportedOptions(IResource resource) {
		if (this.newMgr != null) {
			IResourceHandle handle = ((ResourceImpl) resource).getHandle();
			return new SupportedOptionSet(
				com.sapportals.wcm.repository.wrapping.support.SupportedOptionTransformer.transform(
					this.newMgr.getSupportedOptions(handle)));
		}
		else {
			return this.oldMgr.getSupportedOptions(resource);
		}
	}

	public NameInfo getNameInfo() {
		if (this.newMgr != null) {
			return NameInfoTransformer.transform(this.newMgr.getNameInfo());
		}
		else {
			return this.oldMgr.getNameInfo();
		}
	}

	public boolean checkExistence(RID rid, IResourceContext context) throws ResourceException {
		if (context == null) {
			throw new ResourceException("No resource context");
		}
		if (rid == null) {
			return false;
		}
		if (this.newMgr != null) {
			// Note:
			// The behaviour regarding InternalLinkException for collection links is different
			// from findResource().
			// This is ignored because the javadoc in IResourceFactory ignores it.
			// Let's hope the clients ignore it too.
			try {
				return (((IManager) this.newMgr).lookup(RidTransformer.transform(rid)) != null);
			}
			catch (com.sap.netweaver.bc.rf.common.exception.ResourceException ex) {
				throw ExceptionTransformer.transform(ex);
			}
		}
		else {
			if (this.oldMgr instanceof IExistenceChecker) {
				return ((IExistenceChecker) this.oldMgr).checkExistence(rid, context);
			}
			else {
				try {
					return (CmAdapter.getInstance().findResource(this, rid, context, false, false) != null);
				}
				catch (ResourceException ex) {
					throw ex;
				}
				catch (Exception ex) {
					throw new ResourceException(ex);
				}
			}
		}
	}

	// --------------------------------------------------------------------------------
	// For methods not in IRepositoryManager interface but in AbstractRepositoryManager
	// we must also check if it is  a ILegacyComponent.
	// --------------------------------------------------------------------------------

	public IResource getResource(RID rid, IResourceContext context) throws ResourceException {

		if (this.newMgr != null) {
			// >= SP6
			IResourceHandle handle = null;
			RMAdapter.createAccessContext(context);
			try {
				handle = this.newMgr.lookup(RidTransformer.transform(rid));
			}
			catch (com.sap.netweaver.bc.rf.common.exception.ResourceException ex) {
				throw ExceptionTransformer.transform(ex);
			}
			catch (Exception exception) {
				throw new ResourceException("Lookup failed: " + LoggingFormatter.extractCallstack(exception));
			}
			if (handle == null) {
				return null;
			}
			else {
				return RMAdapter.createResourceInstance(handle, this.newMgr, context);
			}
		}
		else {
			if (oldMgr instanceof ILegacyComponent) {
				// <= SP4
				AbstractRepositoryManager am = (AbstractRepositoryManager) ((ILegacyComponent) oldMgr).getLegacyComponent();
				return am.getResource(rid, context);
			}
			else {
				// SP5
				return this.oldMgr.getResource(rid, context);
			}
		}
	}

	public ResourceList getResources(
		IRidList ridList,
		Map propertyMaps,
		ResourceErrors errors,
		String[] permissionNames,
		IResourceContext context)
		throws ResourceException {

		if (this.newMgr != null) {
			// >= SP6
			RMAdapter.createAccessContext(context);
			List handles = new ArrayList();
			try {
				// Transform rid list
				List newRidList = new ArrayList();
				for (IRidIterator iter = ridList.listIterator(); iter.hasNext();) {
					RID rid = (RID) iter.next();
					newRidList.add(RidTransformer.transform(rid));
				}

				// Transform property maps
				Map newPropertyMaps = null;
				if (propertyMaps == null || propertyMaps.size() == 0) {
					newPropertyMaps = Collections.EMPTY_MAP;
				}
				else {
					newPropertyMaps = new HashMap(propertyMaps.size());
					for (Iterator iter = propertyMaps.entrySet().iterator(); iter.hasNext();) {
						Map.Entry entry = (Map.Entry) iter.next();
						newPropertyMaps.put(
							RidTransformer.transform((RID) entry.getKey()),
							PropertyTransformer.transform((IPropertyMap) entry.getValue()));
					}
				}

				handles = this.newMgr.lookup(newRidList, new HashMap(), newPropertyMaps, permissionNames);
			}
			catch (com.sap.netweaver.bc.rf.common.exception.OperationNotCompletedException ex) {
				throw IncompleteOperationTransformer.transform(ex);
			}
			catch (com.sap.netweaver.bc.rf.common.exception.ResourceException ex) {
				throw ExceptionTransformer.transform(ex);
			}
			catch (Exception exception) {
				throw new ResourceException("Lookup failed: " + LoggingFormatter.extractCallstack(exception));
			}

			ResourceList resourceList = new ResourceList();
			for (Iterator iter = handles.iterator(); iter.hasNext();) {
				resourceList.add(RMAdapter.createResourceInstance((IResourceHandle) iter.next(), this.newMgr, context));
			}
			return resourceList;
		}
		else {
			if (oldMgr instanceof ILegacyComponent) {
				// <= SP4
				AbstractRepositoryManager am = (AbstractRepositoryManager) ((ILegacyComponent) oldMgr).getLegacyComponent();
        return this.callOldGetResources(am, ridList, propertyMaps, errors, permissionNames, context);
			}
			else {
				// SP5
        return this.callOldGetResources(this.oldMgr, ridList, propertyMaps, errors, permissionNames, context);
			}
		}
	}

  private ResourceList callOldGetResources(
    AbstractRepositoryManager am,
    IRidList ridList,
    Map propertyMaps,
    ResourceErrors errors,
    String[] permissionNames,
    IResourceContext context)
    throws ResourceException {
      
    if (propertyMaps != null) {
      if (permissionNames != null) {
        return am.getResources(ridList, propertyMaps, errors, permissionNames, context); 
      }
      else {
        return am.getResources(ridList, propertyMaps, errors, context);
      }
    }
    else {
      if (permissionNames != null) {
        return am.getResources(ridList, errors, permissionNames, context);
      }
      else {
        return am.getResources(ridList, errors, context);
      }
    }      
  }

  public boolean sendEvent(IFrameworkTransaction ft, IResource res, int type, String correlationId, Object param) {    
    if (this.newMgr != null) {
      return true;
    }
    else {
      if (oldMgr instanceof ILegacyComponent) {
        AbstractRepositoryManager am = (AbstractRepositoryManager) ((ILegacyComponent) oldMgr).getLegacyComponent();
        return am.sendEvent(ft, res, type, correlationId, param);
      }
      else {
        return this.getOldAbstract().sendEvent(ft, res, type, correlationId, param);
      }
    }    
  }
  
	public IResourceEvent sendEvent(IResource res, int type, String correlationId, Object param) {
		if (this.newMgr != null) {
			return this.getNewAbstract().sendEvent(res, type, correlationId, param);
		}
		else {
			if (oldMgr instanceof ILegacyComponent) {
				AbstractRepositoryManager am = (AbstractRepositoryManager) ((ILegacyComponent) oldMgr).getLegacyComponent();
				return am.sendEvent(res, type, correlationId, param);
			}
			else {
				return this.getOldAbstract().sendEvent(res, type, correlationId, param);
			}
		}
	}

	public boolean mustSendResourceEventType(int type) {
		if (this.newMgr != null) {
			return this.getNewAbstract().mustSendResourceEventType(type);
		}
		else {
			if (oldMgr instanceof ILegacyComponent) {
				AbstractRepositoryManager am = (AbstractRepositoryManager) ((ILegacyComponent) oldMgr).getLegacyComponent();
				return am.mustSendResourceEventType(type);
			}
			else {
				return this.getOldAbstract().mustSendResourceEventType(type);
			}
		}
	}

	public boolean eventsEnabled() {
		if (this.newMgr != null) {
			return this.getNewAbstract().eventsEnabled();
		}
		else {
			if (oldMgr instanceof ILegacyComponent) {
				AbstractRepositoryManager am = (AbstractRepositoryManager) ((ILegacyComponent) oldMgr).getLegacyComponent();
				return am.eventsEnabled();
			}
			else {
				return this.getOldAbstract().eventsEnabled();
			}
		}
	}

	/**
	 * Creates the access context (thread local variable) from the resource context.
	 */
	static void createAccessContext(IResourceContext context) {
		AccessContextFactory.getInstance().setContext(
			new AccessContext(
				Thread.currentThread().getName(),
				context.getUser().getUMEUser(),
				context.getLocale(),
				context.getValues(),
				com.sap.netweaver.bc.rf.common.context.AccessContext.DEFAULT_ALLOWED_STALE_DATA_TIMESPAN));
	}

	/**
	 * Creates the access context if it is not already set.
	 */
	static void checkAndCreateAccessContext(IResourceContext context) {
		IAccessContext ac = null;
		try {
			ac = AccessContextFactory.getInstance().getContext();
		}
		catch (ContextUnavailableException e) {
			; // ignored
		}
		if (ac == null) {
			RMAdapter.createAccessContext(context);
		}
	}

	static IResource createResourceInstance(IResourceHandle handle, AbstractManager mgr, IResourceContext context)
		throws ResourceException {

		boolean isCollection = false;
		com.sap.netweaver.bc.rf.mi.namespace.INamespaceManager nsMgr = mgr.getNamespaceManager();
		if (nsMgr != null) {
			try {
				isCollection = nsMgr.isCollection(handle);
			}
			catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
				throw new ResourceException(exception);
			}
		}

		if (isCollection) {
			return new CollectionImpl2(handle, mgr, context);
		}
		else {
			return new ResourceImpl2(handle, mgr, context);
		}
	}

	public String toString() {
		if (this.newMgr != null) {
			return this.newMgr.toString();
		}
		else {
			return this.oldMgr.toString();
		}
	}

	public boolean equals(Object other) {
    if (other == null) {
      return false;
    }
    if (! (other instanceof RMAdapter)) {
      return false;
    }
		if (this.newMgr != null) {
			return this.newMgr.equals(((RMAdapter)other).getNewAbstract());
		}
		else {
			return this.oldMgr.equals(((RMAdapter)other).getOldAbstract());
		}
	}
  
  public int hashCode() {
    if (this.newMgr != null) {
      return this.newMgr.hashCode();
    }
    else {
      return this.oldMgr.hashCode();
    }    
  }
}
