/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import com.sapportals.wcm.repository.runtime.*;
import java.util.*;

/**
 * Contains some PropertyName helper functions.
 */
public class PropertyNameTool {

  private static Set someSystemPropNames = new HashSet();
  static {
    // Displayname: Must not be included because some property filter need to filter it.
    PropertyNameTool.someSystemPropNames.add(PropertyName.createContentLanguage());
    PropertyNameTool.someSystemPropNames.add(PropertyName.createContentLength());
    PropertyNameTool.someSystemPropNames.add(PropertyName.createContentType());
    PropertyNameTool.someSystemPropNames.add(PropertyName.createCreatedBy());
    PropertyNameTool.someSystemPropNames.add(PropertyName.createCreationDate());
    PropertyNameTool.someSystemPropNames.add(PropertyName.createDescription());
    PropertyNameTool.someSystemPropNames.add(PropertyName.createEnhancedCollection());
    PropertyNameTool.someSystemPropNames.add(PropertyName.createETag());
    PropertyNameTool.someSystemPropNames.add(PropertyName.createHidden());
    PropertyNameTool.someSystemPropNames.add(PropertyName.createLastModified());
    PropertyNameTool.someSystemPropNames.add(PropertyName.createLastModifiedBy());
    PropertyNameTool.someSystemPropNames.add(PropertyName.createReadOnly());
    PropertyNameTool.someSystemPropNames.add(PropertyName.createResourceType());
    PropertyNameTool.someSystemPropNames.add(PropertyName.createVersioned());
    PropertyNameTool.someSystemPropNames.add(PropertyName.createCollection());
    PropertyNameTool.someSystemPropNames.add(PropertyName.createLinkType());
  }

  public static IPropertyNameList copyPropertyNameListWithoutSystemPropertyNames(IPropertyNameList propNameList) {
    IPropertyNameList newPropNameList = null;

    for (int i = 0; i < propNameList.size(); i++) {
      if (!PropertyNameTool.someSystemPropNames.contains(propNameList.get(i))) {
        if (null == newPropNameList)
          newPropNameList = new PropertyNameList();
        newPropNameList.add(propNameList.get(i));
      }
    }

    return newPropNameList;
  }

  public static IPropertyMap copyPropertyMapWithoutSystemPropertyNames(
    IPropertyMap propMap,
    IMutablePropertyMap systemProps) {
    IMutablePropertyMap newPropMap = new MutablePropertyMap();

    IPropertyIterator iter = propMap.iterator();

    while (iter.hasNext()) {
      IProperty prop = (IProperty) (iter.next());
      if (!PropertyNameTool.someSystemPropNames.contains(prop.getPropertyName())) {
        newPropMap.put(prop);
      }
      else {
        systemProps.put(prop);
      }
    }

    return newPropMap;
  }

  public static boolean checkIfSystemPropertyName(IPropertyName pn) {
    return PropertyNameTool.someSystemPropNames.contains(pn);
  }

  // special methods for property list smuggling

  /**
   * Used in CollectionImpl.getChildren()
   */
  public static void smuggleListWithoutSystemPropertyNames(IPropertyNameList pnl) {
    if (com.sapportals.wcm.repository.runtime.CmFilterHandler.KMC_SYS_PROP_FILTER) {
      return;
    }
    if (pnl instanceof PropertyNameList) {
      IPropertyNameList nameListWithoutSP = PropertyNameTool.copyPropertyNameListWithoutSystemPropertyNames(pnl);
      ((PropertyNameList)pnl).setListWithoutSystemPropertyNames(nameListWithoutSP);
    }
  }

  /**
   * Used in CmFilterHandler.applyPropertyReadFilter()
   */
  public static IPropertyNameList getSmuggledListWithoutSystemPropertyNames(IPropertyNameList pnl) {
    if (com.sapportals.wcm.repository.runtime.CmFilterHandler.KMC_SYS_PROP_FILTER) {
      return pnl;
    }
    IPropertyNameList result = null;
    if (pnl instanceof PropertyNameList) {
      result = ((PropertyNameList)pnl).getListWithoutSystemPropertyNames();
    }
    if (result == null) {
      result = PropertyNameTool.copyPropertyNameListWithoutSystemPropertyNames(pnl);
    }
    return result;
  }
  
  // methods for namelist handling with filtered properties
  
  private static CmFilterHandler filterHandler;

  private static CmFilterHandler getFilterHandler() throws ResourceException {
    if (PropertyNameTool.filterHandler == null) {
      try {
        PropertyNameTool.filterHandler = CmSystem.getInstance().getFilterHandler();
      }
      catch (CmStartupException ex) {
        throw new ResourceException(ex);
      }
    }
    return PropertyNameTool.filterHandler;
  }

  public static boolean checkFilteredPropertyName(String repID, IPropertyName propName) throws ResourceException {
    Set set = PropertyNameTool.getFilterHandler().getFilteredPropertyNames(repID);
    if (set == null) {
      return false;
    }
    else {
      return set.contains(propName);
    }
  }  
  
  /**
   * Remove properties from the list that will never be persisted in a repository but
   * created by property filters only.
   * If the list has to be modified a copy is created so that existing clients will not break:
   * They might expect/imply that IPropertyNameList is unchanged after the API call
   * although it is not read-only
   */
  public static IPropertyNameList removeFilteredPropertyNames(String repID, IPropertyNameList pnl) throws ResourceException {
    if (pnl.size() == 0) {
      return pnl;
    }
    Set set = PropertyNameTool.getFilterHandler().getFilteredPropertyNames(repID);
    if (set == null) {
      return pnl;
    }
    List newList = null;
    for (int i = pnl.size() - 1; i >= 0; i--) {
      IPropertyName n = pnl.get(i);
      if (set.contains(n)) {
        if (newList == null) {
          newList = new ArrayList(pnl.toList());
        }
        newList.remove(i);
      }
    }
    if (newList != null) {
      return new PropertyNameList(newList);
    }
    else {
      return pnl;
    }
  }
  
}