/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import com.sapportals.wcm.*;
import com.sapportals.wcm.repository.enum.*;
import com.sapportals.wcm.util.content.*;

import java.io.*;
import java.util.*;

/**
 * A WCM link is used to create links to repository resources and URL resources
 * on external servers if a repository does not support links.
 *
 * @author SAP AG
 * @version $Revision:$
 */
public class LinkImpl extends WcmObject implements ILink, IContent {

  private final static String TYPE_INTERNAL = "internal";
  private final static String TYPE_EXTERNAL = "external";

  private final static String SYNTAX_ERROR_MSG = "Error reading link resource";

  private String m_url = null;
  private LinkType m_linkType;
  private String m_encoding = null;

  byte[] m_bytes = null;

  /**
   * Constructs a new LinkImpl
   *
   * @param url The link URL
   * @param linkType The link type
   * @param encoding The ID of an encoding supported by Java. <code>null</code>
   *      if the default encoding is to be used.
   * @exception ResourceException Exception raised in failure situation
   */
  public LinkImpl(String url, LinkType linkType, String encoding)
    throws ResourceException {
    if (linkType.equals(LinkType.NONE)) {
      throw new ResourceException("Invalid link type");
    }
    if (url == null) {
      throw new ResourceException("Invalid parameter: url is null");
    }
    if (encoding != null) {
      m_encoding = encoding;
    }
    else {
      m_encoding = "UTF-8";
    }
    m_url = url;
    m_linkType = linkType;
  }

  /**
   * Constructs a new LinkImpl
   */
  public LinkImpl() { }

  /**
   * Constructs a new LinkImpl from an input stream
   *
   * @param is An input stream to read the link resource content from
   * @param encoding The ID of an encoding supported by Java. <code>null</code>
   *      if the default encoding is to be used.
   * @exception ResourceException Exception raised in failure situation
   * @exception UnsupportedEncodingException Exception raised in failure
   *      situation
   */
  public LinkImpl(InputStream is, String encoding)
    throws ResourceException, UnsupportedEncodingException {
    try {
      if (encoding != null) {
        m_encoding = encoding;
      }
      else {
        m_encoding = "UTF-8";
      }
      BufferedReader in = new BufferedReader(new InputStreamReader(is, m_encoding));

      // Read url line
      String firstLine = in.readLine();
      if (firstLine == null) {
        throw new ResourceException(SYNTAX_ERROR_MSG);
      }
      firstLine = firstLine.trim();
      if (!firstLine.startsWith("url=") || firstLine.length() <= 4) {
        throw new ResourceException(SYNTAX_ERROR_MSG);
      }
      m_url = firstLine.substring(4).trim();

      // Read line with link type
      String secondLine = in.readLine();
      if (secondLine == null) {
        throw new ResourceException(SYNTAX_ERROR_MSG);
      }
      if (!secondLine.startsWith("type=") || secondLine.length() <= 5) {
        throw new ResourceException(SYNTAX_ERROR_MSG);
      }
      String sType = secondLine.substring(5).trim();
      if (sType.equals(TYPE_INTERNAL)) {
        m_linkType = LinkType.INTERNAL;
      }
      else if (sType.equals(TYPE_EXTERNAL)) {
        m_linkType = LinkType.EXTERNAL;
      }
      else {
        throw new ResourceException(SYNTAX_ERROR_MSG);
      }
    }
    catch (ContentException ex) {
      throw new ResourceException(ex);
    }
    catch (IOException ex) {
      throw new ResourceException(ex);
    }

  }

  public String toString() {
    return m_url + " - " + m_linkType.toString();
  }

  public String getURL()
    throws ResourceException {
    return m_url;
  }

  public LinkType getLinkType()
    throws ResourceException {
    return m_linkType;
  }


  public InputStream getInputStream()
    throws ContentException {
    convertToBytes();
    return new ByteArrayInputStream(m_bytes);
  }

  public long getContentLength()
    throws ContentException {
    convertToBytes();
    return m_bytes.length;
  }

  public String getContentType()
    throws ContentException {
    return "text/plain";
  }

  public String getEncoding()
    throws ContentException {
    return m_encoding;
  }

  public void close() { }


  // ------------------------ private ------------------------------

  private void convertToBytes()
    throws ContentException {
    if (m_bytes != null) {
      return;
    }
    StringBuffer buff = new StringBuffer("url=");
    buff.append(m_url);
    buff.append(System.getProperty("line.separator"));
    buff.append("type=");
    if (m_linkType.equals(LinkType.INTERNAL)) {
      buff.append(TYPE_INTERNAL);
    }
    else {
      buff.append(TYPE_EXTERNAL);
    }
    try {
      m_bytes = buff.toString().getBytes(m_encoding);
    }
    catch (UnsupportedEncodingException ex) {
      throw new ContentException(ex.getMessage(), ex);
    }
  }

}
