/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import com.sap.netweaver.bc.rf.common.*;
import com.sap.netweaver.bc.rf.common.namespace.*;
import com.sap.netweaver.bc.rf.common.property.*;

import com.sap.netweaver.bc.rf.mi.content.*;
import com.sap.netweaver.bc.rf.mi.lock.*;
import com.sap.netweaver.bc.rf.mi.namespace.*;
import com.sap.netweaver.bc.rf.mi.property.*;
import com.sap.netweaver.bc.rf.mi.security.*;
import com.sap.netweaver.bc.rf.mi.version.*;

import com.sapportals.portal.security.usermanagement.*;
import com.sapportals.wcm.repository.enum.LinkType;

import com.sapportals.wcm.repository.wrapping.RidTransformer;
import com.sapportals.wcm.repository.wrapping.content.*;
import com.sapportals.wcm.repository.wrapping.exception.*;
import com.sapportals.wcm.repository.wrapping.lock.*;
import com.sapportals.wcm.repository.wrapping.namespace.*;
import com.sapportals.wcm.repository.wrapping.property.*;
import com.sapportals.wcm.util.content.*;
import com.sapportals.wcm.util.uri.*;

import java.util.*;

/**
 * Framework resource implementation for new repository manager interfaces
 * This class exists to avoid duplicate code in ResourceImpl2 and CollectionImpl2.
 * The static methods are called in the common internalXXX methods of ResourceImpl2
 * and CollectionImpl2.
 */
public class GeneralImpl2 {

  // ---------------------------------------------------------------------------
  // General
  // ---------------------------------------------------------------------------

  /**
   * Returns the namespace manager for this resource.
   *
   * @param required throw NotSupportedException if the sub-manager is not
   *      available.
   * @param resourceImpl TBD: Description of the incoming method parameter
   * @return the namespace manager for this resource.
   * @exception NotSupportedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  protected final static INamespaceManager getNamespaceManager2(ResourceImpl resourceImpl, boolean required)
    throws NotSupportedException, ResourceException {
    INamespaceManager nsm = resourceImpl.rmAdapter.getNewAbstract().getNamespaceManager();
    if (required == true && nsm == null) {
      throw new NotSupportedException();
    }
    return nsm;
  }

  /**
   * Returns the lock manager for this resource.
   *
   * @param required throw NotSupportedException if the sub-manager is not
   *      available.
   * @param resourceImpl TBD: Description of the incoming method parameter
   * @return the lock manager for this resource.
   * @exception NotSupportedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  protected final static ILockManager getLockManager2(ResourceImpl resourceImpl, boolean required)
    throws NotSupportedException, ResourceException {
    ILockManager lm = resourceImpl.rmAdapter.getNewAbstract().getLockManager();
    if (required == true && lm == null) {
      throw new NotSupportedException();
    }
    return lm;
  }

  /**
   * Returns the property manager for this resource.
   *
   * @param required throw NotSupportedException if the sub-manager is not
   *      available.
   * @param resourceImpl TBD: Description of the incoming method parameter
   * @return the property manager for this resource.
   * @exception NotSupportedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  protected final static IPropertyManager getPropertyManager2(ResourceImpl resourceImpl, boolean required)
    throws NotSupportedException, ResourceException {
    IPropertyManager pm = resourceImpl.rmAdapter.getNewAbstract().getPropertyManager();
    if (required == true && pm == null) {
      throw new NotSupportedException();
    }
    return pm;
  }

  /**
   * Returns the versioning manager for this resource.
   *
   * @param required throw NotSupportedException if the sub-manager is not
   *      available.
   * @param resourceImpl TBD: Description of the incoming method parameter
   * @return the versioning manager for this resource.
   * @exception NotSupportedException Description of Exception
   * @exception ResourceException Description of Exception
   */
  protected final static IBasicVersioningManager getBasicVersioningManager(ResourceImpl resourceImpl, boolean required)
    throws NotSupportedException, ResourceException {
    IBasicVersioningManager vm = resourceImpl.rmAdapter.getNewAbstract().getBasicVersioningManager();
    if (required == true && vm == null) {
      throw new NotSupportedException();
    }
    return vm;
  }

  /**
   * Returns the security manager for this resource.
   *
   * @param required throw NotSupportedException if the sub-manager is not
   *      available.
   * @param resourceImpl TBD: Description of the incoming method parameter
   * @return the security manager for this resource.
   * @exception NotSupportedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  protected final static ISecurityManager getSecurityManager2(ResourceImpl resourceImpl, boolean required)
    throws NotSupportedException, ResourceException {
    ISecurityManager sm = resourceImpl.rmAdapter.getNewAbstract().getSecurityManager();
    if (required == true && sm == null) {
      throw new NotSupportedException();
    }
    return sm;
  }

  /**
   * Returns the content manager for this resource.
   *
   * @param required throw NotSupportedException if the sub-manager is not
   *      available.
   * @param resourceImpl TBD: Description of the incoming method parameter
   * @return the content manager for this resource.
   * @exception NotSupportedException Description of Exception
   * @exception ResourceException Description of Exception
   */
  protected final static IContentManager getContentManager2(ResourceImpl resourceImpl, boolean required)
    throws NotSupportedException, ResourceException {
    IContentManager cm = resourceImpl.rmAdapter.getNewAbstract().getContentManager();
    if (required == true && cm == null) {
      throw new NotSupportedException();
    }
    return cm;
  }

  protected static void internalRename(ResourceImpl resourceImpl, String newName)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    try {
      INamespaceManager nsm = GeneralImpl2.getNamespaceManager2(resourceImpl, true);
      if (!(nsm instanceof IMutableNamespaceManager)) {
        throw new NotSupportedException();
      }
      else {
        IRid parentIRid = resourceImpl.getHandle().getRid().parent();
        MoveDescriptor desc =
          new MoveDescriptor(
            new com.sap.netweaver.bc.rf.common.namespace.DestinationDescriptor(parentIRid, newName, null, false),
            false);
        ((IMutableNamespaceManager)nsm).moveResource(resourceImpl.getHandle(), desc);
      }
    }
    catch (com.sap.netweaver.bc.rf.common.exception.OperationNotCompletedException ex) {
      throw IncompleteOperationTransformer.transform(ex);
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException ex) {
      throw ExceptionTransformer.transform(ex);
    }
  }

  protected static void internalDeleteProperty(ResourceImpl resourceImpl, IPropertyName propName)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    try {
      ResourcePropertyCacheHandler.removeCachedPrefilledProperties(resourceImpl, propName);
      IPropertyManager pm = GeneralImpl2.getPropertyManager2(resourceImpl, true);
      if (!(pm instanceof IMutablePropertyManager)) {
        throw new NotSupportedException();
      }
      else {
        PropertyRemoveDescriptor desc = new PropertyRemoveDescriptor(PropertyNameTransformer.transform(propName));
        ((IMutablePropertyManager)pm).updateProperty(resourceImpl.getHandle(), desc);
      }
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

  protected static void internalSetTargetURL(ResourceImpl resourceImpl, URL url)
    throws ResourceException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    try {
      INamespaceManager nsm = GeneralImpl2.getNamespaceManager2(resourceImpl, true);
      if (!(nsm instanceof IMutableNamespaceManager)) {
        throw new NotSupportedException();
      }
      else {
        if (resourceImpl.internalGetLinkType().equals(LinkType.INTERNAL)) {
          ((IMutableNamespaceManager)nsm).setLinkDescriptor(
            resourceImpl.getHandle(),
            new com.sap.netweaver.bc.rf.common.namespace.LinkDescriptor(
              com.sap.netweaver.bc.rf.common.namespace.LinkType.INTERNAL_STATIC,
              new Rid(url.toString())));
        }
        else if (resourceImpl.internalGetLinkType().equals(LinkType.EXTERNAL)) {
          ((IMutableNamespaceManager)nsm).setLinkDescriptor(
            resourceImpl.getHandle(),
            new com.sap.netweaver.bc.rf.common.namespace.LinkDescriptor(
              com.sap.netweaver.bc.rf.common.namespace.LinkType.EXTERNAL_STATIC,
              com.sap.netweaver.bc.rf.common.UriFactory.parseUri(url.toString())));
        }
        else {
          throw new ResourceException("Resource is not a link", resourceImpl.getRID());
        }
      }
    }
    catch (java.net.MalformedURLException ex) {
      throw new ResourceException(ex.getMessage());
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

 protected static void internalSetLinkType(ResourceImpl resourceImpl, LinkType linkType)
    throws ResourceException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    try {
      INamespaceManager nsm = GeneralImpl2.getNamespaceManager2(resourceImpl, true);
      if (!(nsm instanceof IMutableNamespaceManager)) {
        throw new NotSupportedException();
      }
      else {
        ILinkDescriptor linkDescriptor = ((IMutableNamespaceManager)nsm).getLinkDescriptor(resourceImpl.getHandle());
        if (linkDescriptor.getTargetRid() != null) {
          linkDescriptor = new LinkDescriptor(LinkTypeTransformer.transform(linkType), linkDescriptor.getTargetRid());
        }
        if (linkDescriptor.getTargetUri() != null) {
          linkDescriptor = new LinkDescriptor(LinkTypeTransformer.transform(linkType), linkDescriptor.getTargetUri());
        }
        ((IMutableNamespaceManager)nsm).setLinkDescriptor(resourceImpl.getHandle(), linkDescriptor);
      }
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

  protected static IProperty internalGetProperty(ResourceImpl resourceImpl, IPropertyName propName)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    //ResourceImpl.PrefilledProperty p = resourceImpl.getCachedPrefilledProperty(propName);
    //if (p != null)
    //  return p.get();
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    try {
      return PropertyTransformer.transform(
        GeneralImpl2.getPropertyManager2(resourceImpl, true).getProperty(
          resourceImpl.getHandle(),
          PropertyNameTransformer.transform(propName)));
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

  protected static IPropertyMap internalGetProperties(ResourceImpl resourceImpl)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    try {
      Map map = GeneralImpl2.getPropertyManager2(resourceImpl, true).getAllProperties(resourceImpl.getHandle());
      if (map == null) {
        return new PropertyMap();
      }
      else {
        return PropertyTransformer.transform(map);
      }
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

  protected static IPropertyMap internalGetProperties(IPropertyNameList propNameList, ResourceImpl resourceImpl)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    IPropertyManager propManager = GeneralImpl2.getPropertyManager2(resourceImpl, true);
    IMutablePropertyMap props = new MutablePropertyMap();
    IPropertyNameList pNameList = new PropertyNameList();

    for (int i = 0; i < propNameList.size(); i++) {
      ResourcePropertyCacheHandler.PrefilledProperty p =
        ResourcePropertyCacheHandler.getCachedPrefilledProperty(resourceImpl, propNameList.get(i));
      if (p != null && p.property != null) {
        props.put(p.property);
      }
      else {
        pNameList.add(propNameList.get(i));
      }
    }

    IPropertyMap uncachedProps = new MutablePropertyMap();

    if (pNameList.size() > 0) {
      try {
        Map map =
          GeneralImpl2.getPropertyManager2(resourceImpl, true).getListedProperties(
            resourceImpl.getHandle(),
            PropertyNameListTransformer.transform(pNameList));
        if (map == null) {
          uncachedProps = new PropertyMap();
        }
        else {
          uncachedProps = PropertyTransformer.transform(map);
        }
      }
      catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
        throw ExceptionTransformer.transform(exception);
      }

      ResourcePropertyCacheHandler.setCachedPrefilledProperties(resourceImpl, uncachedProps);
      IPropertyIterator iter = uncachedProps.iterator();
      while (iter.hasNext()) {
        props.put(iter.next());
      }
    }
    return props;
  }

  protected static void internalSetProperties(ResourceImpl resourceImpl, IPropertyMap props)
    throws ResourceException, NotSupportedException, AccessDeniedException, SetPropertiesException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    try {
      IPropertyManager pm = GeneralImpl2.getPropertyManager2(resourceImpl, true);
      if (!(pm instanceof IMutablePropertyManager)) {
        throw new NotSupportedException();
      }
      else {
        LinkedList updates = new LinkedList();
        IPropertyIterator it = props.iterator();
        while (it.hasNext()) {
          updates.add(new PropertySetDescriptor(PropertyTransformer.transform(it.next())));
        }
        ((IMutablePropertyManager)pm).updateProperties(resourceImpl.getHandle(), updates);
      }
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

  protected static void internalSetProperties(ResourceImpl resourceImpl, List propChangeList)
    throws ResourceException, NotSupportedException, AccessDeniedException, SetPropertiesException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    try {
      IPropertyManager pm = GeneralImpl2.getPropertyManager2(resourceImpl, true);
      if (!(pm instanceof IMutablePropertyManager)) {
        throw new NotSupportedException();
      }
      else {
        LinkedList updates = new LinkedList();
        IPropertyUpdateDescriptor desc = null;
        for (int i = 0, s = propChangeList.size(); i < s; i++) {
          Object o = propChangeList.get(i);
          if (o instanceof IProperty) {
            desc = new PropertySetDescriptor(PropertyTransformer.transform((IProperty)o));
          }
          else if (o instanceof IPropertyName) {
            desc = new PropertyRemoveDescriptor(PropertyNameTransformer.transform((IPropertyName)o));
          }
          else {
            throw new ResourceException("Invalid instance in property change list");
          }
          updates.add(desc);
        }
        ((IMutablePropertyManager)pm).updateProperties(resourceImpl.getHandle(), updates);
      }
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

  protected static void internalSetProperty(ResourceImpl resourceImpl, IProperty prop)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    try {
      // Must be called because this method is used internally e.g. to set
      // enhanced coll. property
      ResourcePropertyCacheHandler.removeCachedPrefilledProperties(resourceImpl, prop.getPropertyName());
      IPropertyManager pm = GeneralImpl2.getPropertyManager2(resourceImpl, true);
      if (!(pm instanceof IMutablePropertyManager)) {
        throw new NotSupportedException();
      }
      else {
        PropertySetDescriptor desc = new PropertySetDescriptor(PropertyTransformer.transform(prop));
        ((IMutablePropertyManager)pm).updateProperty(resourceImpl.getHandle(), desc);
      }
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

  protected static void internalUpdateContent(ResourceImpl resourceImpl, IContent newContent)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    IContentManager cm = GeneralImpl2.getContentManager2(resourceImpl, true);
    if (!(cm instanceof IMutableContentManager)) {
      throw new NotSupportedException();
    }
    else {
      ResourcePropertyCacheHandler.removeCachedPrefilledContentProperties(resourceImpl);
      try {
        ((IMutableContentManager)cm).setContent(
          resourceImpl.getHandle(),
          ContentTransformer.transform(newContent),
          false);
      }
      catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
        throw ExceptionTransformer.transform(exception);
      }
    }
  }

  protected static void internalUpdate(ResourceImpl resourceImpl, IContent newContent, IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException, SetPropertiesException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    IContentManager cm = GeneralImpl2.getContentManager2(resourceImpl, true);
    if (!(cm instanceof IMutableContentManager)) {
      throw new NotSupportedException();
    }
    else {
      if (properties != null) {
        if (properties != null) {
          ResourcePropertyCacheHandler.removeCachedPrefilledProperties(resourceImpl, properties);
        }
        ResourcePropertyCacheHandler.removeCachedPrefilledContentProperties(resourceImpl);
      }
      try {
        ((IMutableContentManager)cm).setContent(
          resourceImpl.getHandle(),
          ContentTransformer.transform(newContent),
          false);
      }
      catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
        throw ExceptionTransformer.transform(exception);
      }
      GeneralImpl2.internalSetProperties(resourceImpl, properties);
    }
  }

  protected static void internalDelete(ResourceImpl resourceImpl)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    try {
      INamespaceManager nsm = GeneralImpl2.getNamespaceManager2(resourceImpl, true);
      if (!(nsm instanceof IMutableNamespaceManager)) {
        throw new NotSupportedException();
      }
      else {
        ((IMutableNamespaceManager)nsm).deleteResource(resourceImpl.getHandle());
      }
    }
    catch (com.sap.netweaver.bc.rf.common.exception.OperationNotCompletedException exception) {
      throw IncompleteOperationTransformer.transform(exception);
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

  protected static URL internalGetTargetURL(ResourceImpl resourceImpl) throws ResourceException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    try {
      if (resourceImpl.internalGetLinkType().equals(LinkType.INTERNAL)) {
        return new URL(
          GeneralImpl2
            .getNamespaceManager2(resourceImpl, true)
            .getLinkDescriptor(resourceImpl.getHandle())
            .getTargetRid()
            .toString());
      }
      else if (resourceImpl.internalGetLinkType().equals(LinkType.EXTERNAL)) {
        return new URL(
          GeneralImpl2
            .getNamespaceManager2(resourceImpl, true)
            .getLinkDescriptor(resourceImpl.getHandle())
            .getTargetUri()
            .toString());
      }
      else {
        return null;
      }
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

  protected static LinkType internalGetLinkType(ResourceImpl resourceImpl) throws ResourceException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    try {
      LinkType lt = LinkType.NONE;
      lt =
        LinkTypeTransformer.transform(
          GeneralImpl2
            .getNamespaceManager2(resourceImpl, true)
            .getLinkDescriptor(resourceImpl.getHandle())
            .getLinkType());
      return lt;
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

  protected static IResource internalCopyManager(
    ResourceImpl resourceImpl,
    RID destinationRID,
    IPosition position,
    ICopyParameter param)
    throws ResourceException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    try {
      INamespaceManager nsm = GeneralImpl2.getNamespaceManager2(resourceImpl, true);
      if (!(nsm instanceof IMutableNamespaceManager)) {
        throw new NotSupportedException();
      }
      else {
        CollectionImpl2 parent =
          (CollectionImpl2)ResourceFactory.getInstance().getResource(
            destinationRID.parent(),
            resourceImpl.getContext());
        CopyFlagsDescriptor copyFlagsDescriptor = new CopyFlagsDescriptor(param.getCopyChildren(), null);
        if (param.getCopyInternalLinks()) {
          copyFlagsDescriptor.addLinkTypeToBeCopied(com.sap.netweaver.bc.rf.common.namespace.LinkType.INTERNAL_STATIC);
          copyFlagsDescriptor.addLinkTypeToBeCopied(
            com.sap.netweaver.bc.rf.common.namespace.LinkType.REPOSITORY_DYNAMIC);
        }
        if (param.getCopyExternalLinks()) {
          copyFlagsDescriptor.addLinkTypeToBeCopied(com.sap.netweaver.bc.rf.common.namespace.LinkType.EXTERNAL_STATIC);
        }
        CopyDescriptor desc =
          new CopyDescriptor(
            new com.sap.netweaver.bc.rf.common.namespace.DestinationDescriptor(
              parent.getHandle().getRid(),
              destinationRID.name().toString(),
              OrderPositionTransformer.transform(position),
              param.getOverwrite()),
            param.getIgnorePropertyFailures(),
            copyFlagsDescriptor);
        List resultList = null;
        try {
          resultList = ((IMutableNamespaceManager)nsm).copyResource(resourceImpl.getHandle(), desc);
        }
        catch (com.sap.netweaver.bc.rf.common.exception.OperationNotCompletedException exception) {
          ((CopyParameter)param).setResourceErrors(IncompleteOperationTransformer.getResourceErrors(exception));
        }
        IResourceHandle newHandle = (IResourceHandle)resultList.get(0);
        ResourceImpl newRes = null;
        if (nsm.isCollection(newHandle)) {
          newRes = new CollectionImpl2(newHandle, resourceImpl.rmAdapter.getNewAbstract(), resourceImpl.getContext());
        }
        else {
          newRes = new ResourceImpl2(newHandle, resourceImpl.rmAdapter.getNewAbstract(), resourceImpl.getContext());
        }
        return newRes;
      }
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

  protected static IResource internalMoveManager(
    ResourceImpl resourceImpl,
    RID destinationRID,
    IPosition position,
    ICopyParameter param)
    throws ResourceException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    try {
      INamespaceManager nsm = GeneralImpl2.getNamespaceManager2(resourceImpl, true);
      if (!(nsm instanceof IMutableNamespaceManager)) {
        throw new NotSupportedException();
      }
      else {
        CollectionImpl2 parent =
          (CollectionImpl2)ResourceFactory.getInstance().getResource(
            destinationRID.parent(),
            resourceImpl.getContext());
        MoveDescriptor desc =
          new MoveDescriptor(
            new com.sap.netweaver.bc.rf.common.namespace.DestinationDescriptor(
              parent.getHandle().getRid(),
              destinationRID.name().toString(),
              OrderPositionTransformer.transform(position),
              param.getOverwrite()),
            param.getIgnorePropertyFailures());
        List resultList = null;
        try {
          resultList = ((IMutableNamespaceManager)nsm).moveResource(resourceImpl.getHandle(), desc);
        }
        catch (com.sap.netweaver.bc.rf.common.exception.OperationNotCompletedException exception) {
          ((CopyParameter)param).setResourceErrors(IncompleteOperationTransformer.getResourceErrors(exception));
        }
        IResourceHandle newHandle = (IResourceHandle)resultList.get(0);
        ResourceImpl newRes = null;
        if (nsm.isCollection(newHandle)) {
          newRes = new CollectionImpl2(newHandle, resourceImpl.rmAdapter.getNewAbstract(), resourceImpl.getContext());
        }
        else {
          newRes = new ResourceImpl2(newHandle, resourceImpl.rmAdapter.getNewAbstract(), resourceImpl.getContext());
        }
        return newRes;
      }
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

  protected static void internalChangePosision(ResourceImpl resourceImpl, IPosition pos) throws ResourceException {    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    try {
      INamespaceManager ns = GeneralImpl2.getNamespaceManager2(resourceImpl, true);
      if (!(ns instanceof IMutableNamespaceManager)) {
        throw new NotSupportedException();
      }
      else {
        LinkedList list = new LinkedList();
        list.add(OrderPositionTransformer.transform(new Positioning(resourceImpl.getName(), pos)));
        ((IMutableNamespaceManager)ns).reorderCollection(
          ((ResourceImpl2)resourceImpl.getParentCollection()).getHandle(),
          list);
      }
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

  protected static ILockInfo internalLock(ResourceImpl resourceImpl, ILockProperties lockProperties)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    ILockManager lm = GeneralImpl2.getLockManager2(resourceImpl, true);
    if (!(lm instanceof IMutableLockManager)) {
      throw new NotSupportedException();
    }
    try {
      return LockInfoTransformer.transform(
        ((IMutableLockManager)lm).lock(resourceImpl.getHandle(), LockDescriptorTransformer.transform(lockProperties)));
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

  protected static void internalUnlock(ResourceImpl resourceImpl, ILockInfo lockInfo)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    ILockManager lm = GeneralImpl2.getLockManager2(resourceImpl, true);
    if (!(lm instanceof IMutableLockManager)) {
      throw new NotSupportedException();
    }
    try {
      ((IMutableLockManager)lm).unlock(resourceImpl.getHandle(), lockInfo.getLockToken());
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

  protected static void internalRefreshLock(ResourceImpl resourceImpl, ILockInfo lockInfo)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    ILockManager lm = GeneralImpl2.getLockManager2(resourceImpl, true);
    if (!(lm instanceof IMutableLockManager)) {
      throw new NotSupportedException();
    }
    try {
      ((IMutableLockManager)lm).refreshLock(resourceImpl.getHandle(), lockInfo.getLockToken());
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

  protected static ILockInfo internalGetLockByToken(ResourceImpl resourceImpl, String lockToken)
    throws ResourceException, NotSupportedException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    ILockManager lm = GeneralImpl2.getLockManager2(resourceImpl, true);
    if (!(lm instanceof IMutableLockManager)) {
      throw new NotSupportedException();
    }
    List list;
    try {
      list = ((IMutableLockManager)lm).getLocks(resourceImpl.getHandle());
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
    if (list == null) {
      throw new LockNotFoundException("Lock not found: " + lockToken + " (no list returned)", resourceImpl.getRID());
    }

    ILockInfoCollection coll = LockInfoTransformer.transformCollection(list);
    ILockInfoIterator it = coll.iterator();
    while (it.hasNext()) {
      ILockInfo li = it.next();
      if (li.getLockToken().equals(lockToken)) {
        return li;
      }
    }

    throw new LockNotFoundException("Lock not found: " + lockToken, resourceImpl.getRID());
  }

  protected static ILockInfoCollection internalGetLocks(ResourceImpl resourceImpl)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    List list;
    try {
      list = GeneralImpl2.getLockManager2(resourceImpl, true).getLocks(resourceImpl.getHandle());
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw new ResourceException(exception);
    }
    if (list == null) {
      return new LockInfoCollection();
    }
    else {
      return LockInfoTransformer.transformCollection(list);
    }
  }

  protected static boolean internalIsLocked(ResourceImpl resourceImpl) throws ResourceException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    ILockManager lockMgr = GeneralImpl2.getLockManager2(resourceImpl, false);
    if (lockMgr == null) {
      return false;
    }
    List locks;
    try {
      locks = lockMgr.getLocks(resourceImpl.getHandle());
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw new ResourceException(exception);
    }
    if (locks != null) {
      return (locks.size() > 0);
    }
    else {
      return false;
    }
  }

  protected static boolean internalIsLockedByMe(ResourceImpl resourceImpl) throws ResourceException {
    RMAdapter.checkAndCreateAccessContext(resourceImpl.getContext());
    ILockManager lockManager = GeneralImpl2.getLockManager2(resourceImpl, false);
    if (null == lockManager) {
      return false;
    }

    List list;
    try {
      list = lockManager.getLocks(resourceImpl.getHandle());
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException ex) {
      throw ExceptionTransformer.transform(ex);
    }
    ListIterator it = list.listIterator();
    boolean lockedByMe = false;
    while (it.hasNext()) {
      com.sap.netweaver.bc.rf.common.lock.ILockInfo li = (com.sap.netweaver.bc.rf.common.lock.ILockInfo)it.next();
      String owner = li.getOwner();
      String userid = null;
      IUser user = resourceImpl.getContext().getUser();
      if (user != null) {
        userid = user.getId();
      }
      else {
        throw new ResourceException("No user ID in context", resourceImpl.getRID());
      }

      if (owner != null && userid != null) {
        if (owner.equals(userid)) {
          lockedByMe = true;
          break;
        }
      }
    }
    return lockedByMe;
  }

  protected static boolean internalIsCheckedOut(ResourceImpl resource) throws ResourceException {
    RMAdapter.checkAndCreateAccessContext(resource.getContext());
    IBasicVersioningManager vmgr = GeneralImpl2.getBasicVersioningManager(resource, false);
    try {
      return (vmgr != null) ? vmgr.isCheckedOut(resource.getHandle()) : false;
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException ex) {
      throw ExceptionTransformer.transform(ex);
    }
  }

  protected static ICheckOutInfo internalCheckOut(ResourceImpl resource)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resource.getContext());
    try {
      IBasicVersioningManager vmgr = GeneralImpl2.getBasicVersioningManager(resource, false);
      if (!(vmgr instanceof IMutableBasicVersioningManager)) {
        throw new NotSupportedException();
      }
      else {
        com.sap.netweaver.bc.rf.common.version.ICheckOutInfo checkOutInfo =
          ((IMutableBasicVersioningManager)vmgr).checkOutInPlace(resource.getHandle(), false);
        return new com.sapportals.wcm.repository.CheckOutInfo2(
          RidTransformer.transform(checkOutInfo.getExpectedRid()),
          checkOutInfo.getExpectedRevisionId(),
          RidTransformer.transform(checkOutInfo.getWorkingRid()));
      }
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
  }

  protected static void internalUndoCheckOut(ResourceImpl resource)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resource.getContext());
    IBasicVersioningManager vmgr = GeneralImpl2.getBasicVersioningManager(resource, false);
    if (!(vmgr instanceof IMutableBasicVersioningManager)) {
      throw new NotSupportedException();
    }
    else {
      try {
        ((IMutableBasicVersioningManager)vmgr).undoCheckOut(resource.getHandle());
      }
      catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
        throw ExceptionTransformer.transform(exception);
      }
    }
  }

  protected static ICheckInInfo internalCheckIn(
    ResourceImpl resource,
    IContent newContent,
    IPropertyMap properties,
    boolean ignorePropertyFailures,
    RID expectedCheckInRID)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resource.getContext());
    IBasicVersioningManager vmgr = GeneralImpl2.getBasicVersioningManager(resource, false);
    if (!(vmgr instanceof IMutableBasicVersioningManager)) {
      throw new NotSupportedException();
    }
    else {
      ResourcePropertyCacheHandler.setCachedPrefilledProperties(resource, null);
      if (properties != null) {
        GeneralImpl2.internalSetProperties(resource, properties);
      }
      if (newContent != null) {
        GeneralImpl2.internalUpdateContent(resource, newContent);
      }
      com.sap.netweaver.bc.rf.common.version.ICheckInInfo checkInInfo = null;
      try {
        checkInInfo =
          ((IMutableBasicVersioningManager)vmgr).checkIn(
            resource.getHandle(),
            RidTransformer.transform(expectedCheckInRID),
            false,
            false);
      }
      catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
        throw ExceptionTransformer.transform(exception);
      }

      if (checkInInfo != null) {
        if (checkInInfo.getRevisionId() != null) {
          resource.revID = checkInInfo.getRevisionId();
        }
        return new com.sapportals.wcm.repository.CheckInInfo2(
          RidTransformer.transform(checkInInfo.getRid()),
          checkInInfo.getRevisionId(),
          null);
      }
      else {
        return null;
      }
    }
  }

  protected static IVersionHistory internalGetVersionHistory(ResourceImpl resource)
    throws ResourceException, NotSupportedException {
    RMAdapter.checkAndCreateAccessContext(resource.getContext());
    INamespaceManager nsMgr = GeneralImpl2.getNamespaceManager2(resource, true);
    IBasicVersioningManager vm = GeneralImpl2.getBasicVersioningManager(resource, true);

    List list = null;
    IVersionHistory vh = null;
    try {
      vm.getVersionHistory(resource.getHandle());

      vh = new VersionHistory();
      for (int i = 0, s = list.size(); i < s; i++) {
        IResourceHandle h = (IResourceHandle)list.get(i);
        if (nsMgr.isCollection(h)) {
          vh.add(new CollectionImpl2(h, resource.rmAdapter.getNewAbstract(), resource.getContext()));
        }
        else {
          vh.add(new ResourceImpl2(h, resource.rmAdapter.getNewAbstract(), resource.getContext()));
        }
      }
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }
    return vh;
  }

  protected static IResourceList internalGetCheckedOutResources(ResourceImpl resource)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resource.getContext());
    IBasicVersioningManager vm = GeneralImpl2.getBasicVersioningManager(resource, true);
    if (!(vm instanceof IWorkspaceManager)) {
      throw new NotSupportedException();
    }
    try {
      ((IWorkspaceManager)vm).getCheckedOutResources(resource.getHandle());
    }
    catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
      throw ExceptionTransformer.transform(exception);
    }

    IResourceList result = new ResourceList();
    /*
     * IMultiResultIterator it = ms.getIterator();
     * while (it.hasNext()) {
     * try {
     * IResourceHandle h = (IResourceHandle)it.getNext();
     * if (nsMgr.isCollection(h)) {
     * result.add(new CollectionImpl2(h, resourceImpl.rmAdapter.getNewAbstract(), resourceImpl.getContext()));
     * }
     * else {
     * result.add(new ResourceImpl2(h, resourceImpl.rmAdapter.getNewAbstract(), resourceImpl.getContext()));
     * }
     * }
     * catch (Throwable ignored) {
     * }
     * }
     */
    return result;
  }

  protected static void internalEnableVersioning(ResourceImpl resource, boolean enable)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    RMAdapter.checkAndCreateAccessContext(resource.getContext());
    IBasicVersioningManager vmgr = GeneralImpl2.getBasicVersioningManager(resource, false);
    if (!(vmgr instanceof IMutableBasicVersioningManager)) {
      throw new NotSupportedException();
    }
    else {
      try {
        ((IMutableBasicVersioningManager)vmgr).setVersionControlEnabled(resource.getHandle(), true);
      }
      catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
        throw ExceptionTransformer.transform(exception);
      }
    }
  }

  protected static boolean internalIsVersioned(ResourceImpl resource) throws ResourceException {
    RMAdapter.checkAndCreateAccessContext(resource.getContext());
    IBasicVersioningManager vm = GeneralImpl2.getBasicVersioningManager(resource, false);
    if (vm == null) {
      return false;
    }
    else {
      try {
        return vm.isVersionControlEnabled(resource.getHandle());
      }
      catch (com.sap.netweaver.bc.rf.common.exception.ResourceException exception) {
        throw ExceptionTransformer.transform(exception);
      }
    }
  }

}
