/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: FrameworkTransaction.java,v 1.1 2005/01/21 11:49:20 mbo Exp $
 */
package com.sapportals.wcm.repository;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.sap.tc.logging.Location;
import com.sapportals.wcm.IFrameworkTransaction;
import com.sapportals.wcm.ITransactionEndsListener;
import com.sapportals.wcm.util.jdbc.connectionpool.JDBCConnectionPoolManager;


/**
 * This class handles framework wide transactions.
 * <p>
 * Copyright (c) SAP AG 2004
 * @author martin.boettcher@greenbytes.de
 */
public class FrameworkTransaction implements IFrameworkTransaction {
  private static final Location log = Location.getLocation(FrameworkTransaction.class);
  
  private final static boolean useJ2EE = false;
  
  protected Connection con = null;  
  private Map objects;
  private List listeners;
  protected IFrameworkTransaction orgTransaction = null;  
  private boolean veto = false;
  private static final ThreadLocal transactionContext = new ThreadLocal();
  
  public static IFrameworkTransaction supports() {
    return (FrameworkTransaction)transactionContext.get();
  }
  
  public static IFrameworkTransaction required() throws ResourceException {
    if (useJ2EE) {
      return new J2EETransaction();
    }
    else {
      return new FrameworkTransaction();
    }
  }
  
  protected FrameworkTransaction() throws ResourceException {
    if (log.beDebug()) log.debugT("creating framework transaction...");
    Object o = transactionContext.get();
    if (o == null) {        
      transactionContext.set(this);
      if (log.beDebug()) log.debugT("created framework transaction");        
    }
    else  {      
      this.orgTransaction = (IFrameworkTransaction)o;       
      if (log.beDebug()) log.debugT("created nested framework transaction");
    }    
    
    this.objects = new HashMap();
    this.listeners = new ArrayList();            
  }
  
  protected void invalidate() {
    transactionContext.set(null);
  }
  
  /* (non-Javadoc)
   * @see com.sapportals.wcm.repository.IFrameworkTransaction#getConnection()
   */
  public Connection getConnection() {
    synchronized(this) {
      if (this.con == null) {
        if (this.orgTransaction != null) {
          this.con = this.orgTransaction.getConnection();
        }
        else {
          try{
            this.con = JDBCConnectionPoolManager.getInstance().getPool(null).getConnection();
            if (this.con.getAutoCommit()) this.con.setAutoCommit(false);
            if (this.con.getTransactionIsolation() != Connection.TRANSACTION_READ_COMMITTED) this.con.setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);
          }
          catch(SQLException e) {
            log.errorT("Can't create connection for framework transaction - " + e);
          }
        }
      }
    }
    return this.con;
  }    

  /* (non-Javadoc)
   * @see com.sapportals.wcm.repository.IFrameworkTransaction#addTransactionObject(java.lang.String, java.lang.Object)
   */
  public void addTransactionObject(Object key, Object o) {
    if (this.orgTransaction == null) {
      synchronized(this.objects) {
        this.objects.put(key, o);
      }
    }
    else
      this.orgTransaction.addTransactionObject(key, o);
  }

  /* (non-Javadoc)
   * @see com.sapportals.wcm.repository.IFrameworkTransaction#getTransactionObject(java.lang.String)
   */
  public Object getTransactionObject(Object key) {
    if (this.orgTransaction == null) {
      synchronized(this.objects) {
        return this.objects.get(key);
      }
    }
    else
      return this.orgTransaction.getTransactionObject(key);
  }

  /* (non-Javadoc)
   * @see com.sapportals.wcm.repository.IFrameworkTransaction#addEndListener(com.sapportals.wcm.repository.ITransactionEndsListener)
   */
  public void addEndListener(ITransactionEndsListener listener) throws ResourceException {
    if (this.orgTransaction == null) {
      synchronized(this.listeners) {
        this.listeners.add(listener);
      }
    }
    else
      this.orgTransaction.addEndListener(listener);
  }

  public ITransactionEndsListener getEndListener(String key) {
    if (this.orgTransaction != null) {
      return this.orgTransaction.getEndListener(key);
    }
    
    for (int i = 0; i < this.listeners.size(); i++) {
      ITransactionEndsListener l = (ITransactionEndsListener) this.listeners.get(i);
      if (key.equals(l.getKey())) {
        return l;
      }
    }
    return null;
  }
  
  /* (non-Javadoc)
   * @see com.sapportals.wcm.repository.IFrameworkTransaction#addEndListener(com.sapportals.wcm.repository.ITransactionEndsListener)
   */
  public boolean isEndListenerRegistered(ITransactionEndsListener listener) {
    if (this.orgTransaction == null) {
      return this.listeners.contains(listener);
    }
    else
      return this.orgTransaction.isEndListenerRegistered(listener);
  }

  protected boolean closeConnection(boolean commit) {        
    if (this.orgTransaction != null) return !this.veto;
    
    boolean ok = true;
    boolean inform = true;
    try{
      this.invalidate();
      
      if (commit && this.con != null) {
        this.con.commit();
        this.con.close();
        this.con = null;        
      }
    }
    catch(Exception e) {
      ok = false;      
    }
    finally{
      if (this.con != null) {
        try{
          this.con.rollback();
        }
        catch(SQLException e) {
          if (log.beWarning()) log.warningT("can't rollback connection." + e.getMessage());
        }
        try{
          this.con.close();
        }
        catch(SQLException e) {
          if (log.beWarning()) log.warningT("can't close connection." + e.getMessage());
        }
        this.con = null;
        ok = false;
      }
    }
    
    if (inform) {
      synchronized(this.listeners) {
        for (int i=0; i<this.listeners.size(); i++) {
          ITransactionEndsListener l = (ITransactionEndsListener)this.listeners.get(i);
          if (ok)
            l.wasCommitted(this);
          else
            l.wasRolledback(this);                    
        }
      }    
    }
    return ok;
  }
  
  /* (non-Javadoc)
   * @see com.sapportals.wcm.repository.IFrameworkTransaction#isCommitted()
   */
  public boolean commit() {
    return closeConnection(!this.willBeRollbacked());                    
  }

  /* (non-Javadoc)
   * @see com.sapportals.wcm.repository.IFrameworkTransaction#isRollbacked()
   */
  public void rollback() {    
    closeConnection(false);    
  }
  
  /* (non-Javadoc)
   * @see com.sapportals.wcm.repository.IFrameworkTransaction#setVeto()
   */
  public void setVeto() {    
    this.veto = true;
    if (this.orgTransaction != null) this.orgTransaction.setVeto();        
  }
  
  /* (non-Javadoc)
   * @see com.sapportals.wcm.repository.IFrameworkTransaction#willBeRollbacked()
   */
  public boolean willBeRollbacked() {
    boolean ret = this.veto;
    return ret;    
  }

  public String toString() {
    if (this.orgTransaction != null) return this.orgTransaction.toString();
    return super.toString();
  }
  
}
