/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import com.sapportals.wcm.IFrameworkTransaction;
import com.sapportals.wcm.repository.enum.LinkType;
import com.sapportals.wcm.repository.manager.IExtendedNamespaceManager;
import com.sapportals.wcm.repository.manager.IExtendedVersioningManager;
import com.sapportals.wcm.util.uri.IUri;
import com.sapportals.wcm.util.uri.RID;

/**
 * extended collection resource object impl. <p>
 *
 * Copyright (c) SAP AG 2002-2003
 *
 * @author julian.reschke@greenbytes.de
 * @version $Id: ExtendedCollectionImpl.java,v 1.7 2003/02/17 14:24:04 jre Exp $
 */

public final class ExtendedCollectionImpl implements IExtendedCollection {

  private final ICollection collection;
  private final IExtendedNamespaceManager mgr;
  private final IExtendedVersioningManager vmgr;

  public ExtendedCollectionImpl(ICollection collection, IExtendedNamespaceManager mgr, IExtendedVersioningManager vmgr)
    throws ResourceException {
    this.collection = collection;
    this.mgr = mgr;
    this.vmgr = vmgr;
  }

  public IResource createExternalLink(String name, IUri absoluteTargetUri, boolean isPermanent,
    IPosition position, IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException {

    IFrameworkTransaction ft = FrameworkTransaction.required();
    try
    {
      IResource res = this.mgr.createExternalLink(name, absoluteTargetUri, isPermanent, position,
        properties, ignorePropertyFailures, this.collection);
      ft.commit();
      ft = null;
      return res;
    }
    finally {
      if (ft != null) ft.rollback();
    }
  }

  public IResource createInternalDynamicLink(String name, RID targetRid, IPosition position,
    IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException {

    IFrameworkTransaction ft = FrameworkTransaction.required();
    try
    {
      IResource res = this.mgr.createInternalDynamicLink(name, targetRid, position, properties,
        ignorePropertyFailures, this.collection);
      ft.commit();
      ft = null;
      return res;
    }
    finally {
      if (ft != null) ft.rollback();
    }
  }

  public IResource createInternalStaticLink(String name, RID targetRid, boolean isPermanent,
    IPosition position, IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException {

    IFrameworkTransaction ft = FrameworkTransaction.required();
    try
    {
      IResource res = this.mgr.createInternalStaticLink(name, targetRid, isPermanent,
        position, properties, ignorePropertyFailures, this.collection);
      ft.commit();
      ft = null;
      return res;
    }
    finally {
      if (ft != null) ft.rollback();
    }
  }

  public ICollection createWorkspace(String name, IPosition position)
    throws ResourceException {
    IFrameworkTransaction ft = FrameworkTransaction.required();
    try
    {
      ICollection col = this.vmgr.createWorkspace(name, position, this.collection);
      ft.commit();
      ft = null;
      return col;
    }
    finally {
      if (ft != null) ft.rollback();
  }
  }

  public IResource createVersionControlledResource(RID baseRevisionRid, String name, IPosition position)
    throws ResourceException {
    IFrameworkTransaction ft = FrameworkTransaction.required();
    try
    {
      IResource res = this.vmgr.createVersionControlledResource(baseRevisionRid, name, position, this.collection);
      ft.commit();
      ft = null;
      return res;
    }
    finally {
      if (ft != null) ft.rollback();
    }
  }

  public IResourceList getChildren(Boolean visible, Boolean collections, Boolean links)
    throws ResourceException, AccessDeniedException {
    try {
      return this.mgr.getChildren(visible, collections, links, this.collection);
      // TODO: Event !
    }
    catch (NotSupportedException ex) {
      IResourceList children = this.collection.getChildren();
      for (int i = children.size() - 1; i >= 0; i--) {
        IResource child = children.get(i);
        IProperty hiddenProp = child.getProperty(PropertyName.createHidden());
        boolean hidden = (hiddenProp != null && hiddenProp.getBooleanValue()) ? true : false;
        boolean isLink = (child.getLinkType() != null && !child.getLinkType().equals(LinkType.NONE)) ? true : false;

        if (visible != null && (visible.booleanValue() && hidden || !visible.booleanValue() && !hidden) ||
          collections != null && (collections.booleanValue() && !child.isCollection() || !collections.booleanValue() && child.isCollection()) ||
          links != null && (links.booleanValue() && !isLink || !links.booleanValue() && isLink)) {
          children.remove(i);
        }
      }
      return children;
    }
  }
}
