/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Date;

import com.sap.tc.logging.Location;
import com.sapportals.wcm.repository.manager.IContentManager;
import com.sapportals.wcm.repository.manager.ResourceEvent;
import com.sapportals.wcm.repository.runtime.CmStartupException;
import com.sapportals.wcm.repository.runtime.CmSystem;
import com.sapportals.wcm.util.content.ContentException;
import com.sapportals.wcm.util.content.IContent;
import com.sapportals.wcm.util.content.IExtendedContent;
import com.sapportals.wcm.util.logging.LoggingFormatter;

/**
 * Access to the content data of a resource. <p>
 *
 * Copyright (c) SAP AG 2001-2004
 * @author m.breitenfelder@sap.com
 * @author julian.reschke@greenbytes.de
 * @author stefan.eissing@greenbytes.de
 * @version $Revision: 1.4 $
 */
public class ContentImpl implements IContent, IExtendedContent, IEntityMetadata {

	private final static Location LOG = Location.getLocation(ContentImpl.class);

	private final static Content EMPTY = new Content(new ByteArrayInputStream(new byte[0]), "", 0L);

	/**
	 * Reference to the resource object that contains this content object
	 */
	protected final ResourceImpl resource;

	/**
	 * Apply read filters to the content
   * "true": call all filters, "false": call only readonly-filters
	 */
	private final boolean notUnfiltered;

	/**
	 * Cached content object received from the NamespaceManager
	 */
	private IContent content = null;

	private boolean deliveredStream = false;

	/**
	 * Constructs a new content instance
	 * @param resource The resource instance the content belongs to
	 * @param notUnfiltered "true": call all filters, "false": call only readonly-filters
	 */
	public ContentImpl(ResourceImpl resource, boolean notUnfiltered) throws ResourceException {
		this.resource = resource;
		this.notUnfiltered = notUnfiltered;
    // julian.reschke@greenbytes.de: init content right away to make sure that
    // the caller has the necessary access rights
    this.internalGetContent();
	}

	//////////////////////////////////////////////////////////////////////////////
	// IContent interface
	//////////////////////////////////////////////////////////////////////////////

	public final InputStream getInputStream(String encoding) throws ContentException {
		this.resource.checkDeleted();

		InputStream is = null;
		// Clear cached content object => initContent will call content manager
		if (this.deliveredStream) {
			this.content = null;
      this.internalContent = null;
			this.deliveredStream = false;
		}

		this.initContent(true);

		is =
			(encoding != null && this.content instanceof IExtendedContent)
				? ((IExtendedContent) this.content).getInputStream(encoding)
				: this.content.getInputStream();

		this.deliveredStream = true;
		this.resource.sendEvent(ResourceEvent.GET, null, null);
		return is;
	}

	public final InputStream getInputStream() throws ContentException {
		return this.getInputStream(null);
	}

	public final long getContentLength()
		throws ContentException, NotSupportedException, AccessDeniedException, NoContentException {
		this.resource.checkDeleted();
		return this.initContent(false).getContentLength();
	}

	public final String getContentType()
		throws ContentException, NotSupportedException, AccessDeniedException, NoContentException {
		this.resource.checkDeleted();
		return this.initContent(false).getContentType();
	}

	public final String getEncoding()
		throws ContentException, NotSupportedException, AccessDeniedException, NoContentException {
		this.resource.checkDeleted();
		return initContent(false).getEncoding();
	}

	public final void close() {
		if (this.content != null) {
			this.content.close(); // Will close input stream
			this.content = null;            
		}
		this.internalContent = null;
	}

	// IEntityMetadata interface

	public String getEntityTag() throws ResourceException {
		this.resource.checkDeleted();
		try {
			initContent(false);
			return this.content instanceof IEntityMetadata ? ((IEntityMetadata) this.content).getEntityTag() : null;
		}
		catch (ResourceException ex) {
			throw ex;
		}
		catch (ContentException ex) {
			throw new ResourceException(ex);
		}
	}

	public java.util.Date getExpiryDate() throws ResourceException {
		this.resource.checkDeleted();
		try {
			initContent(false);
			return this.content instanceof IEntityMetadata ? ((IEntityMetadata) this.content).getExpiryDate() : null;
		}
		catch (ResourceException ex) {
			throw ex;
		}
		catch (ContentException ex) {
			throw new ResourceException(ex);
		}
	}

	public String getLanguage() throws ResourceException {
		this.resource.checkDeleted();
		try {
			initContent(false);
			return this.content instanceof IEntityMetadata ? ((IEntityMetadata) this.content).getLanguage() : null;
		}
		catch (ResourceException ex) {
			throw ex;
		}
		catch (ContentException ex) {
			throw new ResourceException(ex);
		}
	}

	public java.util.Date getLastModified() throws ResourceException {
		this.resource.checkDeleted();
		try {
			initContent(false);
			return this.content instanceof IEntityMetadata ? ((IEntityMetadata) this.content).getLastModified() : null;
		}
		catch (ResourceException ex) {
			throw ex;
		}
		catch (ContentException ex) {
			throw new ResourceException(ex);
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//  private / protected
	//////////////////////////////////////////////////////////////////////////////

  private IContent internalContent = null;

  /**
   * Gets the underlying content object. Treats null content and NoContentException
   * as empty content. Keeps content object until close() is called explicitly to
   * avoid re-getting the content from the RM multiple times.
   */
  protected IContent internalGetContent() throws ResourceException {

    if (internalContent == null) {
      try{
        internalContent = this.resource.getContentManager(true).getContent(this.resource);
      }    
      catch (NoContentException ex) {
        // Handle this case like manager returned null, see below
        if (LOG.beDebug()) {
          LOG.debugT(ex.getMessage());
        }
      }
      if (internalContent == null) {
        internalContent = ContentImpl.EMPTY;
      }
    }
    return internalContent;
  }
  
  /**
   * Call {@link IContentManager#getContent(IResource)}, apply content read filter and cache the
   * content object to avoid filtering the content on each call to
   * getContentLength() or getContentType().
   * @param willGetInputStream will need to be set to <code>true</code> if subsequently, the input stream will be fetched
   */
  private IContent initContent(boolean willGetInputStream) throws ContentException {
    if (this.content == null) {
      
      this.content = this.internalGetContent();
      
      // if we will need the input stream later, get it now (to take
      // care of IContent's specific API regarding the allowed calling
      // sequences for getInputStream and obtaining metadata).
      if (willGetInputStream && this.content != ContentImpl.EMPTY) {
        // if this fails with a ContentException, the request on the wrapped content object will fail as well
        // (except for special cases such as isVarying). This is by design.
        this.content = new ContentWithPrefetchedInputStream(this.content);
      }

      if (this.content != ContentImpl.EMPTY) {
        this.content = this.applyContentReadFilter(this.content);
      }
      
    }
    return this.content;
  }

	/**
	 * Apply the read filter to the content.
	 * 
	 * @param content
	 *          content to apply filter to
	 * @return filtered content
	 * @exception ResourceException
	 *              Exception raised in failure situation
	 */
	private IContent applyContentReadFilter(IContent content) throws ResourceException {
		try {
			return CmSystem.getInstance().getFilterHandler().applyContentReadFilter(
				this.resource.getRepositoryManager(),
				this.resource,
				content,
				!this.notUnfiltered);
		}
		catch (CmStartupException ex) {
			throw new ResourceException(ex);
		}
	}
  
  /**
   * Wrapper for {@link IContent} that holds a prefetched {@link InputStream} that will be handed out
   * for each call to {@link IContent#getInputStream()}. 
   */
  private static class ContentWithPrefetchedInputStream implements IContent, IEntityMetadata {
    
    private final IContent content;
    private final InputStream is;
    
    public ContentWithPrefetchedInputStream(IContent content) throws ContentException {
      this.content = content;
      this.is = content.getInputStream();
    }

    public InputStream getInputStream() throws ContentException {
      return this.is;
    }

    public long getContentLength() throws ContentException {
      return this.content.getContentLength();
    }

    public String getContentType() throws ContentException {
      return this.content.getContentType();
    }

    public String getEncoding() throws ContentException {
      return this.content.getEncoding();
    }

    public void close() {
      this.content.close();
    }

    public String getEntityTag() throws ResourceException {
      return this.content instanceof IEntityMetadata ? ((IEntityMetadata) this.content).getEntityTag() : null;
    }

    public Date getExpiryDate() throws ResourceException {
      return this.content instanceof IEntityMetadata ? ((IEntityMetadata) this.content).getExpiryDate() : null;
    }

    public String getLanguage() throws ResourceException {
      return this.content instanceof IEntityMetadata ? ((IEntityMetadata) this.content).getLanguage() : null;
    }

    public Date getLastModified() throws ResourceException {
      return this.content instanceof IEntityMetadata ? ((IEntityMetadata) this.content).getLastModified() : null;
    }

  }

}
