/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import com.sap.tc.logging.Location;
import com.sapportals.wcm.repository.enum.LinkType;
import com.sapportals.wcm.repository.enum.OrderType;
import com.sapportals.wcm.repository.enum.SupportedOption;
import com.sapportals.wcm.repository.manager.IRepositoryManager;
import com.sapportals.wcm.util.content.IContent;
import com.sapportals.wcm.util.events.IEventList;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.uri.RID;
import com.sapportals.wcm.util.uri.URI;
import com.sapportals.wcm.util.uri.URL;

import java.util.*;

/**
 * Represents a internal link to a collection resource. <p>
 *
 * All methods from IResource are forwarded to the link itself. All methods from
 * ICollections are forwarded to the target collection. <p>
 *
 * Copyright (c) SAP AG 2002-2003
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Revision: 1.8 $
 */
public class CollectionLinkImpl implements IResource, ICollection, ISortItem {

  private final static Location log = Location.getLocation(com.sapportals.wcm.repository.CollectionLinkImpl.class);

  private final IResource link;
  private IResource target;
  private IPropertyMap sortProps;

  public CollectionLinkImpl(IResource link, IResource target) {
    this.link = link;
    this.target = target;
  }

  public void setAccessRID(RID accessRID)
    throws ResourceException {
    if (this.link instanceof ResourceImpl) {
      ((ResourceImpl)this.link).setAccessRID(accessRID);
    }
  }

  public boolean equals(Object o) {
    return this.link.equals(o);
  }

  public int hashCode() {
    return this.link.hashCode();
  }

  public String toString() {
    return this.link.toString();
  }

  //------------------------- ICollection--------------------------------------------

  public IResourceList getChildren()
    throws ResourceException, AccessDeniedException {
    IResourceList rl = targetCollection().getChildren();
    this.setAccessRidsForChildren(rl);
    return rl;
  }

  public IResourceList getChildren(boolean visible, boolean collections, boolean links)
    throws ResourceException, AccessDeniedException {
    IResourceList rl = this.targetCollection().getChildren(visible, collections, links);
    this.setAccessRidsForChildren(rl);
    return rl;
  }

  public int getChildrenCount(boolean visible, boolean collections, boolean links)
    throws ResourceException, AccessDeniedException {
    return this.targetCollection().getChildrenCount(visible, collections, links);
  }

  public IResourceList getChildren(Selector childrenSelector, Collator sortBy, IPropertyNameList propertyPrefill, String[] permissionNames)  throws ResourceException, AccessDeniedException {
    IResourceList rl = this.targetCollection().getChildren(childrenSelector, sortBy, propertyPrefill, permissionNames);
    this.setAccessRidsForChildren(rl);
    return rl;
  }


  public IResourceList getChildren(Selector childrenSelector, Collator sortBy, IPropertyNameList propertyPrefill)  throws ResourceException, AccessDeniedException {
    IResourceList rl = this.targetCollection().getChildren(childrenSelector, sortBy, propertyPrefill);
    this.setAccessRidsForChildren(rl);
    return rl;
  }


  public boolean hasChildren()
    throws ResourceException {
    return this.targetCollection().hasChildren();
  }

  public OrderType getOrderType()
    throws ResourceException {
    return this.targetCollection().getOrderType();
  }

  public void setOrderType(OrderType orderType)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.targetCollection().setOrderType(orderType);
  }

  public void reorder(IReorderList list)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.targetCollection().reorder(list);
  }

  public ICollection createCollection(String name, IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.targetCollection().createCollection(name, properties);
  }

  public ICollection createCollection(String name, IPosition position, IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.targetCollection().createCollection(name, position, properties);
  }

  public ICollection createCollection(String name, IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.targetCollection().createCollection(name, properties, ignorePropertyFailures);
  }

  public ICollection createCollection(String name, IPosition position, IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.targetCollection().createCollection(name, position, properties, ignorePropertyFailures);
  }

  public IResource createResource(String name, IPropertyMap properties, IContent content)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.targetCollection().createResource(name, properties, content);
  }

  public IResource createResource(String name, IPosition position, IPropertyMap properties,
    IContent content)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.targetCollection().createResource(name, position, properties, content);
  }

  public IResource createResource(String name, IPropertyMap properties, IContent content, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.targetCollection().createResource(name, properties, content, ignorePropertyFailures);
  }

  public IResource createResource(String name, IPosition position, IPropertyMap properties,
    IContent content, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.targetCollection().createResource(name, position, properties, content, ignorePropertyFailures);
  }

  public ILockInfo createResource(String name, ILockProperties lockProperties)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.targetCollection().createResource(name, lockProperties);
  }

  public IResource createLink(String name, URL targetURL, LinkType linkType,
    IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.targetCollection().createLink(name, targetURL, linkType, properties);
  }

  public IResource createLink(String name, URL targetURL, LinkType linkType,
    IPosition position, IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.targetCollection().createLink(name, targetURL, linkType, position, properties);
  }

  public IResource createLink(String name, URL targetURL, LinkType linkType,
    IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.targetCollection().createLink(name, targetURL, linkType, properties, ignorePropertyFailures);
  }

  public IResource createLink(String name, URL targetURL, LinkType linkType,
    IPosition position, IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.targetCollection().createLink(name, targetURL, linkType, position, properties, ignorePropertyFailures);
  }

  public IResourceList search(IQueryExpression query, int depth, int maxResults, boolean includeRevisions)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.targetCollection().search(query, depth, maxResults, includeRevisions);
  }

  //------------------------- IResource ---------------------------------------------

  public IEventList getSupportedEvents()
    throws ResourceException {
    return this.link.getSupportedEvents();
  }

  public Collection listTypes() {
    return this.link.listTypes();
  }

  public ISupportedOptionSet getSupportedOptions() {
    return new CollectionLinkImplSupportedOptionSet(this.link.getSupportedOptions(), this.target.getSupportedOptions());
  }

  public IResourceContext getContext() {
    return this.link.getContext();
  }

  public IRepositoryManager getRepositoryManager() {
    return this.link.getRepositoryManager();
  }

  public String getName()
    throws ResourceException {
    return this.link.getName();
  }

  public Date getCreationDate() {
    return this.link.getCreationDate();
  }

  public Date getLastModified() {
    return this.link.getLastModified();
  }

  public String getDescription() {
    return this.link.getDescription();
  }

  public String getDisplayName() {
    return this.link.getDisplayName();
  }

  public final String getResourceType() {
    return this.link.getResourceType();
  }

  public String getDisplayName(boolean orNameIfNull) {
    return this.link.getDisplayName(orNameIfNull);
  }

  public String getCreatedBy() {
    return this.link.getCreatedBy();
  }

  public String getLastModifiedBy() {
    return this.link.getLastModifiedBy();
  }

  public String getETag() {
    return this.link.getETag();
  }

  public String getLanguage() {
    return this.link.getLanguage();
  }

  public URI getURI()
    throws ResourceException {
    return this.link.getURI();
  }

  public RID getRID()
    throws ResourceException {
    return this.link.getRID();
  }

  public URI getAccessURI()
    throws ResourceException {
    return this.link.getAccessURI();
  }

  public RID getAccessRID()
    throws ResourceException {
    return this.link.getAccessRID();
  }

  public boolean isReadOnly() {
    return this.link.isReadOnly();
  }

  public boolean isHidden() {
    return this.link.isHidden();
  }

  public boolean isCollection() {
    return this.target.isCollection();
  }

  public void rename(String newName)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.link.rename(newName);
  }

  public ICollection getParentCollection()
    throws ResourceException, AccessDeniedException {
    return this.link.getParentCollection();
  }

  public IContent getContent()
    throws ResourceException, AccessDeniedException {
    return this.link.getContent();
  }

  public IContent getContent(boolean handleExternalLink)
    throws ResourceException, AccessDeniedException {
    return this.link.getContent(handleExternalLink);
  }

  public IContent getUnfilteredContent()
    throws ResourceException, AccessDeniedException {
    return this.link.getUnfilteredContent();
  }

  public IContent getUnfilteredContent(boolean handleExternalLink)
    throws ResourceException, AccessDeniedException {
    return this.link.getUnfilteredContent(handleExternalLink);
  }

  public void delete()
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.link.delete();
  }

  public void updateContent(IContent newContent)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.link.updateContent(newContent);
  }

  public void update(IContent newContent, IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException, SetPropertiesException {
    this.link.update(newContent, properties);
  }

  public IResource copy(URI destinationURI, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.link.copy(destinationURI, param);
  }

  public IResource copy(URI destinationURI, IPosition position, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.link.copy(destinationURI, position, param);
  }

  public IResource move(URI destinationURI, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.link.move(destinationURI, param);
  }

  public IResource move(URI destinationURI, IPosition position, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.link.move(destinationURI, param);
  }

  public IResource copy(RID destinationRID, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.link.copy(destinationRID, param);
  }

  public IResource copy(RID destinationRID, IPosition position, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.link.copy(destinationRID, position, param);
  }

  public IResource move(RID destinationRID, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.link.move(destinationRID, param);
  }

  public IResource move(RID destinationRID, IPosition position, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.link.move(destinationRID, position, param);
  }

  public IPropertyMap getProperties()
    throws ResourceException, NotSupportedException, AccessDeniedException {
    IMutablePropertyMap props = this.link.getProperties().getMutable();
    props.put(new Property(PropertyName.createCollection(), new Boolean(this.target.isCollection())));
    return props;
  }

  public IPropertyMap getProperties(IPropertyNameList propNameList)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    IPropertyMap props = this.link.getProperties(propNameList);    
    if (propNameList.contains(PropertyName.createCollection())) {
      IMutablePropertyMap mprops = props.getMutable();
      mprops.put(new Property(PropertyName.createCollection(), new Boolean(this.target.isCollection())));
      return mprops;
    }
    return props;
  }

  public void setProperties(IPropertyMap props)
    throws ResourceException, NotSupportedException, AccessDeniedException, SetPropertiesException {
    this.link.setProperties(props);
  }

  public void setProperties(List propChangeList)
    throws ResourceException, NotSupportedException, AccessDeniedException, SetPropertiesException {
    this.link.setProperties(propChangeList);
  }

  public IProperty getProperty(IPropertyName propName)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    if (propName.equals(PropertyName.createCollection())) {
      return new Property(PropertyName.createCollection(), new Boolean(this.target.isCollection()));
    }
    else {
      return this.link.getProperty(propName);
    }
  }

  public IProperty getInheritedProperty(IPropertyName propName)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.link.getInheritedProperty(propName);
  }

  public void setProperty(IProperty prop)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.link.setProperty(prop);
  }

  public void deleteProperty(IPropertyName propName)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.link.deleteProperty(propName);
  }

  public ILockInfo lock()
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.link.lock();
  }

  public ILockInfo lock(ILockProperties lockProperties)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.link.lock(lockProperties);
  }

  public void unlock(ILockInfo lockInfo)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.link.unlock(lockInfo);
  }

  public void refreshLock(ILockInfo lockInfo)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.link.refreshLock(lockInfo);
  }

  public ILockInfo getLockByToken(String lockToken)
    throws ResourceException, NotSupportedException {
    return this.link.getLockByToken(lockToken);
  }

  public ILockInfoCollection getLocks()
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.link.getLocks();
  }

  public boolean isLocked()
    throws ResourceException {
    return this.link.isLocked();
  }

  public boolean isLockedByMe()
    throws ResourceException {
    return this.link.isLockedByMe();
  }

  public URL getTargetURL()
    throws ResourceException {
    return this.link.getTargetURL();
  }

  public void setTargetURL(URL url)
    throws ResourceException, AccessDeniedException {
    this.link.setTargetURL(url);
  }

  public IResource getTargetResource()
    throws ResourceException {
    return this.link.getTargetResource();
  }

  public LinkType getLinkType()
    throws ResourceException {
    return this.link.getLinkType();
  }

  public void setLinkType(LinkType linkType)
    throws ResourceException, AccessDeniedException {
    this.link.setLinkType(linkType);
    revalidateTarget();
  }

  public void enableVersioning(boolean enable)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.link.enableVersioning(enable);
  }

  public boolean isVersioned()
    throws ResourceException {
    return this.link.isVersioned();
  }

  public boolean isRevision() {
    return this.link.isRevision();
  }

  public String getRevisionID() {
    return this.link.getRevisionID();
  }

  public boolean isCheckedOut()
    throws ResourceException {
    return this.link.isCheckedOut();
  }

  public ICheckOutInfo checkOut()
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.link.checkOut();
  }

  public void undoCheckOut()
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.link.undoCheckOut();
  }

  public ICheckInInfo checkIn(IContent newContent, IPropertyMap properties,
    boolean ignorePropertyFailures, URI expectedCheckInURI)
    throws ResourceException, NotSupportedException, AccessDeniedException, ExpectedCheckInURIException {
    return this.link.checkIn(newContent, properties, ignorePropertyFailures, expectedCheckInURI);
  }

  public ICheckInInfo checkIn(IContent newContent, IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.link.checkIn(newContent, properties);
  }

  public ICheckInInfo checkIn(IContent newContent, IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.link.checkIn(newContent, properties, ignorePropertyFailures);
  }

  public ICheckInInfo checkIn(IContent newContent, IPropertyMap properties,
    boolean ignorePropertyFailures, RID expectedCheckInRID)
    throws ResourceException, NotSupportedException, AccessDeniedException, ExpectedCheckInRIDException {
    return this.link.checkIn(newContent, properties, ignorePropertyFailures, expectedCheckInRID);
  }

  public IVersionHistory getVersionHistory()
    throws ResourceException, NotSupportedException {
    return this.link.getVersionHistory();
  }

  public void setAsCurrentVersion()
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.link.setAsCurrentVersion();
  }

  public IResourceList getCheckedOutResources()
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.link.getCheckedOutResources();
  }

  public boolean isA(Class classtest) {
    return this.link.isA(classtest);
  }

  public Object as(Class classref)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return this.link.as(classref);
  }

  //------------------------- protected --------------------------------------------

  protected IResource internalCopy(RID destinationRID, IPosition position, ICopyParameter param, boolean isMove)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    if (this.link instanceof ResourceImpl) {
      return map(((ResourceImpl)this.link).internalCopy(destinationRID, position, param, isMove));
    }
    else {
      if (isMove) {
        return map(this.link.move(destinationRID, position, param));
      }
      else {
        return map(this.link.copy(destinationRID, position, param));
      }
    }
  }

  protected IResource internalCreateLink(String name, URL targetURL, LinkType linkType,
    IPosition position, IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    ICollection target = targetCollection();
    if (target instanceof CollectionImpl) {
      return map(((CollectionImpl)target).internalCreateLink(name, targetURL, linkType, position, properties, ignorePropertyFailures));
    }
    else {
      return map(target.createLink(name, targetURL, linkType, position, properties, ignorePropertyFailures));
    }
  }

  protected IResource internalCreateResource(String name, IPosition position, IPropertyMap properties,
    IContent content, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    ICollection target = targetCollection();
    if (target instanceof CollectionImpl) {
      return ((CollectionImpl)target).internalCreateResource(name, position, properties, content, ignorePropertyFailures);
    }
    else {
      return target.createResource(name, position, properties, content, ignorePropertyFailures);
    }
  }

  protected ICollection internalCreateCollection(String name, IPosition position,
    IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    ICollection target = targetCollection();
    if (target instanceof CollectionImpl) {
      return ((CollectionImpl)target).internalCreateCollection(name, position, properties, ignorePropertyFailures);
    }
    else {
      return target.createCollection(name, position, properties, ignorePropertyFailures);
    }
  }


  //------------------------- private --------------------------------------------

  private ICollection targetCollection()
    throws ResourceException {
    if (this.target == null) {
      revalidateTarget();
    }

    if (this.target == null) {
      throw new InvalidTargetException("target of link not found: " + this.link.getTargetURL(), this.link.getRID());
    }
    else if (this.target instanceof ICollection) {
      return (ICollection)this.target;
    }
    else {
      throw new InvalidArgumentException("target of link is not collection: " + this.link.getTargetURL(), this.link.getRID());
    }
  }

  private void revalidateTarget() {
    try {
      IResource target = this.link.getTargetResource();
      if (target != null) {
        this.target = target;
      }
    }
    catch (ResourceException e) {
      if (log.beDebug()) {
        log.infoT("revalidateTarget(643)", "revalidating target of " + this.link + " - " + LoggingFormatter.extractCallstack(e));
      }
    }
  }

  private IResource map(IResource resource)
    throws ResourceException {
    try {
      return ResourceImpl.mapLink(resource, false);
    }
    catch (ResourceException e) {
      if (log.beDebug()) {
        log.debugT("map(655)", "mapping resource " + resource + " - " + LoggingFormatter.extractCallstack(e));
      }
      return resource;
    }
  }

  private void setAccessRidsForChildren(IResourceList children)
    throws ResourceException {
    RID thisAccessRID = this.getAccessRID();
    IResource child = null;
    for (int i = 0, s = children.size(); i < s; i++) {
      child = children.get(i);
      if (child instanceof ResourceImpl) {
        ((ResourceImpl)child).setAccessRID(thisAccessRID.add(child.getRID().name()));
      }
      else if (child instanceof CollectionLinkImpl) {
        ((CollectionLinkImpl)child).setAccessRID(thisAccessRID.add(child.getRID().name()));
      }
      else {
        // this should not happen
        CollectionLinkImpl.log.errorT("setAccessRidsForChildren(675)", "Failed to set AccessRID for child: " + child.getRID() + ", resouce=" + child.getClass().getName());
      }
    }
  }

  // set of SupportedOptions that is taken from link target instead of link itself

  private final static Set TARGETOPTIONS;
  static {
    Set tmp = new HashSet();
    tmp.add(SupportedOption.CREATE_RESOURCE);
    TARGETOPTIONS = tmp;
  }

  // wrapper object for ISupportedOptionSet(s)

  /**
   * TBD: Description of the class.
   */
  private class CollectionLinkImplSupportedOptionSet implements ISupportedOptionSet {

    private final ISupportedOptionSet link;
    private final ISupportedOptionSet target;

    public CollectionLinkImplSupportedOptionSet(ISupportedOptionSet link, ISupportedOptionSet target) {
      this.link = link;
      this.target = target;
    }

    public void add(SupportedOption option) {
      if (TARGETOPTIONS.contains(option)) {
        target.add(option);
      }
      else {
        link.add(option);
      }
    }

    public boolean isSupported(SupportedOption option) {
      return TARGETOPTIONS.contains(option) ? target.isSupported(option) : link.isSupported(option);
    }

    public Iterator iterator() {
      Set tmp = new HashSet();

      // get options from link resource that aren't "target" options
      for (Iterator linkit = this.link.iterator(); linkit.hasNext(); ) {
        SupportedOption opt = (SupportedOption)linkit.next();
        if (!TARGETOPTIONS.contains(opt)) {
          tmp.add(opt);
        }
      }

      // get options from target resouce that *are* "target" options
      for (Iterator targetit = this.target.iterator(); targetit.hasNext(); ) {
        SupportedOption opt = (SupportedOption)targetit.next();
        if (TARGETOPTIONS.contains(opt)) {
          tmp.add(opt);
        }
      }

      return tmp.iterator();
    }

  }
  
  // package-local access to intenal link resource
  ResourceImpl getLinkResourceImpl() {
    return (ResourceImpl)this.link;
  }
  
  public IPropertyMap getSortProperties() {
    return this.sortProps;
  }
  
  public void setSortProperties(IPropertyMap props) {
    this.sortProps = props;
  }    
}
