/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository;

import java.util.*;

import com.sap.netweaver.bc.rf.common.IResourceHandle;
import com.sap.netweaver.bc.rf.mi.AbstractManager;
import com.sap.tc.logging.Location;
import com.sapportals.wcm.IFrameworkTransaction;
import com.sapportals.wcm.repository.enum.*;
import com.sapportals.wcm.repository.manager.*;
import com.sapportals.wcm.repository.runtime.CmFilterHandler;
import com.sapportals.wcm.repository.runtime.CmStartupException;
import com.sapportals.wcm.repository.runtime.CmSystem;
import com.sapportals.wcm.util.content.IContent;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.uri.*;

/**
 * <p>
 *
 * Copyright (c) SAP AG 2001-2004
 * @author m.breitenfelder@sap.com
 * @author frank.renkes@sap.com
 * @version $Revision: 1.13 $
 */
public class CollectionImpl extends ResourceImpl implements ICollection {

  private static Location log = Location.getLocation(com.sapportals.wcm.repository.CollectionImpl.class);

  private final static Position ORDERED_COLL_LAST = new Position(null, OrderPosition.LAST);

  /**
   * @exception ResourceException Exception raised in failure situation
   * @deprecated as of NW04. Use the constructor that just takes a revisionID parameter, no
   *      revisionRID
   */
  public CollectionImpl(
    RID rid,
    String targetURI,
    LinkType linkType,
    String revisionID,
    RID revisionRID,
    IRepositoryManager repositoryManager,
    IResourceContext context)
    throws ResourceException {
    super(rid, targetURI, linkType, revisionID, revisionRID, repositoryManager, context);
  }

  /**
   * Constructs a new collection.
   *
   * @param rid The resource identifier of the resource, must not be <code>null
   *      </code>.
   * @param repositoryManager The repository manager of the resource, must not
   *      be <code>null</code> .
   * @param context The resource context, must not be <code>null</code> .
   * @exception ResourceException Exception raised in failure situation
   */
  public CollectionImpl(RID rid, IRepositoryManager repositoryManager, IResourceContext context)
    throws ResourceException {
    super(rid, null, null, LinkType.NONE, repositoryManager, context);
  }

  /**
   * Constructs a new resource which will be a link and/or a revision. The
   * resource will be a link
   *
   * @param rid The resource identifier of the resource, must not be <code>null
   *      </code>.
   * @param revisionID Specified the revision identified (version number) if the
   *      resource is a revision, <code>null</code> otherwise
   * @param linkType Specifies the link type ({@link
   *      com.sapportals.wcm.repository.enum.LinkType}).
   * @param repositoryManager The repository manager of the resource, must not
   *      be <code>null</code> .
   * @param context The resource context, must not be <code>null</code> .
   * @exception ResourceException If required parameters are not specified.
   */
  public CollectionImpl(
    RID rid,
    String revisionID,
    String targetURI,
    LinkType linkType,
    IRepositoryManager repositoryManager,
    IResourceContext context)
    throws ResourceException {
    super(rid, revisionID, targetURI, linkType, repositoryManager, context);
  }

  /**
   * Framework-internal constructor
   *
   * @exception ResourceException Exception raised in failure situation
   */
  CollectionImpl(IResourceHandle handle, AbstractManager manager, IResourceContext context) throws ResourceException {
    super(handle, manager, context);
  }

  // ---------------------------------------------------------------------------
  // ICollection interface
  // ---------------------------------------------------------------------------

  public final IResourceList getChildren() throws ResourceException, AccessDeniedException {
    this.checkDeleted();

    IResourceList filteredChildren = null;
    IResourceList children = this.internalGetChildren();
    this.completeResourceList(children);
    filteredChildren = this.applyNamespaceReadFilter(children);
    
    this.sendEvent(ResourceEvent.GET_CHILDREN, null);
    return filteredChildren;
  }
  
  public final IResourceList getChildren(boolean visible, boolean collections, boolean links)
    throws ResourceException, AccessDeniedException {
    this.checkDeleted();

    IResourceList children = null;
    if (this.getExtendedNamespaceManager() != null) {
      try {
        // Map old semantic to new one
        Boolean v = visible ? Boolean.TRUE : null;
        Boolean c = collections ? Boolean.TRUE : null;
        Boolean l = links ? Boolean.TRUE : null;

        // If the repository does not support detecting the targets of links to collection
        // must do additional mapping: Links to collections will be included although the "links" flag is set to "false" !
        // This is the behaviour of the "old" filter code (prior SP5) that must be supported to be compatible.
        boolean supportsRL =
          this.getRepositoryManager().getSupportedOptions(this).isSupported(SupportedOption.RESOLVES_LINKS);
        boolean mapCollectionLinkSemantic = !supportsRL && collections && !links;
        if (mapCollectionLinkSemantic) {
          c = null;
        }

        children = this.internalGetChildren(v, c, l);
        this.completeResourceList(children);

        if (mapCollectionLinkSemantic) {
          IResourceListIterator it = children.listIterator();
          while (it.hasNext()) {
            IResource r = it.next();
            if (!r.isCollection()) {
              it.remove();
            }
          }
        }
      }
      catch (NotSupportedException ex) {
        if (log.beDebug()) {
          log.debugT(ex.getMessage());
        }
      }
    }

    // if IExtendedNamespaceManager not implemented
    // or ExtendedNamespaceManager.getChildren() throws NotSupportedException
    // apply "old" filter code (prior SP5).
    if (children == null) {
      children = this.internalGetChildren();
      this.completeResourceList(children);

      IResourceListIterator it = children.listIterator();
      while (it.hasNext()) {
        IResource r = it.next();
        if (visible && r.isHidden()) {
          it.remove();
        }
        else if (collections && !r.isCollection()) {
          it.remove();
        }
        else if (links && (!LinkType.NONE.equals(r.getLinkType()))) {
          it.remove();
        }
      }
    }

    this.applyNamespaceReadFilter(children);
    this.sendEvent(ResourceEvent.GET_CHILDREN, null, null);
    return children;
  }

  public final int getChildrenCount(boolean visible, boolean collections, boolean links)
    throws ResourceException, AccessDeniedException {
    return this.getChildren(visible, collections, links).size();
  }

  private class CollatorComperator implements Comparator {

    private Collator collator = null;
    private List collatorItemList = null;
    private int ruleCount = 0;

    public CollatorComperator(Collator collator) {
      this.collator = collator;
      this.collatorItemList = collator.getPropertyCollatorItemList();
      this.ruleCount = this.collatorItemList.size();
    }

    private int compareWithRule(ISortItem i1, ISortItem i2, int rule) {
      PropertyCollatorItem ci = (PropertyCollatorItem)collatorItemList.get(rule);
      IProperty prop1 = i1.getSortProperties().get(ci.getPropertyName());
      IProperty prop2 = i2.getSortProperties().get(ci.getPropertyName());
      return ci.compare(prop1, prop2);
    }

    public int compare(Object obj1, Object obj2) {
      ISortItem si1 = (ISortItem)obj1;
      ISortItem si2 = (ISortItem)obj2;
      int rule = 0;
      int compare = 0;

      while (compare == 0 && rule < this.ruleCount) {
        compare = this.compareWithRule(si1, si2, rule);
        rule++;
      }

      return compare;
    }

    public boolean equals(Object obj) {
      return false;
    }
  }

  public IResourceList getChildren(Selector childrenSelector, Collator sortBy, IPropertyNameList propertyPrefill)
    throws ResourceException, AccessDeniedException {
    return this.getChildren(childrenSelector, sortBy, propertyPrefill, null);
  }

  public IResourceList getChildren(Selector childrenSelector, Collator sortBy, IPropertyNameList propertyPrefill, String[] permissionNames)
    throws ResourceException, AccessDeniedException {
    long start = 0;

    if (true == CollectionImpl.KM_PERF_MON) {
      start = System.currentTimeMillis();
    }

    IResourceList children = this.getChildren();
    IPropertyNameList nameList = null;

    // remove resource without required permissions for the user in context
    if (permissionNames != null && permissionNames.length > 0 && children.size() > 0) {
      CollectionImpl.removeResourcesWithoutPermission(
        this.getSecurityManager(),
        children,
        permissionNames,
        this.getContext());
    }

    if ((null != sortBy) && (sortBy.getPropertyCollatorItemList().size() == 0))
      sortBy = null;
    if ((null != childrenSelector) && (childrenSelector.isEmpty()))
      childrenSelector = null;

    if ((null != childrenSelector) && (null != sortBy)) {
      nameList = childrenSelector.getPropertyNameList();

      PropertyNameList propNameList = sortBy.getPropertyNameList();

      for (int i = 0; i < propNameList.size(); i++) {
        nameList.add(propNameList.get(i));
      }
    }
    else if (null != childrenSelector) {
      nameList = childrenSelector.getPropertyNameList();
    }
    else if (null != sortBy) {
      nameList = sortBy.getPropertyNameList();
    }

    if (null != propertyPrefill) {
      if (null == nameList)
        nameList = propertyPrefill;
      else {
        for (int i = 0; i < propertyPrefill.size(); i++) {
          nameList.add(propertyPrefill.get(i));
        }
      }
    }

    if (null == nameList)
      nameList = new PropertyNameList();

    Comparator sortByComperator = null;
    SortedSet sortedFolders = null;
    SortedSet sortedResources = null;

    if (null != sortBy) {
      sortByComperator = new CollatorComperator(sortBy);
      sortedFolders = new TreeSet(sortByComperator);
      sortedResources = new TreeSet(sortByComperator);
    }

    if (children != null) {
      // special super-optimization: smuggle a list without system property names 
      // into the filter handler.
      PropertyNameTool.smuggleListWithoutSystemPropertyNames(nameList);
      for (int i = children.size() - 1; i >= 0; i--) {
        IResource res = children.get(i);
        IPropertyMap props = null;

        if (log.beDebug()) {
          log.debugT("Start filtering for resource <" + res.getName() + ">");
        }
        try {
          props = res.getProperties(nameList);
        }
        catch (AccessDeniedException ex) {
          if (log.beDebug()) {
            log.debugT("Removed resource <" + res.getName() + "> because no access to properties");
          }
          if (sortBy == null) {
            children.remove(i);
          }
          continue;
        }

        if (childrenSelector != null && !childrenSelector.isEmpty()) {
          if (false == childrenSelector.check(props)) {
            if (log.beDebug()) {
              log.debugT("Removed Resource <" + res.getName() + "> because it did not pass the property query");
            }
            if (sortBy == null) {
              children.remove(i);
            }
            continue;
          }
          else {
            if (log.beDebug()) {
              log.debugT("Resource <" + res.getName() + "> passed the property query");
            }
          }
        }

        if (sortBy != null) {
          ((ISortItem)res).setSortProperties(props);
          if (res.isCollection()) {
            sortedFolders.add(res);
          }
          else {
            sortedResources.add(res);
          }
        }
      }

      if (sortBy != null) {
        children = new ResourceList(sortedFolders);
        ((ResourceList)children).addAll(sortedResources);
      }
    }

    if (true == CollectionImpl.KM_PERF_MON) {
      RFStatisticCounter.increaseGetChildrenFilterAndSortTimer(System.currentTimeMillis() - start);
    }

    return children;
  }

  public final boolean hasChildren() throws ResourceException {
    IResourceList children = this.getChildren();
    return (children != null && children.size() > 0);
  }

  public final OrderType getOrderType() throws ResourceException {
    this.checkDeleted();
    return this.internalGetOrderType();
  }

  public final void setOrderType(OrderType orderType)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    checkDeleted();
    if (isRevision()) {
      throw new RevisionResourceException("resource is revision", getRID());
    }
    internalSetOrderType(orderType);
  }

  public final void reorder(IReorderList list) throws ResourceException, NotSupportedException, AccessDeniedException {
    checkDeleted();
    if (isRevision()) {
      throw new RevisionResourceException("resource is revision", getRID());
    }
    IFrameworkTransaction ft = FrameworkTransaction.required();
    try {
      internalReorder(list);
      ft.commit();
      ft = null;
    }
    finally {
      if (ft != null) ft.rollback();
    }
  }

  public final ICollection createCollection(String name, IPropertyMap properties)
    throws ResourceException, NotSupportedException {
    return createCollection(name, null, properties, false);
  }

  public final ICollection createCollection(String name, IPosition position, IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return createCollection(name, position, properties, false);
  }

  public final ICollection createCollection(String name, IPropertyMap properties, boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return createCollection(name, null, properties, ignorePropertyFailures);
  }

  public final ICollection createCollection(
    String name,
    IPosition position,
    IPropertyMap properties,
    boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    checkDeleted();
    if (isRevision()) {
      throw new RevisionResourceException("resource is revision", getRID());
    }
    checkNewName(name);

    IPropertyMap filteredProps = properties;
    if (properties != null) {

      RID newRID = getRID().add(name);
      filteredProps = applyPropertyWriteFilterForCreate(newRID.toString(), properties);
    }
    
    ICollection c = null;
    
    IFrameworkTransaction ft = FrameworkTransaction.required();
    try {
      c = internalCreateCollection(name, position, filteredProps, ignorePropertyFailures);
      if (sendEvent(ft, ResourceEvent.CREATE_COLLECTION, null, c)) { 
        ft.commit();        
        ft = null;
      }
    }
    finally{
      if (ft != null) ft.rollback();
    }
    sendEvent(ResourceEvent.CREATE_COLLECTION, null, c);
    
    return c;
  }

  public final IResource createResource(String name, IPropertyMap properties, IContent content)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return createResource(name, null, properties, content, false);
  }

  public final IResource createResource(String name, IPosition position, IPropertyMap properties, IContent content)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return createResource(name, position, properties, content, false);
  }

  public final IResource createResource(
    String name,
    IPropertyMap properties,
    IContent content,
    boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return createResource(name, null, properties, content, ignorePropertyFailures);
  }

  public final IResource createResource(
    String name,
    IPosition position,
    IPropertyMap properties,
    IContent content,
    boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    checkDeleted();
    if (isRevision()) {
      throw new RevisionResourceException("resource is revision", getRID());
    }
    checkNewName(name);

    RID newRID = getRID().add(name);

    // Create empty InputStream if content == null
    if (content == null) {
      content = ResourceImpl.EMPTY_CONTENT;
    }

    IContent filteredContent = null;
    if (content != null) {
      filteredContent = applyContentWriteFilterForCreate(newRID.toString(), content);
    }

    IPropertyMap filteredProps = null;
    if (properties != null) {
      filteredProps = applyPropertyWriteFilterForCreate(newRID.toString(), properties);
    }

    try {
      IResource newRes = null;
      IFrameworkTransaction ft = FrameworkTransaction.required();
      try {
        newRes = internalCreateResource(name, position, filteredProps, filteredContent, ignorePropertyFailures);
        if (sendEvent(ft, ResourceEvent.CREATE_CHILD, null, newRes)) {
          ft.commit();        
          ft = null;
        }
      }
      finally{
        if (ft != null) ft.rollback();
      }
      sendEvent(ResourceEvent.CREATE_CHILD, null, newRes);
      return newRes;
    }
    catch (ResourceException ex) {
      throw ex;
    }
  }

  public final ILockInfo createResource(String name, ILockProperties lockProperties)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    checkDeleted();
    if (isRevision()) {
      throw new RevisionResourceException("resource is revision", getRID());
    }
    checkNewName(name);
    
    IFrameworkTransaction ft = FrameworkTransaction.required();
    try {
      ILockInfo lockInfo = internalCreateResource(name, lockProperties);
      ft.commit();        
      ft = null;
      return lockInfo;
    }
    finally{
      if (ft != null) ft.rollback();
    }
  }

  public final IResource createLink(String name, URL targetURL, LinkType linkType, IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return createLink(name, targetURL, linkType, null, properties, false);
  }

  public final IResource createLink(
    String name,
    URL targetURL,
    LinkType linkType,
    IPosition position,
    IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return createLink(name, targetURL, linkType, position, properties, false);
  }

  public final IResource createLink(
    String name,
    URL targetURL,
    LinkType linkType,
    IPropertyMap properties,
    boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    return createLink(name, targetURL, linkType, null, properties, ignorePropertyFailures);
  }

  public final IResource createLink(
    String name,
    URL targetURL,
    LinkType linkType,
    IPosition position,
    IPropertyMap properties,
    boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    this.checkDeleted();
    if (this.isRevision()) {
      throw new RevisionResourceException("resource is revision", getRID());
    }
    this.checkNewName(name);

    if (linkType.equals(LinkType.INTERNAL)) {
      this.checkLinkTarget(RID.getRID(targetURL.toString()), this.getRID().add(name));
    }

    IPropertyMap filteredProps = properties;
    if (properties != null) {
      RID newRID = this.getRID().add(name);
      filteredProps = this.applyPropertyWriteFilterForCreate(newRID.toString(), properties);
    }

    IResource link = null;
    IFrameworkTransaction ft = FrameworkTransaction.required();
    try {
      link = this.map(this.internalCreateLink(name, targetURL, linkType, position, filteredProps, ignorePropertyFailures));
      if (this.sendEvent(ft, ResourceEvent.CREATE_LINK, null, link)) {
        ft.commit();
        ft = null;
      }
    }
    finally{
      if (ft != null) ft.rollback();
    }
    this.sendEvent(ResourceEvent.CREATE_LINK, null, link);
    return link;
  }

  public boolean isCollection() {
    return true;
  }

  // ---------------------------------------------------------------------------
  // private / protected
  // ---------------------------------------------------------------------------

  protected void internalUpdateContent(IContent newContent)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    throw new NotSupportedException();
  }

  protected void internalUpdate(IContent newContent, IPropertyMap properties)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    throw new NotSupportedException();
  }

  protected IResource internalMove(RID destinationRID, IPosition position, ICopyParameter param)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    // If source and destination are the same change position in ordered collection
    if (destinationRID.equals(getRID())) {
      getNamespaceManager(true).changePosition(position, this);
      return null;
    }

    // Is destination folder a subfolder of the source folder ?
    if (getRID().isAncestorOf(destinationRID)) {
      throw new InvalidTargetException(
        "The destination collection is a subcollection of the source collection: " + destinationRID,
        destinationRID);
    }

    // If target resource is within the same repository find out if the manager supports the move method
    RID destCollectionURI = destinationRID.removeName();
    ICollection destColl = null;
    IResource res = ResourceFactory.getInstance().getResource(destCollectionURI, getContext());
    if (res == null) {
      throw new InvalidTargetException("Destination collection does not exist:" + destCollectionURI, destCollectionURI);
    }
    if (!res.getLinkType().equals(LinkType.NONE)) {
      res = res.getTargetResource();
    }
    if (res == null) {
      throw new InvalidTargetException("Destination collection does not exist:" + destCollectionURI, destCollectionURI);
    }
    if (res.isCollection()) {
      destColl = (ICollection)res;
    }
    else {
      throw new InvalidTargetException("Not a collection: " + res.getRID(), res.getRID());
    }
    if (destColl.getRepositoryManager().getID().equals(getRepositoryManager().getID())) {
      if (this.getRepositoryManager().getSupportedOptions(this).isSupported(SupportedOption.INTERNAL_MOVE)) {
        return internalMoveManager(destColl.getRID().add(destinationRID.name()), position, param);
      }
    }
    else if (this.getRepositoryManager().getSupportedOptions(this).isSupported(SupportedOption.COPY_MOVE_EXTERNAL)) {      
      try {
        return internalMoveManager(destColl.getRID().add(destinationRID.name()), position, param);
      }
      catch (NotSupportedException ex) {
        // The RM has decided it can not handle this external copy operation: continue
      }
    }

    // Default move
    IResource newRes = this.internalCopy(destinationRID, position, param, true);

    // JRE: only delete source collection if there were no errors!!!!!!!
    if (param.getFirstResourceException() == null) {
      try {
        internalDeleteCollection();
      }
      catch (ResourceNotFoundException ex) {
        // ignore it -- may happen as result of the preceding copy operation
        if (log.beDebug()) {
          log.debugT(ex.getMessage());
        }        
      }
    }
    else {
      if (log.beDebug()) {
        log.debugT(
          "internalMove(486)",
          "internalDelete: not removing source collection because errors occured, first was: "
            + LoggingFormatter.extractCallstack(param.getFirstResourceException()));
      }
    }

    return newRes;
  }

  protected IResource internalCopy(RID destinationRID, IPosition position, ICopyParameter param, boolean isMove)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    // Are the source and destination the same ?
    if (destinationRID.equals(getRID())) {
      throw new InvalidTargetException("Source is the same as destination: " + destinationRID, destinationRID);
    }

    // Is destination folder a subfolder of the source folder ?
    if (param.getCopyChildren() && getRID().isAncestorOf(destinationRID)) {
      throw new InvalidTargetException(
        "The destination collection is a subcollection of the source collection: " + destinationRID,
        destinationRID);
    }

    // Get destination collection
    RID destParentRID = destinationRID.removeName();
    String destName = destinationRID.name().toString();
    ICollection destColl = null;
    IResource res = ResourceFactory.getInstance().getResource(destParentRID, getContext());
    if (res == null) {
      throw new InvalidTargetException("Destination collection does not exist:" + destParentRID, destParentRID);
    }
    // Destination URI might contain links
    if (!res.getLinkType().equals(LinkType.NONE)) {
      res = res.getTargetResource();
    }
    if (res == null) {
      throw new InvalidTargetException("Destination collection does not exist:" + destParentRID, destParentRID);
    }
    if (res.isCollection()) {
      destColl = (ICollection)res;
    }
    else {
      throw new InvalidTargetException("Not a collection: " + res.getRID(), res.getRID());
    }
    IRepositoryManager destinationRepMgr = destColl.getRepositoryManager();

    // Calculate supported options
    IRepositoryManager mgr = getRepositoryManager();
    ISupportedOptionSet supportedOptionsSrc = mgr.getSupportedOptions(this);
    ISupportedOptionSet supportedOptionsDst = destinationRepMgr.getSupportedOptions(null);

    boolean internalCopy = (destinationRepMgr.getID().equals(mgr.getID()));
    boolean supportsInternalCopy = supportedOptionsSrc.isSupported(SupportedOption.INTERNAL_COPY);
    boolean supportsCopyMoveExternal = supportedOptionsSrc.isSupported(SupportedOption.COPY_MOVE_EXTERNAL);
    boolean supportsInternalCopyDeep = supportedOptionsSrc.isSupported(SupportedOption.INTERNAL_COPY_DEEP);
    boolean supportsSetProperties = supportedOptionsDst.isSupported(SupportedOption.SET_PROPERTIES);
    boolean supportsLinking = supportedOptionsDst.isSupported(SupportedOption.LINKING);
    boolean supportsOrderedColl = supportedOptionsDst.isSupported(SupportedOption.ORDERED_COLLECTIONS);

    // Init. resource errors
    ResourceErrors re = ((CopyParameter)param).getResourceErrors();
    if (re == null) {
      re = new ResourceErrors();
      ((CopyParameter)param).setResourceErrors(re);
    }

    // If target resource is within the same repository and manager supports a deep copy
    // AND there is no namespace filter registered for this repository (only "visible" resources must be copied!)
    CmFilterHandler fh = null;
    try {
      fh = CmSystem.getInstance().getFilterHandler();
    }
    catch (CmStartupException ex) {
      throw new ResourceException(ex);
    }
    if (internalCopy && supportsInternalCopyDeep && fh.getNamespaceFilterCount(mgr.getID()) == 0) {

      // jre: make sure that links will be copied no matter what the move request said
      ICopyParameter cp = param;
      if (isMove) {
        cp =
          new CopyParameter(
            param.getOverwrite(),
            param.getIgnorePropertyFailures(),
            param.getCopyChildren(),
            true,
            true);
      }

      return this.internalCopyManager(destColl.getRID().add(destinationRID.name()), position, cp);
    }
    else if (!destColl.getRepositoryManager().getID().equals(getRepositoryManager().getID()) && supportsCopyMoveExternal) {      
      try {
        ResourceImpl newRes =
          (ResourceImpl)this.internalCopyManager(destColl.getRID().add(destinationRID.name()), position, param);
        if (newRes == null) {
          // The repository manager misbehaved: Must throw an exception if it is unclear whether a resource was created or not because
          // we need the resource for the event and during a "move" the source will be deleted after the "copy" !
          throw new ResourceException("Repository manager returned null in createXXX() method: " + this.getRepositoryManager().getID());
        }          
        this.sendCreateEvent(newRes);
        return newRes;
      }
      catch (NotSupportedException ex) {
        // The RM has decided it can not handle this external copy operation: continue
        if (CollectionImpl.log.beDebug()) {
          CollectionImpl.log.debugT(LoggingFormatter.extractCallstack(ex));
        }
      }
    }

    // The collection we are going to create
    IResource newCollection = null;
    IResource destination = null;
    Map autoDeleteMap = Collections.EMPTY_MAP;

    // If target resource is within the same repository and manager supports the copy method then call it
    if (internalCopy && supportsInternalCopy) {
      // Must not set copy-children-flag (RM does not support it !!)
      CopyParameter tmpParam =
        new CopyParameter(
          param.getOverwrite(),
          param.getIgnorePropertyFailures(),
          false,
          param.getCopyInternalLinks(),
          param.getCopyExternalLinks());
      newCollection = this.internalCopyManager(destColl.getRID().add(destinationRID.name()), position, tmpParam);
      if (newCollection == null) {
        throw new ResourceException("Repository manager returned null in copy() method: " + this.getRepositoryManager().getID());
      }                
      autoDeleteMap = this.getInternalMembersOf((ICollection)newCollection);
    }
    else {

      if (destinationRID.isAncestorOf(getRID())) {
        throw new InvalidTargetException("destination is ancestor", destinationRID);
      }

      // Does the destination exist and is it a collection ?
      boolean destinationExists = false;
      destination = ResourceFactory.getInstance().getResource(destinationRID, getContext());
      if (destination != null) {
        if (!destination.isCollection()) {
          if (param.getOverwrite()) {
            destination.delete();
            destination = null;
          }
          else {
            throw new InvalidTargetException("Not a collection: " + destinationRID, destinationRID);
          }
        }
        else {
          destinationExists = true;
        }
      }

      // Get this resource's properties if the target supports custom properties
      IPropertyMap properties = null;
      if (supportsSetProperties) {
        properties = this.internalGetProperties();
      }

      ICollection parent = (ICollection)ResourceFactory.getInstance().getResource(destParentRID, getContext());
      if (parent == null) {
        throw new InvalidTargetException("Parent collection does not exist: " + destParentRID, destParentRID);
      }

      if (isLink()) {

        // Target repository does not support links
        if ((param.getCopyInternalLinks() || param.getCopyExternalLinks()) && !supportsLinking) {
          re.append(new NotSupportedException("Target repository does not support links"));
        }
        else if (!isMove && !param.getCopyInternalLinks() && isInternalLink()) {
          re.append(new ResourceIsLinkException(getRID(), getLinkType(), getTargetURL().toString()));
        }
        else if (!isMove && !param.getCopyExternalLinks() && isExternalLink()) {
          re.append(new ResourceIsLinkException(getRID(), getLinkType(), getTargetURL().toString()));
        }
        try {
          if (parent instanceof CollectionImpl) {
            newCollection =
              ((CollectionImpl)parent).internalCreateLink(
                destName,
                getTargetURL(),
                getLinkType(),
                position,
                properties,
                param.getIgnorePropertyFailures());
          }
          else if (parent instanceof CollectionLinkImpl) {
            newCollection =
              ((CollectionLinkImpl)parent).internalCreateLink(
                destName,
                getTargetURL(),
                getLinkType(),
                position,
                properties,
                param.getIgnorePropertyFailures());
          }
          else {
            newCollection =
              parent.createLink(
                destName,
                getTargetURL(),
                getLinkType(),
                position,
                properties,
                param.getIgnorePropertyFailures());
          }
          if (newCollection == null) {
            throw new ResourceException("Repository manager returned null in createXXX() method: " + this.getRepositoryManager().getID());
          }                   
          this.sendCreateEvent((ResourceImpl)newCollection);
        }
        catch (SetPropertiesException ex) {
          if (!param.getIgnorePropertyFailures()) {
            re.append(ex);
          }
        }
        catch (AccessDeniedException ex) {
          // No create permission in target collection
          throw ex;
        }
        catch (ResourceException ex) {
          re.append(ex);
        }
      }

      else { // Not a link

        try {
          // Try to overwrite the destination collection
          if (destinationExists) {
            if (!param.getOverwrite()) {
              throw new NameAlreadyExistsException("Name already exists", destinationRID);
            }
            else {
              // just update the dead properties, do not delete target. Also
              // keep track of internal members to be deleted at the end
              // of the operation
              log.debugT("internalCopy(680)", "overwriting target: " + destinationRID);
              newCollection = (ICollection)destination;
              if (properties != null) {
                newCollection.setProperties(properties);
              }
              autoDeleteMap = this.getInternalMembersOf((ICollection)newCollection);
            }
          }
          else {
            if (parent instanceof CollectionImpl) {
              newCollection =
                ((CollectionImpl)parent).internalCreateCollection(
                  destName,
                  position,
                  properties,
                  param.getIgnorePropertyFailures());
            }
            else if (parent instanceof CollectionLinkImpl) {
              newCollection =
                ((CollectionLinkImpl)parent).internalCreateCollection(
                  destName,
                  position,
                  properties,
                  param.getIgnorePropertyFailures());
            }
            else {
              newCollection =
                parent.createCollection(destName, position, properties, param.getIgnorePropertyFailures());
            }
            this.sendCreateEvent((ResourceImpl)newCollection);
          }
        }
        catch (SetPropertiesException ex) {
          if (!param.getIgnorePropertyFailures()) {
            re.append(ex);
          }
        }
        catch (AccessDeniedException ex) {
          // No create permission in target collection
          throw ex;
        }
        catch (ResourceException ex) {
          re.append(ex);
        }
      } // if (isLink())
    }

    // Copy all children
    if (newCollection != null && !isLink() && param.getCopyChildren()) {

      Position pos = null;
      if (supportsOrderedColl && !OrderType.NONE.equals(((ICollection)newCollection).getOrderType())) {
        pos = ORDERED_COLL_LAST;
      }

      // Get children to copy - namespacefilter must be applied!
      IResourceList children = this.getChildren();
      IResourceListIterator it = children.listIterator();
      while (it.hasNext()) {
        IResource copy = it.next();
        autoDeleteMap.remove(copy.getName());

        if (!copy.getLinkType().equals(LinkType.NONE)
          && (param.getCopyInternalLinks() || param.getCopyExternalLinks())
          && !supportsLinking) {
          // Target repository does not support links
          re.append(new NotSupportedException("Target repository does not support links", copy.getRID()));
        }
        else if (!param.getCopyInternalLinks() && copy.getLinkType().equals(LinkType.INTERNAL)) {
          re.append(new ResourceIsLinkException(copy.getRID(), copy.getLinkType(), copy.getTargetURL().toString()));
        }
        else if (!param.getCopyExternalLinks() && copy.getLinkType().equals(LinkType.EXTERNAL)) {
          re.append(new ResourceIsLinkException(copy.getRID(), copy.getLinkType(), copy.getTargetURL().toString()));
        }
        else if (isMove && (copy.isRevision() || copy.isVersioned() || copy.isA(IVersionHistoryResource.class))) {
          re.append(new NotSupportedException("cannot move this type of resource using default method", copy.getRID()));
        }
        else {
          try {
            if (copy instanceof CollectionImpl) {
              ((CollectionImpl)copy).internalCopy(destinationRID.add(RID.getRID(copy.getName())), pos, param, isMove);
            }
            else if (copy instanceof ResourceImpl) {
              ((ResourceImpl)copy).internalCopy(destinationRID.add(RID.getRID(copy.getName())), pos, param, isMove);
            }
            else if (copy instanceof CollectionLinkImpl) {
              ((CollectionLinkImpl)copy).internalCopy(
                destinationRID.add(RID.getRID(copy.getName())),
                pos,
                param,
                isMove);
            }
            else {
              if (isMove) {
                return copy.move(destinationRID, position, param);
              }
              else {
                return copy.copy(destinationRID, position, param);
              }
            }
          }
          catch (ResourceException ex) {
            re.append(ex);
          }
        }
      } // while
    } // if (copyChildren)

    // auto-remove of non-affected members in target collection
    if (!autoDeleteMap.isEmpty()) {
      log.debugT("internalCopy(778)", "auto-deleting: " + autoDeleteMap);
      Iterator it = autoDeleteMap.values().iterator();
      while (it.hasNext()) {
        IResource todel = (IResource)it.next();
        log.debugT("internalCopy(782)", "deleting: " + todel.getRID());
        todel.delete();
      }
    }

    return newCollection;
  }

  protected IRidList internalDeleteCollectionManager() throws ResourceException {
    return this.getNamespaceManager(true).delete(this);
  }

  protected IRidList internalDeleteCollection()
    throws ResourceException, NotSupportedException, AccessDeniedException {
    if (this.isLink()) {
      this.internalDelete();
      return new RidList();
    }
    else {
      // Test for root collection - can not be deleted
      RID rid = getRID();
      if (rid.equals(getRepositoryManager().getPrefix())) {
        throw new AccessDeniedException("repository root cannot be deleted", rid, "delete", null);
      }

      // Does the manager support deep delete ?
      if (getRepositoryManager().getSupportedOptions(this).isSupported(SupportedOption.DELETE_DEEP)) {
        return this.internalDeleteCollectionManager();
      }

      IRidList deleted = new RidList();

      // Delete all children recursively
      IResourceListIterator it = this.internalGetChildren().listIterator();
      while (it.hasNext()) {
        IResource res = it.next();

        if (res instanceof CollectionLinkImpl) {
          ((CollectionLinkImpl)res).delete();
          deleted.add(res.getRID());
        }
        else if (res instanceof CollectionImpl) {
          IRidList subUris = ((CollectionImpl)res).internalDeleteCollection();
          deleted.addAll(subUris);
        }
        else if (res instanceof ResourceImpl) {
          ((ResourceImpl)res).internalDelete();
          deleted.add(res.getRID());
        }
        else {
          log.debugT(
            "internalDeleteCollection(833)",
            "IResource is not instanceof ResourceImpl|CollectionImpl|CollectionLinkImpl, falling back to IResource.delete");
          res.delete();
          deleted.add(res.getRID());
        }
      }

      // Delete this collection
      this.internalDelete();

      deleted.add(getRID());
      return deleted;
    }
  }

  protected OrderType internalGetOrderType() throws ResourceException {
    if (isInternalLink()) {
      return ((ICollection)getTargetResource()).getOrderType();
    }
    else {
      return getNamespaceManager(true).getOrderType(this);
    }
  }

  protected void internalSetOrderType(OrderType orderType)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    if (isInternalLink()) {
      ((ICollection)getTargetResource()).setOrderType(orderType);
    }
    else {
      getNamespaceManager(true).setOrderType(orderType, this);
    }
  }

  protected void internalReorder(IReorderList list)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    if (isInternalLink()) {
      ((ICollection)getTargetResource()).reorder(list);
    }
    else {
      getNamespaceManager(true).reorder(list, this);
    }
  }

  protected ICollection internalCreateCollection(
    String name,
    IPosition position,
    IPropertyMap properties,
    boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    if (isInternalLink()) {
      return ((ICollection)getTargetResource()).createCollection(name, position, properties, ignorePropertyFailures);
    }
    else {
      return getNamespaceManager(true).createCollection(name, position, properties, ignorePropertyFailures, this);
    }
  }

  protected IResource internalCreateResource(
    String name,
    IPosition position,
    IPropertyMap properties,
    IContent content,
    boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    if (isInternalLink()) {
      return ((ICollection)getTargetResource()).createResource(
        name,
        position,
        properties,
        content,
        ignorePropertyFailures);
    }
    else {
      return getNamespaceManager(true).createResource(
        name,
        position,
        properties,
        content,
        ignorePropertyFailures,
        this);
    }
  }

  protected ILockInfo internalCreateResource(String name, ILockProperties lockProperties)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    if (isInternalLink()) {
      return ((ICollection)getTargetResource()).createResource(name, lockProperties);
    }
    else {
      return getNamespaceManager(true).createResource(name, lockProperties, this);
    }
  }

  protected IResource internalCreateLink(
    String name,
    URL targetURL,
    LinkType linkType,
    IPosition position,
    IPropertyMap properties,
    boolean ignorePropertyFailures)
    throws ResourceException, NotSupportedException, AccessDeniedException {
    if (isInternalLink()) {
      return ((ICollection)getTargetResource()).createLink(
        name,
        targetURL,
        linkType,
        position,
        properties,
        ignorePropertyFailures);
    }
    else {
      return getNamespaceManager(true).createLink(
        name,
        targetURL.toString(),
        linkType,
        position,
        properties,
        ignorePropertyFailures,
        this);
    }
  }

  protected IResourceList internalGetChildrenManager() throws ResourceException {
    return this.getNamespaceManager(true).getChildren(this);
  }

  protected IResourceList internalGetChildren() throws ResourceException, AccessDeniedException {
    if (this.isInternalLink()) {
      IResource target = this.getTargetResource();
      if (target == null) {
        throw new ResourceNotFoundException("link target not found", RID.getRID(getTargetURL().toString()));
      }
      else if (target instanceof ICollection) {
        return ((ICollection)getTargetResource()).getChildren();
      }
      else {
        throw new InvalidTargetException("link target is not collection", getRID());
      }
    }
    else {
      IResourceList children = this.internalGetChildrenManager();
      ResourceImpl.mapLink(children, this.getRepositoryManager());
      return children;
    }
  }

  protected IResourceList internalGetChildren(Boolean visible, Boolean collections, Boolean links)
    throws ResourceException, AccessDeniedException {
    if (this.isInternalLink()) {
      IResource target = this.getTargetResource();
      if (target == null) {
        throw new ResourceNotFoundException("link target not found", RID.getRID(getTargetURL().toString()));
      }
      else if (target instanceof ICollection) {
        return ((ICollection)getTargetResource()).getChildren();
      }
      else {
        throw new InvalidTargetException("link target is not collection", getRID());
      }
    }
    else {
      IResourceList children = this.getExtendedNamespaceManager(true).getChildren(visible, collections, links, this);
      ResourceImpl.mapLink(children, this.getRepositoryManager());
      return children;
    }
  }

  // ---------------------------------------------------------------------------

  private Map getInternalMembersOf(ICollection collection) throws ResourceException {

    Map map = new HashMap();

    if (((CollectionImpl)collection).isInternalLink()) {
      return map;
    }

    IResourceListIterator it = collection.getChildren().listIterator();
    while (it.hasNext()) {
      IResource member = it.next();
      map.put(member.getName(), member);
    }

    return map;
  }

  /**
   * Apply the write filter to the content
   *
   * @exception ResourceException Exception raised in failure situation
   */
  protected IContent applyContentWriteFilterForCreate(String uri, IContent content) throws ResourceException {
    try {
      return CmSystem.getInstance().getFilterHandler().applyContentWriteFilterOnCreate(
        getRepositoryManager(),
        uri,
        content);
    }
    catch (CmStartupException ex) {
      throw new ResourceException(ex);
    }
  }

  /**
   * Apply write filter to properties
   *
   * @exception ResourceException Exception raised in failure situation
   */
  protected IPropertyMap applyPropertyWriteFilterForCreate(String uri, IPropertyMap properties)
    throws ResourceException {
    try {
      return CmSystem.getInstance().getFilterHandler().applyPropertyWriteFilterOnCreate(
        getRepositoryManager(),
        uri,
        properties,
        PropertyFilterMode.ALL_PROPERTIES);
    }
    catch (CmStartupException ex) {
      throw new ResourceException(ex);
    }
  }

  /**
   * Apply namespace filter
   *
   * @exception ResourceException Exception raised in failure situation
   */
  protected IResourceList applyNamespaceReadFilter(IResourceList resources) throws ResourceException {
    try {
      return CmSystem.getInstance().getFilterHandler().applyNamespaceReadFilter(
        getRepositoryManager(),
        this,
        resources,
        NamespaceFilterMode.GET_CHILDREN);
    }
    catch (CmStartupException ex) {
      throw new ResourceException(ex);
    }
  }

  private IResource map(IResource resource) throws ResourceException {
    try {
      return ResourceImpl.mapLink(resource, false);
    }
    catch (ResourceException e) {
      if (log.beDebug()) {
        log.debugT("map(1054)", "mapping resource " + resource + ": " + LoggingFormatter.extractCallstack(e));
      }
      return resource;
    }
  }

  /**
   * Set the AccessRIDs for the children.
   *
   * @exception ResourceException Exception raised in failure situation
   */
  private void completeResourceList(IResourceList children) throws ResourceException {
    if (!(children instanceof ICompletedResourceList)) {
      RID thisAccessRID = this.getAccessRID();
      IResource child = null;

      /*
       * ISecurityManager sm = this.getRepositoryManager().getSecurityManager(this);
       * IRidList ridList = null;
       * if (sm != null) ridList = new RidList();
       */
      for (int i = 0, s = children.size(); i < s; i++) {
        child = children.get(i);
        if (child instanceof ResourceImpl) {
          ((ResourceImpl)child).setAccessRID(thisAccessRID.add(child.getRID().name()));
        }
        else if (child instanceof CollectionLinkImpl) {
          ((CollectionLinkImpl)child).setAccessRID(thisAccessRID.add(child.getRID().name()));
        }
        else {
          // this should not happen
          CollectionImpl.log.errorT(
            "completeResourceList(1088)",
            "Failed to set AccessRID for child: " + child.getRID() + ", resouce=" + child.getClass().getName());
        }
        //if (sm != null) ridList.add(((IResource)child).getRID());
      }

      // Pre-fill ACL caches
      // 18.3.2002: too expensive (first call is 5 times slower with this code)
      /*
       * if (sm != null && ridList.size() > 0) {
       * IResourceContext ctxt = this.getContext();
       * IUser user = ctxt.getUser();
       * if (user != null) {
       * PermissionList permissions = new PermissionList();
       * permissions.add(sm.getPermission(IPermission.PERMISSION_READ_PROPERTIES));
       * try {
       * sm.isAllowed(ridList, ctxt.getUser(), permissions, ctxt);
       * }
       * catch (NotSupportedException ignored) {}
       * }
       * }
       */
    }
  }

  /**
   * RF internal method - also used in AbstractRepositoryManager.
   * Remove resources from a resource list for which the user in the context does
   * not have ALL of the specified permissions.
   */
  public static void removeResourcesWithoutPermission(
    ISecurityManager sm,
    IResourceList list, String[] permissionNames, IResourceContext context)
    throws ResourceException {
    if (list == null) return;
    if (permissionNames == null) return;
    if (sm == null) return;

    // Get list of permission objects
    PermissionList permList = new PermissionList();
    for (int i = 0; i < permissionNames.length; i++) {
      IPermission perm = sm.getPermission(permissionNames[i]);
      if (perm == null) {
        log.errorT("removeResourcesWithoutPermission(827)", "unknown permission: " + permissionNames[i]);
      }
      else {
        permList.add(perm);
      }
    }

    // Make a list of RIDs for the ACL mass call
    IRidList rids = new RidList();
    for (int i = 0; i < list.size(); i++) {
      rids.add(list.get(i).getRID());
    }
    IRidSet allowedRIDs = null;
    try {
      allowedRIDs = sm.isAllowed(rids, context.getUser(), permList, context);
    }
    catch (NotSupportedException ex) {
      // Fallback: Use call without RidList or PermissionList parameter
      allowedRIDs = new RidSet(list.size());
      for (int i = 0; i < list.size(); i++) {
        IResource res = list.get(i);
        boolean allowed = true;
        try {
          allowed = sm.isAllowed(res, context.getUser(), permList);
        }
        catch (NotSupportedException e) {
          IPermissionListIterator it = permList.iterator();
          while (it.hasNext() && allowed) {
            IPermission perm = it.next();
            allowed = allowed && sm.isAllowed(res, context.getUser(), perm);
          }
        }
        if (allowed) {
          allowedRIDs.add(res.getRID());
        }
      }      
    }
    
    // Remove resources from list that are not in the set of allowed ones
    for (int i = list.size() - 1; i >= 0; i--) {
      if (!allowedRIDs.contains(list.get(i).getRID())) {
        list.remove(i);
      }
    }
  }

}
