/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.sf.service/60NW_SP_COR/src/_taskqueue/java/api/com/sapportals/wcm/service/taskqueue/TaskState.java#3 $
 */

package com.sapportals.wcm.service.taskqueue;

import com.sapportals.wcm.util.enum.AbstractEnum;

import java.util.HashMap;

/**
 * This class defines the identifiers for the different lifecycle states of a
 * task in the task queue service. <p>
 *
 * If a new task is added to the task queue the task's state will be set to
 * {@link #NEW}. If a task is fetched from the queue it's state will be set to
 * {@link #ACTIVE} to indicate that the task is currently being processed. The
 * processor itself should set the task's state to {@link #DONE} ifthe task was
 * processed sucessfully or {@link #ERROR} if the task failed. <p>
 *
 * If an active task is added to ther queue an exception will be raised. It is
 * possible to add tasks that are done (have the {@link #DONE}-state) to run
 * them once again. <p>
 *
 * <!--COPYRIGHT TAG:-->Copyright 2004 SAP AG<!-- -->
 *
 * @author Jens Kaiser
 * @version $Id: //kmgmt/bc.sf.service/dev/src/_taskqueue/java/api/com/sapportals/wcm/service/taskqueue/TaskState.java#3
 *      $
 */
public class TaskState extends AbstractEnum {

  /**
   * A task added to the task queue is set to this state.
   */
  public final static TaskState NEW = new TaskState("NEW");

  /**
   * A task fetched from the task queue is set to this state by the task queue
   * service.
   *
   * @see ITaskQueueReader#get()
   * @see ITaskQueue#get()
   */
  public final static TaskState ACTIVE = new TaskState("ACTIVE");

  /**
   * A processor that failed to execute this tasks sets this state.
   */
  public final static TaskState ERROR = new TaskState("ERROR");

  /**
   * A processor that sucessfully finished the task sets this state.
   */
  public final static TaskState DONE = new TaskState("DONE");

  /**
   * Task state name as an identifier.
   */
  private String name;

  /**
   * Look-up table to resolve the task state object out of its order number.
   *
   * @see #getTaskState(int)
   */
  private static HashMap stateMap;

  /**
   * An order-number counter
   */
  private static int state;


  //----------------------------------------------------------------------------
  // Init
  //----------------------------------------------------------------------------



  /**
   * Constructor for the TaskState object
   *
   * @param name The human-readable name of this state
   */
  private TaskState(String name) {
    super(++TaskState.state);
    if (stateMap == null) {
      stateMap = new HashMap();
    }
    this.name = name.trim().toUpperCase();
    stateMap.put(new Integer(TaskState.state), this);
  }


  //----------------------------------------------------------------------------
  // Properties
  //----------------------------------------------------------------------------



  /**
   * Gets the value attribute of the TaskState object
   *
   * @return The value value
   */
  public int getValue() {
    return super.getValue();
  }


  //----------------------------------------------------------------------------
  // Object
  //----------------------------------------------------------------------------



  /**
   * Overrides java.lan.Object.toString
   *
   * @return TBD: Description of the outgoing return value
   */
  public String toString() {
    return name;
  }


  //----------------------------------------------------------------------------
  // Helpers
  //----------------------------------------------------------------------------



  /**
   * Converts a task state string representation to a task state object
   *
   * @param state Description of the Parameter
   * @return The task state object or <code>null</code> if this state is unknown
   */
  public static TaskState getTaskState(int state) {
    return (TaskState)stateMap.get(new Integer(state));
  }

}
