/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.sf.service/60NW_SP_COR/src/_taskqueue/java/api/com/sapportals/wcm/service/taskqueue/ITaskQueueWriter.java#2 $
 */

package com.sapportals.wcm.service.taskqueue;

import com.sapportals.wcm.WcmException;

import java.util.Properties;

/**
 * A writer provides write-acces to a task queue belonging to a specific
 * namespace. In addition, it acts as a validating factory for tasks. <p>
 *
 * @author Jens Kaiser
 * @version $Revision: #2 $
 */
public interface ITaskQueueWriter {

  /**
   * Adds an entry to the end of the queue. Raises an excpetion if the entry is
   * already added to the queue. <p>
   *
   * The task' state is set {@link TaskState#NEW NEW} during this operation.
   * According to the persistence layer a correct task id will be created.
   *
   * @param task The task to add
   * @exception WcmException raised in failure situation
   */
  ITask add(ITask task) throws WcmException;

  /**
   * Sets the task's state to a new <code>state</code> . The task must belong to
   * the writer's namespace. The new state must be a legal transition of the old
   * state (see {@link ITask#getState() ITask.getState()}) for details.
   *
   * @param state The state to set.
   * @param task The task to set the state.
   * @exception WcmException Exception raised in failure situation
   */
  void setState(ITask task, TaskState state) throws WcmException;

  /**
   * Factory method to create task objects. Constructs a task object with
   * correct settings for user, namespace, date of creation, priortiy set to
   * {@link ITask#NORM_PRIORITY normal}.
   *
   * @return A task belonging to the writer's namespace.
   * @exception WcmException Exception raised in failure situation
   */
  ITask createTask() throws WcmException;

  /**
   * Factory method to create task objects. Adds a set of properties to the
   * task. Only string values are allowed as property values.
   *
   * @param properties Adds a set of properties to the task.
   * @return The created task
   * @exception WcmException Exception raised in failure situation
   * @see #createTask()
   */
  ITask createTask(Properties properties) throws WcmException;

  /**
   * Factory method to create task objects. Adds a short descritption <code>
   * description</code> to the task. The description string may be <code>null
   * </code>.
   *
   * @param properties Adds a set of properties to the task.
   * @param description Adds a short descritption <code>description</code> to
   *      the task.
   * @return The created task
   * @exception WcmException Exception raised in failure situation
   * @see #createTask(Properties)
   */
  ITask createTask(Properties properties, String description) throws WcmException;

  /**
   * Factory method to create task objects. Sets a priority <code>priority
   * </code> value to the task. The priory value must lie between {@link
   * ITask#MIN_PRIORITY 0} and {@link ITask#MAX_PRIORITY 100}.
   *
   * @param properties Adds a set of properties to the task.
   * @param description Adds a short descritption <code>description</code> to
   *      the task.
   * @param priority The priority value that affect the processing order of
   *      tasks.
   * @return The created task
   * @exception WcmException Exception raised in failure situation
   * @see #createTask(Properties,String)
   */
  ITask createTask(Properties properties, String description, int priority) throws WcmException;

  /**
   * Returns the user of this writer.
   *
   * @return Returns the user of this writer.
   */
  String getUser();

  /**
   * Returns the namespace of this writer.
   *
   * @return Returns the namespace of this writer.
   */
  String getNamespace();

}
