/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.sf.service/60NW_SP_COR/src/_taskqueue/java/api/com/sapportals/wcm/service/taskqueue/ITaskQueueService.java#3 $
 */

package com.sapportals.wcm.service.taskqueue;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.service.IService;

/**
 * The task queue is a base service for clustered systems. <p>
 *
 * It provides interfaces to write and to read {@link ITask tasks} in a
 * platform-spreading queue and to monitor the tasks and (above all) their
 * processing state. <p>
 *
 * Access to the task queue is provided by a {@link ITaskQueueReader} and a
 * {@link ITaskQueueWriter} interface. This thin interfaces support the
 * implementation of highly spezialised and customizable distributed services.
 * (For example a service could implement a customized service writer as a
 * wrapper of the task queue writer and a customized service daemon reader as a
 * wrapper of the reader). <p>
 *
 * Reader and writer are assigned to a special namespace and a service user. A
 * writer can only write tasks belonging to its namespace. A reader can only
 * read task of its namespace. A namespace is a unique application identifier.
 * <p>
 *
 * A task queue consists of a list (a sequence) of tasks. The tasks can be added
 * to and removed from the list on a priority basis ({@link ITask#MIN_PRIORITY
 * lowest} priority values first, first-in-first-out (FIFO) basis within
 * priority). <p>
 *
 * This services provides support of different persistence layers. It is
 * recommended to use the database persistence layer in production systems. File
 * system and memory persistence layers are only recommended for testing to
 * offer a high-performance test plattform with some restrictions. <p>
 *
 * The monitor methods will return only a snapshot of the current task queue.
 * For maintenance a remove method is provided to remove completly progressed or
 * incorrectly aborted tasks from the queue.
 *
 * @author Jens Kaiser
 * @version $Id: //kmgmt/bc.sf.service/60NW_SP_COR/src/_taskqueue/java/api/com/sapportals/wcm/service/taskqueue/ITaskQueueService.java#3 $ 
 * 
 * Copyright (c) SAP AG 2003
 */
public interface ITaskQueueService extends IService {

  /**
   * identifier for TaskQueue Service
   */
  String TASKQUEUE_SERVICE = "TaskQueueService";


  /**
   * Returns a task queue writer for write acccess to the queue service.
   *
   * @param user The user of this interface
   * @param namespace The namespace this writer is assigned to.
   * @return A new task queue writer.
   * @exception WcmException Exception raised in failure situation
   */
  ITaskQueueWriter getTaskQueueWriter(String user, String namespace)
    throws WcmException;


  /**
   * Returns a task queue reader for read acccess to the queue service.
   *
   * @param user The user of this interface
   * @param namespace The namespace this reader is assigned to.
   * @return A new task queue reader.
   * @exception WcmException Exception raised in failure situation
   */
  ITaskQueueReader getTaskQueueReader(String user, String namespace)
    throws WcmException;


  /**
   * Returns an iterator for tasks belonging to <code>service</code> . This is a
   * snapshot of the current task queue. It does not change during time.
   *
   * @param namespace The namespace to monitor
   * @return An iterator that includes the snapshot.
   * @exception WcmException Exception raised in failure situation
   */
  ITaskQueueIterator monitorTaskQueue(String namespace)
    throws WcmException;


  /**
   * Returns an iterator for tasks that have task state <code>state</code> .
   * This is a snapshot of the current task queue. It does not change during
   * time.
   *
   * @param state The state to monitor
   * @return An iterator that includes the snapshot.
   * @exception WcmException Exception raised in failure situation
   */
  ITaskQueueIterator monitorTaskQueue(TaskState state)
    throws WcmException;


  /**
   * Returns a task with id <code>id</code>
   */
  ITask monitorTask(String id)
    throws WcmException;
  
  /**
   * Returns an iterator for all queued tasks. This is a snapshot of the current
   * task queue. It does not change during time.
   *
   * @return The total queued tasks.
   * @exception WcmException Exception raised in failure situation
   */
  ITaskQueueIterator monitorTaskQueues()
    throws WcmException;


  /**
   * Removes a task that is queued by this service. The task to remove must have
   * been processed (task state was set to {@link TaskState#DONE} or {@link
   * TaskState#ERROR}).
   *
   * @param task The task to remove
   * @return indicates if the task was removed (if it was contained in the queue
   *      service)
   * @exception WcmException Raised on illegal task states, persistence
   *      failures,...
   */
  boolean removeTask(ITask task)
    throws WcmException;
}
