/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.sf.service/60NW_SP_COR/src/_taskqueue/java/api/com/sapportals/wcm/service/taskqueue/ITaskQueueReader.java#2 $
 */

package com.sapportals.wcm.service.taskqueue;

import com.sapportals.wcm.WcmException;

/**
 * A reader provides read-acces to a task queue belonging to a specific
 * namespace. <p>
 *
 * Copyright (c) SAP AG 2003
 *
 * @author Jens Kaiser
 * @version $Id: //kmgmt/bc.sf.service/dev/src/_taskqueue/java/api/com/sapportals/wcm/service/taskqueue/ITaskQueueReader.java#3
 *      $
 */
public interface ITaskQueueReader {

  /**
   * Removes the task at the top of the task queue and returns it. 
   * The head-element is the first added (and currently not fetched) task with 
   * the highest priority.<p>
   *
   * The task' state (former set to {@link TaskState#NEW NEW } by a {@link
   * ITaskQueueWriter writer} is set {@link TaskState#ACTIVE ACTIVE} during this
   * operation.
   *
   * @return Returns the next task to process or <code>null</code> if no task
   *      exists.
   * @exception WcmException Exception raised in failure situation
   */
  ITask get()
    throws WcmException;


  /**
   * Removes the task at the top of the task queue and returns it. 
   * The head-element is the first added (and currently not fetched) task with 
   * the highest priority.<p>
   *
   * This method returns when a task is found in the queue or a <code>timeout
   * </code> delay is reached.<code> The task' state (former set to {@link
   * TaskState#NEW NEW } by a {@link ITaskQueueWriter writer} is set {@link
   * TaskState#ACTIVE ACTIVE} during this operation.
   *
   * @param timeout The timeout delay in ms. If <code>timeout</code> is 0 the
   *      method behaves like {@link #get()}. When set to a negative value it
   *      will never timeout.
   * @return Returns the next task to process or <code>null</code> if no task
   *      exists.
   * @exception WcmException Exception raised in failure situation
   * @see #get()
   */
  ITask get(long timeout)
    throws WcmException;


  /**
   * Returns the top elemet of the task queue. The head-element is the first added (and currently not fetched) task with 
   * the highest priority.<p>
   * @return
   */
  ITask top()
    throws WcmException;

  /**
   * Sets the task's state to a new <code>state</code> . The task must belong to
   * the reader's namespace. The new state must be a legal transition of the old
   * state (see {@link ITask#getState() ITask.getState()}) for details. <p>
   *
   * Interactive users of the taskqueue service are only allowed to set the
   * state to {@link TaskState#DONE} and {@link TaskState#ERROR}. Other states
   * are set by the task queue service.
   *
   * @param state The state to set.
   * @param task The task to set the state.
   * @exception WcmException Exception raised in failure situation
   */
  void setState(ITask task, TaskState state)
    throws WcmException;


  /**
   * Indicates if the queue still holds some tasks to process (and the {@link
   * #get() get-method} will return a task.
   *
   * @return Indicates if the queue still holds some tasks to process.
   */
  boolean isEmpty();


  /**
   * Returns the user of this reader.
   *
   * @return Returns the user of this reader.
   */
  String getUser();


  /**
   * Returns the namespace of this reader.
   *
   * @return Returns the namespace of this reader.
   */
  String getNamespace();

}
