/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.sf.service/60NW_SP_COR/src/_taskqueue/java/api/com/sapportals/wcm/service/taskqueue/ITaskQueue.java#2 $
 */

package com.sapportals.wcm.service.taskqueue;

import com.sapportals.wcm.WcmException;

/**
 * Priority queue for tasks. <p>
 *
 * This interface provides a FIFO container for {@link ITask}. Tasks that are
 * {@link #add(ITask) added} to this queue first are {@link #get() fetched} first if they
 * have the same or higher priority than other tasks in the queue. <p>
 *
 * A queue is assigned to one and only one (case-sensitive} {@link #getNamespace()
 * namespace} by the underlying task queue service. Therefore a queue manages
 * read-write access for tasks belonging to one specific namespace. <p>
 *
 * The following example provides a possible usage scenario for a simple
 * single-task system: <p>
 *
 * An application or service (named <i>A</i> ) may write the following tasks to
 * the queue: <pre>
 *   ITaskQueueReader writer = this.service.getTaskQueueWriter( this.user, SERVICE_ID );
 *   // create some tasks
 *   ITask entry1 = writer.createTask( properties1, "entry1" );
 *   ITask entry2 = writer.createTask( properties2, "entry2" );
 *   ITask entry3 = writer.createTask( properties3, "entry3" );
 *   ITask entry4 = writer.createTask( properties4, "entry4" );
 *   ITask maxPrio = writer.createTask( properties5, "maxPrio", ITask.MAX_PRIORITY );
 *   ITask minPrio = writer.createTask( properties6, "maxPrio", ITask.MIN_PRIORITY );
 *
 *   // add 4 tasks to the queue (all have the same priority value)
 *   writer.add( entry1 );
 *   writer.add( entry2 );
 *   writer.add( entry3 );
 *   writer.add( entry4 );
 * </pre> Somewhere else another application (named <i>B</i> ) tries to read
 * some tasks: <pre>
 *   ITaskQueueReader reader = this.service.getTaskQueueReader( this.user, SERVICE_ID );
 *
 *   // fetch the tasks
 *   ITask task1 = reader.get();   // gets entry1
 *   ITask task2 = reader.get();   // gets entry2
 *   ITask task3 = reader.get();   // gets entry3
 *   ITask task4 = reader.get();   // gets entry4
 * </pre> The reading results may differ if another application writes tasks at
 * the same time to the queue. <p>
 *
 * Some other examples: <p>
 *
 * <i>A</i> writes tasks in the order <code>minPrio, entry1, maxPrio, entry3,
 * entry2</code> . <i>B</i> should read: <code>maxPrio, entry1, entry3, entry2,
 * minPrio</code> (in this order). <p>
 *
 * <i>A</i> writes tasks in the order <code>entry1, entry2, entry3, maxPrio,
 * minPrio</code> . <i>B</i> should read: <code>maxPrio, entry1, entry2, entry3,
 * minPrio</code> <p>
 *
 * Copyright (c) SAP AG 2003
 *
 * @author Jens Kaiser
 * @version $Revision: #2 $
 */
public interface ITaskQueue {

  /**
   * Returns the service namespace this queue is assigned.
   *
   * @return The service value
   */
  String getNamespace();


  /**
   * Adds an entry to the end of the queue. Raises an excpetion if the entry was
   * already added to the queue. <p>
   *
   * A queue consists of a list (a sequence) of tasks. The tasks can be added to
   * and removed from the list on a priority basis (lowest priority values
   * first, first-in-first-out (FIFO) basis within priority). <p>
   *
   * The task's state is set to {@link TaskState#NEW} during this operation. A
   * {@link TaskStateTransitionException} will be thrown if the task's state
   * already was set to an illegal state.
   *
   * @param task The entry
   * @exception WcmException Exception raised in failure situation
   * @see TaskState
   */
  ITask add(ITask task)
    throws WcmException;


  /**
   * Removes and returns a task from the head of the queue. The task's state is set to {@link
   * TaskState#ACTIVE} during this operation.
   *
   * @return The head entry
   * @exception WcmException Exception raised in failure situation
   */
  ITask get()
    throws WcmException;


  /** 
   * Returns a task from the head of the queue.
   *
   * @return The head entry
   * @exception WcmException Exception raised in failure situation
   */
  ITask top()
    throws WcmException;


  /**
   * Removes a task that is queued by this service. The task to remove must have
   * been processed (task state was set to {@link TaskState#DONE} or {@link
   * TaskState#ERROR}) to avoid non-consistend runtime states.
   *
   * @param task The task to remove
   * @return true if the task was successful removed
   * @exception WcmException Exception raised in failure situation
   */
  boolean removeTask(ITask task)
    throws WcmException;


  /**
   * Creates an iterator for the task queue {@link ITask entries}.
   *
   * @return A new iterator object
   */
  ITaskQueueIterator iterator();


  /**
   * Indicates if the queue is empty.
   *
   * @return queue is empty.
   * @exception WcmException Exception raised in failure situation
   */
  boolean isEmpty()
    throws WcmException;
}
