package com.sap.caf.core.besrv.principal;

import com.sap.caf.rt.exception.ServiceException;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.security.api.IPrincipal;
import com.sap.security.api.IPrincipalFactory;
import com.sap.security.api.IUser;
import com.sap.security.api.UMException;
import com.sap.security.api.UMFactory;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;
import com.sap.caf.rt.bol.EntityServiceBase;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.EJBLocalObject;
import javax.ejb.RemoveException;

import com.sap.caf.rt.bol.IBusinessObject;
import com.sap.caf.rt.exception.CAFPermissionException;
import com.sap.caf.rt.security.acl.impl.CAFPermission;

/**
 * @ejbLocal <{com.sap.caf.authorization.api.services.principal.PrincipalServiceLocal}>
 * @ejbLocalHome <{com.sap.caf.authorization.api.services.principal.PrincipalServiceLocalHome}>
 * @stateless 
 */
public class PrincipalServiceBean extends EntityServiceBase implements SessionBean {

        private static final String APPLICATION = PrincipalServiceBean.class.getName();
        private static final Location logger = Location.getLocation(APPLICATION);
        private static final String jARMReqPrefix = "CAF:CORE:security:";
        private static final String jARMRequest = jARMReqPrefix+APPLICATION;

        private static final String UNDEFINED_VALUE = "<undefined>";

	   /**
		* Reads a principal from the UME by given id.
		* @param principalID principal id.
		* @return <code>Principal</code> data object.
		* @throws ServiceException if some error is encountered.
		*/
        public Principal read(String principalID) throws ServiceException {
                final String method = jARMRequest + ":read(String principalID)";
                CAFPublicLogger.entering(null, jARMRequest, method, logger, 1);
        
                Principal principal = null;
                try {
                        IPrincipalFactory principalFactory = UMFactory.getPrincipalFactory();
                        IPrincipal iPrincipal = principalFactory.getPrincipal(principalID);
                        
                        principal = new Principal();
                        principal.setPrincipalID(principalID);
                        principal.setPrincipalName(iPrincipal.getDisplayName());
                        principal.setPrincipalType(principalFactory.getPrincipalType(principalID));
                        String[] description = iPrincipal.getAttribute(IPrincipal.DEFAULT_NAMESPACE, IPrincipal.DESCRIPTION);
                        if ((description!=null) && (description.length>0)) {
                                principal.setDescription(description[0]);
                        }
                        if (iPrincipal instanceof IUser) {
                                principal.setEmail(((IUser)iPrincipal).getEmail());
                        } else {
                                principal.setEmail(UNDEFINED_VALUE);
                        }
                } catch (UMException e) {
                        CAFPublicLogger.traceThrowable(Severity.ERROR, logger, method, e);
                        throw new ServiceException(e);
                } finally {
                        CAFPublicLogger.exiting(null, jARMRequest, method, logger, 1);
                }
                return principal;
        }
	
	
	protected SessionContext getSessionContext() {
			return sessionContext;
	}

	public void ejbRemove() {
	}

	public void ejbCreate() throws CreateException {
	}

	protected final static void checkParameter(String parameterValue, String name)
			throws ServiceException {
			if (parameterValue == null || parameterValue.length() == 0)
					throw new ServiceException(new Exception("Parameter " + name + " is missed"));
	}

	protected String getKey(String pk) {
			return (pk == null || pk.length() <= 2) ? pk : pk.substring(1, pk.length() - 1);

	}

	private static final String[] STRING_ARRAY_TYPE = new String[0];

	protected String getCurrentUserID() {
			final String method = jARMRequest + ":getCurrentUserID()";
			String loginUserID = null;
			try {
					String callerName = getSessionContext().getCallerPrincipal().getName();
					if (callerName != null) {
							final IUser user = UMFactory.getUserFactory().getUserByLogonID(callerName);
							loginUserID = user.getUniqueID();
					}
			} catch (UMException e) {
					CAFPublicLogger.traceThrowable(Severity.ERROR, logger, method, e);
			}
			return loginUserID;
	}

	public static void release(EJBLocalObject object) {
			final String method = jARMRequest + ":release(EJBLocalObject object)";
			if (object != null) {
					try {
							object.remove();
					} catch (EJBException e) {
							CAFPublicLogger.traceThrowable(Severity.ERROR, logger, method, e);
					} catch (RemoveException e) {
							CAFPublicLogger.traceThrowable(Severity.ERROR, logger, method, e);
					}
			}
	}

	public String getUser() {
			String user = null;
			final String method = jARMRequest + ":getUser()";
			try {
					user = (null != sessionContext) ? sessionContext.getCallerPrincipal().getName() : null;
			} catch (Exception e) {
					CAFPublicLogger.traceThrowable(Severity.ERROR, logger, method, e);
			}
			return user;
	}
    
	public String getApplicationName() {
			return APPLICATION;
	}

	public boolean checkPermission(String permission, IBusinessObject object, String user, String objectName) {
			try {
					if (!CAFPermission.checkAclPermission(object, user, permission, objectName)) {
							return false;
					}
			} catch (CAFPermissionException e) {
					return false;
			}
			return true;
	}
	
}