package com.sap.caf.core.besrv.document;

import java.util.*;
import java.util.Collection;
import java.util.Date;

import javax.ejb.SessionBean;
import javax.ejb.CreateException;
import javax.naming.Context;
import javax.naming.InitialContext;

import com.sap.caf.km.da.DocumentDataAccessService;
import com.sap.caf.km.da.KMDataAccessHelper;
import com.sap.caf.km.da.RidUtils;
import com.sap.caf.km.ejb.svc.idxsearch.IIndexSearch;
import com.sap.caf.km.ejb.svc.idxsearch.bean.IndexSearchLocalHome;
import com.sap.caf.metamodel.Application;
import com.sap.caf.metamodel.DataObject;
import com.sap.caf.rt.bol.EntityServiceBase;
import com.sap.caf.rt.bol.IBusinessObject;
import com.sap.caf.rt.bol.context.CAFContext;
import com.sap.caf.rt.bol.da.IDataAccessService;
import com.sap.caf.rt.bol.pk.PrimaryKeyFactory;
import com.sap.caf.rt.bol.util.IntQueryFilter;
import com.sap.caf.rt.bol.util.QueryFilter;
import com.sap.caf.rt.exception.*;
import com.sap.caf.rt.metamodel.MetaModel;
import com.sap.caf.rt.security.acl.impl.CAFPermission;
import com.sap.caf.rt.security.util.CAFPermissionName;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.tc.logging.Location;
import com.sap.tc.logging.Severity;

/**
 * @ejbLocal <{com.sap.caf.core.besrv.document.DocumentServiceLocal}>
 * @ejbLocalHome <{com.sap.caf.core.besrv.document.DocumentServiceLocalHome}>
 * @stateless 
 */
public class DocumentServiceBean extends EntityServiceBase implements SessionBean {

	private IDataAccessService dataAccessService;
	private IIndexSearch m_indexSearch;
	protected String _objectName = "sap.com/caf.core/Document";

	private static final String APPLICATION = DocumentServiceBean.class.getName();
	private static final String jARMReqPrefix = "XAP:BO:"; 
	private static final String JARM_REQUEST =  jARMReqPrefix + APPLICATION;		
	private static final Location location = Location.getLocation(APPLICATION);
	
	public void ejbRemove() {
		final String method = JARM_REQUEST + ":ejbRemove()";
		CAFPublicLogger.entering(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_DETAIL);
		dataAccessService.destroy();
		CAFPublicLogger.exiting(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_DETAIL);
	}

	public void ejbCreate() throws CreateException {
		final String method = JARM_REQUEST + ":ejbCreate()";
		CAFPublicLogger.entering(null, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_BASIC);
		int backendType;
		try {
			dataAccessService = DocumentDataAccessService.getInstance();
			dataAccessService.activate();
		} catch (Exception e) {
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error in ejbCreate()",
				e);
			throw new CreateException();
		} finally {
			CAFPublicLogger.exiting(
				null,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_BASIC);
		}
	}


	/**
	 * Create document instance.
	 * 
	 * @param kmParentFolder	Parent KM Folder RID
	 * @param title				Document title
	 * @return					New Document instance
	 * @throws CAFCreateException
	 */
	public Document create(String kmParentFolder, String title) throws CAFCreateException {
		final String method = JARM_REQUEST + ":create(String, String)";
		String user = sessionContext.getCallerPrincipal().getName();
		CAFPublicLogger.entering(
			user,
			JARM_REQUEST,
			method,
			location,
			new Object[] { kmParentFolder, title},
			CAFPublicLogger.LEVEL_MEDIUM);
			
		if (!checkPermission(CAFPermissionName.create, null, user, Document.OBJECT_NAME)) {
			Object[] permissionExceptionArgs = {user, CAFPermissionName.create, null, Document.OBJECT_NAME};
			CAFPublicLogger.exiting(user, Document.JARM_REQUEST, method, location);
			throw new CAFCreateException("BO_PERMISSION", permissionExceptionArgs);
		}
						
		Document document = new Document();
		document.setTitle(title);		

		// Set the administrational attributes
		Date changeDate = new Date();
		document.setCreatedBy(user);
		document.setCreatedAt(changeDate);
		document.setLastChangedBy(user);
		document.setLastChangedAt(changeDate);		
		
		try {
			String parentFolder=null;
			if (kmParentFolder==null || kmParentFolder.length()==0) {
				parentFolder = KMDataAccessHelper.getTempFolderRid();
			} else {
				parentFolder = kmParentFolder;
			}
			
			document.setParentFolder(parentFolder);
		
			DocumentPK primaryKey;
			primaryKey =
						new DocumentPK(PrimaryKeyFactory.getInstance().getPrimaryKey());

			document.setKey(primaryKey.key);

			//For new documents set KM DocumentID equal to CAF Document KEY 
			document.setDocumentId(primaryKey.key);						

//			Next line was added as workaround due to description is mandatory attribute for createResource operation in EP6SP6
			document.setDescription("Created by " + _objectName + " CAF service");
			
			dataAccessService.create(document);
			Document newDoc = read(document.getKey());
			document.setCreatedAt(newDoc.getCreatedAt());
			document.setLastChangedAt(newDoc.getLastChangedAt());
			
			CAFPermission.createOwnerPermission(user, primaryKey.key);
		} catch (DataAccessException e) {						
			location.throwing(method, e);
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new CAFCreateException("BO_CREATE", e);
		} catch (Exception e) {
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error in " + method,
				e);
			location.throwing(method, e);
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new CAFCreateException("BO_CREATE", e);
		}

		if (CAFContext.PUBLISH_ENABLED) {
			try {
				if (pub == null) {
					initPublishing();
				}
				pub.publishCreated(getPublishingArgs(document));
			} catch (BEException e) {
				location.throwing(method, e);
				CAFPublicLogger.exiting(
					user,
					JARM_REQUEST,
					method,
					location,
					CAFPublicLogger.LEVEL_MEDIUM);
				throw new CAFCreateException("BO_PUBLISH", e);
			} catch (Exception e) {
				CAFPublicLogger.traceThrowableT(
					Severity.DEBUG,
					location,
					method,
					"Error in " + method,
					e);
				location.throwing(method, e);
				CAFPublicLogger.exiting(
					user,
					JARM_REQUEST,
					method,
					location,
					CAFPublicLogger.LEVEL_MEDIUM);
				throw new CAFCreateException("BO_PUBLISH", e);
			}
		}
		
		CAFPublicLogger.exiting(
			user,
			JARM_REQUEST,
			method,
			location,
			new Object[] { document },
			CAFPublicLogger.LEVEL_MEDIUM);
		return document;		
	}
	

	/**
	 * The method gets Document instance by it KEY
	 * 
	 * @param key	Document KEY 
	 * @return		Instance of Document with specified KEY
	 * @throws CAFRetrieveException
	 */
	public Document read(String key) throws CAFRetrieveException {
		final String method = JARM_REQUEST + ":read(String)";
		String user = sessionContext.getCallerPrincipal().getName();
		CAFPublicLogger.entering(
			user,
			JARM_REQUEST,
			method,
			location,
			new Object[] { key },
			CAFPublicLogger.LEVEL_MEDIUM);
	
		DocumentPK documentPk = new DocumentPK(key);
		Document doc;
		try {
			doc =
				(Document) dataAccessService.load(documentPk, Document.class);
							
		} catch (DataAccessException e) {
			location.throwing(method, e);
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new CAFRetrieveException("BO_READ", e);
		} catch (Exception e) {
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error in " + method,
				e);
			location.throwing(method, e);
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new CAFRetrieveException("BO_READ", e);
		}
		
		if (!checkPermission(CAFPermissionName.read, doc, user, Document.OBJECT_NAME)) {
			Object[] permissionExceptionArgs = { user, CAFPermissionName.read, null, Document.OBJECT_NAME };
			CAFPublicLogger.exiting(user, Document.JARM_REQUEST, method, location);
			throw new CAFRetrieveException("BO_PERMISSION",	permissionExceptionArgs);
		}		

		CAFPublicLogger.exiting(
			user,
			JARM_REQUEST,
			method,
			location,
			new Object[] { doc },
			CAFPublicLogger.LEVEL_MEDIUM);
			
		return doc;
	}

	/**
	 * Store updated document. 
	 * 
	 * @param document		Document instance to update
	 * @throws CAFUpdateException	If document has already been updated or something else goes wrong 
	 */
	public void update(Document doc) throws CAFUpdateException {
		final String method = JARM_REQUEST + ":update(Document)";
		String user = sessionContext.getCallerPrincipal().getName();
		CAFPublicLogger.entering(
			user,
			JARM_REQUEST,
			method,
			location,
			new Object[] { doc },
			CAFPublicLogger.LEVEL_MEDIUM);
			
		if (!checkPermission(CAFPermissionName.update, doc, user, Document.OBJECT_NAME)) {
			Object[] permissionExceptionArgs = {user, CAFPermissionName.update, null, Document.OBJECT_NAME};
			CAFPublicLogger.exiting(user, Document.JARM_REQUEST, method, location);
			throw new CAFUpdateException("BO_PERMISSION", permissionExceptionArgs);
		}			

		// optimistic locking
		Document oldDocument;
		
		try {
			oldDocument = read(doc.getKey());			
		} catch (CAFRetrieveException e) {
			location.throwing(method, e);
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new CAFUpdateException("BO_UPDATE", e);
		} catch (Exception e) {
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error in " + method,
				e);
			location.throwing(method, e);
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new CAFUpdateException("BO_UPDATE", e);
		}

		// check if document has already been updated by other thread
		Date dOldChanged = oldDocument.getLastChangedAt();
		Date dChanged = doc.getLastChangedAt();
	
		if (
			(dOldChanged!=null && !dOldChanged.equals(dChanged)) 
			|| (dOldChanged==null && dChanged!=null))
		{
			CAFUpdateException e = new CAFUpdateException("BO_UPDATE");
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error in " + method,
				e);
			location.throwing(method, e);
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw e;
		}

		// Set the administrational attributes
		doc.setLastChangedBy(user);
		doc.setLastChangedAt(new Date());		

		try {
			dataAccessService.store(doc);
			reindex(doc);
		} catch (DataAccessException e) {
			location.throwing(method, e);
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new CAFUpdateException("BO_UPDATE", e);
		} catch (Exception e) {
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error in " + method,
				e);
			location.throwing(method, e);
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new CAFUpdateException("BO_UPDATE", e);
		}

		if (CAFContext.PUBLISH_ENABLED) {
			Object[] publArgs = null;
			try {
				if (pub == null) {
					initPublishing();
				}
				publArgs = getPublishingArgs(doc);
				pub.publishChanged(publArgs);
			} catch (BEException e) {
				location.throwing(method, e);
				CAFPublicLogger.exiting(
					user,
					JARM_REQUEST,
					method,
					location,
					CAFPublicLogger.LEVEL_MEDIUM);
				throw new CAFUpdateException("BO_PUBLISH", e);
			} catch (Exception e) {
				CAFPublicLogger.traceThrowableT(
					Severity.DEBUG,
					location,
					method,
					"Error in " + method,
					e);
				location.throwing(method, e);
				CAFPublicLogger.exiting(
					user,
					JARM_REQUEST,
					method,
					location,
					CAFPublicLogger.LEVEL_MEDIUM);
				throw new CAFUpdateException("BO_PUBLISH", e);
			}
		}
		CAFPublicLogger.exiting(user, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM);
	}

	/**
	 * Delete Document.
	 * 
	 * @param document	Document instance to delete
	 * @throws CAFDeleteException
	 */
	public void delete(Document doc) throws CAFDeleteException {
		final String method = JARM_REQUEST + ":delete(Document)";
		String user = sessionContext.getCallerPrincipal().getName();
		CAFPublicLogger.entering(
			user,
			JARM_REQUEST,
			method,
			location,
			new Object[] { doc },
			CAFPublicLogger.LEVEL_MEDIUM);
			
		if (!checkPermission(CAFPermissionName.delete, doc, user, Document.OBJECT_NAME)) {
			Object[] permissionExceptionArgs = {user, CAFPermissionName.delete, null, Document.OBJECT_NAME};
			CAFPublicLogger.exiting(user, Document.JARM_REQUEST, method, location);
			throw new CAFDeleteException("BO_PERMISSION", permissionExceptionArgs);
		} 			

		try {
			dataAccessService.remove(doc);
		} catch (DataAccessException e) {
			location.throwing(method, e);
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new CAFDeleteException("BO_DELETE", e);
		} catch (Exception e) {
			CAFPublicLogger.traceThrowableT(
				Severity.DEBUG,
				location,
				method,
				"Error in " + method,
				e);
			location.throwing(method, e);
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
			throw new CAFDeleteException(e);
		}

		if (CAFContext.PUBLISH_ENABLED) {
			Object[] publArgs = null;
			try {
				if (pub == null) {
					initPublishing();
				}
				publArgs = getPublishingArgs(doc);
				pub.publishRemoved(publArgs);
			} catch (BEException e) {
				location.throwing(method, e);
				CAFPublicLogger.exiting(
					user,
					JARM_REQUEST,
					method,
					location,
					CAFPublicLogger.LEVEL_MEDIUM);
				throw new CAFDeleteException("BO_PUBLISH", e);
			} catch (Exception e) {
				CAFPublicLogger.traceThrowableT(
					Severity.DEBUG,
					location,
					method,
					"Error in " + method,
					e);
				location.throwing(method, e);
				CAFPublicLogger.exiting(
					user,
					JARM_REQUEST,
					method,
					location,
					CAFPublicLogger.LEVEL_MEDIUM);
				throw new CAFDeleteException("BO_PUBLISH", e);
			}
		}
		CAFPublicLogger.exiting(user, JARM_REQUEST, method, location, CAFPublicLogger.LEVEL_MEDIUM);
	}

	/**
	 * Find Document by it KEY
	 * 
	 * @param documentKey	Document KEY
	 * @return				Collection of obtained documents
	 * @throws CAFRetrieveException
	 */	
	public Collection findByDocumentKey(String documentKey) throws CAFFindException {
		final String method = JARM_REQUEST + ":findByDocumentKey(String)";
			String user = sessionContext.getCallerPrincipal().getName();
			CAFPublicLogger.entering(
				user,
				JARM_REQUEST,
				method,
				location,
				new Object[] { documentKey },
				CAFPublicLogger.LEVEL_MEDIUM);
		try {
//			Document doc = read(documentKey);
			Document doc = (Document)dataAccessService.findByPrimaryKey(new DocumentPK(documentKey), Document.class);
			
			Collection list = new ArrayList(1);
			
			if (doc!=null && checkPermission(CAFPermissionName.read, doc, user, Document.OBJECT_NAME)) {
				list.add(doc);
			}								
				
			return list;
		} catch (Exception ex) {
			location.throwing(ex);
			throw new CAFFindException(ex);
		} finally {
				CAFPublicLogger.exiting(
							user,
							JARM_REQUEST,
							method,
							location,
							CAFPublicLogger.LEVEL_MEDIUM);
		}
	}
	
	/**
	 * Find Document by it title.
	 * The method performs search in KM
	 * 
	 * @param parentFolder	KM folder to search 
	 * @param title			Document Title to search
	 * @return	Collection of obtained documents
	 * @throws	CAFFindException
	 */
	public Collection findByTitle(String parentFolder, String title) throws CAFFindException {
		final String method = JARM_REQUEST + ":findByTitle(String, String)";
		String user = sessionContext.getCallerPrincipal().getName();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { parentFolder, title},
					CAFPublicLogger.LEVEL_MEDIUM);

		try {			
			IntQueryFilter fParentFolder = new IntQueryFilter(QueryFilter.createStringFilter(
							QueryFilter.ACTION_EXACT, parentFolder, "parentFolder", true));
			fParentFolder.setAttribute("parentFolder");					
			fParentFolder.condition = QueryFilter.CONDITION_EQ;
						
			IntQueryFilter fTitle = new IntQueryFilter(QueryFilter.createStringFilter(
							QueryFilter.ACTION_EXACT, title, "title", true));
			fTitle.setAttribute("title");
			fTitle.condition = QueryFilter.CONDITION_EQ;
//			
			Collection resultQuery = dataAccessService.query(Document.class, new IntQueryFilter[] {fTitle, fParentFolder}, null);					
			return checkPermissions(resultQuery, CAFPermissionName.create, user, Document.OBJECT_NAME);
		} catch (Exception ex) {
			location.throwing(ex);
			throw new CAFFindException(ex);
		} finally {
			CAFPublicLogger.exiting(
						user,
						JARM_REQUEST,
						method,
						location,
						CAFPublicLogger.LEVEL_MEDIUM);
		}
	}
	
	/**
	 * The method searches for Document by KM Document ID and KM Parent folder RID.  
	 * If Document isn't found in CAF(JDO) KM search will be perfomed.
	 * 
	 * @param parentFolder		KM Folder RID to search
	 * @param documentId		KM DocumentId to search
	 * @return	Collection of obtained documents
	 * @throws CAFRetrieveException
	 */	
	public Collection findByDocumentId(String parentFolder, String documentId) throws CAFFindException {
		final String method = JARM_REQUEST + ":findByDocumentId(String, String)";
		String user = sessionContext.getCallerPrincipal().getName();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { parentFolder, documentId },
					CAFPublicLogger.LEVEL_MEDIUM);
	
		try {			
			IntQueryFilter fParentFolder = new IntQueryFilter(QueryFilter.createStringFilter(
							QueryFilter.ACTION_EXACT, parentFolder, "parentFolder", true));
			fParentFolder.setAttribute("parentFolder");					
			fParentFolder.condition = QueryFilter.CONDITION_EQ;
						
			IntQueryFilter fDocumentId = new IntQueryFilter(QueryFilter.createStringFilter(
							QueryFilter.ACTION_EXACT, documentId, "documentId", true));
			fDocumentId.setAttribute("documentId");
			fDocumentId.condition = QueryFilter.CONDITION_EQ;

			Collection result = dataAccessService.query(Document.class, new IntQueryFilter[] {fDocumentId, fParentFolder}, null); 
			return checkPermissions(result, CAFPermissionName.create, user, Document.OBJECT_NAME);
		} catch (Exception ex) {
			location.throwing(ex);
			throw new CAFFindException(ex);
		} finally {
			CAFPublicLogger.exiting(
						user,
						JARM_REQUEST,
						method,
						location,
						CAFPublicLogger.LEVEL_MEDIUM);
		}

	}

	/**
	 * The method performs search for Document   
	 * 
	 * @param parentFolder		KM Folder RID to search
	 * @param documentId		KM DocumentId to search
	 * @return	Collection of obtained documents
	 * @throws CAFRetrieveException
	 */	
	public Collection findByRid(String rid) throws CAFFindException {
		final String method = JARM_REQUEST + ":findByRid(String)";
		String user = sessionContext.getCallerPrincipal().getName();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { rid },
					CAFPublicLogger.LEVEL_MEDIUM);

		try {
			return findByDocumentId(RidUtils.getPath(rid), RidUtils.getName(rid));
		} catch (CAFFindException ex) {
			location.throwing(method, ex);
			throw ex;
		} finally {
			CAFPublicLogger.exiting(
						user,
						JARM_REQUEST,
						method,
						location,
						CAFPublicLogger.LEVEL_MEDIUM);
		}
	}

	private boolean checkPermission(String permission, IBusinessObject object, String user, String objectName) {
			try {
				if (!CAFPermission.checkAclPermission(object, user, permission, objectName)) {
					return false;
				}
			} catch (CAFPermissionException e) {
				return false;
			}
			return true;
	}

	/**
	 * Check permissions for each object of collection
	 * 
	 * @param objectList	Objects to check
	 * @param permission	Permission to check
	 * @param user			User to check permissions for
	 * @param objectName	
	 * @return				Filtered list of objects
	 */	
	private Collection checkPermissions(Collection objectList, String permission, String user, String objectName) {
		Collection filteredResult = new ArrayList();
		for (Iterator i = objectList.iterator();i.hasNext();) {
			IBusinessObject doc = (IBusinessObject)i.next();
			if (checkPermission(permission, doc, user, Document.OBJECT_NAME)) {
				filteredResult.add(doc);
			}
		}
		return filteredResult;
	}	

	private void reindex(Document doc) throws Exception {
		Context context = new InitialContext();

		if (m_indexSearch == null) {
			String INDEX_SEARCH_JNDI_NAME =
				"localejbs/sap.com/caf~com.sap.caf.runtime/IndexSearchBean";
			Object obj = context.lookup(INDEX_SEARCH_JNDI_NAME);
			IndexSearchLocalHome home = (IndexSearchLocalHome) obj;
			m_indexSearch = home.create();
		}
		String relRid = doc.getRid();

		Collection col = doc.getRelatedObjectRids();
		if (!col.isEmpty()) {
			Object oBOName = col.iterator().next();
			String sBOName = (String) oBOName;
			StringTokenizer st = new StringTokenizer(sBOName, "/");
			st.nextToken();
			sBOName = st.nextToken();
			MetaModel mm = new MetaModel();
			DataObject dobj = mm.getDataObjectByGUID(sBOName);
			Application app =
				dobj.getBusinessEntityInterface().getApplication();
			sBOName =
				app.getProviderName()
					+ '/'
					+ app.getObjectName()
					+ '/'
					+ dobj.getObjectName();
			m_indexSearch.indexRelatedObject(sBOName, relRid);
		}
	}
}
