package com.sap.caf.core.appsrv.doccontent;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;

import javax.activation.MimetypesFileTypeMap;
import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.SessionContext;

import com.sap.caf.core.CoreServices;
import com.sap.caf.core.appsrv.datatypes.*;
import com.sap.caf.core.besrv.document.Document;
import com.sap.caf.core.besrv.document.DocumentPK;
import com.sap.caf.core.besrv.document.DocumentServiceLocal;
import com.sap.caf.km.da.DocumentDataAccessService;
import com.sap.caf.km.da.KMDataAccessHelper;
import com.sap.caf.km.ejb.data.document.DocumentProxyEJBLocal;
import com.sap.caf.km.ejb.data.util.NodeContent;
import com.sap.caf.km.da.RidUtils;
//import com.sap.caf.km.ejb.svc.upload.UploadProperties;
import com.sap.caf.metamodel.DataObject;
import com.sap.caf.rt.bol.context.CAFContext;
import com.sap.caf.rt.bol.context.CAFProperties;
import com.sap.caf.rt.exception.CAFBaseException;
import com.sap.caf.rt.exception.CAFCreateException;
import com.sap.caf.rt.exception.CAFFindException;
import com.sap.caf.rt.exception.CAFRetrieveException;
import com.sap.caf.rt.exception.CAFUpdateException;
import com.sap.caf.rt.exception.DataAccessException;
import com.sap.caf.rt.exception.ServiceException;
import com.sap.caf.rt.metamodel.MetaModel;
import com.sap.caf.rt.srv.ApplicationServiceBean;
import com.sap.caf.rt.util.CAFPublicLogger;
import com.sap.tc.logging.Location;
import com.sap.tc.webdynpro.services.sal.url.api.WDWebResourceType;
import com.sap.tc.logging.Severity;

/**
 * @author C5047347
 */
public class DocContentBean extends ApplicationServiceBean {

	protected String _objectName = "sap.com/caf.core/DocContent";			
	private static final String APPLICATION = DocContentBean.class.getName();
	private static final String jARMReqPrefix = "XAP:BO:"; 
	private static final String JARM_REQUEST =  jARMReqPrefix + APPLICATION;		
	private static final Location location = Location.getLocation(APPLICATION);
			
						
	private static final String DEF_MIME_TYPE = "application/octet-stream";
	private static final String PARAM_ROOT_FOLDER = "KM_TMP_REPOSITORY_FOLDER";
	private static final char RID_SEPARATOR = '/';
	private static final MimetypesFileTypeMap s_mimeTypes = new MimetypesFileTypeMap();
	private String m_rootFolder;
	private MetaModel mmrProxy;

	public void ejbCreate() throws  CreateException  {
	}

	public void ejbActivate() throws EJBException, RemoteException {
	}

	public void ejbPassivate() throws EJBException, RemoteException {
	}

	public void ejbRemove() throws EJBException, RemoteException {
		mmrProxy = null;
	}

	public void setSessionContext(SessionContext arg0)
		throws EJBException, RemoteException {
			sessionContext = arg0;
	}

	/**
	 * Copy document into BO folder
	 * 
	 * @param srcRid	Resource ID to copy
	 * @param dstBOName	BO Name
	 * @param dstBOGuid	BO GUID
	 * @return	New document id
	 */
	public String copyDocument(String srcRid, String dstBOName, String dstBOGuid) throws CAFUpdateException { 
		final String method = JARM_REQUEST + ":copyDocument(String, String, String)";
		String user = getUser();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { srcRid, dstBOName, dstBOGuid},
					CAFPublicLogger.LEVEL_MEDIUM);

		try {
				return KMDataAccessHelper.getDocumentProxyService().copyBODocument(dstBOName, dstBOGuid, srcRid);
		} catch (Exception e) {
			location.throwing(method, e);
			throw new CAFUpdateException(e);
		} finally {
			CAFPublicLogger.exiting(
							user,
							JARM_REQUEST,
							method,
							location,
							CAFPublicLogger.LEVEL_MEDIUM);			
		}
	}
	
	/**
	 * Upload document content
	 * 
	 * @param key		Document KEY to upload content
	 * @param content	Document content to upload
	 * @return			Document KEY
	 * @throws CAFCreateException
	 */
	public String uploadDocument(String key, byte[] content) throws CAFCreateException {
		final String method = JARM_REQUEST + ":uploadDocument(String, byte[])";
		String user = getUser();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { key, content},
					CAFPublicLogger.LEVEL_MEDIUM);
		
		try {
			DocumentDataAccessService service = DocumentDataAccessService.getInstance();

			Document doc = 
				(Document)service
					.load(new DocumentPK(key), Document.class);			

			if (content==null) {
				content = new byte[0];
			}
			
			KMDataAccessHelper.getDocumentProxyService()
				.saveDocumentContent(doc.getDocumentId(),
						 doc.getParentFolder(),
						 content.length,
						 null,
						 null,
						 content);

			return doc.getKey();
		}
		catch (Exception e) {
			location.throwing(method, e);
			throw new CAFCreateException(e);
		}
		finally {
			CAFPublicLogger.exiting(
							user,
							JARM_REQUEST,
							method,
							location,
							CAFPublicLogger.LEVEL_MEDIUM);
		}
	}
	
	/**
	 * Uploads external link
	 * 
	 * @param key		Document KEY to upload external link
	 * @param url		External link URL
	 * @return			Document KEY
	 */
	public String uploadExtLink(String key, String url) throws CAFCreateException {
		final String method = JARM_REQUEST + ":uploadExtLink(String, String)";
		String user = getUser();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { key, url},
					CAFPublicLogger.LEVEL_MEDIUM);
						
		try {			
			Document doc = (Document)DocumentDataAccessService.getInstance().load(new DocumentPK(key), Document.class);
			doc.setLink(url);
			
			DocumentDataAccessService.getInstance().store(doc);
			
			return doc.getKey();
		}
		catch (Exception e) {
			location.throwing(method, e);
			throw new CAFCreateException(e);
		}
		finally {
			CAFPublicLogger.exiting(
							user,
							JARM_REQUEST,
							method,
							location,
							CAFPublicLogger.LEVEL_MEDIUM);
		}
	}	

	/**
	 * Returns TRUE if resource is attached to BO 
	 * 
	 * @param rid	Resource RID
	 * @return TRUE if specified resource is attached to BO
	 */
	public boolean isAttached(String rid) throws CAFBaseException {
		final String method = JARM_REQUEST + ":isAttached(String)";
		String user = getUser();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { rid },
					CAFPublicLogger.LEVEL_MEDIUM);
		
		try {
			this.readProperties();
			return rid.startsWith(m_rootFolder);			
		} catch (CAFBaseException e) {			
			location.throwing(method, e);
			throw new CAFFindException(e);			
		} finally {
			CAFPublicLogger.exiting(
										user,
										JARM_REQUEST,
										method,
										location,
										CAFPublicLogger.LEVEL_MEDIUM);						
		}
	}	

	/**
	 * Relate Document to Business Object 
	 * 
	 * @param documentKey	Document KEY to relate
	 * @param boName		Business Object Name. See <i>IDependentObject.getObjectType()</i>
	 * @param boGuid		Business Object Instance GUID. See <i>IDependentObject.getKey()</i>
	 * @param boRid			Business Object Instance RID. See <i>IBusinessObject.getRid()</i>
	 * @return				Document KEY
	 * @throws CAFUpdateException
	 */
	public String addRelatedObjectRid(String documentKey, String boName, String boGuid, String boRid) throws CAFUpdateException {
		final String method = JARM_REQUEST + "addRelatedObjectRid(String, String, String, String)";
		String user = getUser();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { documentKey, boName, boGuid, boRid },
					CAFPublicLogger.LEVEL_MEDIUM);
				
		try {
			DocumentServiceLocal documentService = CoreServices.getDocumentService();
			
			Collection result = documentService.findByDocumentKey(documentKey);
			Document doc = (Document)result.iterator().next();
				
			String sOldRid = doc.getRid();			
			Collection docRids = new ArrayList(1);
			docRids.add(sOldRid);

			//copy document into BO folder
			Collection newDocRids = KMDataAccessHelper.getDocumentProxyService()
				.copyAttachedKMDocuments(boName, boGuid, docRids);

			//get new RID for document							 
			String newDocRid = (String) newDocRids.iterator().next();
			
			//store new KM parentFolder and documentId into JDO 
			doc.setParentFolder(RidUtils.getPath(newDocRid));
			doc.setDocumentId(RidUtils.getName(newDocRid));			
			DocumentDataAccessService.getInstance().store(doc);
			
			//re-read document from new location (in order to get lastUpdatedDate from KM)
			result = documentService.findByDocumentKey(documentKey);
			doc = (Document)result.iterator().next();

			doc.addRelatedObjectRids( createRid(boName, boGuid) );

			//store changes and rise notification
			documentService.update(doc);		

			//delete doc if it was uploaded into temp folder
			if (!newDocRid.equals(sOldRid)) { 				
				try  {
					KMDataAccessHelper.getDocumentProxyService()
						.deleteDocument(RidUtils.getName(sOldRid), RidUtils.getPath(sOldRid));
				}
				catch (Throwable e) {
					location.throwing(method, e);
				}
			}			
						
			return doc.getKey();
		} catch (Exception e) {
			location.throwing(method, e);
			throw new CAFUpdateException(e);
		} finally {
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
	}
	
	/**
	 * Remove relation between Business Object and Document 
	 * 
	 * @param documentKey	Document KEY to relate
	 * @param boName		Business Object Name. See <i>IDependentObject.getObjectType()</i>
	 * @param boRid			Business Object Instance RID. See <i>IBusinessObject.getRid()</i>
	 * @throws CAFUpdateException
	 */	
	public void removeRelatedObjectRid(String documentKey, String boName, String boInstanceGuid) throws CAFUpdateException {
		final String method = JARM_REQUEST + ":removeRelatedObjectRid(String, String, String)";
		String user = getUser();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { documentKey, boName, boInstanceGuid},
					CAFPublicLogger.LEVEL_MEDIUM);

		try {
			DocumentServiceLocal documentService = CoreServices.getDocumentService();
			
			Collection result = documentService.findByDocumentKey(documentKey);
			Document doc = (Document)result.iterator().next();
			
			doc.removeRelatedObjectRids( createRid(boName, boInstanceGuid) );			
//			DocumentDataAccessService.getInstance().store(doc);
			documentService.update(doc);

			Collection relatedObjects = doc.getRelatedObjectRids();

			if (relatedObjects.isEmpty()) {
				//documents needs to be removed only if it is under caf bo folder
				String docRepository = CAFContext.KM_TMP_REPOSITORY_FOLDER; //UploadProperties.getInstance().getDocRepositoryRid();
				String rootBORid = RidUtils.getBORootFolder(docRepository, documentKey); 
				if (rootBORid!=null) {
					DocumentDataAccessService.getInstance().remove(doc);
//					KMDataAccessHelper.getDocumentProxyService()
//						.deleteDocument(doc.getDocumentId(), doc.getDocumentId());
					KMDataAccessHelper.getFolderProxyService().deleteEmptyFolders(RidUtils.getName(rootBORid), RidUtils.getPath(rootBORid));
				}
			}
		} catch (Exception e) {
			location.throwing(method, e);
			throw new CAFUpdateException(e);
		}
		finally {
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
	}
	
	/**
	 * Relate document to specified BO
	 * 
	 * @param srcRid	Document RID to relate
	 * @param dstBOName	BO Name
	 * @param dstBOGuid	BO GUID
	 * @return	
	 * @throws CAFUpdateException
	 */
	public String relateDocument(String srcRid, String dstBOName, String dstBOGuid) throws CAFUpdateException {
		final String method = JARM_REQUEST + ":relateDocument(String, String, String)";
		String user = getUser();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { srcRid, dstBOName, dstBOGuid},
					CAFPublicLogger.LEVEL_MEDIUM);

		try {
			return KMDataAccessHelper.getDocumentProxyService().relateKMDocument(dstBOName, dstBOGuid, srcRid); //getUploadService().relateKMDocument(dstBOName, dstBOGuid, srcRid);
		} catch (Exception e) {
			location.throwing(method, e);
			throw new CAFUpdateException(e);
		}	finally {
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
	}

	/**
	 * 
	 * @param documentKey
	 * @return
	 * @throws CAFRetrieveException
	 */
	public byte[] readDocumentContent(String documentKey) throws CAFRetrieveException {
		final String method = JARM_REQUEST + ":readDocumentContent(String)";
		String user = getUser();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { documentKey },
					CAFPublicLogger.LEVEL_MEDIUM);

		try {
			DocumentDataAccessService service = DocumentDataAccessService.getInstance();

			Document doc = 
				(Document)service
							.load(new DocumentPK(documentKey), Document.class);							
			
			NodeContent data = KMDataAccessHelper
									.getDocumentProxyService() 
										.readDocumentContent(doc.getDocumentId(), 
															 doc.getParentFolder());
		
			return data.getContent();
					
		} catch (Exception e) {
			location.throwing(method, e);
			throw new CAFRetrieveException(e);
		}	finally {
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}		
		
	}

	/**
	 * 
	 * @param rid
	 * @return
	 * @throws ServiceException
	 */
	public boolean isDocumentLocked (String rid) throws ServiceException {
		final String method = JARM_REQUEST + ":isDocumentLocked(String)";
		String user = getUser();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { rid },
					CAFPublicLogger.LEVEL_MEDIUM);
			
		try {
			return KMDataAccessHelper.getDocumentProxyService().isDocumentLocked(rid);
		}
		catch (Exception e) {
			location.throwing(e);
			throw new ServiceException(e);
		}
		finally {
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);

		}
	}
	
	/**
	 * 
	 * @param rid
	 * @return
	 * @throws ServiceException
	 */
	public IDocumentLockInfo getLockInfo (String rid) throws ServiceException {
		final String method = JARM_REQUEST + ":getLockInfo(String)";
		String user = getUser();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { rid },
					CAFPublicLogger.LEVEL_MEDIUM);

		try {
			DocumentProxyEJBLocal docProxy = KMDataAccessHelper.getDocumentProxyService();
			
			String lockOwnwer = docProxy.getLockOwner(rid);
			boolean allowUnlock = docProxy.authorizationCheck(rid);
			return new DocumentLockInfo(lockOwnwer, allowUnlock); 
		}
		catch (Exception e) {
			location.throwing(e);
			throw new ServiceException(e);
		}
		finally {
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
	}
	
	/**
	 * 
	 * @param rid
	 * @return
	 * @throws ServiceException
	 */
	public boolean unLockDocument (String rid) throws ServiceException {
		final String method = JARM_REQUEST + ":unLockDocument(String)";
		String user = getUser();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { rid },
					CAFPublicLogger.LEVEL_MEDIUM);

			try {
				return KMDataAccessHelper.getDocumentProxyService().unlockDocument(rid);
			}
			catch (Exception e) {
				location.throwing(e);
				throw new ServiceException(e);
			}
			finally {
				CAFPublicLogger.exiting(
					user,
					JARM_REQUEST,
					method,
					location,
					CAFPublicLogger.LEVEL_MEDIUM);
			}
	
	}
	
	/**
	 * 
	 * @param rid
	 * @throws ServiceException
	 */
	public void disableDocumentVersioning(String rid) throws ServiceException {
		final String method = JARM_REQUEST + ":disableDocumentVersioning(String)";
		String user = getUser();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { rid },
					CAFPublicLogger.LEVEL_MEDIUM);

		try {
			KMDataAccessHelper.getDocumentProxyService().disableDocumentVersioning(rid);
		}
		catch (Exception e) {
			location.throwing(e);
			throw new ServiceException(e);
		}
		finally {
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}		
	}
	
	/**
	 * 
	 * @param rid
	 * @throws ServiceException
	 */
	public void enableDocumentVersioning (String rid) throws ServiceException {
		final String method = JARM_REQUEST + ":enableDocumentVersioning(String)";
		String user = getUser();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { rid },
					CAFPublicLogger.LEVEL_MEDIUM);

		try {
			KMDataAccessHelper.getDocumentProxyService().enableDocumentVersioning(rid);
		}
		catch (Exception e) {
			location.throwing(e);
			throw new ServiceException(e);
		}
		finally {
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}		
	}
	
	/**
	 * 
	 * @param rid
	 * @return
	 * @throws ServiceException
	 */
	public boolean isDocumentVersioned (String rid) throws ServiceException {
		final String method = JARM_REQUEST + ":isDocumentVersioned(String)";
		String user = getUser();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { rid },
					CAFPublicLogger.LEVEL_MEDIUM);

		try {
			return KMDataAccessHelper.getDocumentProxyService().isDocumentVersioned(rid);
		}
		catch (Exception e) {
			location.throwing(e);
			throw new ServiceException(e);
		}
		finally {
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
	}
	
	/**
	 * 
	 * @param rid
	 * @return
	 * @throws ServiceException
	 */
	public String[] getVersionHistory(String rid)throws ServiceException {
		final String method = JARM_REQUEST + ":getVersionHistory(String)";
		String user = getUser();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { rid },
					CAFPublicLogger.LEVEL_MEDIUM);

		try {
			return KMDataAccessHelper.getDocumentProxyService().getVersionHistory(rid);
		}
		catch (Exception e) {
			location.throwing(e);
			throw new ServiceException(e);
		}
		finally {
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}

	}
	
	/**
	 * 
	 * @param historyRid
	 * @param rid
	 * @throws ServiceException
	 */
	public void setAsCurrentVersion (String historyRid, String rid) throws ServiceException {
		final String method = JARM_REQUEST + ":setAsCurrentVersion(String)";
		String user = getUser();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { rid },
					CAFPublicLogger.LEVEL_MEDIUM);

		try {
			KMDataAccessHelper.getDocumentProxyService().setAsCurrentVersion(historyRid, rid);
		}
		catch (Exception e) {
			location.throwing(e);
			throw new ServiceException(e);
		}
		finally {
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
		
	}
	
	/**
	 * 
	 * @param historyRid
	 * @throws ServiceException
	 */
	public void deleteVersion (String historyRid) throws ServiceException {
		final String method = JARM_REQUEST + ":deleteVersion(String)";
		String user = getUser();
		CAFPublicLogger.entering(
					getUser(),
					JARM_REQUEST,
					method,
					location,
					new Object[] { historyRid },
					CAFPublicLogger.LEVEL_MEDIUM);

		try {
			KMDataAccessHelper.getDocumentProxyService().deleteVersion(historyRid);
		}
		catch (Exception e) {
			location.throwing(e);
			throw new ServiceException(e);
		}
		finally {
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		} 		
	}
	
	/**
	 * 
	 * @param rid
	 * @return
	 * @throws ServiceException
	 */
	public String getCurrentVersion(String rid) throws ServiceException {
		final String method = JARM_REQUEST + ":getCurrentVersion(String)";
		String user = getUser();
		CAFPublicLogger.entering(
					user,
					JARM_REQUEST,
					method,
					location,
					new Object[] { rid },
					CAFPublicLogger.LEVEL_MEDIUM);

		try {
			return KMDataAccessHelper.getDocumentProxyService().getCurrentVersion(rid);
		}
		catch (Exception e) {
			location.throwing(e);
			throw new ServiceException(e);
		}
		finally {
			CAFPublicLogger.exiting(
				user,
				JARM_REQUEST,
				method,
				location,
				CAFPublicLogger.LEVEL_MEDIUM);
		}
	}

	/**
	 * Read CAF properties
	 */
	private void readProperties() throws DataAccessException {
		CAFProperties properties = new CAFProperties();
		properties.loadApplicationProperties();
		m_rootFolder = properties.getProperty(PARAM_ROOT_FOLDER);			
	}		
	
	/**
	 * Transforms source root folder path and source file pathname into
	 * resulted root folder and file name moving extra path information to resulted root folder name 
	 * i.e.
	 * operands: "/documents/temp" and "countries/US/states/NY/report.doc" will be transformed into
	 * results: "/documents/temp/countries/US/states/NY" and "report.doc"
	 *  
	 * @param folderPath 	 Source root folder
	 * @param filePathname Source pathname probably containing extra path information
	 * @return An array of two elements: 0 is a root folder and 1 is a file name
	 */
	private String[] normalizePathname(String folderPath, String filePathname) throws Exception {
		if (folderPath==null || filePathname==null) {
			throw new Exception("Method parameters can not be null!");
		}
		int i = filePathname.lastIndexOf(RID_SEPARATOR);
		if (i>0) {
			folderPath = folderPath+RID_SEPARATOR+filePathname.substring(0, i);
			filePathname = filePathname.substring(i+1);
		}
		return new String[] {folderPath, filePathname};
	}

	/**
	 * Determine Mime Type according to file extention
	 * 
	 * @param sFileName		FileName
	 * @return	String MimeType representation
	 */
	private String getMimeType(String sFileName) {
		int i = sFileName.lastIndexOf('.');
		if(i < 0)
			return DEF_MIME_TYPE;
		String sFileExt = sFileName.substring(i + 1);
		if(sFileExt.length() == 0)
			return DEF_MIME_TYPE;
		WDWebResourceType wdMimeType = WDWebResourceType.getWebResourceTypeForFileExtentsion(sFileExt);
		String sMimeType = (wdMimeType!=null? wdMimeType.getHtmlMime(): s_mimeTypes.getContentType(sFileName));
		return (sMimeType!=null? sMimeType: DEF_MIME_TYPE); 
	}

	/**
	 * Create BO Rid
	 * 
	 * @param boName			Business Object name
	 * @param boInstanceGuid	Business Object Instance GUID
	 * @return					KM BO RID
	 * @throws DataAccessException
	 */
	private String createRid(String boName, String boInstanceGuid) throws DataAccessException {
		if (mmrProxy==null) {
			mmrProxy = new MetaModel();
		} 			
		DataObject obj = mmrProxy.getDataObject(boName);

		return CAFContext.KM_REPOSITORY 
						+ "/" 
						+ obj.getObjectId() 
						+ "/" 
						+ boInstanceGuid.substring(0,2) 
						+ "/" 
						+ boInstanceGuid.substring(0,4) 
						+ "/r" 
						+ boInstanceGuid; 	
	}
	
	private String getUser() {
		return sessionContext.getCallerPrincipal().getName();
	}		

    public Object execute(String operationName, java.util.ArrayList parameters)	throws com.sap.caf.rt.exception.ServiceException {
	String method = "execute(String, ArrayList)";
	String user = sessionContext.getCallerPrincipal().getName();
	Object result = null;
	CAFPublicLogger.entering(user, JARM_REQUEST, method, location, new Object[] {operationName, parameters}, CAFPublicLogger.LEVEL_MEDIUM);
		if (operationName == null) {
		CAFPublicLogger.exiting(user, JARM_REQUEST, method, location);
		throw new com.sap.caf.rt.exception.ServiceException("APPSRV_NULL_OPERATION_ERROR");
	} else {
		try {
			if("".equals(operationName)){
				
			} else {				
				throw new com.sap.caf.rt.exception.ServiceException("Operation not defined");
			}
		} catch (Exception e) {
			CAFPublicLogger.traceThrowableT(Severity.DEBUG, location, method, "Error in " + method, e);
			location.throwing(method, e);
			throw new com.sap.caf.rt.exception.ServiceException(e);
		} finally {
			CAFPublicLogger.exiting(user, JARM_REQUEST, method, location, new Object[] {result}, CAFPublicLogger.LEVEL_MEDIUM);
		}
		return result;
	}

    }

}
