package com.sap.tc.loggingStandard;

/**
 * Title:        LoggingStandard
 * Description:
 * Copyright:    Copyright (c) 2001
 * Company:      SAP Markets, Inc
 * @author
 * @version $Id: //sapmarkets/Logging/630_VAL_REL/src/_StdLogging1.3/java/com/sap/tc/loggingStandard/PropertiesConfigurator13.java#1 $
 */


import java.util.*;
import java.text.*;
import java.io.*;

import com.sap.tc.logging.*;
/**
 * <p>
 *   This is based on <code>com.sap.tc.logging.PropertiesConfigurator</code>
 *   but with modified code to wrap around the configuration for JSR 47 API as in
 *   JDK 1.4. It is in the format of key-value pair, and the syntax rule follows that
 *   defined by SAP but with different semantics. In particular, it replaces the SAP
 *   terminology with the corresponding ones used in SUN <em>'java.util.logging'</em>,
 *   for the ease of use.
 * </p>
 * <p>
 *   The syntax defined by SAP is similar to the default configuration file <em>
 *   "lib/logging.properties"</em> provided by SUN under the JRE directory,
 *   but it is even enhanced to support more
 *   flexible and clear definition, as shown in the example below.
 * </p>
 * <p>
 *   In general, it is in the form of: <br>
 *   <i>&lt;Name&gt;</i><code>.</code><i>&lt;Attribute&gt;</i> <code>=</code>
 *   <i>&lt;Value&gt;</i><br>
 *   You can learn more about the basic syntax concept described in
 *   {@link com.sap.tc.logging.PropertiesConfigurator}. The specific
 *   syntax rule used in this wrapper class is listed after the example as well.
 * </p>
 * <p>
 *   The concept of global handlers and level are not implemented in this wrapper, however,
 *   this can still be achieved, though with a slightly difference mechanism with the SAP
 *   logging tool. This can easily be specified in the configuration file, basically
 *   identical to the SUN syntax (shown in example below).<br>
 *   In addition to the global handler specification, you can specify
 *   local handler for each Logger class. Note that this is not mentioned/available
 *   in the SUN API.
 * </p>
 * <p>
 *   Here is a simple example, followed by the syntax rule:
 * <code><pre>
 *  .handlers = ConsoleHandler		#simulate global handlers, same syntax as in SUN
 *
 * 					#simulate global severity level:
 *  .level = INFO			#everybody under this parent Logger will have threshold INFO.
 *  foo.parent.child1.level = WARNING 	#except all descendants of 'child1'
 *
 *
 *  #SAP syntax: allows configuration of this handler before assigning to the Logger
 *  handler[simpleFile]			= FileHandler
 *  handler[simpleFile].pattern		= C:\\temp\\result\\trace.txt
 *  handler[simpleFile].formatter 	= formatter[Simple]
 *
 *  formatter[Simple]		= SimpleFormatter
 *  formatter[Simple].pattern	= %24d %-38l %s: %m		#this follows SAP style
 *
 *  #Assign local handler for this Logger in addition to the global handler
 *  #Note that in this case, output to console will be done twice.
 *  foo.parent.child1.localHandlers =  handler[Console], handler[simpleFile]
 *
 * </code></pre>
 * </p>
 * <p>
 *   Below shows the specific subset of the syntax of the properties definition used in this version:
 *   <blockquote>
 *     <table>
 *       <tr>
 *         <td><i>&lt;Configuration&gt;</i>    </td><td> = </td><td> </td><td><i>&lt;Entry&gt;</i>*.</td>
 *       </tr>
 *       <tr>
 *         <td><i>&lt;Entry&gt;</i>            </td><td> = </td><td> </td><td><i>&lt;Key&gt;</i> <code>'='</code> <i>&lt;Value&gt;</i></td>
 *       </tr>
 *       <tr><td><pre> </pre></td></tr>
 *       <tr>
 *         <td><i>&lt;Key&gt;</i>              </td><td> = </td><td> </td><td>[<i>&lt;Logger&gt;</i>] <code>'.'</code> <i>&lt;LoggerAttr&gt;</i></td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><i>&lt;Handler&gt;</i> [<code>'.'</code> <i>&lt;HandlerAttr&gt;</i>]</td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><i>&lt;Formatter&gt;</i> [<code>'.'</code> <i>&lt;FormatterAttr&gt;</i>].</td>
 *       </tr>
 *       <tr>
 *         <td><i>&lt;Logger&gt;</i>         </td><td> = </td><td> </td><td><i>&lt;Identifier&gt;</i> (<code>'.'</code> <i>&lt;Identifier&gt;</i>)*.</td>
 *       </tr>
 *       <tr>
 *         <td><i>&lt;LoggerAttr&gt;</i></td><td> = </td><td> </td><td><code>'level'</code></td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><code>'handlers'</code></td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><code>'localHandlers'</code>.</td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><code>'filters'</code>.</td>
 *       </tr>
 *       <tr>
 *         <td><i>&lt;Handler&gt;</i>              </td><td> = </td><td> </td><td><code>'handler'</code> <code>'['</code> <i>&lt;Identifier&gt;</i> <code>']'</code>.</td>
 *       </tr>
 *       <tr>
 *         <td><i>&lt;HandlerAttr&gt;</i>          </td><td> = </td><td> </td><td><code>'level'</code></td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><code>'formatter'</code></td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><code>'filters'</code></td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><code>'pattern'</code></td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><code>'limit'</code></td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><code>'cnt'</code>.</td>
 *       </tr>
 *       <tr>
 *         <td><i>&lt;Formatter&gt;</i>        </td><td> = </td><td> </td><td><code>'formatter'</code> <code>'['</code> <i>&lt;Identifier&gt;</i> <code>']'</code>.</td>
 *       </tr>
 *       <tr>
 *         <td><i>&lt;FormatterAttr&gt;</i>    </td><td> = </td><td> </td><td><code>'pattern'</code>.</td>
 *       </tr>
 *       <tr><td><pre> </pre></td></tr>
 *       <tr>
 *         <td><i>&lt;Value&gt;</i>            </td><td> = </td><td> </td><td><i>&lt;Level&gt;</i></td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><i>&lt;Objects&gt;</i></td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><i>&lt;String&gt;</i></td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><i>&lt;Identifier&gt;</i>.</td>
 *       </tr>
 *       <tr>
 *         <td><i>&lt;Level&gt;</i>         </td><td> = </td><td> </td><td nowrap><code>'OFF'</code> | <code>'SEVERE'</code> |<code>'WARNING'</code> | <code>'INFO'</code> | <code>'CONFIG'</code> | <code>'FINE'</code> | <code>'FINER'</code> | <code>'FINEST'</code>.</td>
 *       </tr>
 *       <tr>
 *         <td><i>&lt;Objects&gt;</i>          </td><td> = </td><td> </td><td>[<code>'+'</code>] <i>&lt;Object&gt;</i> (<code>','</code> <i>&lt;Object&gt;</i>)*.</td>
 *       </tr>
 *         <td><i>&lt;Object&gt;</i>           </td><td> = </td><td> </td><td><i>&lt;Class&gt;</i></td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><i>&lt;Handler&gt;</i></td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><i>&lt;Formatter&gt;</i>.</td>
 *       </tr>
 *       <tr>
 *         <td><i>&lt;Class&gt;</i>            </td><td> = </td><td> </td><td><code>'FileHandler'</code></td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><code>'ConsoleHandler'</code></td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><code>'SimpleFormatter'</code></td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><code>'XMLFormatter'</code>.</td>
 *       </tr>
 *       <tr>
 *         <td>                                </td><td>   </td><td>|</td><td><i>&lt;Identifier&gt;</i> (<code>'.'</code> <i>&lt;Identifier&gt;</i>)*.</td>
 *       </tr>
 *     </table>
 *   </blockquote>
 * </p>
 * <p>
 *   Note: the order of the configuration does not matter in most cases, <br>except</br>
 *   severity assignment. This is due to the severity inheritance behavior described
 *   in the JSR47. Therefore, level settings for child nodes in the tree should come after
 *   settings for their parents.<br>
 *   Order of other assignments are ok, eg. you can assign a <i>Handler 'vairable'</i>
 *   to a <i>Logger</i> before you specify the configuration of the <i>Handler</i> itself.
 *   Refer to the example above that shows the configuration of the <i>Formatter</i> does
 *   come after its assignment to a <i>FileHandler</i>.
 * </p>
 * <p>
 *   Options to specify <i>pattern</i> for both <i>FileHandler</i>(that is, filename) and
 *   the <i>SimpleFormatter</i> should refer to the Javadoc for the corresponding SAP classes:
 *   {@link com.sap.tc.logging.FileLog} and
 *   {@link com.sap.tc.logging.TraceFormatter}.
 * </p>
 * <p>
 *   Limitation: <br>
 *   For the Handler object, only <em>ConsoleHandler</em> and <em>FileHandler</em>
 *   is supported in this wrapper.<br>
 *   For the Filter object, again, if users intend to implement any filters in
 *   this wrapper, this has to follow the interface of
 *   {@link com.sap.tc.logging.Filter}, working on the
 *   {@link com.sap.tc.logging.LogRecord}. Since this is most
 *   likely a filter class implemented by the user, so, assign the fully
 *   qualified classname as the value:
 *   <blockquote>
 *    <pre>
 *    handler[simpleFile].filters = com.sap.foo.FooFilter</pre>
 *   </blockquote>
 * </p>
 * <br>
 * <br>
 * <p>
 *   Comparison: <br>
 *   This properties class is also compatible to the original SAP logging tool, in case
 *   users would like define configuration here with SAP terminology.
 *   A brief mapping is listed here for your convenience:
 *   <blockquote><pre>
 *	SUN				SAP
 *      --------------------------------------------------
 *      Logger				Location
 *      Handler				Log
 *	  -ConsoleHandler		  -ConsoleLog
 *	  -FileHandler			  -FileHandler
 *      Level				Severity
 *	  -OFF				  -NONE
 *	  -SEVERE			  -FATAL
 * 	  -WARNING			  -WARNING
 *	  -INFO				  -INFO
 *	  -CONFIG
 *	  -FINE
 *	  -FINER			  -PATH
 *	  -FINEST			  -DEBUG
 *	  -ALL				  -ALL
 *	Formatter			Formatter
 *	  -SimpleFormatter		  -TraceFormatter
 *	  -XMLFormatter			  -XMLFormatter
 *   </pre></blockquote>
 * </p>
 * <p>
 * Note: If users do use the SAP style, the configuration behavior will be identical to
 * that described in {@link com.sap.tc.logging.PropertiesConfigurator}, except the
 * update of 'limit' and 'cnt' attributes of FileLog for a periodic reloading. The
 * current limitation in this version is new values of 'limit' and 'cnt' are ignored
 * upon a periodic reloading.
 * </p>
 */
public class PropertiesConfigurator13 extends Configurator {

  static String version = "$Id: //sapmarkets/Logging/630_VAL_REL/src/_StdLogging1.3/java/com/sap/tc/loggingStandard/PropertiesConfigurator13.java#1 $";

  protected static final String NO_PROPERTIES_EMSG       = "No properties",
                                INVALID_PROPS_FILE       = "Invalid logging properties filename: {0}, but application should still run properly",
                                SYNTAX_ERR_EMSG          = "Syntax error: see console for details",
                                SEMANTICS_ERR_EMSG       = "Semantics error: see console for details",
                                SYNTAX_ERR_IN_KEY_MSG    = "key ''{0}'' has wrong syntax",
                                SYNTAX_ERR_IN_VAL_MSG    = "value of key ''{0}'' has wrong syntax",
                                UNKNOWN_CLASS_MSG        = "unknown class ''{0}''",
                                NO_PATTERN_MSG           = "no pattern for log[{0}]",
                                NO_LIMIT_MSG             = "no limit for log[{0}]",
                                NO_CNT_MSG               = "no count for log[{0}]",
                                UNKNOWN_FORMATTER_MSG    = "unknown formatter[{0}] for log[{1}]",
                                UNKNOWN_LOG_MSG          = "unknown log[{0}] for controller {1}",
                                LIMIT_CNT_IGNORED_MSG    = "new limit and count setting ignored for log [{0}]";

  /**
   * Configures this properties configurator with a properties object and the
   * class loader used for this class.
   * @param  properties Properties object
   * @see    #PropertiesConfigurator13(java.util.Properties,
   *                                 java.lang.ClassLoader)
   */
  public PropertiesConfigurator13(Properties properties) {
    super();

    final String method = "PropertiesConfigurator13(java.util.Properties)";

    if (properties != null) {
      this.properties = properties;
      this.file       = null;
    } else {
      LogManager.getIntErrorCat().errorT(classLoc,
                                     method,
                                     "No properties object");
      handleException(new IllegalArgumentException(NO_PROPERTIES_EMSG));
    }
  }

  /**
   * Configures this properties configurator with a properties object and a
   * custom class loader.
   * @param  properties  Properties object
   * @param  classLoader Custom class loader
   * @see    #PropertiesConfigurator13(java.util.Properties)
   */
  public PropertiesConfigurator13(Properties  properties,
                                ClassLoader classLoader) {
    super(classLoader);

    final String method = "PropertiesConfigurator13(java.util.Properties,"
                                               + "java.lang.ClassLoader)";

    if (properties != null) {
      this.properties = properties;
      this.file       = null;
    } else {
      LogManager.getIntErrorCat().errorT(classLoc,
                                     method,
                                     "No properties object");
      handleException(new IllegalArgumentException(NO_PROPERTIES_EMSG));
    }
  }

  /**
   * Configures this properties configurator with a properties file and the
   * class loader used for this class.
   * @param  file Properties file
   * @see    #PropertiesConfigurator13(java.io.File,
   *                                 java.lang.ClassLoader)
   */
  public PropertiesConfigurator13(File file) {
    super();

    final String method = "PropertiesConfigurator13(java.io.File)";

    if (file != null) {
      this.properties  = null;
      this.file        = file;
    } else {
      LogManager.getIntErrorCat().errorT(classLoc,
                                     method,
                                     "No properties file");
      handleException(new IllegalArgumentException(NO_PROPERTIES_EMSG));
    }
  }

  /**
   * Configures this properties configurator with a properties file and a custom
   * class loader.
   * @param  file        Properties file
   * @param  classLoader Custom class loader
   * @see    #PropertiesConfigurator13(java.io.File)
   */
  public PropertiesConfigurator13(File        file,
                                ClassLoader classLoader) {
    super(classLoader);

    final String method = "PropertiesConfigurator13(java.io.File,"
                                               + "java.lang.ClassLoader)";

    if (file != null) {
      this.properties  = null;
      this.file        = file;
    } else {
      LogManager.getIntErrorCat().errorT(classLoc,
                                     method,
                                     "No properties file");
      handleException(new IllegalArgumentException(NO_PROPERTIES_EMSG));
    }
  }

  /**
   * Gets the properties object used for this configurator.
   * @return Properties object, or <code>null</code> if there is none
   * @see    #setProperties(java.util.Properties)
   */
  public Properties getProperties() {
    return properties;
  }

  /**
   * Sets the properties object used for this configurator.
   * @param  properties Properties object
   * @see    #getProperties()
   */
  public void setProperties(Properties properties) {
    final String method = "setProperties(java.util.Properties)";

    resetException();
    if (properties != null) {
      this.properties = properties;
      this.file       = null;
    } else {
      LogManager.getIntErrorCat().errorT(classLoc,
                                     method,
                                     "No properties object");
      handleException(new IllegalArgumentException(NO_PROPERTIES_EMSG));
    }
  }

  /**
   * Gets the file used for this configurator.
   * @return File, or <code>null</code> if there is none
   * @see    #setFile(java.io.File)
   */
  public File getFile() {
    return file;
  }

  /**
   * Sets the file used for this configurator.
   * @param  file File
   * @throws IllegalArgumentException No file
   * @see    #getProperties()
   */
  public void setFile(File file) {
    final String method = "setFile(java.io.File)";

    resetException();
    if (file != null) {
      this.properties = null;
      this.file       = file;
    } else {
      LogManager.getIntErrorCat().errorT(classLoc,
                                     method,
                                     "No properties file");
      handleException(new IllegalArgumentException(NO_PROPERTIES_EMSG));
    }
  }

  public void configure() {
    final String method = "configure()";

    resetException();
    if (   properties != null
        || file != null) {
      Properties currProperties = null;
      Iterator   iter;

      if (properties != null) {
        currProperties = properties;
      } else if (file != null) {
        currProperties = new Properties();
        // Load properties from file if it has been touched since the last
        // read via this configurator.
        try {
          //YUE: check explicitly. See next comments about 'modification' comparison
          if(!file.exists()){
            LogManager.getIntErrorCat().setMaximumSeverity(Severity.INFO);
            LogManager.getIntErrorCat().infoT(classLoc,
                                           method,
                                           INVALID_PROPS_FILE,
                                           new Object[] {file.getAbsoluteFile()});
            LogManager.getIntErrorCat().setMaximumSeverity(Severity.ERROR);
            handleException(new FileNotFoundException(file.getAbsolutePath()));
            return;    //nothing to be done
          }

          //YUE: this comparison masks the error of non-existing properties file
          if (file.lastModified() > lastConfigured) {
            currProperties.load(new FileInputStream(file));
            lastConfigured = file.lastModified();
          }
        }
        catch (Exception exc) {
          handleException(exc);
        }
      }

      // Iterate over properties to pick up data. The data is accumulated in
      // provisional objects. Real objects like formatters and controllers are
      // instantiated in the second phase if their specifications are complete.
      HashMap formatters  = new HashMap(),
              logs        = new HashMap(),
              controllers = new HashMap();

     //YUE: ORDER DOES MATTER
      LinkedList controllersList = new LinkedList();
     //end YUE

      iter = currProperties.entrySet().iterator();
      while (iter.hasNext()) {
        Map.Entry entry = (Map.Entry) iter.next();
        Key       key   = parseKey((String) entry.getKey());
        String    value = (String) entry.getValue();

        if (key != null) {
          if (key instanceof LogControllerKey) {
            LogControllerKey   controllerKey = (LogControllerKey) key;
            LogController      controller    = controllerKey.getController();
            LogControllerEntry accEntry;

            accEntry = (LogControllerEntry) controllers.get(controller.getName());
            if (accEntry == null) {
              accEntry = new LogControllerEntry(controller);
              controllers.put(controller.getName(),
                              accEntry);
              //YUE
              controllersList.addLast(accEntry);
              //end YUE
            }

            // Store attribute.
            Attr          attr     = controllerKey.getAttr();
            LogController relative;

            switch (attr.getAttr()) {
              case TokenType.EFF_SEVERITY: {
                relative = ((SeverityAttr) attr).getRelative();
                if (relative == null) {
                  //YUE, this is done with local data structure only. See next 'YUE'
                  accEntry.setEffectiveSeverity(parseSeverity(value,
                                                              (String) entry.getKey()));

                } else {  //redundant for JDK1.3 wrapper
                  accEntry.setEffectiveRelationSeverity(relative,
                                                        parseSeverity(value,
                                                                      (String) entry.getKey()));
                }
                break;
              }
              case TokenType.MIN_SEVERITY: {
                relative = ((SeverityAttr) attr).getRelative();
                if (relative == null) {
                  accEntry.setMinimumSeverity(parseSeverity(value,
                                                            (String) entry.getKey()));
                } else {
                  accEntry.setMinimumRelationSeverity(relative,
                                                      parseSeverity(value,
                                                                    (String) entry.getKey()));
                }
                break;
              }
              case TokenType.MAX_SEVERITY: {
                relative = ((SeverityAttr) attr).getRelative();
                if (relative == null) {
                  accEntry.setMaximumSeverity(parseSeverity(value,
                                                            (String) entry.getKey()));
                } else {
                  accEntry.setMaximumRelationSeverity(relative,
                                                      parseSeverity(value,
                                                                    (String) entry.getKey()));
                }
                break;
              }
              case TokenType.LOGS: {
                accEntry.setLogs(parseObjects(value,
                                              (String) entry.getKey()));
                break;
              }
              case TokenType.LOCAL_LOGS: {
                accEntry.setLocalLogs(parseObjects(value,
                                                   (String) entry.getKey()));
                break;
              }

              case TokenType.PRIVATE_LOGS: {
                accEntry.setPrivateLogs(parseObjects(value,
                                                     (String) entry.getKey()));
                break;
              }
              case TokenType.FILTERS: {
                accEntry.setFilters(parseObjects(value,
                                                 (String) entry.getKey()));
                break;
              }
              case TokenType.BUNDLE_NAME: {
                accEntry.setResourceBundleName(value);
                break;
              }
            }
          } else if (key instanceof LogRefKey) {
            LogRefKey logRefKey = (LogRefKey) key;
            LogEntry  accEntry;

            accEntry = (LogEntry) logs.get(logRefKey.getName());
            if (accEntry == null) {
              accEntry = new LogEntry();
              logs.put(logRefKey.getName(),
                       accEntry);
            }

            // Check whether there is an attribute and store it.
            Attr attr = logRefKey.getAttr();

            if (attr != null) {
              switch (attr.getAttr()) {
                case TokenType.EFF_SEVERITY: {
                  accEntry.setEffectiveSeverity(parseSeverity(value,
                                                              (String) entry.getKey()));
                  break;
                }
                case TokenType.ENCODING: {
                  accEntry.setEncoding(value);
                  break;
                }
                //YUE: additional attr 'desc' for Log
                case TokenType.DESCRIPTION: {
                  accEntry.setDesc(value);
                  break;
                }
                case TokenType.FILTERS: {
                  accEntry.setFilters(parseObjects(value,
                                                   (String) entry.getKey()));
                  break;
                }
                case TokenType.FORMATTER: {
                  accEntry.setFormatter(parseObjects(value,
                                                     (String) entry.getKey()));
                  break;
                }
                case TokenType.PATTERN: {
                  accEntry.setPattern(value);
                  break;
                }
                case TokenType.LIMIT: {
                  try {
                    accEntry.setLimit(Integer.parseInt(value));
                  }
                  catch (NumberFormatException e) {
                    // Syntax error: wrong number format
                    LogManager.getIntErrorCat().errorT(classLoc,
                                                   method,
                                                   SYNTAX_ERR_IN_VAL_MSG,
                                                   new Object[] {(String) entry.getKey()});
                    handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
                  }
                  break;
                }
                case TokenType.CNT: {
                  try {
                    accEntry.setCnt(Integer.parseInt(value));
                  }
                  catch (NumberFormatException e) {
                    // Syntax error: wrong number format
                    LogManager.getIntErrorCat().errorT(classLoc,
                                                   method,
                                                   SYNTAX_ERR_IN_VAL_MSG,
                                                   new Object[] {(String) entry.getKey()});
                    handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
                  }
                  break;
                }
              }
            } else {
              accEntry.setObject(parseObjects(value,
                                              (String) entry.getKey()));
            }
          } else if (key instanceof FormatterRefKey) {
            FormatterRefKey formatterRefKey = (FormatterRefKey) key;
            FormatterEntry  accEntry;

            accEntry = (FormatterEntry) formatters.get(formatterRefKey.getName());
            if (accEntry == null) {
              accEntry = new FormatterEntry();
              formatters.put(formatterRefKey.getName(),
                             accEntry);
            }

            // Check whether there is an attribute and store it.
            Attr attr = formatterRefKey.getAttr();

            if (attr != null) {
              switch (attr.getAttr()) {
                case TokenType.PATTERN: {
                  accEntry.setPattern(value);
                  break;
                }
              }
            } else {
              accEntry.setObject(parseObjects(value,
                                              (String) entry.getKey()));
            }
          }
        } else {
          // Ignore entry with incorrect key syntax: nothing to be done.
        }
      }

      // Instantiate objects to completed specifications, in the order of
      // possible referencing.

      // First handle formatters.
      iter = formatters.values().iterator();
      while (iter.hasNext()) {
        FormatterEntry entry  = (FormatterEntry) iter.next();

        if (entry.getObject() != null) {
          ObjectVal object = (ObjectVal) entry.getObject().getObjects().iterator().next();

          if (object instanceof ClassVal) {
            ClassVal cl = (ClassVal) object;

            switch (cl.getType()) {
              case TokenType.TRACE_FORMATTER: {
                if (entry.getPattern() != null) {
                  entry.setFormatter(new TraceFormatter(entry.getPattern()));
                } else {
                  entry.setFormatter(new TraceFormatter());
                }
                break;
              }
              case TokenType.LIST_FORMATTER: {
                entry.setFormatter(new ListFormatter());
                break;
              }
              case TokenType.XML_FORMATTER: {
                entry.setFormatter(new XMLFormatter());
                break;
              }
              case TokenType.IDENT: {
                try {
                  entry.setFormatter((Formatter) getClassLoader().loadClass(((CustomClassVal) cl).getIdent()).newInstance());
                }
                catch (Exception e) {
                  // Semantics error: wrong class specification
                  LogManager.getIntErrorCat().errorT(classLoc,
                                                 method,
                                                 UNKNOWN_CLASS_MSG,
                                                 new Object[] {((CustomClassVal) cl).getIdent()});
                  handleException(new IllegalArgumentException(SEMANTICS_ERR_EMSG));
                }
                break;
              }
              default: {
                // Specification of formatter incorrect: drop it.
                iter.remove();
                break;
              }
            }
          } else {
            // Specification of formatter incorrect: drop it.
            iter.remove();
          }
        } else {
          // Specification of formatter incomplete: drop it.
          iter.remove();
        }
      }

      // Then handle logs which might refer to formatters.
      iter = logs.entrySet().iterator();
      while (iter.hasNext()) {
        Map.Entry mapEntry = (Map.Entry) iter.next();
        String    name     = (String) (mapEntry).getKey();
        LogEntry  entry    = (LogEntry) (mapEntry).getValue();
        Log       log      = null;

        if (entry.getObject() != null) {
          ObjectVal object  = (ObjectVal) entry.getObject().getObjects().iterator().next();

          if (object instanceof ClassVal) {
            ClassVal cl = (ClassVal) object;

            switch (cl.getType()) {
              case TokenType.FILE_LOG: {
                if (entry.getPattern() != null) {
                  int limit = FileLog.NO_LIMIT,
                      cnt   = FileLog.NO_CNT;

                  if (   entry.getLimit() > FileLog.NO_LIMIT
                      && entry.getCnt() > FileLog.NO_CNT) {
                    limit = entry.getLimit();
                    cnt = entry.getCnt();
                  } else {
                    if (entry.getLimit() > FileLog.NO_LIMIT) {
                      LogManager.getIntErrorCat().errorT(classLoc,
                                                     method,
                                                     NO_CNT_MSG,
                                                     new Object[] {name});
                      handleException(new IllegalArgumentException(SEMANTICS_ERR_EMSG));
                    }
                    if (entry.getCnt() > FileLog.NO_CNT) {
                      LogManager.getIntErrorCat().errorT(classLoc,
                                                     method,
                                                     NO_LIMIT_MSG,
                                                     new Object[] {name});
                      handleException(new IllegalArgumentException(SEMANTICS_ERR_EMSG));
                    }
                  }
                  log = new FileLog(entry.getPattern(),
                                    limit,
                                    cnt);
                } else {
                  // Semantics error: incomplete log specification
                  iter.remove();
                  LogManager.getIntErrorCat().errorT(classLoc,
                                                 method,
                                                 NO_PATTERN_MSG,
                                                 new Object[] {name});
                  handleException(new IllegalArgumentException(SEMANTICS_ERR_EMSG));
                }
                break;
              }
              case TokenType.CONSOLE_LOG: {
                log = new ConsoleLog();
                break;
              }
              case TokenType.IDENT: {
                try {
                  log = (Log) getClassLoader().loadClass(((CustomClassVal) cl).getIdent()).newInstance();
                }
                catch (Exception e) {
                  // Semantics error: wrong class specification
                  LogManager.getIntErrorCat().errorT(classLoc,
                                                 method,
                                                 UNKNOWN_CLASS_MSG,
                                                 new Object[] {((CustomClassVal) cl).getIdent()});
                  handleException(new IllegalArgumentException(SEMANTICS_ERR_EMSG));
                }
                break;
              }
              default: {
                // Specification of log incomplete: drop it.
                iter.remove();
                break;
              }
            }
          } else {
            // Specification of formatter incorrect: drop it.
            iter.remove();
          }

          if (log != null) {
            if (entry.getEffectiveSeverity() != null) {
              log.setEffectiveSeverity(entry.getEffectiveSeverity().getSeverity());
            }

            if (entry.getEncoding() != null) {
              log.setEncoding(entry.getEncoding());
            }

            //YUE: additional attr 'desc' for Log
            if (entry.getDesc() != null) {
              log.setLogName(entry.getDesc());
            }

            if (entry.getFilters() != null) {
              if (!entry.getFilters().isAdding()) {
                log.removeFilters();
              }

              Iterator filterIter = entry.getFilters().getObjects().iterator();

              while (filterIter.hasNext()) {
                object = (ObjectVal) filterIter.next();

                if (object instanceof ClassVal) {
                  ClassVal filter = (ClassVal) object;

                  if (filter.getType() == TokenType.IDENT) {
                    try {
                      log.addFilter((Filter) getClassLoader().loadClass(((CustomClassVal) filter).getIdent()).newInstance());
                    }
                    catch (Exception e) {
                      // Semantics error: wrong class specification
                      LogManager.getIntErrorCat().errorT(classLoc,
                                                     method,
                                                     UNKNOWN_CLASS_MSG,
                                                     new Object[] {((CustomClassVal) filter).getIdent()});
                      handleException(new IllegalArgumentException(SEMANTICS_ERR_EMSG));
                    }
                  }
                }
              }
            }

            if (entry.getFormatter() != null) {
              Iterator formatterIter = entry.getFormatter().getObjects().iterator();

              if (formatterIter.hasNext()) {
                object = (ObjectVal) formatterIter.next();

                if (object instanceof ClassVal) {
                  ClassVal formatter = (ClassVal) object;

                  switch (formatter.getType()) {
                    case TokenType.TRACE_FORMATTER: {
                      log.setFormatter(new TraceFormatter());
                      break;
                    }
                    case TokenType.LIST_FORMATTER: {
                      log.setFormatter(new ListFormatter());
                      break;
                    }
                    case TokenType.XML_FORMATTER: {
                      log.setFormatter(new XMLFormatter());
                      break;
                    }
                    case TokenType.IDENT: {
                      try {
                        log.setFormatter((Formatter) getClassLoader().loadClass(((CustomClassVal) formatter).getIdent()).newInstance());
                      }
                      catch (Exception e) {
                        // Semantics error: wrong class specification
                        LogManager.getIntErrorCat().errorT(classLoc,
                                                       method,
                                                       UNKNOWN_CLASS_MSG,
                                                       new Object[] {((CustomClassVal) formatter).getIdent()});
                        handleException(new IllegalArgumentException(SEMANTICS_ERR_EMSG));
                      }
                      break;
                    }
                  }
                } else if (object instanceof FormatterRefVal) {
                  FormatterRefVal ref = (FormatterRefVal) object;

                  try {
                    log.setFormatter(((FormatterEntry) formatters.get(ref.getName())).getFormatter());
                  }
                  catch (NullPointerException e) {
                    // Semantics error: unknown formatter reference
                    LogManager.getIntErrorCat().errorT(classLoc,
                                                   method,
                                                   UNKNOWN_FORMATTER_MSG,
                                                   new Object[] {ref.getName(),
                                                                 name});
                    handleException(new IllegalArgumentException(SEMANTICS_ERR_EMSG));
                  }
                } else {
                  // Specification of formatter incorrect: nothing to be done.
                }
              }
            }

            entry.setLog(log);
          }
        }
      }

      // Finally handle controllers which might refer to logs.
      //YUE
      //iter = controllers.values().iterator();      //order screwed up
      iter = controllersList.iterator();
      //end YUE
      while (iter.hasNext()) {
        LogControllerEntry entry      = (LogControllerEntry) iter.next();
        LogController      controller = entry.getController();

        if (entry.getEffectiveSeverity() != null) {
          //YUE
          SeverityVal sVal = entry.getEffectiveSeverity();
          LogManager.getLogManager().setLevel(controller.getName(),
                                              Level.toLevel(sVal.getSeverity()));
//          controller.setEffectiveSeverity(entry.getEffectiveSeverity().getSeverity());
          //end YUE
        }
        if (entry.getMinimumSeverity() != null) {
          controller.setMinimumSeverity(entry.getMinimumSeverity().getSeverity());
        }

        if (entry.getMaximumSeverity() != null) {
          controller.setMaximumSeverity(entry.getMaximumSeverity().getSeverity());
        }

        Iterator severitiesIter;

        severitiesIter = entry.getEffectiveRelationSeverities().entrySet().iterator();
        while (severitiesIter.hasNext()) {
//*** Not possible here. Redundant. NO relative severity concept in SUN API ***//
          Map.Entry severity = (Map.Entry) severitiesIter.next();

//          controller.setEffectiveSeverity((LogController) severity.getKey(),
//                                          ((SeverityVal) severity.getValue()).getSeverity());
        }

        severitiesIter = entry.getMinimumRelationSeverities().entrySet().iterator();
        while (severitiesIter.hasNext()) {
          Map.Entry severity = (Map.Entry) severitiesIter.next();
//*** Not possible here. Redundant. NO relative severity concept in SUN API ***//
//          controller.setMinimumSeverity((LogController) severity.getKey(),
//                                        ((SeverityVal) severity.getValue()).getSeverity());
        }

        severitiesIter = entry.getMaximumRelationSeverities().entrySet().iterator();
        while (severitiesIter.hasNext()) {
          Map.Entry severity = (Map.Entry) severitiesIter.next();
//*** Not possible here. Redundant. NO relative severity concept in SUN API ***//
//          controller.setMaximumSeverity((LogController) severity.getKey(),
//                                        ((SeverityVal) severity.getValue()).getSeverity());
        }

        ObjectsVal objects;

        objects = entry.getLogs();
        if (objects != null) {
          LinkedList newLogs = computeLogs(NORMAL_LOGS,
                                           objects,
                                           controller,
                                           logs);

          // Attach the logs from the new collection, swapped for existing logs,
          // to the controller.
          Iterator newLogsIter = newLogs.iterator();

          while (newLogsIter.hasNext()) {
            controller.addLog((Log) newLogsIter.next());
          }
        }

        objects = entry.getLocalLogs();
        if (objects != null) {
          LinkedList newLogs = computeLogs(LOCAL_LOGS,
                                           objects,
                                           controller,
                                           logs);

          // Attach the local logs from the new collection, swapped for existing
          // logs, to the controller.
          Iterator newLogsIter = newLogs.iterator();

          while (newLogsIter.hasNext()) {
            controller.addLocalLog((Log) newLogsIter.next());
          }
        }

        objects = entry.getPrivateLogs();
        if (objects != null) {
          LinkedList newLogs = computeLogs(PRIVATE_LOGS,
                                           objects,
                                           controller,
                                           logs);

          // Attach the private logs from the new collection, swapped for
          // existing logs, to the controller.
          Iterator newLogsIter = newLogs.iterator();

          while (newLogsIter.hasNext()) {
            controller.addPrivateLog((Log) newLogsIter.next());
          }
        }

        if (entry.getFilters() != null) {
          if (!entry.getFilters().isAdding()) {
            controller.removeFilters();
          }

          Iterator filterIter = entry.getFilters().getObjects().iterator();

          while (filterIter.hasNext()) {
            ObjectVal object = (ObjectVal) filterIter.next();

            if (object instanceof ClassVal) {
              ClassVal filter = (ClassVal) object;

              if (filter.getType() == TokenType.IDENT) {
                  try {
                    controller.addFilter((Filter) getClassLoader().loadClass(((CustomClassVal) filter).getIdent()).newInstance());
                  }
                  catch (Exception e) {
                    // Semantics error: wrong class specification
                    LogManager.getIntErrorCat().errorT(classLoc,
                                                   method,
                                                   UNKNOWN_CLASS_MSG,
                                                   new Object[] {((CustomClassVal) filter).getIdent()});
                    handleException(new IllegalArgumentException(SEMANTICS_ERR_EMSG));
                  }
              }
            }
          }
        }

        if (entry.getResourceBundleName() != null) {
          controller.setResourceBundleName(entry.getResourceBundleName());
        }
      }
    } else {
      // Nothing to be done.
    }
  }

  private LinkedList computeLogs(int           type,
                                 ObjectsVal    objects,
                                 LogController controller,
                                 HashMap       logs) {
    final String method = "computeLogs(ObjectsVal,"
                                    + "LogController"
                                    + "java.util.HashMap)";

    // To avoid forgetting the current state of logs, for example the current
    // file of a rotating file set or whether a header has been written, logs
    // are kept if attached to the controller, according to the result of the
    // method equals. The procecure is as follows. First we make all the logs
    // described via the configuration properties and put them into a new
    // collection. Afterwards, we iterate through the logs attached to the
    // controller. If a log is not found in the new collection and the adding
    // flag is not set, it is removed. Otherwise, it is removed from the new
    // collection instead. The first time, the log attached to the controller is
    // added to the new collection. This is done in order to, for example,
    // switch normals logs to local logs.
    Iterator   logsIter = objects.getObjects().iterator();
    LinkedList newLogs  = new LinkedList();

    while (logsIter.hasNext()) {
      ObjectVal object = (ObjectVal) logsIter.next();

      if (object instanceof ClassVal) {
        ClassVal log = (ClassVal) object;

        switch (log.getType()) {
          case TokenType.CONSOLE_LOG: {
            newLogs.add(new ConsoleLog());
            break;
          }
          case TokenType.IDENT: {
            try {
              newLogs.add(getClassLoader().loadClass(((CustomClassVal) log).getIdent()).newInstance());
            }
            catch (Exception e) {
              // Semantics error: wrong class specification
              LogManager.getIntErrorCat().errorT(classLoc,
                                             method,
                                             UNKNOWN_CLASS_MSG,
                                             new Object[] {((CustomClassVal) log).getIdent()});
              handleException(new IllegalArgumentException(SEMANTICS_ERR_EMSG));
            }
            break;
          }
        }
      } else if (object instanceof LogRefVal) {
        LogRefVal ref = (LogRefVal) object;

        // Add complete log entry to keep explicit log attributes available
        // for copying to existing log.
        Object entry = logs.get(ref.getName());
        if (entry != null) {
          newLogs.add(entry);
        }
        else{
          // Semantics error: unknown log reference
          LogManager.getIntErrorCat().errorT(classLoc,
                                         method,
                                         UNKNOWN_LOG_MSG,
                                         new Object[] {ref.getName(),
                                                       controller.getName()});
          handleException(new IllegalArgumentException(SEMANTICS_ERR_EMSG));
        };
      } else {
        // Specification of log incorrect, nothing to be done.
      }
    }

    // Remove superfluous logs.
    for (int curr = NORMAL_LOGS;
             curr < NO_LOGS;
             ++curr) {
      Iterator prevLogsIter;

      switch (curr) {
        case NORMAL_LOGS: {
          prevLogsIter = controller.getLogs().iterator();
          break;
        }
        case LOCAL_LOGS: {
          prevLogsIter = controller.getLocalLogs().iterator();
          break;
        }
        case PRIVATE_LOGS: {
          prevLogsIter = controller.getPrivateLogs().iterator();
          break;
        }
        default: {
          prevLogsIter = null;
          break;
        }
      }

      while (prevLogsIter.hasNext()) {
        Log     log   = (Log) prevLogsIter.next();
        boolean found = false;

        // Look for identical log in new logs.
        ListIterator newLogsIter = newLogs.listIterator();

        while (newLogsIter.hasNext()) {
          Object nextLog = newLogsIter.next();

          if (nextLog instanceof Log) {
            if (log.equals((Log) nextLog)) {
              found = true;
              newLogsIter.remove();
            }
          } else { // if nextLog instanceof LogEntry
            LogEntry newLogEntry = (LogEntry) nextLog;
            Log      newLog      = newLogEntry.getLog();

            if (log.equals(newLog)) {
              found = true;
              newLogsIter.remove();

              // Copy attributes from new log.
              if (newLogEntry.getEffectiveSeverity() != null) {
                log.setEffectiveSeverity(newLogEntry.getEffectiveSeverity().getSeverity());
              }

              if (newLogEntry.getEncoding() != null) {
                log.setEncoding(newLog.getEncoding());
              }

              //YUE: additional attr 'desc' for Log
              if (newLogEntry.getDesc() != null) {
                log.setLogName(newLog.getLogName());
              }

              if (newLogEntry.getFilters() != null) {
                Iterator filterIter = newLog.getFilters().iterator();

                log.removeFilters();
                while (filterIter.hasNext()) {
                  log.addFilter((Filter) filterIter.next());
                }
              }

              if (newLogEntry.getFormatter() != null) {
                log.setFormatter(newLog.getFormatter());
              }

              // Copy file log attributes from new log if applicable.
              if (log instanceof FileLog) {
                if (   newLogEntry.isLimitSet()
                    || newLogEntry.isCntSet()) {
                    LogManager.getIntErrorCat().setMaximumSeverity(Severity.WARNING);
                    LogManager.getIntErrorCat().warningT(classLoc,
                                                         method,
                                                         LIMIT_CNT_IGNORED_MSG,
                                                         new Object[] {log.getLogName()});
                    LogManager.getIntErrorCat().setMaximumSeverity(Severity.ERROR);
//TODO:: discrepancy from SAP Logging, new limit & count are ignored
//     : setRotation method is not accessible by this 1.3 package
//                  ((FileLog) log).setRotation(((FileLog) newLog).getLimit(),
//                                              ((FileLog) newLog).getCnt());
                }
              }
            }
          }
        }
        if (found) {
          // Reattach log to move it from current to new visibility.
          newLogs.add(log);
        } else {
          // Forget log if it is of the type of the property, and the latter
          // does not specify adding.
          if (   curr == type
              && !objects.isAdding()) {
            controller.removeLog(log);
          }
        }
      }
    }

    // Cast new logs from LogEntry to Log.
    ListIterator newLogsIter = newLogs.listIterator();

    while (newLogsIter.hasNext()) {
      Object newLog = newLogsIter.next();

      if (newLog instanceof LogEntry) {
        newLogsIter.remove();
        newLogsIter.add(((LogEntry) newLog).getLog());
      }
    }

    return newLogs;
  }

  static class TokenType extends LogManager.TokenType {

    public static final int BUNDLE_NAME     = LogManager.TokenType.MAX + 1,
                            CNT             = BUNDLE_NAME + 1,
                            CONSOLE_LOG     = CNT + 1,
                            EFF_SEVERITY    = CONSOLE_LOG + 1,
                            ENCODING        = EFF_SEVERITY + 1,
                            //YUE: additional attr 'desc' for Log
                            DESCRIPTION     = ENCODING + 1,
                            FILE_LOG        = DESCRIPTION + 1,
                            FILTERS         = FILE_LOG + 1,
                            FORMATTER       = FILTERS + 1,
                            LIMIT           = FORMATTER + 1,
                            LIST_FORMATTER  = LIMIT + 1,
                            LOCAL_LOGS      = LIST_FORMATTER + 1,
                            LOG             = LOCAL_LOGS + 1,
                            LOGS            = LOG + 1,
                            MAX_SEVERITY    = LOGS + 1,
                            MIN_SEVERITY    = MAX_SEVERITY + 1,
                            PATTERN         = MIN_SEVERITY + 1,
                            PLUS            = PATTERN + 1,
                            PRIVATE_LOGS    = PLUS + 1,
                            SEVERITY        = PRIVATE_LOGS + 1,
                            TRACE_FORMATTER = SEVERITY + 1,
                            XML_FORMATTER   = TRACE_FORMATTER + 1;
    public static final int MAX             = XML_FORMATTER;

  }

  static class Tokenizer extends LogManager.Tokenizer {

    public Tokenizer(String name) {
      super(name);
    }

    public Token nextTokenInt() {
      Token token   = super.nextTokenInt();
      int   type    = token.getType();

      if (type != TokenType.UNKNOWN) {
        if (type == TokenType.IDENT) {
          String val = token.getValue();

          if (val.equals("bundleName")) {
            return new Token(TokenType.BUNDLE_NAME,
                             val);
          } else if (val.equals("cnt")) {
            return new Token(TokenType.CNT,
                             val);
          } else if (val.equals("ConsoleLog")
                  || val.equals("ConsoleHandler")) {   //YUE
            return new Token(TokenType.CONSOLE_LOG,
                             val);
          } else if (   val.equals("severity")
                     || val.equals("effSeverity")
                     || val.equals("level")   //YUE
                     ) {
            return new Token(TokenType.EFF_SEVERITY,
                             val);
          } else if (val.equals("encoding")) {
            return new Token(TokenType.ENCODING,
                             val);
          } else if (val.equals("FileLog")
                  || val.equals("FileHandler")) {   //YUE
            return new Token(TokenType.FILE_LOG,
                             val);
          } else if (val.equals("filters")) {
            return new Token(TokenType.FILTERS,
                             val);
          } else if (val.equals("formatter")) {
            return new Token(TokenType.FORMATTER,
                             val);
          } else if (val.equals("limit")) {
            return new Token(TokenType.LIMIT,
                             val);
          } else if (val.equals("ListFormatter")) {
            return new Token(TokenType.LIST_FORMATTER,
                             val);
          } else if (val.equals("localLogs")) {
            return new Token(TokenType.LOCAL_LOGS,
                             val);
          } else if (val.equals("log")
                  || val.equals("handler")) {   //YUE
            return new Token(TokenType.LOG,
                             val);
          } else if (val.equals("logs")
                  || val.equals("handlers"))    //YUE
          {
            return new Token(TokenType.LOGS,
                             val);
          } else if (val.equals("maxSeverity")) {
            return new Token(TokenType.MAX_SEVERITY,
                             val);
          } else if (val.equals("minSeverity")) {
            return new Token(TokenType.MIN_SEVERITY,
                             val);
          } else if (val.equals("pattern")) {
            return new Token(TokenType.PATTERN,
                             val);
          } else if (val.equals("privateLogs")
                  || val.equals("localHandlers")) {   //YUE
            return new Token(TokenType.PRIVATE_LOGS,
                             val);
          }
          else if (val.equals("Severity")) {
            return new Token(TokenType.SEVERITY,
                             val);
          } else if (val.equals("TraceFormatter")
                  || val.equals("SimpleFormatter")) {   //YUE
            return new Token(TokenType.TRACE_FORMATTER,
                             val);
          } else if (val.equals("XMLFormatter")) {
            return new Token(TokenType.XML_FORMATTER,
                             val);
          } else {
            return token;
          }
        } else {
          return token;
        }
      } else {
        char val = token.getValue().charAt(0);

        switch (val) {
          case ',': {
            return new Token(TokenType.COMMA,
                             ",");
          }
          case '+': {
            return new Token(TokenType.PLUS,
                             "+");
          }
          default: {
            return token;
          }
        }
      }
    }

  }

  protected Key parseKey(String key) {
    final String method = "parseKey(java.lang.String)";

    Tokenizer tokenizer = new Tokenizer(key);

    switch (tokenizer.getLookahead().getType()) {
      case TokenType.IDENT:
      case TokenType.DOT:
      case TokenType.SLASH: {
        // Look for dot in front of attribute. This is done in order to not make
        // attribute names reserved but available for as package or category
        // names.
        ReverseTokenizer reverseTokenizer = new ReverseTokenizer(tokenizer);
        Token            token            = reverseTokenizer.nextToken();

        if (token.getType() == TokenType.RIGHT_BRACKET) {
          // Look for matching left bracket and nothing else in order to ignore
          // concrete log controller name syntax.
          int level = 1;

          while (level > 0) {
            token = reverseTokenizer.nextToken();
            if (token.getType() == TokenType.LEFT_BRACKET) {
              --level;
            } else if (token.getType() == TokenType.RIGHT_BRACKET) {
              ++level;
            } else if (token.getType() == TokenType.NONE) {
              // Syntax error: nothing to be done.
              LogManager.getIntErrorCat().errorT(classLoc,
                                             method,
                                             SYNTAX_ERR_IN_KEY_MSG,
                                             new Object[] {tokenizer.getFullContent()});
              handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
              return null;
            }
          }

          token = reverseTokenizer.nextToken();
          if (   token.getType() != TokenType.EFF_SEVERITY
              && token.getType() != TokenType.MIN_SEVERITY
              && token.getType() != TokenType.MAX_SEVERITY) {
            // Syntax error: nothing to be done.
            LogManager.getIntErrorCat().errorT(classLoc,
                                           method,
                                           SYNTAX_ERR_IN_KEY_MSG,
                                           new Object[] {tokenizer.getFullContent()});
            handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
            return null;
          }
        } else if (   token.getType() != TokenType.EFF_SEVERITY
                   && token.getType() != TokenType.MIN_SEVERITY
                   && token.getType() != TokenType.MAX_SEVERITY
                   && token.getType() != TokenType.LOGS
                   && token.getType() != TokenType.LOCAL_LOGS
                   && token.getType() != TokenType.PRIVATE_LOGS
                   && token.getType() != TokenType.FILTERS
                   && token.getType() != TokenType.BUNDLE_NAME) {
          // Syntax error: nothing to be done.
          LogManager.getIntErrorCat().errorT(classLoc,
                                         method,
                                         SYNTAX_ERR_IN_KEY_MSG,
                                         new Object[] {tokenizer.getFullContent()});
          handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
          return null;
        }

        if (reverseTokenizer.nextToken().getType() == TokenType.DOT) {
          LogController controller = parseLogController(reverseTokenizer.getContent(),
                                                        tokenizer.getFullContent());

          if (controller != null) {
            Attr attr;

            // Parse the remainder following the accepted controller name but
            // skip known dot.
            tokenizer = new Tokenizer (reverseTokenizer.getAcceptedContent());
            tokenizer.nextToken();

            attr = parseLogControllerAttr(tokenizer);
            if (attr != null) {
              return new LogControllerKey(controller,
                                          attr);
            } else {
              // Syntax error: nothing to be done.
              return null;
            }
          } else {
            return null;
          }
        }
      }
      case TokenType.EFF_SEVERITY:
      case TokenType.MIN_SEVERITY:
      case TokenType.MAX_SEVERITY:
      case TokenType.LOGS:
      case TokenType.LOCAL_LOGS:
      case TokenType.PRIVATE_LOGS:
      case TokenType.FILTERS:
      case TokenType.BUNDLE_NAME: {
        // TODO: get location of current component.
        return new LogControllerKey(Location.getRoot(),
                                    parseLogControllerAttr(tokenizer));
      }
      case TokenType.LOG: {
        String name = parseLog(tokenizer,
                               null);

        if (name != null) {
          Token token = tokenizer.nextToken();

          if (token.getType() == TokenType.DOT) {
            Attr attr = parseLogAttr(tokenizer);

            if (attr != null) {
              if (tokenizer.nextToken().getType() == TokenType.NONE) {
                return new LogRefKey(name,
                                     attr);
              } else {
                // Syntax error: nothing to be done.
                LogManager.getIntErrorCat().errorT(classLoc,
                                               method,
                                               SYNTAX_ERR_IN_KEY_MSG,
                                               new Object[] {tokenizer.getFullContent()});
                handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
                return null;
              }
            } else {
              // Syntax error: nothing to be done.
              return null;
            }
          } else if (token.getType() == TokenType.NONE) {
            return new LogRefKey(name);
          } else {
            // Syntax error: nothing to be done.
            LogManager.getIntErrorCat().errorT(classLoc,
                                           method,
                                           SYNTAX_ERR_IN_KEY_MSG,
                                           new Object[] {tokenizer.getFullContent()});
            handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
            return null;
          }
        } else {
          // Syntax error: nothing to be done.
          return null;
        }
      }
      case TokenType.FORMATTER: {
        String name = parseFormatter(tokenizer,
                                     null);

        if (name != null) {
          Token token = tokenizer.nextToken();

          if (token.getType() == TokenType.DOT) {
            Attr attr = parseFormatterAttr(tokenizer);

            if (attr != null) {
              if (tokenizer.nextToken().getType() == TokenType.NONE) {
                return new FormatterRefKey(name,
                                           attr);
              } else {
                // Syntax error: nothing to be done.
                LogManager.getIntErrorCat().errorT(classLoc,
                                               method,
                                               SYNTAX_ERR_IN_KEY_MSG,
                                               new Object[] {tokenizer.getFullContent()});
                handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
                return null;
              }
            } else {
              // Syntax error: nothing to be done.
              return null;
            }
          } else if (token.getType() == TokenType.NONE) {
            return new FormatterRefKey(name);
          } else {
            // Syntax error: nothing to be done.
            LogManager.getIntErrorCat().errorT(classLoc,
                                           method,
                                           SYNTAX_ERR_IN_KEY_MSG,
                                           new Object[] {tokenizer.getFullContent()});
            handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
            return null;
          }
        } else {
          // Syntax error: nothing to be done.
          return null;
        }
      }
      default: {
        // Syntax error: nothing to be done.
        LogManager.getIntErrorCat().errorT(classLoc,
                                       method,
                                       SYNTAX_ERR_IN_KEY_MSG,
                                       new Object[] {tokenizer.getFullContent()});
        handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
        return null;
      }
    }
  }

  protected LogController parseLogController(String name,
                                             String forKey) {
    final String method = "parseLogController(java.lang.String)";

    try {
      if ((new Tokenizer(name)).getLookahead().getType() != TokenType.SLASH) {
        return Location.getLocation(name);
      } else {
        return Category.getCategory(name);
      }
    }
    catch (IllegalArgumentException e) {
      LogManager.getIntErrorCat().errorT(classLoc,
                                     method,
                                     SYNTAX_ERR_IN_KEY_MSG,
                                     new Object[] {forKey});
      handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
      return null;
    }
  }

  protected Attr parseLogControllerAttr(Tokenizer tokenizer) {
    final String method = "parseLogControllerAttr(Tokenizer)";

    Token token = tokenizer.nextToken();

    switch (token.getType()) {
      case TokenType.EFF_SEVERITY:
      case TokenType.MIN_SEVERITY:
      case TokenType.MAX_SEVERITY: {
        if (tokenizer.getLookahead().getType() == TokenType.LEFT_BRACKET) {
          ReverseTokenizer reverseTokenizer;

          tokenizer.nextToken();
          reverseTokenizer = new ReverseTokenizer(tokenizer);
          if (reverseTokenizer.nextToken().getType() == TokenType.RIGHT_BRACKET) {
            LogController controller = parseLogController(reverseTokenizer.getContent(),
                                                          tokenizer.getFullContent());

            if (controller != null) {
              return new SeverityAttr(token.getType(),
                                      controller);
            } else {
              // Syntax error: nothing to be done.
              return null;
            }
          } else {
            // Syntax error: nothing to be done.
            LogManager.getIntErrorCat().errorT(classLoc,
                                           method,
                                           SYNTAX_ERR_IN_KEY_MSG,
                                           new Object[] {tokenizer.getFullContent()});
            handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
            return null;
          }
        } else {
          return new SeverityAttr(token.getType());
        }
      }
      case TokenType.LOGS:
      case TokenType.LOCAL_LOGS:
      case TokenType.PRIVATE_LOGS:
      case TokenType.FILTERS:
      case TokenType.BUNDLE_NAME : {
        return new Attr(token.getType());
      }
      default: {
        // Syntax error: nothing to be done.
        LogManager.getIntErrorCat().errorT(classLoc,
                                       method,
                                       SYNTAX_ERR_IN_KEY_MSG,
                                       new Object[] {tokenizer.getFullContent()});
        handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
        return null;
      }
    }
  }

  protected String parseLog(Tokenizer tokenizer,
                            String    forKey) {
    final String method = "parseLog(Tokenizer,"
                                 + "java.lang.String)";

    if (   tokenizer.nextToken().getType() == TokenType.LOG
        && tokenizer.nextToken().getType() == TokenType.LEFT_BRACKET) {
      Token token;

      token = tokenizer.nextToken();
      if (token.getType() == TokenType.IDENT) {
        String name = (String) token.getValue();

        if (tokenizer.nextToken().getType() == TokenType.RIGHT_BRACKET) {
          return name;
        } else {
          // Syntax error: nothing to be done.
          if (forKey == null) {
            LogManager.getIntErrorCat().errorT(classLoc,
                                           method,
                                           SYNTAX_ERR_IN_KEY_MSG,
                                           new Object[] {tokenizer.getFullContent()});
          } else {
            LogManager.getIntErrorCat().errorT(classLoc,
                                           method,
                                           SYNTAX_ERR_IN_VAL_MSG,
                                           new Object[] {forKey});
          }
          handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
          return null;
        }
      } else {
        // Syntax error: nothing to be done.
        if (forKey == null) {
          LogManager.getIntErrorCat().errorT(classLoc,
                                         method,
                                         SYNTAX_ERR_IN_KEY_MSG,
                                         new Object[] {tokenizer.getFullContent()});
        } else {
          LogManager.getIntErrorCat().errorT(classLoc,
                                         method,
                                         SYNTAX_ERR_IN_VAL_MSG,
                                         new Object[] {forKey});
        }
        handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
        return null;
      }
    } else {
      // Syntax error: nothing to be done.
      if (forKey == null) {
        LogManager.getIntErrorCat().errorT(classLoc,
                                       method,
                                       SYNTAX_ERR_IN_KEY_MSG,
                                       new Object[] {tokenizer.getFullContent()});
      } else {
        LogManager.getIntErrorCat().errorT(classLoc,
                                       method,
                                       SYNTAX_ERR_IN_VAL_MSG,
                                       new Object[] {forKey});
      }
      handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
      return null;
    }
  }

  protected Attr parseLogAttr(Tokenizer tokenizer) {
    final String method = "parseLogAttr(Tokenizer)";

    Token token = tokenizer.nextToken();

    switch (token.getType()) {
      case TokenType.EFF_SEVERITY:
      case TokenType.ENCODING:
      case TokenType.FILTERS:
      case TokenType.FORMATTER:
      case TokenType.PATTERN:
      case TokenType.LIMIT:
      case TokenType.CNT: {
        return new Attr(token.getType());
      }
      default: {
        // Syntax error: nothing to be done.
        LogManager.getIntErrorCat().errorT(classLoc,
                                       method,
                                       SYNTAX_ERR_IN_KEY_MSG,
                                       new Object[] {tokenizer.getFullContent()});
        handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
        return null;
      }
    }
  }

  protected String parseFormatter(Tokenizer tokenizer,
                                  String    forKey) {
    final String method = "parseFormatter(Tokenizer,"
                                       + "java.lang.String)";

    if (   tokenizer.nextToken().getType() == TokenType.FORMATTER
        && tokenizer.nextToken().getType() == TokenType.LEFT_BRACKET) {
      Token token = tokenizer.nextToken();

      if (token.getType() == TokenType.IDENT) {
        String name = (String) token.getValue();

        if (tokenizer.nextToken().getType() == TokenType.RIGHT_BRACKET) {
          return name;
        } else {
          // Syntax error: nothing to be done.
          if (forKey == null) {
            LogManager.getIntErrorCat().errorT(classLoc,
                                           method,
                                           SYNTAX_ERR_IN_KEY_MSG,
                                           new Object[] {tokenizer.getFullContent()});
          } else {
            LogManager.getIntErrorCat().errorT(classLoc,
                                           method,
                                           SYNTAX_ERR_IN_VAL_MSG,
                                           new Object[] {forKey});
          }
          handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
          return null;
        }
      } else {
        // Syntax error: nothing to be done.
        if (forKey == null) {
          LogManager.getIntErrorCat().errorT(classLoc,
                                         method,
                                         SYNTAX_ERR_IN_KEY_MSG,
                                         new Object[] {tokenizer.getFullContent()});
        } else {
          LogManager.getIntErrorCat().errorT(classLoc,
                                         method,
                                         SYNTAX_ERR_IN_VAL_MSG,
                                         new Object[] {forKey});
        }
        handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
        return null;
      }
    } else {
      // Syntax error: nothing to be done.
      if (forKey == null) {
        LogManager.getIntErrorCat().errorT(classLoc,
                                       method,
                                       SYNTAX_ERR_IN_KEY_MSG,
                                       new Object[] {tokenizer.getFullContent()});
      } else {
        LogManager.getIntErrorCat().errorT(classLoc,
                                       method,
                                       SYNTAX_ERR_IN_VAL_MSG,
                                       new Object[] {forKey});
      }
      handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
      return null;
    }
  }

  protected Attr parseFormatterAttr(Tokenizer tokenizer) {
    final String method = "parseFormatterAttr(Tokenizer)";

    Token token = tokenizer.nextToken();

    switch (token.getType()) {
      case TokenType.PATTERN: {
        return new Attr(token.getType());
      }
      default: {
        // Syntax error: nothing to be done.
        LogManager.getIntErrorCat().errorT(classLoc,
                                       method,
                                       SYNTAX_ERR_IN_KEY_MSG,
                                       new Object[] {tokenizer.getFullContent()});
        handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
        return null;
      }
    }
  }

  protected SeverityVal parseSeverity(String value,
                                      String forKey) {
    final String method = "parseSeverity(java.lang.String,"
                                      + "java.lang.String)";

    Tokenizer tokenizer = new Tokenizer(value);
    Token     token     = tokenizer.nextToken();

    //the following if block is ~redundant, as not taught in the parser syntax
    if (token.getType() == TokenType.SEVERITY) {
      if (tokenizer.nextToken().getType() == TokenType.DOT) {
        token = tokenizer.nextToken();
      } else {
        // Syntax error: nothing to be done.
        LogManager.getIntErrorCat().errorT(classLoc,
                                       method,
                                       SYNTAX_ERR_IN_VAL_MSG,
                                       new Object[] {forKey});
        handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
        return null;
      }
    }

    //YUE: we will accommodate both SUN & SAP severity level definition
    int severity = Severity.NONE;
    try {
      severity = Level.mapLevel(Level.parse(token.getValue()));
    }
    catch (IllegalArgumentException e) {
      try{
        severity = Severity.parse(token.getValue());
      }
      catch (IllegalArgumentException e2) {
          // Syntax error: nothing to be done.
        LogManager.getIntErrorCat().errorT(classLoc,
                                     method,
                                     SYNTAX_ERR_IN_VAL_MSG,
                                     new Object[] {forKey});
        handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
        return null;
      }
    }
    //end YUE

    if (tokenizer.nextToken().getType() == TokenType.NONE) {
      return new SeverityVal(severity);
    } else {
        // Syntax error: nothing to be done.
      LogManager.getIntErrorCat().errorT(classLoc,
                                       method,
                                       SYNTAX_ERR_IN_VAL_MSG,
                                       new Object[] {forKey});
      handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
      return null;
    }
  }

  protected ObjectsVal parseObjects(String value,
                                    String forKey) {
    final String method = "parseObjects(java.lang.String,"
                                     + "java.lang.String)";

    Tokenizer  tokenizer = new Tokenizer(value);
    boolean    isAdding  = false;
    ObjectsVal objects;

    //new feature: allows empty value
    if (tokenizer.getLookahead().getType() != TokenType.NONE) {
      if (tokenizer.getLookahead().getType() == TokenType.PLUS) {
        tokenizer.nextToken();
        isAdding = true;
      }
      objects = parseObjectList(tokenizer,
                                forKey);
      if (objects != null) {
        if (tokenizer.nextToken().getType() == TokenType.NONE) {
          objects.setAdding(isAdding);
          return objects;
        } else {
        // Syntax error: nothing to be done.
          LogManager.getIntErrorCat().errorT(classLoc,
                                         method,
                                         SYNTAX_ERR_IN_VAL_MSG,
                                         new Object[] {forKey});
          handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
          return null;
        }
      } else {
        // Syntax error: nothing to be done.
        return null;
      }
     } else {
       // Remove all attached objects.
       return new ObjectsVal();
     }
  }

  protected ObjectsVal parseObjectList(Tokenizer tokenizer,
                                       String    forKey) {
    final String method = "parseObjectList(Tokenizer,"
                                        + "java.lang.String)";

    ObjectsVal objects = null;
    ObjectVal  object  = parseObject(tokenizer,
                                     forKey);

    if (object != null) {
      objects = new ObjectsVal();
      objects.addObject(object);
      while (tokenizer.getLookahead().getType() == TokenType.COMMA) {
        tokenizer.nextToken();
        object = parseObject(tokenizer,
                             forKey);
        if (object != null) {
          objects.addObject(object);
        }
      }
    }
    return objects;
  }

  protected ObjectVal parseObject(Tokenizer tokenizer,
                                  String    forKey) {
    final String method = "parseObject(Tokenizer,"
                                    + "java.lang.String)";

    switch (tokenizer.getLookahead().getType()) {
      case TokenType.FILE_LOG:
      case TokenType.CONSOLE_LOG:
      case TokenType.TRACE_FORMATTER:
      case TokenType.LIST_FORMATTER:
      case TokenType.XML_FORMATTER:
      case TokenType.IDENT: {
        return parseClass(tokenizer,
                          forKey);
      }

      case TokenType.LOG: {
        String name = parseLog(tokenizer,
                               forKey);

        if (name != null) {
          return new  LogRefVal(name);
        } else {
          return null;
        }
      }
      case TokenType.FORMATTER: {
        String name = parseFormatter(tokenizer,
                                     forKey);

        if (name != null) {
          return new FormatterRefVal(name);
        } else {
          return null;
        }
      }
      default: {
        // Syntax error: nothing to be done.
        LogManager.getIntErrorCat().errorT(classLoc,
                                       method,
                                       SYNTAX_ERR_IN_VAL_MSG,
                                       new Object[] {forKey});
        handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
        return null;
      }
    }
  }

  protected ClassVal parseClass(Tokenizer tokenizer,
                                String    forKey) {
    final String method = "parseClass(Tokenizer,"
                                   + "java.lang.String)";

    Token token = tokenizer.nextToken();

    switch (token.getType()) {
      case TokenType.FILE_LOG:
      case TokenType.CONSOLE_LOG:
      case TokenType.TRACE_FORMATTER:
      case TokenType.LIST_FORMATTER:
      case TokenType.XML_FORMATTER:
      {
        return new ClassVal(token.getType());
      }
      case TokenType.IDENT: {
        String ident = token.getValue();

        while (tokenizer.getLookahead().getType() == TokenType.DOT) {
          tokenizer.nextToken();
          token = tokenizer.nextToken();
          if (token.getType() == TokenType.IDENT) {
            ident +=   "."
                     + token.getValue();
          } else {
            // Syntax error: nothing to be done.
            LogManager.getIntErrorCat().errorT(classLoc,
                                           method,
                                           SYNTAX_ERR_IN_VAL_MSG,
                                           new Object[] {forKey});
            handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
            return null;
          }
        }
        return new CustomClassVal(ident);
      }
      default: {
        // Syntax error: nothing to be done.
        LogManager.getIntErrorCat().errorT(classLoc,
                                       method,
                                       SYNTAX_ERR_IN_VAL_MSG,
                                       new Object[] {forKey});
        handleException(new IllegalArgumentException(SYNTAX_ERR_EMSG));
        return null;
      }
    }
  }

  private static class Element {
  }

  private static class Key extends Element {

    public Key() {
      this.attr = null;
    }

    public Key(Attr attr) {
      this.attr = attr;
    }

    public Attr getAttr() {
      return attr;
    }

    public void setAttr(Attr attr) {
      this.attr = attr;
    }

    private Attr attr;

  }

  private static class LogControllerKey extends Key {

    public LogControllerKey(LogController controller,
                            Attr          attr) {
      super(attr);

      this.controller = controller;
    }

    public LogController getController() {
      return controller;
    }

    private LogController controller;

  }

  private static class RefKey extends Key {

    public RefKey(String name) {
      this.name = name;
    }

    public RefKey(String name,
                  Attr   attr) {
      super(attr);

      this.name = name;
    }

    public String getName() {
      return name;
    }

    private String name;

  }

  private static class LogRefKey extends RefKey {

    public LogRefKey(String name) {
      super(name);
    }

    public LogRefKey(String name,
                     Attr   attr) {
      super(name,
            attr);
    }

  }

  private static class FormatterRefKey extends RefKey {

    public FormatterRefKey(String name) {
      super(name);
    }

    public FormatterRefKey(String name,
                           Attr   attr) {
      super(name,
            attr);
    }

  }

  private static class Attr extends Element {

    public Attr(int attr) {
      this.attr = attr;
    }

    public int getAttr() {
      return attr;
    }

    private int attr;

  }

  private static class SeverityAttr extends Attr {

    public SeverityAttr(int attr) {
      super(attr);
    }

    public SeverityAttr(int           attr,
                        LogController relative) {
      super(attr);

      this.relative = relative;
    }

    public LogController getRelative() {
      return relative;
    }

    private LogController relative;

  }

  private static class Val extends Element {
  }

  private static class SeverityVal extends Val {

    public SeverityVal(int severity) {
      this.severity = severity;
    }

    public int getSeverity() {
      return severity;
    }

    private int severity;

  }

  private static class ObjectsVal extends Val {

    public boolean isAdding() {
      return isAdding;
    }

    public void setAdding(boolean isAdding) {
      this.isAdding = isAdding;
    }

    public Collection getObjects() {
      return objects;
    }

    public void addObject(ObjectVal object) {
      objects.add(object);
    }

    private boolean    isAdding = false;
    private LinkedList objects  = new LinkedList();

  }

  private static class ObjectVal extends Val {
  }

  private static class RefVal extends ObjectVal {

    public RefVal(String name) {
      this.name = name;
    }

    public String getName() {
      return name;
    }

    private String name;

  }

  private static class LogRefVal extends RefVal {

    public LogRefVal(String name) {
      super(name);
    }

  }

  private static class FormatterRefVal extends RefVal {

    public FormatterRefVal(String name) {
      super(name);
    }

  }

  private static class ClassVal extends ObjectVal {

    public ClassVal(int type) {
      this.type = type;
    }

    public int getType() {
      return type;
    }

    private int type;

  }

  private static class CustomClassVal extends ClassVal {

    public CustomClassVal(String ident) {
      super(TokenType.IDENT);

      this.ident = ident;
    }

    public String getIdent() {
      return ident;
    }

    private String ident;

  }

  private static class LogControllerEntry {

    public LogControllerEntry(LogController controller) {
      this.controller = controller;
    }

    public LogController getController() {
      return controller;
    }

    public SeverityVal getEffectiveSeverity() {
      return effSeverity;
    }

    public void setEffectiveSeverity(SeverityVal severity) {
      effSeverity = severity;
    }

    public Map getEffectiveRelationSeverities() {
      return effRelationSeverities;
    }

    public void setEffectiveRelationSeverity(LogController controller,
                                             SeverityVal   severity) {
      effRelationSeverities.put(controller,
                                severity);
    }

    public SeverityVal getMinimumSeverity() {
      return minSeverity;
    }

    public void setMinimumSeverity(SeverityVal severity) {
      minSeverity = severity;
    }

    public Map getMinimumRelationSeverities() {
      return minRelationSeverities;
    }

    public void setMinimumRelationSeverity(LogController controller,
                                           SeverityVal   severity) {
      minRelationSeverities.put(controller,
                                severity);
    }

    public SeverityVal getMaximumSeverity() {
      return maxSeverity;
    }

    public void setMaximumSeverity(SeverityVal severity) {
      maxSeverity = severity;
    }

    public Map getMaximumRelationSeverities() {
      return maxRelationSeverities;
    }

    public void setMaximumRelationSeverity(LogController controller,
                                           SeverityVal   severity) {
      maxRelationSeverities.put(controller,
                                severity);
    }

    public ObjectsVal getLogs() {
      return logs;
    }

    public void setLogs(ObjectsVal logs) {
      this.logs = logs;
    }

    public ObjectsVal getLocalLogs() {
      return localLogs;
    }

    public void setLocalLogs(ObjectsVal logs) {
      this.localLogs = logs;
    }

    public ObjectsVal getPrivateLogs() {
      return privateLogs;
    }

    public void setPrivateLogs(ObjectsVal logs) {
      this.privateLogs = logs;
    }

    public ObjectsVal getFilters() {
      return filters;
    }

    public void setFilters(ObjectsVal filters) {
      this.filters = filters;
    }

    public String getResourceBundleName() {
      return resourceBundleName;
    }

    public void setResourceBundleName(String resourceBundleName) {
      this.resourceBundleName = resourceBundleName;
    }

    private LogController controller;
    private SeverityVal   effSeverity,
                          minSeverity,
                          maxSeverity;
    private Map           effRelationSeverities = new HashMap(),
                          minRelationSeverities = new HashMap(),
                          maxRelationSeverities = new HashMap();
    private ObjectsVal    logs,
                          localLogs,
                          privateLogs,
                          filters;
    private String        resourceBundleName;

  }

  private static class LogEntry {

    public ObjectsVal getObject() {
      return object;
    }

    public void setObject(ObjectsVal object) {
      this.object = object;
    }

    public SeverityVal getEffectiveSeverity() {
      return effSeverity;
    }

    public void setEffectiveSeverity(SeverityVal severity) {
      effSeverity = severity;
    }

    public String getEncoding() {
      return enc;
    }

    public void setEncoding(String enc) {
      this.enc = enc;
    }

    //YUE: additional attr 'desc' for Log
    public String getDesc() {
      return desc;
    }

    public void setDesc(String desc) {
      this.desc = desc;
    }

    public ObjectsVal getFilters() {
      return filters;
    }

    public void setFilters(ObjectsVal objects) {
      filters = objects;
    }

    public ObjectsVal getFormatter() {
      return formatter;
    }

    public void setFormatter(ObjectsVal object) {
      formatter = object;
    }

    public String getPattern() {
      return pattern;
    }

    public void setPattern(String pattern) {
      this.pattern = pattern;
    }

    public int getLimit() {
      return limit;
    }

    public void setLimit(int limit) {
      if (limit > FileLog.NO_LIMIT) {
        this.limit = limit;
      }
    }

    public boolean isLimitSet() {
      return limitSet;
    }

    public int getCnt() {
      return cnt;
    }

    public void setCnt(int cnt) {
      if (cnt > FileLog.NO_CNT) {
        this.cnt = cnt;
      }
    }

    public boolean isCntSet() {
      return cntSet;
    }

    public Log getLog() {
      return log;
    }

    public void setLog(Log log) {
      this.log = log;
    }

    private ObjectsVal  object,
                        filters,
                        formatter;
    private SeverityVal effSeverity;
    private String      enc,
                        pattern,
                        desc;   //YUE: short description about the Log;
    private int         limit = FileLog.NO_LIMIT,
                        cnt   = FileLog.NO_CNT;
    private boolean     limitSet = false,
                        cntSet   = false;
    private Log         log;

  }

  private static class FormatterEntry {

    public ObjectsVal getObject() {
      return object;
    }

    public void setObject(ObjectsVal object) {
      this.object = object;
    }

    public String getPattern() {
      return pattern;
    }

    public void setPattern(String pattern) {
      this.pattern = pattern;
    }

    public Formatter getFormatter() {
      return formatter;
    }

    public void setFormatter(Formatter formatter) {
      this.formatter = formatter;
    }

    private ObjectsVal object;
    private String     pattern;
    private Formatter  formatter;

  }


  private Properties properties;
  private File       file;
  private long       lastConfigured = 0;

  private static final int NORMAL_LOGS  = 0,
                           LOCAL_LOGS   = NORMAL_LOGS + 1,
                           PRIVATE_LOGS = LOCAL_LOGS + 1,
                           NO_LOGS      = PRIVATE_LOGS + 1;

  private static Location classLoc = Location.getLocation(PropertiesConfigurator13.class);

}
