/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_eclipse/_core/src/com/sap/tc/jtools/jlint/eclipse/wizards/ResultImportWizard.java#3 $
 */

package com.sap.tc.jtools.jlint.eclipse.wizards;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.net.URL;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jdt.internal.ui.actions.WorkbenchRunnableAdapter;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IImportWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import com.sap.tc.jtools.jlint.eclipse.launch.EclipseResponse;
import com.sap.tc.jtools.jlint.eclipse.launch.JLinRunner;
import com.sap.tc.jtools.jlint.eclipse.launch.MetaMarkerManager;
import com.sap.tc.jtools.jlint.eclipse.util.ExceptionHandler;
import com.sap.tc.jtools.jlint.eclipse.util.ResponseTransformatorTool;
import com.sap.tc.jtools.jtci.PerformResponse;
import com.sap.tc.jtools.jtci.exceptions.BadTreeException;
import com.sap.tc.jtools.jtci.exceptions.NestableRuntimeException;
import com.sap.tc.jtools.jtci.interfaces.ResultInterface;
import com.sap.tc.jtools.util.xml.XMLParseException;
import com.sap.tc.jtools.util.xml.XMLTool;

/**
 * Description of this class
 * 
 * @author Jan Sievers
 * @version $Revision: #3 $ $Date: 2004/07/30 $
 */
public class ResultImportWizard extends Wizard implements IImportWizard {

  private IStructuredSelection sel;
  private ResultImportPage     mainPage = new ResultImportPage(
                                            TextKeyWizards.PAGE_NAME.getText(),
                                            TextKeyWizards.PAGE_TITLE.getText(),
                                            getImageDescriptor("wizban/import_wiz.gif"));//$NON-NLS-1$

  public ResultImportWizard() {
    super();
    mainPage.setDescription(TextKeyWizards.IMPORT_WIZ_DESCR.getText());
    setNeedsProgressMonitor(true);
    setDefaultPageImageDescriptor(getImageDescriptor("wizban/import_wiz.gif"));//$NON-NLS-1$
  }

  public boolean performFinish() {
    IWorkspaceRunnable op = new IWorkspaceRunnable() {

      public void run(IProgressMonitor monitor) throws CoreException,
          OperationCanceledException {
        try {
          finishPage(monitor);
        } catch (BadTreeException e) {
          throw new NestableRuntimeException(e);
        } catch (IOException e) {
          throw new NestableRuntimeException(e);
        } catch (XMLParseException e) {
          throw new NestableRuntimeException(e);
        }
      }
    };
    try {
      getContainer().run(false, false, new WorkbenchRunnableAdapter(op));
    } catch (InvocationTargetException e) {
      ExceptionHandler.handle(e);
      return false;
    } catch (InterruptedException e) {
      return false;
    }
    return true;
  }

  protected void finishPage(IProgressMonitor monitor) throws BadTreeException,
      IOException, XMLParseException, CoreException {
    String resFileName = mainPage.getFileName();
    if (null == resFileName) {
      throw new IllegalArgumentException("resultfile is null");//$NON-NLS-1$
    }
    File resultFile = new File(resFileName);
    if (!resultFile.isFile()) {
      throw new IllegalArgumentException("resultfile " + resFileName
          + " is not a file");
    }
    monitor.beginTask(TextKeyWizards.IMPORT_TASK_DESCR.getText(), 100);
    monitor.worked(1);
    try {
      if (mainPage.clearMarkers()) {
        deleteJLinMarkers(mainPage.getProject());
      }
      PerformResponse response = readResponse(resultFile);
      monitor.worked(9);
      EclipseResponse[] transformedResponses = ResponseTransformatorTool
          .transform(response, mainPage.getProject(), new SubProgressMonitor(
              monitor, 80));
      MetaMarkerManager markerManager = new MetaMarkerManager(
          new Object[]{mainPage.getProject()}, transformedResponses);
      markerManager.createMarkers();
      monitor.worked(10);
      JLinRunner.showTasksView();
      showFinishedMessage(response, markerManager);
    } finally {
      monitor.done();
    }
  }

  private void deleteJLinMarkers(IProject project) throws CoreException {
    project.deleteMarkers(MetaMarkerManager.JLIN_MARKER_ID, true,
        IResource.DEPTH_INFINITE);
  }

  private void showFinishedMessage(PerformResponse response,
      MetaMarkerManager manager) {
    int totalCreatedMsgCount = manager.getPrioHighCount()
        + manager.getPrioLowCount() + manager.getPrioNormalCount();
    ResultInterface[] results = response.getResults();
    int totalMsgCount = 0;
    for (int i = 0; i < results.length; i++) {
      int prio = results[i].getPriority();
      if (prio >= ResultInterface.SEVERITY_ERROR
          && prio <= ResultInterface.SEVERITY_INFO) {
        totalMsgCount++;
      }
    }
    int notFound = totalMsgCount - totalCreatedMsgCount;
    boolean warnNotFound = notFound > 0;
    final String msg = TextKeyWizards.IMPORT_FINISH_MSG.getFormattedText(
    		new String[]{String.valueOf(totalCreatedMsgCount), String.valueOf(notFound)});
    final String title = TextKeyWizards.IMPORT_FINISH_TITLE.getText();
    final Display dsp = Display.getDefault();
    if (dsp != null) {
      Runnable runnable;
      if (warnNotFound) {
        runnable = new Runnable() {

          public void run() {
            MessageDialog.openWarning(dsp.getActiveShell(), title, msg);
          }
        };
      } else {
        runnable = new Runnable() {

          public void run() {
            MessageDialog.openInformation(dsp.getActiveShell(), title, msg);
          }
        };
      }
      dsp.asyncExec(runnable);
    }
  }

  private PerformResponse readResponse(File resultFile)
      throws BadTreeException, IOException, XMLParseException {
    Reader fileReader = null;
    PerformResponse response = null;
    try {
      fileReader = new BufferedReader(new FileReader(resultFile));
      response = new PerformResponse(XMLTool.parseReader(fileReader));
    } finally {
      try {
        if (fileReader != null) {
          fileReader.close();
        }
      } catch (IOException e) {
        ExceptionHandler.handle(e);
      }
    }
    return response;
  }

  public void addPages() {
    super.addPages();
    mainPage.setWizard(this);
    mainPage.setSelection(sel);
    addPage(mainPage);
  }

  /**
   * @see org.eclipse.ui.IWorkbenchWizard#init(IWorkbench,
   *      IStructuredSelection)
   */
  public void init(IWorkbench workbench, IStructuredSelection selection) {
    sel = selection;
  }

  /**
   * Returns the image descriptor with the given relative path.
   */
  private ImageDescriptor getImageDescriptor(String relativePath) {
    String iconPath = "icons/full/"; //$NON-NLS-1$  
    try {
      AbstractUIPlugin plugin = (AbstractUIPlugin) Platform
          .getPlugin(PlatformUI.PLUGIN_ID);
      URL installURL = plugin.getDescriptor().getInstallURL();
      URL url = new URL(installURL, iconPath + relativePath);
      return ImageDescriptor.createFromURL(url);
    } catch (MalformedURLException e) {
      // Should not happen
      return null;
    }
  }
}
