/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_eclipse/_core/src/com/sap/tc/jtools/jlint/eclipse/wizards/ResultImportPage.java#3 $
 */

package com.sap.tc.jtools.jlint.eclipse.wizards;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.debug.internal.ui.SWTUtil;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import com.sap.tc.jtools.jlint.eclipse.JlinPlugin;
import com.sap.tc.jtools.jtci.PerformResponse;
import com.sap.tc.jtools.jtci.exceptions.BadTreeException;
import com.sap.tc.jtools.util.xml.XMLParseException;
import com.sap.tc.jtools.util.xml.XMLTool;

/**
 * Description of this class
 * 
 * @author Jan Sievers
 * @version $Revision: #3 $ $Date: 2004/07/30 $
 */
public class ResultImportPage extends WizardPage {

  Text               fImportText;
  Button             fBrowseButton;
  Combo              fProjectCombo;
  Button             fProjectButton;
  Button             fClearProblemsButton;
  private IProject[] allProjs;
  private String     defaultProjName;
  private String     defaultFileName;
  private boolean    isProjectValid = false;
  private boolean    isResultFileValid = false;

  /* (non-Javadoc)
   * @see org.eclipse.jface.wizard.IWizardPage#isPageComplete()
   */
  public boolean isPageComplete() {
    return isProjectValid && isResultFileValid;
  }
  /**
   * Constructor for ImportResultPage.
   * 
   * @param pageName
   */
  public ResultImportPage(String pageName) {
    super(pageName);
    setPageComplete(false);
    allProjs = getAccessibleProjects();
  }

  private IProject[] getAccessibleProjects() {
    IProject[] projects = ResourcesPlugin.getWorkspace().getRoot()
        .getProjects();
    List accessibleProjs = new ArrayList();
    for (int i = 0; i < projects.length; i++) {
      if (projects[i].isAccessible()) {
        accessibleProjs.add(projects[i]);
      }
    }
    return (IProject[]) accessibleProjs.toArray(new IProject[0]);
  }

  /**
   * Constructor for ImportResultPage.
   * 
   * @param pageName
   * @param title
   * @param titleImage
   */
  public ResultImportPage(String pageName, String title,
      ImageDescriptor titleImage) {
    super(pageName, title, titleImage);
    setPageComplete(false);
    allProjs = getAccessibleProjects();
  }

  /**
   * @see org.eclipse.jface.dialogs.IDialogPage#createControl(Composite)
   */
  public void createControl(Composite parent) {
    Composite base = new Composite(parent, SWT.NONE);
    {
      GridLayout topLayout = new GridLayout();
      topLayout.numColumns = 2;
      base.setLayout(topLayout);
    }
    // "import result from file" selection
    {
      Label importFileLabel = new Label(base, SWT.NONE);
      importFileLabel.setText(TextKeyWizards.IMPORT_LABEL.getText());
      GridData gd = new GridData();
      gd.horizontalSpan = 2;
      importFileLabel.setLayoutData(gd);
    }
    fImportText = new Text(base, SWT.NONE | SWT.BORDER);
    fImportText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
    fImportText.setToolTipText(TextKeyWizards.IMPORT_TOOLTIP.getText());
    if (defaultFileName != null) {
      fImportText.setText(defaultFileName);
    }
    fBrowseButton = new Button(base, SWT.NONE);
    fBrowseButton.setText(TextKeyWizards.BROWSE_LABEL.getText());
    setButtonGridData(fBrowseButton);
    // project selection
    {
      Label projLabel = new Label(base, SWT.NONE);
      projLabel.setText(TextKeyWizards.PROJECT_LABEL.getText());
      GridData gd = new GridData();
      gd.horizontalSpan = 2;
      projLabel.setLayoutData(gd);
    }
    fProjectCombo = new Combo(base, SWT.DROP_DOWN | SWT.READ_ONLY);
    for (int i = 0; i < allProjs.length; i++) {
      String projName = allProjs[i].getName();
      fProjectCombo.add(projName);
      if (projName.equals(defaultProjName)) {
        fProjectCombo.select(i);
      }
    }
    fProjectCombo.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
    fProjectCombo.setToolTipText(TextKeyWizards.PROJECT_TOOLTIP.getText());
    // clear markers checkbox
    {
      fClearProblemsButton = new Button(base, SWT.CHECK);
      fClearProblemsButton
          .setText(TextKeyWizards.CLEAR_ALL_PROBLEMS_LABEL.getText());
      fClearProblemsButton.setSelection(true);
      GridData gd = new GridData();
      gd.horizontalSpan = 2;
      fClearProblemsButton.setLayoutData(gd);
    }
    setControl(base);
    {
      SelectionListener sl = new SelectionAdapter() {

        /**
         * @see org.eclipse.swt.events.SelectionListener#widgetSelected(SelectionEvent)
         */
        public void widgetSelected(SelectionEvent e) {
          FileDialog fd = new FileDialog(JlinPlugin.getActiveWorkbenchShell());
          String initialFile = fImportText.getText();
          if (initialFile != null) {
          	// initial filename must be OS-specific
            fd.setFileName(initialFile.replace('\\', File.separatorChar).replace('/', File.separatorChar));
          }
          fd.setFilterExtensions(new String[]{"*.xml"}); //$NON-NLS-1$
          fd.setText(TextKeyWizards.IMPORT_LABEL.getText());
          String fileName = fd.open();
          if (fileName != null) {
            fImportText.setText(fileName);
          }
        }
      };
      fBrowseButton.addSelectionListener(sl);
    }
    fImportText.addModifyListener(new ModifyListener() {

      public void modifyText(ModifyEvent evt) {
        fImportText.update();
        updateResultFileValid();
        updateProjectValid();
        updateErrorMessage();
        getContainer().updateButtons();
      }
    });
    fProjectCombo.addModifyListener(new ModifyListener() {

      public void modifyText(ModifyEvent evt) {
        fProjectCombo.update();
        updateProjectValid();
        updateErrorMessage();
        getContainer().updateButtons();
      }
    });
    updateProjectValid();
    updateResultFileValid();
    updateErrorMessage();
  }

  public String getFileName() {
    return fImportText.getText();
  }

  public IProject getProject() {
    return allProjs[fProjectCombo.getSelectionIndex()];
  }

  public boolean clearMarkers() {
    return fClearProblemsButton.getSelection();
  }

  public void setSelection(IStructuredSelection sel) {
    Object firstSelectedObject = sel.getFirstElement(); 
    if (firstSelectedObject instanceof IResource) {
      IResource resource = (IResource) firstSelectedObject;
      defaultProjName = resource.getProject().getName();
      IPath path = resource.getLocation();
      if (path != null) {
        defaultFileName = path.toOSString();
      }
    }
  }

  private void updateProjectValid() {
    if (fProjectCombo.getSelectionIndex() < 0) {
      setErrorMessage(TextKeyWizards.MISSING_PROJECT_MSG.getText());
      isProjectValid = false;
    } else {
      isProjectValid = true;
    }
  }
    
  private void updateResultFileValid() {
     String fileName = fImportText.getText();
     if (fileName == null) {
       setErrorMessage(TextKeyWizards.MISSING_FILE_MSG.getText());
       isResultFileValid = false;
       return;
     }
     File resultsFile = new File(fileName);
     if (!resultsFile.isFile()) {
       setErrorMessage(TextKeyWizards.NOT_EXISTING_FILE_MSG.getText());
       isResultFileValid = false;
       return;
     }
     Reader fileReader = null;
     PerformResponse response = null;
     try {
        fileReader = new BufferedReader(new FileReader(resultsFile));
        response = new PerformResponse(XMLTool.parseReader(fileReader));
        // parser for response is quite forgiving, takes any valid XML file...
        if (response.getName() == null && response.getResults().length == 0) {
          setErrorMessage(TextKeyWizards.INVALID_RESULT_FILE_MSG.getText());
          isResultFileValid = false;
          return;
        }
    } catch (BadTreeException e) {
      setErrorMessage(TextKeyWizards.INVALID_RESULT_FILE_MSG.getText());
      isResultFileValid = false;
      return;
    } catch (IOException e) {
      setErrorMessage(TextKeyWizards.INVALID_RESULT_FILE_MSG.getText());
      isResultFileValid = false;
      return;
    } catch (XMLParseException e) {
      setErrorMessage(TextKeyWizards.INVALID_RESULT_FILE_MSG.getText());
      isResultFileValid = false;
      return;
    } finally {
       if (fileReader != null) {
         try {
          fileReader.close();
        } catch (IOException e1) {
          setErrorMessage(TextKeyWizards.INVALID_RESULT_FILE_MSG.getText());
          isResultFileValid = false;
          return;
        }
     }
    }
    isResultFileValid = true;
  }
  
  private void updateErrorMessage() {
    if (isResultFileValid && isProjectValid) {
      setErrorMessage(null);
    }
  }

  private void setButtonGridData(Button button) {
    GridData data = new GridData();
    data.horizontalAlignment = GridData.FILL;
    button.setLayoutData(data);
    // set default button size
    SWTUtil.setButtonDimensionHint(button);
  }

}
