/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_eclipse/_core/src/com/sap/tc/jtools/jlint/eclipse/launch/MetaRequestManager.java#4 $
 */

package com.sap.tc.jtools.jlint.eclipse.launch;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

import com.sap.tc.jtools.jlint.eclipse.ExtensionManager;
import com.sap.tc.jtools.jlint.eclipse.extension.EclipseModule;
import com.sap.tc.jtools.jlint.eclipse.extension.IRequestCreator;
import com.sap.tc.jtools.jlint.eclipse.extension.ISelectionHandler;
import com.sap.tc.jtools.jlint.extension.Module;
import com.sap.tc.jtools.jtci.Test;
import com.sap.tc.jtools.jtci.TestObjectPosition;
import com.sap.tc.jtools.jtci.interfaces.ResultInterface;

/**
 * @author D037913
 */
public class MetaRequestManager {

	private ExtensionManager extensionManager = ExtensionManager
			.getExtensionManager();
	private List requests = null;
	private List requestCreators = new ArrayList();
	private Object[] selectedObjects;
	private Test[] tests;

	private Map data2handler = new HashMap();

	public MetaRequestManager(Test[] tests, Object[] selectedObjects) {
		init(tests, selectedObjects);
	}

	/**
	 * @param tests
	 * @param selectedObjects
	 */
	private void init(Test[] tests, Object[] selectedObjects) {
		this.tests = tests;
		ISelectionHandler[] handlers = extensionManager.getSelectionHandlers();
		// sort objects by selectionHandler
		Map handler2ObjectMap = new HashMap();
		List noHandlerObjects = new ArrayList();
		for (int i = 0; i < selectedObjects.length; i++) {
			boolean handlerFound = false;
			for (int j = 0; j < handlers.length; j++) {
				if (handlers[j].canHandleObject(selectedObjects[i])) {
					handlerFound = true;
					Set objects = (Set) handler2ObjectMap.get(handlers[j]);
					if (objects == null) {
						objects = new HashSet();
						handler2ObjectMap.put(handlers[j], objects);
					}
					objects.add(selectedObjects[i]);
					break;
				}
			}
			if (!handlerFound) {
				noHandlerObjects.add(selectedObjects[i]);
			}
		}
		this.selectedObjects = selectedObjects;
		Module[] mods = extensionManager.getTestProvider().getModules();
		for (int i = 0; i < mods.length; i++) {
			EclipseModule currentModule = (EclipseModule) mods[i];
			IRequestCreator reqCr = currentModule.getRequestCreator();
			reqCr.reset();
			requestCreators.add(reqCr);
			Test[] moduleTests = pickTestsFromModule(tests, currentModule);
			if (moduleTests.length == 0)
				continue;
			// set to defaults
			Set handlerKeys = handler2ObjectMap.keySet();
			for (Iterator iter = handlerKeys.iterator(); iter.hasNext();) {
				ISelectionHandler handler = (ISelectionHandler) iter.next();
				Set objs = (Set) handler2ObjectMap.get(handler);
				handler.setSelection(objs.toArray());
				Object[] data = handler.getData();
				for (int j = 0; j < data.length; j++) {
					data2handler.put(data[j], handler);
				}
				reqCr.addData(data);
				reqCr.addMetaData(handler.getMetaData());
			}
			// objects without selectionHandler
			reqCr.addData(noHandlerObjects.toArray());
		}
	}

	/**
	 * @param tests
	 * @param moduleId
	 * @return
	 */
	private Test[] pickTestsFromModule(Test[] tests, EclipseModule module) {
		List testList = new ArrayList();
		for (int i = 0; i < tests.length; i++) {
			if (module.hasTest(tests[i].getName()))
				testList.add(tests[i]);
		}
		return (Test[]) testList.toArray(new Test[0]);
	}

	public EclipseRequest[] getRequests() {
		if (requests == null) {
			fillRequests();
		}
		return (EclipseRequest[]) requests.toArray(new EclipseRequest[0]);
	}

	public Object[] getSelectedObjects() {
		return selectedObjects;
	}

	public int testObjectCount() {
		if (requests == null) {
			fillRequests();
		}
		int no = 0;
		for (Iterator iter = requests.iterator(); iter.hasNext();) {
			EclipseRequest req = (EclipseRequest) iter.next();
			no += req.testObjectCount();
		}
		return no;
	}

	public void persistRequests(File dir) {
		if (!dir.isDirectory()) {
			throw new IllegalArgumentException(dir.getAbsolutePath()
					+ " is not a directory");
		}
		if (requests == null) {
			fillRequests();
		}
		int i = 0;
		for (Iterator iter = requests.iterator(); iter.hasNext(); i++) {
			EclipseRequest request = (EclipseRequest) iter.next();
			String projectName = request.getProject().getName();
			request.saveTo(new File(dir, "request_" + projectName + "_"
					+ String.valueOf(i) + ".xml"));
		}
	}

	/**
	 * shows warning message dialog box if requestContainer contains no sources
	 */
	public static void warnNoSources() {
		final Display dsp = Display.getDefault();
		if (dsp != null) {
			Runnable runnable = new Runnable() {
				public void run() {
					MessageDialog.openWarning(new Shell(dsp),
							TextKeyLaunch.NO_SOURCES_TITLE.getText(),
							TextKeyLaunch.NO_SOURCES_MSG.getText());
				}
			};
			dsp.asyncExec(runnable);
		}
	}

	/**
	 * shows warning message dialog box in case project(s) are out of sync with
	 * the filesystem
	 */
	public static void warnProjectsOutOfSync(String[] projectNames) {
		final Display dsp = Display.getDefault();
		StringBuffer buf = new StringBuffer();
		for (int i = 0; i < projectNames.length; i++) {
			if (i > 0) {
				buf.append(", ");//$NON-NLS-1$
			}
			buf.append(projectNames[i]);
		}
		final String projectList = buf.toString();
		if (dsp != null) {
			Runnable runnable = new Runnable() {
				public void run() {
					MessageDialog.openWarning(new Shell(dsp),
							TextKeyLaunch.PROJECTS_OUT_OF_SYNC_TITLE.getText(),
							TextKeyLaunch.PROJECTS_OUT_OF_SYNC_MSG
									.getFormattedText(projectList));
				}
			};
			dsp.asyncExec(runnable);
		}
	}

	public Object getObject(ResultInterface result) {
		String toID = result.getTestObjectID();
		TestObjectPosition top = result.getResultPosition();
		Object data = null;
		for (Iterator iter = requestCreators.iterator(); iter.hasNext();) {
			IRequestCreator rc = (IRequestCreator) iter.next();
			data = rc.getData(toID, top);
			if (data != null) {
				break;
			}
		}

		ISelectionHandler handler = (ISelectionHandler) data2handler.get(data);
		if (handler == null) {
			return data;
		} else {
			return handler.getSelection(data);
		}
	}

	public String[] getOutOfSyncProjectNames() {
		Set outOfSyncProjects = new HashSet();
		for (Iterator iter = requestCreators.iterator(); iter.hasNext();) {
			IRequestCreator reqCr = (IRequestCreator) iter.next();
			IResource[] resources = reqCr.getResources();
			for (int i = 0; i < resources.length; i++) {
				outOfSyncProjects.add(resources[i].getProject());
			}
		}
		for (Iterator iter = outOfSyncProjects.iterator(); iter.hasNext();) {
			IProject project = (IProject) iter.next();
			if (project.isSynchronized(IResource.DEPTH_INFINITE)) {
				iter.remove();
			}
		}
		String[] outOfSyncNames = new String[outOfSyncProjects.size()];
		int i = 0;
		for (Iterator iter = outOfSyncProjects.iterator(); iter.hasNext(); i++) {
			outOfSyncNames[i] = ((IProject) iter.next()).getName();
		}
		return outOfSyncNames;
	}

	private void fillRequests() {
		this.requests = new ArrayList();
		Module[] mods = extensionManager.getTestProvider().getModules();
		for (int i = 0; i < mods.length; i++) {
			EclipseModule currentModule = (EclipseModule) mods[i];
			IRequestCreator reqCr = currentModule.getRequestCreator();
			Test[] moduleTests = pickTestsFromModule(tests, currentModule);
			if (moduleTests.length == 0)
				continue;
			EclipseRequest[] moduleRequests = reqCr.createRequests(moduleTests);
			// set module for each request belonging to this module
			for (int k = 0; k < moduleRequests.length; k++) {
				moduleRequests[k].setModule(currentModule);
			}
			requests.addAll(Arrays.asList(moduleRequests));
		}
	}

}