/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_eclipse/_core/src/com/sap/tc/jtools/jlint/eclipse/launch/MetaMarkerManager.java#3 $
 */

package com.sap.tc.jtools.jlint.eclipse.launch;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Display;

import com.sap.tc.jtools.jlint.Result;
import com.sap.tc.jtools.jlint.eclipse.ExtensionManager;
import com.sap.tc.jtools.jlint.eclipse.JlinPlugin;
import com.sap.tc.jtools.jlint.eclipse.extension.IMarkerManager;
import com.sap.tc.jtools.jlint.eclipse.extension.IMarkerPosition;
import com.sap.tc.jtools.jlint.eclipse.util.ExceptionHandler;
import com.sap.tc.jtools.jlint.eclipse.util.JLinSetupCache;
import com.sap.tc.jtools.jtci.Dispatcher;
import com.sap.tc.jtools.jtci.interfaces.ResultInterface;

/**
 * @author D037913
 *
 * To change the template for this generated type comment go to
 * Window>Preferences>Java>Code Generation>Code and Comments
 */
public class MetaMarkerManager implements IWorkspaceRunnable {

  public static final String JLIN_MARKER_ID =
    JlinPlugin.getPluginId() + ".jlin_marker";

  private ExtensionManager extensionManager =
    ExtensionManager.getExtensionManager();

  private int highCount = 0;
  private int normalCount = 0;
  private int lowCount = 0;

  private List markerList = new ArrayList();

  // marker attribute for jlin test name 
  public static final String ATTR_TEST_NAME = "jlin.test";

  private Object[] selectedObjects;
  private EclipseResponse[] responses;


  public int getPrioHighCount() {
    return highCount;
  }

  public int getPrioLowCount() {
    return lowCount;
  }

  public int getPrioNormalCount() {
    return normalCount;
  }
  
  public MetaMarkerManager(
    Object[] selectedObjects,
    EclipseResponse[] responses) {
    this.selectedObjects = selectedObjects;
    this.responses = responses;
  }

  private void clearJlinMarkers() {
    for (int i = 0; i < responses.length; i++) {
      IResource[] resources =
        responses[i]
          .getRequest()
          .getModule()
          .getRequestCreator()
          .getResources();
      clearMarkersFor(resources);
    }
  }

  private void clearMarkersFor(IResource[] resources) {
    for (int i = 0; i < resources.length; i++) {
      if (resources[i] == null
        || !resources[i].exists()
        || !resources[i].getProject().isOpen())
        continue;
      try {
        resources[i].deleteMarkers(
          JLIN_MARKER_ID,
          true,
          IResource.DEPTH_INFINITE);
        // also delete (metrics-) markers on project level
        resources[i].getProject().deleteMarkers(
          JLIN_MARKER_ID,
          true,
          IResource.DEPTH_ZERO);
      } catch (CoreException ce) {
        continue;
      }
    }
    markerList = new ArrayList();
  }

  private void createJlinMarkers() {
    for (int i = 0; i < responses.length; i++) {
      IMarkerManager markerMan =
        responses[i].getRequest().getModule().getMarkerManager();
      IProject correspondingProject = responses[i].getRequest().getProject();
      Dispatcher dispatcher = JLinSetupCache.getDispatcher();
      // compact and localize the results
      ResultInterface[] results =
          responses[i].getResponse().getResults();
      for (int j = 0; j < results.length; j++) {
        IMarkerPosition pos =
          markerMan.getMarkerPosition(
            results[j].getResultPosition(),
            correspondingProject);
        if (pos == null)
          continue;
        createMarker(pos, results[j]);
      }
    }
  }

  /**
   * @param jlinMarker
   */
  private void createMarker(IMarkerPosition pos, ResultInterface result) {
    // we don't display internal errors/messages
    int prio = result.getPriority();
    if (prio < Result.SEVERITY_ERROR || prio > Result.SEVERITY_INFO)
      return;
    IMarker marker = null;
    try {
      marker = pos.getResource().createMarker(JLIN_MARKER_ID);
      prio = mapMarkerPrio(result);
      updatePrioCount(prio);
      marker.setAttribute(IMarker.PRIORITY, prio);
      int line = pos.getLine();
      if (line > -1) {
        marker.setAttribute(IMarker.LINE_NUMBER, line);
      }
      marker.setAttribute(ATTR_TEST_NAME, result.getTestName());
      marker.setAttribute(IMarker.MESSAGE, createErrorMsg(result));
      markerList.add(marker);
    } catch (CoreException e) {
      ExceptionHandler.handle(e);
    }
  }

  /**
   * @param prio
   */
  private void updatePrioCount(int prio) {
    switch (prio) {
      case IMarker.PRIORITY_LOW :
        lowCount++;
        break;
      case IMarker.PRIORITY_NORMAL :
        normalCount++;
        break;
      case IMarker.PRIORITY_HIGH :
        highCount++;
        break;
    }
  }

  /**
   * JLin has arbitrary int priorities, Eclipse only 3:
   * this method maps 1->high, 2->normal, everything else -> low
   */
  private int mapMarkerPrio(ResultInterface result) {
    switch (result.getPriority()) {
      case Result.SEVERITY_ERROR :
        return IMarker.PRIORITY_HIGH;
      case Result.SEVERITY_WARNING :
        return IMarker.PRIORITY_NORMAL;
      default :
        return IMarker.PRIORITY_LOW;
    }
  }

  private String createErrorMsg(ResultInterface result) {
    StringBuffer msg = new StringBuffer();
    msg.append("JLin ");
    msg.append(result.getTestName());
    msg.append(": ");
    msg.append(result.getErrorText());
    return msg.toString();
  }

  public void createMarkers() {
    try {
      ResourcesPlugin.getWorkspace().run(this, null);
    } catch (CoreException e) {
      ExceptionHandler.handle(e);
    }
  }

  public void run(IProgressMonitor monitor) {
    clearJlinMarkers();
    createJlinMarkers();
  }

  public void showFinishDialog(int filesChecked) {
    int totalCount = lowCount + normalCount + highCount;
    final String text =
      TextKeyLaunch.FINISH_MSG.getFormattedText(
        new String[] {
          String.valueOf(filesChecked),
          String.valueOf(totalCount),
          String.valueOf(highCount),
          String.valueOf(normalCount),
          String.valueOf(lowCount)});
    final String title = TextKeyLaunch.FINISH_TITLE.getText();
    final String[] labels = new String[] { IDialogConstants.CLOSE_LABEL };
    final int type;
    if (totalCount == 0) {
      type = MessageDialog.INFORMATION;
    } else {
      type = MessageDialog.WARNING;
    }
    Runnable runnable = new Runnable() {
      public void run() {
        new MessageDialog(null, title, null, text, type, labels, 0).open();
      }
    };
    Display dsp = Display.getDefault();
    if (dsp != null) {
      dsp.syncExec(runnable);
    }
  }

  /**
   * @return
   */
  public IMarker[] getMarkers() {
    return (IMarker[]) markerList.toArray(new IMarker[markerList.size()]);
  }

}
