/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_eclipse/_core/src/com/sap/tc/jtools/jlint/eclipse/launch/LaunchTab.java#3 $
 */

package com.sap.tc.jtools.jlint.eclipse.launch;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaModel;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.ui.dialogs.StatusInfo;
import org.eclipse.jdt.internal.ui.wizards.TypedElementSelectionValidator;
import org.eclipse.jdt.internal.ui.wizards.TypedViewerFilter;
import org.eclipse.jdt.launching.IJavaLaunchConfigurationConstants;
import org.eclipse.jdt.ui.JavaElementLabelProvider;
import org.eclipse.jdt.ui.JavaElementSorter;
import org.eclipse.jdt.ui.StandardJavaElementContentProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;

import com.sap.tc.jtools.jlint.eclipse.JlinPlugin;
import com.sap.tc.jtools.jlint.eclipse.config.JLinPreferencePage;
import com.sap.tc.jtools.jlint.eclipse.config.TestSetTool;
import com.sap.tc.jtools.jtci.TestSet;

/**
 * @author d037913
 */
public class LaunchTab
  extends AbstractLaunchConfigTab
  implements IJavaLaunchConfigurationConstants {

  public static final String CONTAINER_ATTR = JlinPlugin.getPluginId() + ".container"; //$NON-NLS-1$
  public static final String ATTR_TESTSET_NAME = JlinPlugin.getPluginId() + ".testsetname"; //$NON-NLS-1$

  private final ILabelProvider fJavaElementLabelProvider =
    new JavaElementLabelProvider(
      JavaElementLabelProvider.SHOW_ROOT | JavaElementLabelProvider.SHOW_QUALIFIED);

  private Text fContainerText;
  private Button fContainerButton;
  private IJavaElement fContainerElement;

  private Text fVariantText;
  private Button fVariantButton;

  /**
   * @see org.eclipse.debug.ui.ILaunchConfigurationTab#createControl(Composite)
   */
  public void createControl(Composite parent) {
    Composite base = new Composite(parent, SWT.NONE);

    {
      GridLayout topLayout = new GridLayout();
      topLayout.numColumns = 2;
      base.setLayout(topLayout);
    }

    // container selection
    {
      Label projLabel = new Label(base, SWT.NONE);
      projLabel.setText(TextKeyLaunch.CONTAINER_LABEL.getText());
      GridData gd = new GridData();
      gd.horizontalSpan = 2;
      projLabel.setLayoutData(gd);
    }

    fContainerText = new Text(base, SWT.READ_ONLY | SWT.BORDER);
    fContainerText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
    fContainerText.setToolTipText(TextKeyLaunch.CONTAINER_TOOLTIP.getText());
    fContainerButton = new Button(base, SWT.NONE);
    fContainerButton.setText(TextKeyLaunch.BROWSE_LABEL.getText());

    // variant selection

    {
      Label variantLabel = new Label(base, SWT.NONE);
      variantLabel.setText(TextKeyLaunch.VARIANT_LABEL.getText());
      GridData gd = new GridData();
      gd.horizontalSpan = 2;
      variantLabel.setLayoutData(gd);
    }

    fVariantText = new Text(base, SWT.NONE | SWT.BORDER);
    fVariantText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
    fVariantText.setToolTipText(TextKeyLaunch.VARIANT_TOOLTIP.getText());
    fVariantButton = new Button(base, SWT.NONE);
    fVariantButton.setText(TextKeyLaunch.BROWSE_LABEL.getText());

    fContainerButton.addSelectionListener(new SelectionAdapter() {
      public void widgetSelected(SelectionEvent e) {
        handleContainerSearchButtonSelected();
      }
    });
    {
      SelectionListener sl = new SelectionAdapter() {
        /**
         * @see org.eclipse.swt.events.SelectionListener#widgetSelected(SelectionEvent)
         */
        public void widgetSelected(SelectionEvent e) {
          VariantSelection vs = new VariantSelection();
          vs.show(JlinPlugin.getActiveWorkbenchShell());
          String variantName = vs.getResult();
          if (variantName != null) {
            fVariantText.setText(variantName);
          }
        }
      };
      fVariantButton.addSelectionListener(sl);
    }

    fContainerText.addModifyListener(new ModifyListener() {
      public void modifyText(ModifyEvent evt) {
        updateLaunchConfigurationDialog();
      }
    });

    fVariantText.addModifyListener(new ModifyListener() {
      public void modifyText(ModifyEvent evt) {
        updateLaunchConfigurationDialog();
      }
    });
    setControl(base);
  }

  /**
   * @see org.eclipse.debug.ui.ILaunchConfigurationTab#setDefaults(ILaunchConfigurationWorkingCopy)
   */
  public void setDefaults(ILaunchConfigurationWorkingCopy pConfig) {
    IJavaElement je = getContext();
    if (je != null && je.exists()) {
      fContainerElement = je.getJavaProject();
      pConfig.setAttribute(CONTAINER_ATTR, fContainerElement.getHandleIdentifier());
    } else {
      pConfig.setAttribute(CONTAINER_ATTR, ""); //$NON-NLS-1$
    }
    pConfig.setAttribute(ATTR_DEFAULT_CLASSPATH, true);
    pConfig.setAttribute(ATTR_TESTSET_NAME, JLinPreferencePage.getDefaultTestSetName());
  }

  /**
   * @see org.eclipse.debug.ui.ILaunchConfigurationTab#initializeFrom(ILaunchConfiguration)
   */
  public void initializeFrom(ILaunchConfiguration configuration) {

    // container
    {
      String containerHandle;
      try {
        containerHandle = configuration.getAttribute(CONTAINER_ATTR, ""); //$NON-NLS-1$
      } catch (CoreException e) {
        containerHandle = null; //$NON-NLS-1$
      }
      // JavaCore.create("") doesnt return null, so nullify containerHandle
      if ("".equals(containerHandle))
        containerHandle = null;
      fContainerElement = JavaCore.create(containerHandle);
      String container;
      if (fContainerElement != null) {
        container = getPresentationName(fContainerElement);
      } else {
        container = "";
      }
      fContainerText.setText(container);
    }
    // variant
    {
      String vn = getTestSetName(configuration);
      if (null != vn) {
        fVariantText.setText(vn);
      } else {
        fVariantText.setText(JLinPreferencePage.getDefaultTestSetName());
      }
    }
  }

  /**
   * @see org.eclipse.debug.ui.ILaunchConfigurationTab#performApply(ILaunchConfigurationWorkingCopy)
   */
  public void performApply(ILaunchConfigurationWorkingCopy pConfig) {
    if (fContainerElement != null) {
      pConfig.setAttribute(CONTAINER_ATTR, fContainerElement.getHandleIdentifier());
    }
    pConfig.setAttribute(ATTR_TESTSET_NAME, fVariantText.getText());
  }

  /**
   * @see org.eclipse.debug.ui.ILaunchConfigurationTab#getName()
   */
  public String getName() {
    return TextKeyLaunch.LAUNCH_TAB_TITLE.getText();
  }

  public boolean isValid(ILaunchConfiguration pConfig) {
    setErrorMessage(null);
    setMessage(null);
    TestSet testSet = null;

    // check variant
    {
      String name = fVariantText.getText().trim();
      if (0 < name.length()) {
        try {
          testSet = TestSetTool.readTestSet(name);
        } catch (Exception e) {
          setErrorMessage(TextKeyLaunch.CONFIG_BAD_VARIANT.getText());
          return false;
        }
      } else {
        setErrorMessage(TextKeyLaunch.CONFIG_INVALID_VARIANT.getText());
        return false;
      }
    }
    
    // check container
    {
      if (fContainerElement != null) {
        if (!fContainerElement.exists()) {
          setErrorMessage(TextKeyLaunch.CONFIG_INVALID_PROJECT.getText());
          return false;
        }
        try {
          IProject project = (IProject)fContainerElement.getJavaProject().getCorrespondingResource();
          if (!project.isSynchronized(IResource.DEPTH_INFINITE)) {
            setErrorMessage(TextKeyLaunch.PROJECT_OUT_OF_SYNC_MSG.getFormattedText(project.getName()));
            return false;
          }
        } catch (JavaModelException e) {
          setErrorMessage(e.getMessage());
          return false;
        }
        if (testSet != null) {
          if (new MetaRequestManager(testSet.getTests(),
            new IJavaElement[] { fContainerElement })
            .testObjectCount()
            == 0) {
            setErrorMessage(TextKeyLaunch.CONFIG_NO_SOURCES.getText());
            return false;
          }
        }
      } else {
        setErrorMessage(TextKeyLaunch.CONFIG_INVALID_PROJECT.getText());
        return false;
      }
    }
    return true;
  }

  private void handleContainerSearchButtonSelected() {
    IJavaElement javaElement = chooseContainer(fContainerElement);
    if (javaElement != null) {
      fContainerElement = javaElement;
      fContainerText.setText(getPresentationName(javaElement));
    }
  }

  private String getPresentationName(IJavaElement element) {
    return fJavaElementLabelProvider.getText(element);
  }

  private IJavaElement chooseContainer(IJavaElement initElement) {
    Class[] acceptedClasses =
      new Class[] {
        IPackageFragmentRoot.class,
        IJavaProject.class,
        IPackageFragment.class,
        ICompilationUnit.class };
    TypedElementSelectionValidator validator =
      new TypedElementSelectionValidator(acceptedClasses, false) {

      private IStatus STATUS_OK = new StatusInfo();
      private IStatus STATUS_ERROR =
        new StatusInfo(IStatus.ERROR, TextKeyLaunch.SELECTION_BINARY_MSG.getText());

      public IStatus validate(Object[] selection) {
        for (int i = 0; i < selection.length; i++) {
          if (selection[i] instanceof IJavaElement) {
            IPackageFragmentRoot pfr =
              (IPackageFragmentRoot) ((IJavaElement) selection[i]).getAncestor(
                IJavaElement.PACKAGE_FRAGMENT_ROOT);
            if (pfr != null) {
              try {
                if (pfr.getKind() != IPackageFragmentRoot.K_SOURCE) {
                  return STATUS_ERROR;
                }
              } catch (JavaModelException e) {
                return STATUS_ERROR;
              }
            }
          }
        }
        return STATUS_OK;
      }
    };

    acceptedClasses =
      new Class[] {
        IJavaModel.class,
        IPackageFragmentRoot.class,
        IJavaProject.class,
        IPackageFragment.class,
        ICompilationUnit.class };
    ViewerFilter filter = new TypedViewerFilter(acceptedClasses) {
      public boolean select(Viewer viewer, Object parent, Object element) {
        return super.select(viewer, parent, element);
      }
    };

    StandardJavaElementContentProvider provider =
      new StandardJavaElementContentProvider();
    ILabelProvider labelProvider =
      new JavaElementLabelProvider(JavaElementLabelProvider.SHOW_DEFAULT);
    ElementTreeSelectionDialog dialog =
      new ElementTreeSelectionDialog(getShell(), labelProvider, provider);
    dialog.setValidator(validator);
    dialog.setSorter(new JavaElementSorter());
    dialog.setTitle(TextKeyLaunch.CHOOSE_CONTAINER_TITLE.getText());
    dialog.setMessage(TextKeyLaunch.CHOOSE_CONTAINER_LABEL.getText());
    dialog.addFilter(filter);
    dialog.setInput(JavaCore.create(getWorkspaceRoot()));
    dialog.setInitialSelection(initElement);
    dialog.setAllowMultiple(false);

    if (dialog.open() == ElementTreeSelectionDialog.OK) {
      Object element = dialog.getFirstResult();
      return (IJavaElement) element;
    }
    return null;
  }

  /**
   * Convenience method to get the workspace root.
   */
  private IWorkspaceRoot getWorkspaceRoot() {
    return ResourcesPlugin.getWorkspace().getRoot();
  }

  public static String getTestSetName(ILaunchConfiguration configuration) {
    try {
      String testSetName = configuration.getAttribute(ATTR_TESTSET_NAME, ""); //$NON-NLS-1$
      if (0 == testSetName.length()) {
        return null;
      }
      String[] actualTestSets = JLinPreferencePage.getTestSets();
      for (int i = 0; i < actualTestSets.length; i++) {
        if (testSetName.equals(actualTestSets[i])) {
          return testSetName;
        }
      }
      return null;
    } catch (CoreException ce) {
      JlinPlugin.log(ce);
    }
    return null;
  }

}
